
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: desktop_entry.c 1873 2007-03-15 14:39:10Z mschwerin $
 *
 * See 
 *
 * http://standards.freedesktop.org/desktop-entry-spec/desktop-entry-spec-latest.html
 *
 * for details.
 */
#include "config.h"

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "codeset.h"
#include "desktop_entry.h"
#include "heap.h"
#include "logger.h"
#include "utils.h"

double
get_numeric_from_desktop_entry (const char *mrl, const char *group_name,
                                const char *entry_name)
{
    double value = 0;
    char *str_val = get_string_from_desktop_entry (mrl, group_name,
                                                   entry_name);
    if (str_val)
        value = atof (str_val);
    ho_free (str_val);

    return value;
}


int
get_boolean_from_desktop_entry (const char *mrl, const char *group_name,
                                const char *entry_name)
{
    int value = 0;
    char *str_val = get_string_from_desktop_entry (mrl, group_name,
                                                   entry_name);
    if (str_val && (strcmp (str_val, "true") == 0))
        value = 1;
    ho_free (str_val);

    return value;
}


char *
get_localestring_from_desktop_entry (const char *mrl, const char *group_name,
                                     const char *entry_name)
{
    char *value = NULL;
    char *str_val = get_string_from_desktop_entry (mrl, group_name,
                                                   entry_name);
    if (str_val) {
        char *codeset = get_system_encoding ();
        recode_t *xr = recode_init ("utf8", codeset);
        value = recode (xr, str_val);
        recode_done (xr);
        ho_free (codeset);
    }
    ho_free (str_val);

    return value;
}


char *
get_string_from_desktop_entry (const char *mrl, const char *group_name,
                               const char *entry_name)
{
    char *value = NULL;

    int correct_group = 0;
    FILE *f = fopen (mrl, "r");
    if (!f) {
        error (_("Could not open '%s': %s!"), mrl, strerror (errno));
        return NULL;
    }

    while (!feof (f)) {
        /* Read a line */
        char line[256];
        if (!fgets (line, 256, f))
            break;

        /* Trim off whitespace */
        trim_whitespace (line);

        /* It's a comment */
        if ((strlen (line) == 0) || (line[0] == '#'))
            continue;

        /* It's a group header */
        else if (line[0] == '[') {
            /* If we're already in the correct group (we're leaving it
             * now) we haven't found what we were looking for and exit. */
            if (correct_group)
                break;

            /* This is the group we are looking for. */
            if (strncmp (line + 1, group_name, strlen (group_name)) == 0)
                correct_group = 1;
        }

        /* It's an entry and we're in the correct group. */
        else if (correct_group) {
            char key[256];

            /* read key name */
            sscanf (line, "%[A-Za-z0-9-]", key);

            if (strncmp (key, entry_name, strlen (entry_name)) == 0) {
                /* read key value */
                char *val_s = index (line, '=');
                if (!val_s)
                    continue;
                value = ho_strdup (val_s + 1);
            }
        }
    }
    fclose (f);

    return value;
}
