/*
 * CV3D.h
 * $Id: CV3D.h,v 1.4 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Markus Janich, Michael Meissner, Rainer Jaeger
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */



#ifndef __CV3D_H
#define __CV3D_H



// System
///////////
#include <math.h>
#ifdef _MSC_VER
#if _MSC_VER >= 1300
#include <iostream>
#endif
#else
#include <iostream.h>
#endif


// Own
///////////
//#include "CP3D.h"

using namespace std;
// Forward declaration
////////////////////////
class CV4D;



/** This class implements a vector class for 3D coordinates
  * including functionality to vector calculations.
  *
  * @author Markus Janich, Michael Meissner, Rainer Jaeger
  */
class CV3D {
public:
  static double epsilon;

  /** Default constructor.
    * The default value of the instantiated vector will be (0.0,0.0,0.0). */
  CV3D(void) { m_ard[0] = 0.0;
               m_ard[1] = 0.0;
               m_ard[2] = 0.0; };

  /** Construct new vector.
    * The value of the vector will be (rdX, rdY, rdZ). */
  CV3D(double rdX, double rdY, double rdZ) { m_ard[0] = rdX;
                                             m_ard[1] = rdY;
                                             m_ard[2] = rdZ; };

  /** Copyconstructor.
    * Initializes the new vector with the vector passed in 'v'. */
  CV3D(const CV3D& Vector) { m_ard[0] = Vector.m_ard[0];
                             m_ard[1] = Vector.m_ard[1];
                             m_ard[2] = Vector.m_ard[2]; };

  /** Default destructor. */
  ~CV3D(void) {};



  //////////////////////////
  // OVERLOADED OPERATORS //
  //////////////////////////

  /** Cast operator to convert CV3D vectors to CV4D vectors.
    * Initializes the new vector with this vector.
    * The component 'w' is set to 1.0. */
  operator CV4D() const;

  /** Assign one vector to another.*/
  const CV3D& operator=(const CV3D&);

  /** Compares to vectors.
    * The result will be 'true' if the two vector are indentically
    * in each coefficient. Otherwise 'false' will be returned. */ 
  bool operator==(const CV3D&) const;

  /** Compares to vectors.
    * Same as above. Only the result is negated.
    * @see operator==() */
  bool operator!=(const CV3D&) const;

  /** Adds another vector to this vector. */
  CV3D& operator+=(const CV3D&);

  /** Subtracts another vector from this vector. */
  CV3D& operator-=(const CV3D&);

  /** Multiplies a vector with a scalar. */
  CV3D& operator*=(double);

  /** Multiplies a vector with a scalar. */
  CV3D& operator/=(double);

  /** Adds two vectors. The sum will be returned. */
  CV3D operator+(const CV3D&) const;

  /** Subtracts two vectors. The difference will be returned. */
  CV3D operator-(const CV3D&) const;

  /** Negates the vector. */
  CV3D operator-(void) const;

  /** Scalar multiplication of two vectors. The result will be returned. */
  double operator*(const CV3D&) const;

  /** Multiplication of a vector with a scalar. */
  CV3D operator*(double) const;

  /** Division of a vector with a scalar. */
  CV3D operator/(double) const;

  /** crossproduct. */
  CV3D operator|(const CV3D&) const;

  /** Returns the i-th coefficient or the vector.
    * The index goes from 0 to 3, so 0 stands for the
    * x-coordinate, 1 for the y-coordinate and so on. */
  double& operator[](int i) { return m_ard[i]; };

  /** Same as above but for constant vectors. */
  double operator[](int i) const { return m_ard[i]; };

  /** Multiplication of a scalar with a vector. */
  friend CV3D operator*(double, const CV3D&); 



  /////////////
  // METHODS //
  /////////////

  /** Returns the value of the minimal point component. */
  double getMinComponent(void) const    { return m_ard[getMinComponentCoord()]; };
 
  /** Returns the value of the minimal point component. */
  double getAbsMinComponent(void) const { return m_ard[getAbsMinComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getMaxComponent(void) const    { return m_ard[getMaxComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getAbsMaxComponent(void) const { return m_ard[getAbsMaxComponentCoord()]; };
 
  /** Returns the coordinate index of the minial point component. */
  int getMinComponentCoord(void) const;

  /** Returns the coordinate index of the minial point component (using fabs). */
  int getAbsMinComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component. */
  int getMaxComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component (using fabs). */
  int getAbsMaxComponentCoord(void) const;

  /** Returns the x-coordinate of the vector. */
  double getX(void) const { return m_ard[0]; };
  
  /** Returns the y-coordinate of the vector. */
  double getY(void) const { return m_ard[1]; };
  
  /** Returns the z-coordinate of the vector. */
  double getZ(void) const { return m_ard[2]; };

  /** Sets the x-coordinate of the vector to 'x'. */
  void setX(double rdX)   { m_ard[0] = rdX; };

  /** Sets the y-coordinate of the vector to 'y'. */
  void setY(double rdY)   { m_ard[1] = rdY; };

  /** Sets the z-coordinate of the vector to 'z'. */
  void setZ(double rdZ)   { m_ard[2] = rdZ; };

  /** Set the value of the vector.
    * The value of the vector will be (rdX, rdY, rdZ, rdW). */
  void setCoord(double rdX, double rdY, double rdZ) { m_ard[0] = rdX; 
                                                      m_ard[1] = rdY; 
						      m_ard[2] = rdZ; 
                                                      return; };

  /** Returns the euclidian norm of the vector. */
  double getNorm(void) const { return sqrt(m_ard[0]*m_ard[0] + m_ard[1]*m_ard[1] + m_ard[2]*m_ard[2]); };

  /** Normalizes the vector. */
  void normalize(void);

  /** Returns the normalized vector. */
  CV3D getNormalized(void) const;

  /** Prints a vector to the standard output. */ 
  void print(void) const;
  
  /** Same as above. But more useful for streams. */
  friend ostream& operator<<(ostream&, const CV3D&); 

  /** Reads a vector from the given stream. */
  friend istream& operator>>(istream&, CV3D&); 


protected:
  double m_ard[3];

};



// Function   : operator=
// Parameters : const CP3D& p1
// Purpose    : assign another point to this point
// Comments   : 
inline const CV3D& CV3D::operator=(const CV3D& v)
/*******************************************************************/
{
  m_ard[0] = v[0];
  m_ard[1] = v[1];
  m_ard[2] = v[2];
  return *this;
}

#endif // _CV3D_H_
