/* Copyright (C) 2004 Nikos Chantziaras.
 *
 * This file is part of the QTads program.  This program is free
 * software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Qt-specific Tads OS functions.
 *
 * This file *must not* contain C++ code, since it gets included from
 * the portable Tads C code.
 */

#ifndef OSQT_H
#define OSQT_H

#include "config.h"


/* Most systems have typedefs for ushort, uint and ulong.  If not, the
 * qtads.pro project file should be modified to define
 * OS_NO_TYPES_DEFINED at compile time.  On MacOS X, this happens
 * automatically.
 */
#ifndef OS_NO_TYPES_DEFINED
#define OS_USHORT_DEFINED
#define OS_UINT_DEFINED
#define OS_ULONG_DEFINED
#endif


#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

/* Qt doesn't run on the Atari 2600. */
#include "osbigmem.h"

/* Provide some non-standard functions (memicmp(), etc). */
#include "missing.h"

#define TADS_OEM_NAME "Nikos Chantziaras <realnc@users.sourceforge.net>"

/* Non-ANSI compilers are ancient history. */
#define OSANSI

/* ??? */
#define OS_LOADDS

/* Qt doesn't run on systems that have near/far pointers, so we make
 * this a do-nothing macro. */
#define osfar_t

/* Should work on every ANSI-Compiler. */
#define DISCARD (void)

/* ANSI C allows the assignment operator to be used with structs. */
#define OSCPYSTRUCT(x,y) ((x)=(y))

/* We provide our own line-wrapping and "more" prompt handling. */
#define USE_OS_LINEWRAP


/* --------------------------------------------------------------------
 */

/* Link error messages into the application. */
#define ERR_LINK_MESSAGES


/* --------------------------------------------------------------------
 * Platform Identifiers.
 */

#define OS_SYSTEM_NAME "Qt"  /* System identifier. */
#define OS_SYSTEM_LDESC "Trolltech Qt" /* System descriptive name. */


/* --------------------------------------------------------------------
 * Program Exit Codes.
 */

#define OSEXSUCC 0 /* Successful completion. */
#define OSEXFAIL 1 /* Failure. */


/* --------------------------------------------------------------------
 * Font Attributes.
 */

/* Render the "<em>" tag in italics, since this is the best way to
 * emphasize text.
 */
#define OS_ATTR_EM OS_ATTR_ITALIC

/* Render the "<strong>" tag in bold italics.
 */
#define OS_ATTR_STRONG (OS_ATTR_BOLD | OS_ATTR_ITALIC)


/* --------------------------------------------------------------------
 * Basic memory management interface.
 *
 * We simply use the ANSI-C functions.  It's pointless to use the "new"
 * and "delete" operators of C++ since we don't know of what type the
 * allocated memory is supposed to be; all we know is the capacity in
 * bytes.
 */

/* Theoretical maximum osmalloc() size.
 * Qt runs only on systems with at least 32-bit memory space.  Even on
 * 64-bit systems, 2^32 is a good value.
 */
#define OSMALMAX 0xFFFFFFFF


/* Allocate a block of memory of the given size in bytes.
 */
#define osmalloc(siz) malloc(siz)


/* Free memory previously allocated with osmalloc().
 */
#define osfree(block) free(block)


/* Reallocate memory previously allocated with osmalloc() or
 * osrealloc(), changing the block's size to the given number of bytes.
 */
#define osrealloc(block,siz) realloc((block),(siz))


/* --------------------------------------------------------------------
 * Basic file I/O interface.
 */

/* TODO: These are for Unix. */
#define OSFNMAX 63     /* Maximum length of a filename. */
#define OSPATHALT ""   /* Other path separator characters. */
#define OSPATHURL "/"  /* Path separator characters for URL conversions. */
/* This is only used by the Tads 3 make utility; we don't need it. */
/*#define OSPATHSEP ':'*/  /* Directory separator for PATH-style environment variables. */

/* Qt's native path separator is '/', regardless of the current OS. */
#define OSPATHCHAR '/' /* Normal path separator character. */

typedef FILE osfildef; /* File handle structure for osfxxx functions. */

/* Open text file for reading.
 *
 * osfildef *osfoprt(const char *fname, os_filetype_t typ);
 */
#define osfoprt(fname,typ) (fopen((fname),"r"))

/* Open text file for writing.
 */
osfildef*
osfopwt( const char* fname, os_filetype_t typ );

/* Open text file for reading and writing, keeping the file's existing
 * contents if the file already exists or creating a new file if no
 * such file exists.
 */
osfildef*
osfoprwt( const char* fname, os_filetype_t typ );

/* Open text file for reading/writing.  If the file already exists,
 * truncate the existing contents.  Create a new file if it doesn't
 * already exist.
 */
osfildef*
osfoprwtt( const char* fname, os_filetype_t typ );

/* Open binary file for writing.
 *
 * osfildef *osfopwb(const char *fname, os_filetype_t typ);
 */
#define osfopwb(fname,typ) (fopen((fname),"wb"))

/* Open source file for reading - use the appropriate text or binary
 * mode.
 *
 * osfildef *osfoprs(const char *fname, os_filetype_t typ);
 */
#define osfoprs(fname,typ) (osfoprt((fname),(typ)))

/* Open binary file for reading.
 *
 * osfildef *osfoprb(const char *fname, os_filetype_t typ);
 */
#define osfoprb(fname,typ) (fopen((fname),"rb"))

/* Open binary file for reading/writing.  If the file already exists,
 * keep the existing contents.  Create a new file if it doesn't already
 * exist.
 */
osfildef*
osfoprwb( const char* fname, os_filetype_t typ );

/* Open binary file for reading/writing.  If the file already exists,
 * truncate the existing contents.  Create a new file if it doesn't
 * already exist.
 *
 * osfildef *osfoprwtb(const char *fname, os_filetype_t typ);
 */
#define osfoprwtb(fname,typ) (fopen((fname),"w+b"))

/* Get a line of text from a text file.
 *
 * char *osfgets(char *buf, size_t len, osfildef *fp);
 */
#define osfgets(buf,len,fp) (fgets((buf),(len),(fp)))

/* Write a line of text to a text file.
 *
 * int osfputs( const char* buf, osfildef* fp );
 */
#define osfputs(buf,fp) (fputs((buf),(fp)))

/* Write bytes to file.
 *
 * int osfwb( osfildef* fp, const void* buf, int bufl );
 */
#define osfwb(fp,buf,bufl) (fwrite((buf),(bufl),1,(fp))!=1)

/* Read bytes from file.
 *
 * int osfrb(osfildef *fp, uchar *buf, int bufl);
 */
#define osfrb(fp,buf,bufl) (fread((buf),(bufl),1,(fp))!=1)

/* Read bytes from file and return the number of bytes read.
 *
 * size_t osfrbc(osfildef *fp, uchar *buf, size_t bufl);
 */
#define osfrbc(fp,buf,bufl) (fread((buf),1,(bufl),(fp)))

/* Get the current seek location in the file.
 *
 * long osfpos(osfildef *fp);
 */
#define osfpos(fp) (ftell(fp))

/* Seek to a location in the file.
 *
 * int osfseek(osfildef *fp, long pos, int mode);
 */
#define osfseek(fp,pos,mode) (fseek((fp),(pos),(mode)))

/* These values are used for the "mode" parameter to indicate where to
 * seek in the file.
 */
#define OSFSK_SET SEEK_SET /* Set position relative to the start of the file. */
#define OSFSK_CUR SEEK_CUR /* Set position relative to the current file position. */
#define OSFSK_END SEEK_END /* Set position relative to the end of the file. */

/* Close a file.
 */
void
osfcls( osfildef* fp );

/* Delete a file.
 *
 * int osfdel(const char *fname);
 */
#define osfdel(fname) (remove(fname))

/* Access a file - determine if the file exists.
 *
 * We don't use the access() function for this because it's not ANSI-C.
 * (The access() function is listed in SVID, AT&T, POSIX, X/OPEN and
 * BSD 4.3, but not in ANSI-C (C89).  I don't know about C99 though.)
 */
int
osfacc( const char* fname );

/* Get a character from a file.
 *
 * int osfgetc(osfildef *fp);
 */
#define osfgetc(fp) (fgetc(fp))


/* --------------------------------------------------------------------
 * File time stamps.
 */

/* File timestamp type.
 */
struct os_file_time_t {
	time_t t;
};


/* --------------------------------------------------------------------
 * Display routines.
 */

/* The maximum width of a line of text.
 *
 * We ignore this, but the portable Tads code needs it defined.  We use
 * a very large value to avoid drawing the text in "steps" in Tads 3.
 */
#define OS_MAXWIDTH 65535


/* --------------------------------------------------------------------
 */

/* Disable the Tads swap file, since Qt only runs on systems with
 * virtual memory; swapping is handled by the operating system.
 */
#define OS_DEFAULT_SWAP_ENABLED 0

/* We are qtads, not tr.
 *
 * TODO: Don't just use "qtads" but find out how we are called from the
 * command line.
 */
#define OS_TR_USAGE "usage: qtads [options] file"


#endif /* OSQT_H */
