#ifndef TAGCOLL_SERIALIZER_H
#define TAGCOLL_SERIALIZER_H

/** \file
 * Serializer for tags and items
 */

/*
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Consumer.h>
#include <tagcoll/OpSet.h>

namespace Tagcoll
{

/**
 * General class converting items from one representation to another
 */
template<typename IN, typename OUT>
class Converter
{
public:
	virtual ~Converter() {}

	/**
	 * Convert a single item
	 */
	virtual OUT operator()(const IN& item) const = 0;

	/**
	 * Convert a set of items
	 */
	virtual OpSet<OUT> operator()(const OpSet<IN>& items) const
	{
		OpSet<OUT> res;

		for (typename OpSet<IN>::const_iterator i = items.begin();
				i != items.end(); i++)
		{
			OUT t = (*this)(*i);
			if (t != OUT())
				res += t;
		}

		return res;
	}
};

template<typename IN, typename OUT>
class TrivialConverter : public Converter<IN, OUT>
{
public:
	virtual ~TrivialConverter() {}

	/**
	 * Convert a single item
	 */
	virtual OUT operator()(const IN& item) const { return item; }
};


/**
 * Filter converting streams of tagged items among two representations
 */
template<typename IN_ITEM, typename IN_TAG, typename OUT_ITEM, typename OUT_TAG>
class ConversionFilter : public Consumer<IN_ITEM, IN_TAG>
{
	/*
	 * Implementation detail: we cannot derive from Filter because the type of
	 * the output consumer is different from the type of the input consumer.
	 * So we have to reimplement filter methods.
	 */

protected:
	Converter<IN_ITEM, OUT_ITEM>& citem;
	Converter<IN_TAG, OUT_TAG>& ctag;
	Consumer<OUT_ITEM, OUT_TAG>* consumer;

	virtual void consumeItemUntagged(const IN_ITEM& item)
	{
		consumer->consume(citem(item));
	}
	virtual void consumeItem(const IN_ITEM& item, const OpSet<IN_TAG>& tags)
	{
		consumer->consume(citem(item), ctag(tags));
	}
	virtual void consumeItemsUntagged(const OpSet<IN_ITEM>& items)
	{
		consumer->consume(citem(items));
	}
	virtual void consumeItems(const OpSet<IN_ITEM>& items, const OpSet<IN_TAG>& tags)
	{
		consumer->consume(citem(items), ctag(tags));
	}

public:
	ConversionFilter(
			Converter<IN_ITEM, OUT_ITEM>& citem,
			Converter<IN_TAG, OUT_TAG>& ctag) : citem(citem), ctag(ctag), consumer(0) {}
	ConversionFilter(
			Converter<IN_ITEM, OUT_ITEM>& citem,
			Converter<IN_TAG, OUT_TAG>& ctag,
			Consumer<OUT_ITEM, OUT_TAG>& consumer) : citem(citem), ctag(ctag), consumer(&consumer) {}
	virtual ~ConversionFilter() throw () {}

	Consumer<OUT_ITEM, OUT_TAG>& getConsumer() const { return *consumer; }
	void setConsumer(Consumer<OUT_ITEM, OUT_TAG>& consumer) { this->consumer = &consumer; }
};

}

// vim:set ts=4 sw=4:
#endif
