/*
 * Copyright (C) 2007 OpenedHand Ltd
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <config.h>
#include <gtk/gtkwindow.h>

#include "window-util.h"

#define WINDOW_GROUP "Window"
#define WINDOW_X "WindowX"
#define WINDOW_Y "WindowY"
#define WINDOW_WIDTH "WindowWidth"
#define WINDOW_HEIGHT "WindowHeight"


static gboolean
get (GKeyFile *keys, const char *key, int *value)
{
  GError *error = NULL;
  int i;
  i = g_key_file_get_integer (keys, WINDOW_GROUP, key, &error);
  if (error) {
    g_error_free (error);
    return FALSE;
  } else {
    *value = i;
    return TRUE;
  }
}

static GKeyFile *
get_key_file (void)
{
  GError *error = NULL;
  char *filename;
  GKeyFile *keys;

  filename = g_build_filename (g_get_home_dir (), ".gnome2", g_get_prgname (), NULL);
  
  keys = g_key_file_new ();

  if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
    if (!g_key_file_load_from_file (keys, filename, G_KEY_FILE_NONE, &error)) {
      g_warning ("Cannot read key file: %s", error->message);
      goto out;
    }
  }
  
 out:
  g_free (filename);
  return keys;
}

static gboolean
on_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  GError *error = NULL;
  GKeyFile *keys;
  char *filename, *data;
  int x, y, w, h;

  keys = get_key_file ();

  gtk_window_get_position (GTK_WINDOW (widget), &x, &y);
  g_key_file_set_integer (keys, WINDOW_GROUP, WINDOW_X, x);
  g_key_file_set_integer (keys, WINDOW_GROUP, WINDOW_Y, y);

  gtk_window_get_size (GTK_WINDOW (widget), &w, &h);
  g_key_file_set_integer (keys, WINDOW_GROUP, WINDOW_WIDTH, w);
  g_key_file_set_integer (keys, WINDOW_GROUP, WINDOW_HEIGHT, h);

  data = g_key_file_to_data (keys, NULL, NULL);
  filename = g_build_filename (g_get_home_dir (), ".gnome2", g_get_prgname (), NULL);
  if (!g_file_set_contents (filename, data, -1, &error)) {
    g_warning ("Cannot write key file: %s", error->message);
    g_error_free (error);
  }

  g_free (data);
  g_free (filename);
  g_key_file_free (keys);
  return FALSE;
}

void
window_bind_state (GtkWindow *window)
{
  GKeyFile *keys;
  gboolean got_pos, got_size;
  int x, y, w, h;

  g_return_if_fail (GTK_IS_WINDOW (window));

  g_signal_connect (window, "delete-event", G_CALLBACK (on_delete_event), NULL);

  keys = get_key_file ();
  
  got_pos = get (keys, WINDOW_X, &x);
  if (got_pos)
    got_pos &= get (keys, WINDOW_Y, &y);

  if (got_pos) {
    gtk_window_move (window, x, y);
  }
  
  got_size = get (keys, WINDOW_WIDTH, &w);
  if (got_size) 
    got_size &= get (keys, WINDOW_HEIGHT, &h);
  
  if (got_size) {
    gtk_window_set_default_size (window, w, h);
  }
  
  g_key_file_free (keys);
}
