#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <panel-applet.h>

#include "image-loader.h"
#include "properties.h"

static void
set_tooltip (Teatime *teatime)
{
  gchar *tip;

  if (teatime->full)
  {
    tip = g_strdup_printf (_("%s %i s left"),
                           _(teatime->properties.tea_names[teatime->last_tea]),
                           teatime->time_left / 3);
  }
  else
  {
    tip = g_strdup_printf (_("%s (%is)"),
                           _(teatime->properties.tea_names[teatime->last_tea]),
                           teatime->properties.drawing_times[teatime->last_tea]);
  }
  gtk_tooltips_set_tip (teatime->tooltip, teatime->area, tip, NULL);
  g_free (tip);
}

void
teatime_applet_redraw (Teatime *teatime)
{
  GdkPixbuf *active;
  gint size;
  if (!teatime->full)
    active = teatime->highlight ? teatime->cup_empty_hl : teatime->cup_empty;
  else
   active = teatime->highlight ? teatime->cup_full_hl[teatime->frame] : teatime->cup_full[teatime->frame];
  gdk_pixbuf_render_to_drawable_alpha (active, teatime->area->window,
                                       0, 0,
                                       0, 0,
                                       teatime->size, teatime->size,
                                       GDK_PIXBUF_ALPHA_FULL,
                                       1,
                                       GDK_RGB_DITHER_NONE,
                                       0, 0);
  if (!teatime->clock_gc)
  {
    static GdkColor clock_color = {0, 0xFFFF, 0x0000, 0x0000};
    GdkColormap * colormap = gtk_widget_get_colormap (gtk_widget_get_toplevel (teatime->area));
    gdk_colormap_alloc_color (colormap, &clock_color, TRUE, TRUE);
    teatime->clock_gc = gdk_gc_new (teatime->area->window);
    gdk_gc_set_foreground (teatime->clock_gc, &clock_color);
  }
  if (teatime->time_left && teatime->properties.show_clock)
    gdk_draw_arc (teatime->area->window,
                  teatime->clock_gc,
                  FALSE,
                  1, 1,
                  teatime->size -2, teatime->size -2,
                  90 * 64 , (gint)(1 - ((gfloat)teatime->time_left / (gfloat)teatime->drawing_time / 3.0) * 360.0 * 64.0));

  set_tooltip (teatime);
}

static void
enter_event (GtkWidget *widget, GdkEvent *event, Teatime *teatime)
{
  teatime->highlight = TRUE;
  teatime_applet_redraw (teatime);
}

static void
leave_event (GtkWidget *widget, GdkEvent *event, Teatime *teatime)
{
  teatime->highlight = FALSE;
  teatime_applet_redraw (teatime);
}

static void
animate (Teatime *teatime)
{
  teatime->frame++;
  if (teatime->frame == FRAMES)
    teatime->frame = 0;
  teatime_applet_redraw (teatime);
}

static void
popup_hide (GtkWidget *widget, GdkEvent *event, TeatimePopup *popup)
{
  gtk_timeout_remove (popup->timer_handle);
  gtk_widget_hide (popup->cup_window);
  gtk_widget_hide (popup->label_window);
  popup->visible = FALSE;
}

static void
cup_popup (TeatimePopup *popup)
{
  static GdkPixbuf *pixbuf;
  static GdkPixmap *pixmap;
  GtkWidget *hbox;
  GtkWidget *eventbox;
  gint i;
    
  popup->cup_window = gtk_window_new (GTK_WINDOW_POPUP);
  gtk_widget_set_usize (popup->cup_window, CUP_WIDTH, CUP_HEIGHT);
  hbox = gtk_hbox_new (TRUE, 0);
  for (i=0; i<ROT_FRAMES; i++)
    gtk_box_pack_start (GTK_BOX (hbox), popup->image[i], FALSE, FALSE, 0);
  gtk_widget_shape_combine_mask (popup->cup_window, popup->bitmap[0], 0, 0);
  eventbox = gtk_event_box_new ();
  gtk_container_add (GTK_CONTAINER (eventbox), hbox);	
  gtk_container_add (GTK_CONTAINER (popup->cup_window), eventbox);
  g_signal_connect (G_OBJECT (eventbox), "button_release_event",
                    G_CALLBACK (popup_hide), popup);
  gtk_widget_show (eventbox);
  gtk_widget_show (hbox);
  gtk_widget_show (popup->image[0]);
}

static void
draw_layout(GtkWidget *widget, GdkEvent *event, TeatimePopup *popup)
{
  gchar *tea_is_ready = {N_("Your Tea\nis ready")};
  gchar *string;

  string = g_strdup_printf ("<span face=\"bold\" foreground=\"gold\">"
                            "<big><big>%s</big></big></span>", _(tea_is_ready));
  pango_layout_set_markup (popup->layout, string, -1);
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   0, 0,
                   popup->layout);
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   0, 1,
                   popup->layout);
  g_free (string);

  string = g_strdup_printf ("<span face=\"bold\" foreground=\"gold4\">"
                            "<big><big>%s</big></big></span>", _(tea_is_ready));
  pango_layout_set_markup (popup->layout, string, -1); 
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   1, 1,
                   popup->layout);
  gdk_draw_layout (widget->window,
                   widget->style->black_gc,
                   2, 1,
                   popup->layout);
  g_free (string);
}

static void
label_popup (TeatimePopup *popup, GtkWidget *widget)
{
  GtkWidget *area;
  GdkPixbuf *pixbuf;
  GdkPixmap *pixmap;
  GdkBitmap *bitmap;
  guchar *pixels;
  gchar *tea_is_ready = {N_("Your Tea\nis ready")};
  gchar *string; 
  gint x, y;
  GdkGC *fggc, *bggc;
  GdkGC *fggc_bit, *bggc_bit;
  GdkColor black = {0, 0, 0, 0};
  GdkColor white = {1, 0xffff, 0xffff, 0xffff};
 
  popup->label_window = gtk_window_new (GTK_WINDOW_POPUP);
  area = gtk_drawing_area_new ();

  gtk_container_add (GTK_CONTAINER (popup->label_window), area);
  g_signal_connect (G_OBJECT (area), "expose_event",
                    G_CALLBACK (draw_layout), popup);

  popup->layout = gtk_widget_create_pango_layout (widget, "");
  pango_layout_set_alignment (popup->layout, PANGO_ALIGN_CENTER);

  string = g_strdup_printf ("<span face=\"bold\" foreground=\"black\">"
                            "<big><big>%s</big></big></span>", _(tea_is_ready)); 
  pango_layout_set_markup (popup->layout, string, -1);
  g_free (string);
  pango_layout_get_pixel_size (popup->layout, &popup->label_width, &popup->label_height);
  if (popup->label_width%4)
    popup->label_width = popup->label_width - popup->label_width%4 + 4;

  gtk_widget_set_usize (popup->label_window, popup->label_width, popup->label_height); 
  pixmap = gdk_pixmap_new (widget->window, popup->label_width, popup->label_height, -1);
  bitmap = gdk_pixmap_new (NULL, popup->label_width, popup->label_height, 1);  
  
  fggc = gdk_gc_new (pixmap);
  bggc = gdk_gc_new (pixmap);
  fggc_bit = gdk_gc_new (bitmap);
  bggc_bit = gdk_gc_new (bitmap);

  gdk_gc_set_foreground (bggc, &black);
  gdk_gc_set_foreground (fggc, &white);
  gdk_gc_set_foreground (fggc_bit, &black);
  gdk_gc_set_foreground (bggc_bit, &white);
 
  gdk_draw_rectangle (pixmap, widget->style->white_gc, TRUE, 0, 0,
                      popup->label_width, popup->label_height);
  
  gdk_draw_layout (pixmap,
                   fggc,
                   0, 0, popup->layout);
  gdk_draw_layout (pixmap,
                   fggc,
                   1, 1, popup->layout);

  pixbuf = gdk_pixbuf_get_from_drawable (NULL,
                                         pixmap,
                                         gdk_drawable_get_colormap (pixmap),
                                         0, 0, 0, 0,
                                         popup->label_width, popup->label_height);
  pixels = gdk_pixbuf_get_pixels (pixbuf);

  for (y=0; y< popup->label_height; y++)
    for (x=0; x<popup->label_width; x++)
    {
      if (*(pixels+2) <255)
        gdk_draw_point (bitmap, bggc_bit, x, y);
      else
        gdk_draw_point (bitmap, fggc_bit, x, y);
      pixels+=3;      
    }
    
  gtk_widget_shape_combine_mask (popup->label_window, bitmap, 0, 0);
  gtk_widget_show (area);
}

static gint
popup_cup_timer (TeatimePopup *popup)
{
  gint hide;

  hide = popup->frame -1;
  
  if (hide < 0)
    hide = ROT_FRAMES - 1;
  gtk_widget_hide (popup->image[hide]);
  gtk_widget_show (popup->image[popup->frame]);
  gtk_widget_shape_combine_mask (popup->cup_window, popup->bitmap[popup->frame], 0, 0);
  popup->frame++;
  if (popup->frame == ROT_FRAMES)
    popup->frame = 0;
  return TRUE;  
}

static void
stop_sound_thread (GstElement *element,
                   gpointer data)
{
  gst_element_set_state (GST_ELEMENT (data), GST_STATE_NULL);
}

static gint
timer(Teatime *teatime)
{
  GstElement *audio_sink;
  gint center_x, center_y;
 
  if (--teatime->time_left)
  {
    animate (teatime);
    return TRUE;
  }
  
  teatime->full = FALSE;
  if (teatime->timer_handle)
    gtk_timeout_remove (teatime->timer_handle);
  teatime_applet_redraw (teatime);
  if (teatime->properties.play_sound && teatime->properties.sound)
  {
    audio_sink = gst_gconf_get_default_audio_sink ();
    if (GST_IS_ELEMENT (audio_sink))
    {
      teatime->sound.thread = gst_thread_new ("pipe");
      teatime->sound.input = gst_element_factory_make ("filesrc", "input");
      g_signal_connect (G_OBJECT (teatime->sound.input), "eos",
                        G_CALLBACK (stop_sound_thread), teatime->sound.thread);
      teatime->sound.decoder = gst_element_factory_make ("spider", "decoder");
      teatime->sound.output = audio_sink;
      gst_bin_add_many (GST_BIN (teatime->sound.thread),
                        teatime->sound.input,
                        teatime->sound.decoder,
                        teatime->sound.output, NULL);
      gst_element_link_many (teatime->sound.input,
                             teatime->sound.decoder,
                             teatime->sound.output, NULL);   
      g_object_set (G_OBJECT (teatime->sound.input), "location", teatime->properties.sound, NULL);
      gst_element_set_state (teatime->sound.thread,  GST_STATE_PLAYING);
    }
    else
      g_warning ("Could not get default audio sink from GConf");
  }
  if (!teatime->properties.popup)
    return FALSE;
      
  teatime->popup.visible = TRUE;

  if (teatime->popup.cup_window)
  {
    gdk_window_raise (teatime->popup.cup_window->window);
    gdk_window_raise (teatime->popup.label_window->window);
    gtk_widget_hide (teatime->popup.image[teatime->popup.frame - 1]);
    gtk_widget_show (teatime->popup.image[0]);
  }
  else
  {
    cup_popup (&teatime->popup);
    label_popup (&teatime->popup, teatime->border);
  }
  teatime->popup.frame = 0;    
  gdk_window_get_size (gdk_get_default_root_window(), &center_x, &center_y);
  center_x >>= 1;
  center_y >>= 1;
  gtk_widget_set_uposition (teatime->popup.cup_window,
                            center_x - CUP_WIDTH / 2,
                            center_y - (CUP_HEIGHT + teatime->popup.label_height) / 2);
  gtk_widget_set_uposition (teatime->popup.label_window,
                            center_x - teatime->popup.label_width / 2,
                            center_y  - (CUP_HEIGHT + teatime->popup.label_height) / 2
                            + CUP_HEIGHT);

  gtk_widget_show (teatime->popup.cup_window);
  gtk_widget_show (teatime->popup.label_window);

  teatime->popup.timer_handle = g_timeout_add (60,
                                               (GSourceFunc)popup_cup_timer,
                                               &teatime->popup);
  return FALSE;
}

static void
show_website (GtkWidget *dialog,
              const char *url,
              gpointer data)
{
  gnome_vfs_url_show (url);
}

static void
display_about_dialog(BonoboUIComponent *uic,
                     Teatime *teatime,
                     const gchar *verbname){

  GdkPixbuf *logo;
  GError *error = NULL;
  static GtkWidget *about = NULL;
  const gchar *authors [] = {N_("Detlef Reichl\n\tdetlef ! reichl () gmx ! org"), NULL};
  const gchar *website = "http://dr.dhis.org/teatime/index.rbx?r=2.6.0";
  const gchar *website_label = N_("Visit the Teatime website\nhttp://dr.dhis.org/teatime/");
  
  if (about != NULL)
  {
    gdk_window_show(about->window);
    gdk_window_raise(about->window);
    return;
  }
  logo = gdk_pixbuf_new_from_file (teatime->icon_path, &error);
 
  gtk_about_dialog_set_url_hook ((GtkAboutDialogActivateLinkFunc)show_website,
                                  NULL, NULL);
 
  about = gtk_about_dialog_new ();
  gtk_about_dialog_set_name (GTK_ABOUT_DIALOG (about), _("Teatime"));
  gtk_about_dialog_set_version (GTK_ABOUT_DIALOG (about), VERSION);
  gtk_about_dialog_set_copyright (GTK_ABOUT_DIALOG (about), _("Copyright (C) 1998-2005 Detlef Reichl"));
  gtk_about_dialog_set_comments (GTK_ABOUT_DIALOG (about), _("A\ntea-drawing-time-observer.\n"
                                                             "The most useful GNOME program ;-)"));
  gtk_about_dialog_set_website (GTK_ABOUT_DIALOG (about), website);
  gtk_about_dialog_set_website_label (GTK_ABOUT_DIALOG (about), _(website_label));
  gtk_about_dialog_set_authors (GTK_ABOUT_DIALOG (about), authors);
  gtk_about_dialog_set_artists (GTK_ABOUT_DIALOG (about), authors);
  gtk_about_dialog_set_translator_credits (GTK_ABOUT_DIALOG (about), _("This crappy English is from\n"
                                                                "Detlef Reichl <detlef ! reichl () gmx ! org>"));
  gtk_about_dialog_set_logo (GTK_ABOUT_DIALOG (about), logo);

  gtk_window_set_icon_from_file (GTK_WINDOW (about),
                                 teatime->icon_path,
                                 NULL);

  gtk_dialog_run (GTK_DIALOG (about));
  gtk_widget_destroy (about);
  about = NULL;
}

static void
restart (Teatime *teatime)
{
  if (teatime->full)
  {
    GtkWidget *dialog;
      
    dialog = gtk_message_dialog_new (NULL,
                                     0,
                                     GTK_MESSAGE_QUESTION,
                                     GTK_BUTTONS_YES_NO,
                                     _("Restart drawing?"),
                                     NULL);
    if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_YES)
    {
      gtk_widget_destroy (dialog);
      return;
    }
    gtk_widget_destroy (dialog);
  }
  if (teatime->timer_handle)
    gtk_timeout_remove (teatime->timer_handle);
  if (teatime->popup.visible)
    popup_hide (NULL, NULL, &teatime->popup);
  teatime->full = TRUE;
  teatime->time_left = teatime->drawing_time * FRAMES;
  teatime->timer_handle = gtk_timeout_add (333, (GtkFunction)timer, teatime);
  teatime_applet_redraw (teatime);
}

static gboolean
clicked_event (GtkWidget *widget, GdkEventButton *event, Teatime *teatime)
{
  if (event->button != 1 || teatime->popup.visible)
    return FALSE;
  
  teatime->drawing_time = teatime->properties.drawing_times[teatime->last_tea];
  restart (teatime);
  gtk_tooltips_force_window (teatime->tooltip);
  return TRUE;
}

static void
change_orient (GtkWidget *widget, PanelAppletOrient orient, gpointer data)
{
  Teatime *teatime = (Teatime*)data;

  teatime->orient_vert = (orient == PANEL_APPLET_ORIENT_UP
                          || orient == PANEL_APPLET_ORIENT_DOWN) ? FALSE : TRUE;
}

static void
size_allocate (PanelApplet *applet,
               GtkAllocation *allocation,
               Teatime *teatime)
{
  if (teatime->orient_vert)
  {
    if (teatime->size == allocation->width)
      return;
    teatime->size = allocation->width;
  }
  else
  {
    if (teatime->size == allocation->height)
      return;
    teatime->size = allocation->height;
  }
 
  /* in the begining the applets are built with a size of 1 */
  if (teatime->size <= 1)
    return;
  gtk_widget_set_size_request (teatime->area, teatime->size, teatime->size);
  teatime_image_loader_load_applet_images (teatime);
  if (GTK_WIDGET_REALIZED (teatime->area))
    teatime_applet_redraw (teatime);
}

static void
real_start_drawing (Teatime *teatime, gint tea)
{
  if (teatime->popup.visible)
    return;

  teatime->drawing_time = teatime->properties.drawing_times[tea];
  teatime->last_tea = tea;
  restart (teatime);
}

static void
start_drawing (BonoboUIComponent *uic,
              Teatime *teatime,
              const gchar *verbname)
{
  gint tea = (gint) g_strtod(verbname + 1, NULL);
  if (tea > teatime->properties.num_teas)
    tea = teatime->properties.num_teas;
  real_start_drawing (teatime, tea);
}

static void
destroy (GtkWidget *applet, Teatime *teatime)
{
  gint i;
  
  if (teatime->timer_handle)
    gtk_timeout_remove(teatime->timer_handle);

  if (teatime->popup.timer_handle)
      gtk_timeout_remove(teatime->popup.timer_handle);
  
  if (teatime->popup.label_window)
    gtk_widget_destroy (teatime->popup.label_window);
  
  if (teatime->popup.cup_window)
    gtk_widget_destroy (teatime->popup.cup_window);
  
  for (i=0; i<ROT_FRAMES; i++)
    if (teatime->popup.bitmap[i])
      gdk_pixmap_unref (teatime->popup.bitmap[i]);
  
  for (i=0; i<TEACUPS; i++)
    if (teatime->properties.teacups[i])
      g_object_unref (G_OBJECT (teatime->properties.teacups[i]));
  
  if (teatime->cup_empty)
    g_object_unref (G_OBJECT (teatime->cup_empty));
  
  if (teatime->cup_empty_hl)
    g_object_unref (G_OBJECT (teatime->cup_empty_hl));
  
  for (i=0; i<FRAMES; i++)
  {
    if (teatime->cup_full[i])
      g_object_unref (G_OBJECT (teatime->cup_full[i]));
    if (teatime->cup_full_hl[i])
      g_object_unref (G_OBJECT (teatime->cup_full_hl[i]));
  }
  
  g_free (teatime);
}

static void
ui_verb_set_data (BonoboUIVerb *verb,
                  gchar *name,
                  gpointer cb)
{
  verb->cname = (const gchar *)g_strdup (name);
  verb->cb = (BonoboUIVerbFn)cb;
  verb->user_data = NULL;
}

void
teatime_set_menu (Teatime *teatime)
{
  gchar *string = NULL;
  gint i;
  gint num_teas = teatime->properties.num_teas;
  gchar *append;
  gchar *tmp;
  
  BonoboUIVerb *teatime_menu_verbs = g_new (BonoboUIVerb, num_teas + 3);

  /* We have to clear the complete menu, before we can add new stuff. Else old, allready
   * removed entries are displayed. */
  bonobo_ui_component_rm (panel_applet_get_popup_component (teatime->applet), "/", NULL);
  
  ui_verb_set_data (&teatime_menu_verbs[0], "Prop", teatime_display_properties_dialog);
  ui_verb_set_data (&teatime_menu_verbs[1], "About", display_about_dialog);
  ui_verb_set_data (&teatime_menu_verbs[num_teas + 2], NULL, NULL);
  
  string = g_strdup_printf (
  "<popup name=\"button3\">\n"
  "  <menuitem name=\"Item 1\" verb=\"Prop\" _label=\"%s\"\n"
  "     pixtype=\"stock\" pixname=\"gtk-properties\"/>\n"
  "  <menuitem name=\"Item 3\" verb=\"About\" _label=\"%s\"\n"
  "    pixtype=\"stock\" pixname=\"gtk-stock-about\"/>\n"
  "  <separator/>\n",

    _("Properties ..."),
    _("About"));

  for (i=0; i<num_teas; i++)
  {
    gchar *verb = g_strdup_printf ("V%i", i);
    gchar *quoted_tea_name = g_markup_escape_text (teatime->properties.tea_names[i], strlen(teatime->properties.tea_names[i]));
    append = g_strdup_printf ("  <menuitem name=\"Item %i\" verb=\"%s\" _label=\"%s %is\"/>\n",
                               i + 4,
                               verb,
                               quoted_tea_name,
                               teatime->properties.drawing_times[i]);
    tmp = g_strconcat (string, append, NULL);
    g_free (quoted_tea_name);
    g_free (string);
    string = tmp;
    ui_verb_set_data (&teatime_menu_verbs[i + 2], verb, start_drawing);
    g_free (verb);
  }
  tmp = g_strconcat (string, "</popup>\n", NULL);
  g_free (string);
  string = tmp;
  panel_applet_setup_menu (PANEL_APPLET (teatime->applet),
                           (const gchar *)string,
                           teatime_menu_verbs,
                           teatime);
}


static gboolean
teatime_applet_new (PanelApplet *applet)
{
  Teatime *teatime = g_new0 (Teatime, 1);
  gint i;
  gchar *string;
  PanelAppletOrient orient;
  
  teatime->applet = applet;
  teatime->full = FALSE;
  teatime->highlight = FALSE;
  teatime->frame = 0;
  teatime->popup.cup_window = NULL;
  teatime->clock_gc = NULL;
  
  panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);

  panel_applet_add_preferences (applet, "/schemas/apps/teatime/prefs", NULL);
  
  gdk_rgb_init();

  gst_init(NULL, NULL);
  
  teatime_load_properties (teatime);
  teatime_image_loader_load_popup_images (teatime);
  teatime_image_loader_load_pref_images (teatime);

  teatime->border = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (teatime->border), GTK_SHADOW_IN);
  teatime->area = gtk_drawing_area_new ();

  gtk_container_add (GTK_CONTAINER (teatime->border), teatime->area);
  gtk_container_add (GTK_CONTAINER (teatime->applet), teatime->border);
  
  teatime->tooltip = gtk_tooltips_new ();
  gtk_tooltips_set_delay (teatime->tooltip, 50);

  g_signal_connect_data (G_OBJECT (teatime->area), "expose_event",
                         G_CALLBACK (teatime_applet_redraw), teatime,
                         NULL, G_CONNECT_SWAPPED);
   g_signal_connect (G_OBJECT (teatime->area), "size_allocate",
                     G_CALLBACK (size_allocate), teatime);
 
  g_signal_connect (G_OBJECT (applet), "destroy",
                    G_CALLBACK (destroy), teatime);
  g_signal_connect (G_OBJECT (applet), "change_orient",
                    G_CALLBACK (change_orient), teatime);

  gtk_widget_add_events (teatime->area, GDK_BUTTON_PRESS_MASK
                         | GDK_ENTER_NOTIFY_MASK
                         | GDK_LEAVE_NOTIFY_MASK);
  g_signal_connect (G_OBJECT (teatime->area), "enter_notify_event",
                    G_CALLBACK (enter_event), teatime);
  g_signal_connect (G_OBJECT (teatime->area), "leave_notify_event",
                    G_CALLBACK (leave_event), teatime);
  g_signal_connect_data (G_OBJECT (teatime->area), "button_press_event",
                         G_CALLBACK (clicked_event), teatime,
                         NULL, G_CONNECT_AFTER);
  
  orient = panel_applet_get_orient (applet);
  teatime->orient_vert = (orient == PANEL_APPLET_ORIENT_UP
                          || orient == PANEL_APPLET_ORIENT_DOWN) ? FALSE : TRUE;

  gtk_widget_show_all(GTK_WIDGET(applet));
  teatime_set_menu (teatime);
  return TRUE;
}

static gboolean
teatime_factory (PanelApplet *applet,
		 const gchar *iid,
		 gpointer data)
{
  gboolean retval = FALSE;
  if (strcmp (iid, "OAFIID:GNOME_TeatimeApplet") == 0)
    retval = teatime_applet_new (applet);
  return retval;
}

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_TeatimeApplet_Factory",
                             PANEL_TYPE_APPLET,
                             "teatime",
                             "0",
                             teatime_factory,
                             NULL)

