/************************************************************************/
/*									*/
/*  Read an RTF text file into a BufferDocument				*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<appDebugon.h>

#   include	<psFont.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Forward declarations.						*/
/*									*/
/************************************************************************/

static int docRtfReadPntext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfSkipPn(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfReadListtext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfReadUpr(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );
static int docRtfReadPnseclvl(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc );

/************************************************************************/
/*									*/
/*  Special actions for section properties.				*/
/*									*/
/************************************************************************/

static int docRtfTextSpecial(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case RTFidU:
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId);
	    break;
	}

    return 0;
    }

static int docRtfHierarchy(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {

    switch( rcw->rcwId )
	{
	case	RTFidSECT:
	    if  ( rrc->rrcSplitLevel < DOClevSECT )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevSECT- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevSECT- 1;
	    break;

	case	RTFidROW:
	    if  ( rrc->rrcSplitLevel < DOClevROW )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevROW- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevROW- 1;
	    break;

	case	RTFidCELL:
	    if  ( rrc->rrcSplitLevel <= DOClevCELL )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevCELL- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevCELL- 1;

	    break;

	case	RTFidPAR:
	    if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }

	    if  ( docRtfAdjustLevel( rrc, DOClevPARA- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    break;

	case	RTFidTAB:
	    if  ( docSaveSpecialParticule( rrc->rrcBi,
			    rrc->rrcCurrentTextAttribute, DOCkindTAB ) )
		{ LDEB(1); return -1;	}
	    break;

	case	RTFidLINE:
	    if  ( docSaveSpecialParticule( rrc->rrcBi,
			    rrc->rrcCurrentTextAttribute, DOCkindLINEBREAK ) )
		{ LDEB(1); return -1;	}
	    break;

	case	RTFidPAGE:
	    if  ( rrc->rrcJustAfterPntext )
		{ rrc->rrcPageAfterPntext= 1; return 0; }

	    if  ( docSaveSpecialParticule( rrc->rrcBi,
			    rrc->rrcCurrentTextAttribute, DOCkindPAGEBREAK ) )
		{ LDEB(1); return -1;	}
	    break;

	case	RTFidCHFTNSEP:
	    if  ( docSaveSpecialParticule( rrc->rrcBi,
			    rrc->rrcCurrentTextAttribute, DOCkindCHFTNSEP ) )
		{ LDEB(1); return -1;	}
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId); break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Control words to remember border properties.			*/
/*									*/
/************************************************************************/

# define RTF_BORDER_PROP( s, id ) \
		    { s, id, DOClevANY, docRtfBrdrProperty, }
# define RTF_BORDER_STYLE(s,v) \
		    { s, BRDRpropSTYLE, DOClevANY, docRtfBrdrProperty, v, }

static RtfControlWord	docRtfBorderWords[]=
    {
	RTF_BORDER_PROP( "brdrw",		BRDRpropPEN_WIDE ),
	RTF_BORDER_PROP( "brdrcf",		BRDRpropCOLOR ),
	RTF_BORDER_PROP( "brsp",		BRDRpropSPACING ),

	/**/

	RTF_BORDER_STYLE( "brdrnone",		DOCbsNONE ),
	RTF_BORDER_STYLE( "brdrs",		DOCbsS ),
	RTF_BORDER_STYLE( "brdrth",		DOCbsTH ),
	RTF_BORDER_STYLE( "brdrsh",		DOCbsSH ),

	RTF_BORDER_STYLE( "brdrdb",		DOCbsDB ),
	RTF_BORDER_STYLE( "brdrhair",		DOCbsHAIR ),
	RTF_BORDER_STYLE( "brdrdot",		DOCbsDOT ),
	RTF_BORDER_STYLE( "brdrdash",		DOCbsDASH  ),
	RTF_BORDER_STYLE( "brdrdashd",		DOCbsDASHD ),
	RTF_BORDER_STYLE( "brdrdashdd",		DOCbsDASHDD ),
	RTF_BORDER_STYLE( "brdrtriple",		DOCbsTRIPLE ),

	RTF_BORDER_STYLE( "brdrtnthsg",		DOCbsTNTHSG ),
	RTF_BORDER_STYLE( "brdrthtnsg",		DOCbsTHTNSG ),
	RTF_BORDER_STYLE( "brdrtnthtnsg",	DOCbsTNTHTNSG ),
	RTF_BORDER_STYLE( "brdrtnthmg",		DOCbsTNTHMG ),
	RTF_BORDER_STYLE( "brdrthtnmg",		DOCbsTHTNMG ),
	RTF_BORDER_STYLE( "brdrtnthtnmg",	DOCbsTNTHTNMG ),
	RTF_BORDER_STYLE( "brdrtnthlg",		DOCbsTNTHLG ),
	RTF_BORDER_STYLE( "brdrthtnlg",		DOCbsTHTNLG ),
	RTF_BORDER_STYLE( "brdrtnthtnlg",	DOCbsTNTHTNLG ),

	RTF_BORDER_STYLE( "brdrwavy",		DOCbsWAVY ),
	RTF_BORDER_STYLE( "brdrwavydb",		DOCbsWAVYDB ),
	RTF_BORDER_STYLE( "brdrdashsm",		DOCbsDASHSM ),
	RTF_BORDER_STYLE( "brdrdashdotstr",	DOCbsDASHDOTSTR ),
	RTF_BORDER_STYLE( "brdremboss",		DOCbsEMBOSS ),
	RTF_BORDER_STYLE( "brdrengrave",	DOCbsENGRAVE ),

	/**/

	{ 0, 0, 0 }
    };

RtfControlWord	docRtfDocumentWords[]=
    {
	{ "fc",		RTFidFC,	DOClevANY,  docRtfRememberProperty, },
				/****************************************/
				/*  Special characters.			*/
				/****************************************/
	{ "bullet",	RTFidBULLET,	DOClevTEXT, docRtfTextSpecial,	},
	{ "chatn",	RTFidCHATN,	DOClevTEXT, docRtfTextSpecial,	},
	{ "chdate",	RTFidCHDATE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "chdpa",	RTFidCHDPA,	DOClevTEXT, docRtfTextSpecial,	},
	{ "chdpl",	RTFidCHDPL,	DOClevTEXT, docRtfTextSpecial,	},
	{ "chftn",	RTFidCHFTN,	DOClevTEXT, docRtfChftn,	},
	{ "chftnsepc",	RTFidCHFTNSEPC,	DOClevTEXT, docRtfTextSpecial,	},
	{ "chpgn",	RTFidCHPGN,	DOClevTEXT, docRtfSpecialToField, },
	{ "chtime",	RTFidCHTIME,	DOClevTEXT, docRtfTextSpecial,	},
	{ "column",	RTFidCOLUMN,	DOClevTEXT, docRtfTextSpecial,	},
	{ "emdash",	RTFidEMDASH,	DOClevTEXT, docRtfTextSpecial,	},
	{ "emspace",	RTFidEMSPACE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "endash",	RTFidENDASH,	DOClevTEXT, docRtfTextSpecial,	},
	{ "enspace",	RTFidENSPACE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "ldblquote",	RTFidLDBLQUOTE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "lquote",	RTFidLQUOTE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "ltrmark",	RTFidLTRMARK,	DOClevTEXT, docRtfTextSpecial,	},
	{ "rdblquote",	RTFidRDBLQUOTE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "rquote",	RTFidRQUOTE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "rtlmark",	RTFidRTLMARK,	DOClevTEXT, docRtfTextSpecial,	},
	{ "sectnum",	RTFidSECTNUM,	DOClevTEXT, docRtfTextSpecial,	},
	{ "softcol",	RTFidSOFTCOL,	DOClevTEXT, docRtfTextSpecial,	},
	{ "softlheight",RTFidSOFTLHEIGHT, DOClevTEXT, docRtfTextSpecial, },
	{ "softline",	RTFidSOFTLINE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "softpage",	RTFidSOFTPAGE,	DOClevTEXT, docRtfTextSpecial,	},
	{ "zwj",	RTFidZWJ,	DOClevTEXT, docRtfTextSpecial,	},
	{ "zwnj",	RTFidZWNJ,	DOClevTEXT, docRtfTextSpecial,	},

	{ "u",		RTFidU,		DOClevTEXT, docRtfTextSpecial,	},
				/****************************************/
				/*  Hierarchy/Structure tags.		*/
				/****************************************/
	{ "chftnsep",	RTFidCHFTNSEP,	DOClevTEXT, docRtfHierarchy, },
	{ "tab",	RTFidTAB,	DOClevTEXT, docRtfHierarchy, },
	{ "line",	RTFidLINE,	DOClevTEXT, docRtfHierarchy, },
	{ "page",	RTFidPAGE,	DOClevTEXT, docRtfHierarchy, },

	{ "par",	RTFidPAR,	DOClevANY,  docRtfHierarchy, },
	{ "cell",	RTFidCELL,	DOClevANY,  docRtfHierarchy, },
	{ "row",	RTFidROW,	DOClevANY,  docRtfHierarchy, },
	{ "sect",	RTFidSECT,	DOClevANY,  docRtfHierarchy, },

	{ 0, 0, 0 }
    };

RtfControlWord	docRtfDocumentGroups[]=
    {
	{ "upr",	RTFidUPR,	DOClevANY,  docRtfReadUpr, },

	{ "fonttbl",	RTFidFONTTBL,	DOClevDOC,  docRtfFontTable,  },
	{ "colortbl",	RTFidCOLORTBL,	DOClevDOC,  docRtfColorTable, },
	{ "stylesheet",	RTFidSTYLESHEET,DOClevDOC,  docRtfStylesheet,  },
	/*
	{ "listtable",	RTFidLISTTABLE,	DOClevDOC,  docRtfListtable,  },
	*/
	{ "info",	RTFidINFO,	DOClevDOC,  docRtfReadInfo,  },
	{ "revtbl",	RTFidREVTBL,	DOClevDOC,  docRtfRevisionTable,  },
	{ "pn",		RTFidPN,	DOClevANY,  docRtfSkipPn,   },
	{ "pntext",	RTFidPNTEXT,	DOClevPARA, docRtfReadPntext, },
	{ "pict",	RTFidPICT,	DOClevPARA, docRtfReadPict, },
	{ "nonshppict",	RTFidNONSHPPICT,DOClevPARA, docRtfReadNonshppict, },
	{ "shppict",	RTFidSHPPICT,	DOClevPARA, docRtfReadShppict, },
	{ "object",	RTFidOBJECT,	DOClevPARA, docRtfReadObject, },
	{ "field",	RTFidFIELD,	DOClevPARA, docRtfReadField, },
	{ "shp",	RTFidSHP,	DOClevPARA, docRtfReadShape, },
	{ "shpgrp",	RTFidSHPGRP,	DOClevPARA, docRtfReadShape, },
	{ "xe",		RTFidXE,	DOClevPARA, docRtfReadLookupEntry, },
	{ "tc",		RTFidTC,	DOClevPARA, docRtfReadLookupEntry, },
	{ "tcn",	RTFidTCN,	DOClevPARA, docRtfReadLookupEntry, },

				/****************************************/
				/*  Headers and Footers.		*/
				/****************************************/
	{ "footer",	DOCinSECT_FOOTER, DOClevSECT, docRtfReadExtIt, },
	{ "footerf",	DOCinFIRST_FOOTER, DOClevSECT, docRtfReadExtIt, },
	{ "footerl",	DOCinLEFT_FOOTER, DOClevSECT, docRtfReadExtIt, },
	{ "footerr",	DOCinRIGHT_FOOTER, DOClevSECT, docRtfReadExtIt, },

	{ "header",	DOCinSECT_HEADER, DOClevSECT, docRtfReadExtIt, },
	{ "headerf",	DOCinFIRST_HEADER, DOClevSECT, docRtfReadExtIt, },
	{ "headerl",	DOCinLEFT_HEADER, DOClevSECT, docRtfReadExtIt, },
	{ "headerr",	DOCinRIGHT_HEADER, DOClevSECT, docRtfReadExtIt, },

	{ "ftnsep",	DOCinFTNSEP,	DOClevDOC, docRtfReadExtIt, },
	{ "ftnsepc",	DOCinFTNSEPC,	DOClevDOC, docRtfReadExtIt, },
	{ "ftncn",	DOCinFTNCN,	DOClevDOC, docRtfReadExtIt, },

	{ "aftnsep",	DOCinAFTNSEP,	DOClevDOC, docRtfReadExtIt, },
	{ "aftnsepc",	DOCinAFTNSEPC,	DOClevDOC, docRtfReadExtIt, },
	{ "aftncn",	DOCinAFTNCN,	DOClevDOC, docRtfReadExtIt, },

	{ "footnote",	DOCinFOOTNOTE,	DOClevPARA, docRtfReadFootnote, },

				/****************************************/
				/*  Multilevel Lists.			*/
				/****************************************/
	{ "pnseclvl",	RTFidPNSECLVL,	DOClevANY, docRtfReadPnseclvl,	},
				/****************************************/
				/*  Lists.				*/
				/****************************************/
	{ "listtext",	RTFidLISTTEXT,	DOClevANY, docRtfReadListtext, },
				/****************************************/
				/*  Annotation.				*/
				/****************************************/
	{ "chatn",	RTFidCHATN,	DOClevPARA, docRtfSkipGroup, },
				/****************************************/
				/*  Bookmarks				*/
				/****************************************/
	{ "bkmkstart",	RTFidBKMKSTART,	DOClevPARA, docRtfBkmkStart,	},
	{ "bkmkend",	RTFidBKMKEND,	DOClevPARA, docRtfBkmkEnd,	},

	{ 0, 0, 0 }
    };

/************************************************************************/
/*									*/
/*  Read a document as a whole.						*/
/*									*/
/************************************************************************/

int docRtfTextParticule(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    if  ( rrc->rrcInDeletedText )
	{ return 0;	}

    if  ( rrc->rrcJustAfterPntext )
	{
	PropertyMask	ppChgMask;
	PropertyMask	ppUpdMask;

	PROPmaskCLEAR( &ppChgMask );

	PROPmaskCLEAR( &ppUpdMask );
	PROPmaskFILL( &ppUpdMask, PPprop_COUNT );

	if  ( docUpdParaProperties( &ppChgMask, &(rrc->rrcBi->biParaProperties),
				&ppUpdMask, &(rrc->rrcParagraphProperties) ) )
	    { LDEB(1); return -1;	}

	if  ( rrc->rrcPageAfterPntext )
	    { rrc->rrcBi->biParaProperties.ppStartsOnNewPage= 1;	}

	rrc->rrcJustAfterPntext= 0;
	rrc->rrcPageAfterPntext= 0;
	}

    return docSaveParticules( rrc->rrcBi, rrc->rrcCurrentTextAttribute,
								text, len );
    }

static int docRtfReadDoc(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static int docRtfReadUpr(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static int docRtfReadListtext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Consume the description of a Paragraph Numbering level for a	*/
/*  section.								*/
/*									*/
/************************************************************************/

static int docRtfPndescText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    unsigned char *	target;
    int			olen;

    switch( rrc->rrcWhichPnText )
	{
	case RTFidPNTXTA:
	    target= rrc->rrcParagraphNumber.pnTextAfter;
	    break;

	case RTFidPNTXTB:
	    target= rrc->rrcParagraphNumber.pnTextBefore;
	    break;

	default:
	    LDEB(rrc->rrcWhichPnText); return -1;
	}

    olen= strlen( (char *)target );
    if  ( olen+ len >= 32 )
	{ SLDEB((char *)target,len); return -1;	}

    memcpy( target+ olen, text, len );
    target[olen+ len]= '\0';

    return 0;
    }

static int docRtfPndescGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int			res;

    rrc->rrcWhichPnText= rcw->rcwId;

    res= docRtfReadGroup( sis, DOClevANY,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfPndescText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    rrc->rrcWhichPnText= -1;

    return res;
    }


RtfControlWord	docRtfPndescGroups[]=
    {
	{ "pntxta",	RTFidPNTXTA,	DOClevANY, docRtfPndescGroup, },
	{ "pntxtb",	RTFidPNTXTB,	DOClevANY, docRtfPndescGroup, },

	{ 0, 0, 0 }
    };

static int docRtfReadPnseclvl(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int			res;
    int			level= arg- 1;

    if  ( level < 0 )
	{ LDEB(level);	}

    docInitParagraphNumber( &(rrc->rrcParagraphNumber) );

    res= docRtfReadGroup( sis, DOClevANY,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfPndescGroups,
				docRtfRefuseText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res); return res;	}

    rrc->rrcParagraphNumber.pnLevel= level;

    if  ( level >= rrc->rrcParagraphNumberCount )
	{
	ParagraphNumber *	fresh;

	fresh= realloc( rrc->rrcParagraphNumbers,
				( level+ 1 )* sizeof( ParagraphNumber ) );
	if  ( ! fresh )
	    { LXDEB(level,fresh); return -1;	}

	rrc->rrcParagraphNumbers= fresh;

	while( rrc->rrcParagraphNumberCount <= level )
	    {
	    docInitParagraphNumber( rrc->rrcParagraphNumbers+ 
					    rrc->rrcParagraphNumberCount++ );
	    }
	}

    rrc->rrcParagraphNumbers[level]= rrc->rrcParagraphNumber;

    return res;
    }

static RtfControlWord	docRtfPnWords[]=
    {
				/****************************************/
				/*  Bullets and Numbering.		*/
				/*  (No destinations)			*/
				/****************************************/
	{ "pnacross",	RTFidPNACROSS,	DOClevANY, docRtfIgnoreWord, },
	{ "pncf",	RTFidPNCF,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvl",	RTFidPNLVL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlblt",	RTFidPNLVLBLT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlbody",	RTFidPNLVLBODY,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlcont",	RTFidPNLVLCONT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnnumonce",	RTFidPNNUMONCE,	DOClevANY, docRtfIgnoreWord, },
	{ "pnprev",	RTFidPNPREV,	DOClevANY, docRtfIgnoreWord, },
	{ "pnqc",	RTFidPNQC,	DOClevANY, docRtfIgnoreWord, },
	{ "pnql",	RTFidPNQL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnqr",	RTFidPNQR,	DOClevANY, docRtfIgnoreWord, },
	{ "pnrestart",	RTFidPNRESTART,	DOClevANY, docRtfIgnoreWord, },
	{ "pnsp",	RTFidPNSP,	DOClevANY, docRtfIgnoreWord, },

	{ "ls",		RTFidLS,	DOClevANY, docRtfIgnoreWord, },
	{ "ilvl",	RTFidILVL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnrnot",	RTFidPNRNOT,	DOClevANY, docRtfIgnoreWord, },

	{ 0, 0, 0 }
    };

static int docRtfSkipPn(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    docInitParagraphNumber( &(rrc->rrcParagraphNumber) );

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfPnWords, docRtfPndescGroups,
				docRtfRefuseText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Read the result of a bullet.					*/
/*									*/
/*  For one reason or another, Word resets the paragraph properties	*/
/*  inside the result. So save and restore them.			*/
/*									*/
/************************************************************************/

static int docRtfReadPntext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc )
    {
    int				res;
    ParagraphProperties		pp;

    PropertyMask		ppChgMask;
    PropertyMask		ppUpdMask;

    PROPmaskCLEAR( &ppChgMask );

    PROPmaskCLEAR( &ppUpdMask );
    PROPmaskFILL( &ppUpdMask, PPprop_COUNT );

    docInitParagraphProperties( &pp );
    docUpdParaProperties( &ppChgMask, &pp,
				&ppUpdMask, &(rrc->rrcBi->biParaProperties) );

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfEmptyTable,
				docRtfTextParticule );

    PROPmaskCLEAR( &ppChgMask );

    PROPmaskCLEAR( &ppUpdMask );
    PROPmaskFILL( &ppUpdMask, PPprop_COUNT );

    docUpdParaProperties( &ppChgMask, &(rrc->rrcBi->biParaProperties),
							    &ppUpdMask, &pp );
    docCleanParagraphProperties( &pp );

    rrc->rrcJustAfterPntext= 1;
    rrc->rrcPageAfterPntext= 0;

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Read a whole document.						*/
/*									*/
/************************************************************************/

static RtfControlWord	docRtfOutsideGroups[]=
    {
	{ "rtf",	RTFidRTF,	DOClevDOC, docRtfReadDoc,	},
	{ 0, 0, 0 }
    };

BufferDocument * docRtfReadFile(	SimpleInputStream *	sis,
					int			defAnsigpg )
    {
    BufferDocument *		bd;
    RtfReadingContext		rrc;

    int				res;
    const RtfControlWord *	rcw;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				arg= -1;
    int				c;

    const int			listOnly= 1;

    int				noteIndex;
    DocumentNote *		dn;

    bd= (BufferDocument *)malloc( sizeof(BufferDocument) );
    if  ( ! bd )
	{ XDEB(bd); return bd;	}

    docInitDocument( bd );

    docRtfInitReadingContext( &rrc );

    rrc.rrcBd= bd;
    rrc.rrcBi= &(bd->bdItem);
    rrc.rrcLevel= DOClevDOC;
    rrc.rrcSplitLevel= DOClevDOC;

    if  ( defAnsigpg >= 0 )
	{ docRtfReadSetAnsicpg( &rrc, defAnsigpg );	}

    res= docRtfFindControl( sis, &rrc, &c, controlWord, &gotArg, &arg );

    if  ( res != RTFfiCTRLGROUP )
	{ docFreeDocument( bd ); return (BufferDocument *)0; }

    rcw= docRtfFindWord( controlWord, docRtfOutsideGroups, listOnly );
    if  ( ! rcw )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0; }

    res= docRtfApplyControlWord( sis, rcw, gotArg, arg, &rrc );
    if  ( res )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0; }

    docRtfCleanReadingContext( &rrc );

    noteIndex= docGetFirstNoteOfDocument( &dn, bd, DOCinFOOTNOTE );
    if  ( noteIndex >= 0 )
	{
	if  ( docCheckNoteSeparatorItem( bd, DOCinFTNSEP ) )
	    {
	    LDEB(DOCinFTNSEP);
	    docFreeDocument( bd ); return (BufferDocument *)0;
	    }
	}

    noteIndex= docGetFirstNoteOfDocument( &dn, bd, DOCinENDNOTE );
    if  ( noteIndex >= 0 )
	{
	if  ( docCheckNoteSeparatorItem( bd, DOCinAFTNSEP ) )
	    {
	    LDEB(DOCinAFTNSEP);
	    docFreeDocument( bd ); return (BufferDocument *)0;
	    }
	}

    return bd;
    }

/************************************************************************/
/*									*/
/*  Just read a ruler from RTF.						*/
/*									*/
/************************************************************************/
static int docRtfReadRulerGroup(	SimpleInputStream *	sis,
					const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfConsumeGroup( sis, DOClevPARA, rrc,
					    docRtfEmptyTable, NULL, NULL );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static RtfControlWord	docRtfRulerOutsideGroups[]=
    {
	{ "ruler",	RTFidRULER,	DOClevSECT, docRtfReadRulerGroup, },
	{ 0, 0, 0 }
    };

int docRtfReadRuler(	SimpleInputStream *	sis,
			ParagraphProperties *	pp )
    {
    BufferItem			bi;
    RtfReadingContext		rrc;

    int				res;
    const RtfControlWord *	rcw;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				arg= -1;
    int				c;

    const int			listOnly= 1;

    PropertyMask		ppChgMask;
    PropertyMask		ppUpdMask;

    docInitItem( &bi, (BufferItem *)0, (BufferDocument *)0, 0,
						    DOClevCELL, DOCinBODY );
    docRtfInitReadingContext( &rrc );

    rrc.rrcBi= &bi;
    rrc.rrcLevel= bi.biLevel;
    rrc.rrcSplitLevel= bi.biLevel;

    res= docRtfFindControl( sis, &rrc, &c, controlWord, &gotArg, &arg );

    if  ( res != RTFfiCTRLGROUP )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    rcw= docRtfFindWord( controlWord, docRtfRulerOutsideGroups, listOnly );
    if  ( ! rcw )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    res= docRtfApplyControlWord( sis, rcw, gotArg, arg, &rrc );
    if  ( res )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    PROPmaskCLEAR( &ppChgMask );

    PROPmaskCLEAR( &ppUpdMask );
    PROPmaskFILL( &ppUpdMask, PPprop_COUNT );

    docUpdParaProperties( &ppChgMask, pp,
				&ppUpdMask, &(rrc.rrcParagraphProperties) );

    docCleanItem( (BufferDocument *)0, &bi );
    docRtfCleanReadingContext( &rrc );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Remember properties to be used subsequently.			*/
/*									*/
/*  The level should be DOClevANY.					*/
/*									*/
/*  As this is the way Word parses RTF, and this makes an RTF reader	*/
/*  insensitive to dubious RTF, this is the way to go in further	*/
/*  extensions.								*/
/*									*/
/************************************************************************/

/* Not practical
# define rrcBd do not use
*/
# define rrcBi do not use
# define rrcInsertedObject do not use
# define rrcFieldNumber do not use

# define	RTF_REMEMBER(s,id) \
    { s,      id,         DOClevANY,  docRtfRememberProperty, }


# define	RTF_DOCPROP(s,id) \
		{ s, id, DOClevANY, docRtfRememberDocProperty, }
# define	RTF_DOC_ENUM(s,id,v) \
		{ s, id, DOClevANY, docRtfRememberDocProperty, v }


# define	RTF_BORDER(s,id) \
    { s,	id,	DOClevANY,  docRtfRememberProperty, 0, \
				    docRtfBeginBorder, docRtfBorderWords, }


# define	RTF_TEXT_DEFAULT(s) \
		{ s, TAprop_NONE, DOClevANY, docRtfRememberTextProperty, }
# define	RTF_TEXT_VALUE(s,id) \
		{ s, id, DOClevANY, docRtfRememberTextProperty, }
# define	RTF_TEXT_ENUM(s,id,v) \
		{ s, id, DOClevANY, docRtfRememberTextProperty, v, }
# define	RTF_TEXT_FLAG(s,id) \
		{ s, id, DOClevANY, docRtfRememberTextProperty, }


# define	RTF_PARA_DEFAULT(s) \
		{ s, PPprop_NONE, DOClevANY, docRtfRememberParagraphProperty, }
# define	RTF_PARA_VALUE(s,id) \
		{ s, id, DOClevANY, docRtfRememberParagraphProperty, }
# define	RTF_PARA_ENUM(s,id,v) \
		{ s, id, DOClevANY, docRtfRememberParagraphProperty, v, }
# define	RTF_PARA_FLAG(s,id) \
		{ s, id, DOClevANY, docRtfRememberParagraphProperty, }
# define	RTF_PARA_BORDER(s,id) \
		{ s, id, DOClevANY, docRtfRememberParagraphProperty, 0, \
				    docRtfBeginBorder, docRtfBorderWords, }


# define	RTF_SECT_DEFAULT(s) \
		{ s, SPprop_NONE, DOClevANY, docRtfRememberSectionProperty, }
# define	RTF_SECT_VALUE(s,id) \
		{ s, id, DOClevANY, docRtfRememberSectionProperty, }
# define	RTF_SECT_ENUM(s,id,v) \
		{ s, id, DOClevANY, docRtfRememberSectionProperty, v, }
# define	RTF_SECT_FLAG(s,id) \
		{ s, id, DOClevANY, docRtfRememberSectionProperty, }

# define	RTF_FRAME_VALUE(s,id) \
		{ s, id, DOClevANY, docRtfRememberTextFrameProperty, }
# define	RTF_FRAME_ENUM(s,id,v) \
		{ s, id, DOClevANY, docRtfRememberTextFrameProperty, v, }
# define	RTF_FRAME_FLAG(s,id) \
		{ s, id, DOClevANY, docRtfRememberTextFrameProperty, }

RtfControlWord	docRtfPropertyWords[]=
{
				/****************************************/
				/*  Text properties.			*/
				/*  Character Formatting Properties	*/
				/****************************************/
    RTF_TEXT_DEFAULT( "plain" ),

    RTF_TEXT_VALUE( "f",		TApropFONTFAMILY ),
    RTF_TEXT_VALUE( "fs",		TApropFONTSIZE ),

    RTF_TEXT_FLAG( "b",			TApropFONTBOLD ),
    RTF_TEXT_FLAG( "i",			TApropFONTSLANTED ),
    RTF_TEXT_FLAG( "ul",		TApropTEXTUNDERLINED ),

    RTF_TEXT_ENUM( "sub",		TApropSUPERSUB, DOCfontSUBSCRIPT ),
    RTF_TEXT_ENUM( "super",		TApropSUPERSUB, DOCfontSUPERSCRIPT ),
    RTF_TEXT_ENUM( "nosupersub",	TApropSUPERSUB, DOCfontREGULAR ),

    RTF_TEXT_FLAG( "caps",		TApropCAPITALS ),
    RTF_TEXT_FLAG( "scaps",		TApropSMALLCAPS ),
    RTF_TEXT_FLAG( "strike",		TApropSTRIKETHROUGH ),

    /**/

    RTF_REMEMBER( "cf",			RTFidCF ),
    RTF_REMEMBER( "lang",		RTFidLANG ),
    RTF_REMEMBER( "noproof",		RTFidNOPROOF ),
    RTF_REMEMBER( "ulnone",		RTFidULNONE ),
    RTF_REMEMBER( "up",			RTFidUP ),
    RTF_REMEMBER( "cgrid",		RTFidCGRID ),

    RTF_REMEMBER( "cb",			RTFidCB ),
    RTF_REMEMBER( "cchs",		RTFidCCHS ),
    RTF_REMEMBER( "cs",			RTFidCS ),
    RTF_REMEMBER( "deleted",		RTFidDELETED ),
    RTF_REMEMBER( "dn",			RTFidDN ),
    RTF_REMEMBER( "ltrch",		RTFidLTRCH ),
    RTF_REMEMBER( "outl",		RTFidOUTL ),
    RTF_REMEMBER( "revauth",		RTFidREVAUTH ),
    RTF_REMEMBER( "revauthdel",		RTFidREVAUTHDEL ),
    RTF_REMEMBER( "revdttm",		RTFidREVDTTM ),
    RTF_REMEMBER( "revdttmdel",		RTFidREVDTTMDEL ),
    RTF_REMEMBER( "revised",		RTFidREVISED ),
    RTF_REMEMBER( "rtlch",		RTFidRTLCH ),
    RTF_REMEMBER( "shad",		RTFidSHAD ),

    RTF_REMEMBER( "uld",		RTFidULD ),
    RTF_REMEMBER( "uldb",		RTFidULDB ),
    RTF_REMEMBER( "ulw",		RTFidULW ),
    RTF_REMEMBER( "v",			RTFidV ),
    RTF_REMEMBER( "kerning",		RTFidKERNING ),
    RTF_REMEMBER( "expnd",		RTFidEXPND ),
    RTF_REMEMBER( "expndtw",		RTFidEXPNDTW ),
    RTF_REMEMBER( "charscalex",		RTFidCHARSCALEX ),
    RTF_REMEMBER( "highlight",		RTFidHIGHLIGHT ),

    RTF_REMEMBER( "chbghoriz",		RTFidCHBGHORIZ ),
    RTF_REMEMBER( "chbgvert",		RTFidCHBGVERT ),
    RTF_REMEMBER( "chbgfdiag",		RTFidCHBGFDIAG ),
    RTF_REMEMBER( "chbgbdiag",		RTFidCHBGBDIAG ),
    RTF_REMEMBER( "chbgcross",		RTFidCHBGCROSS ),
    RTF_REMEMBER( "chbgdcross",		RTFidCHBGDCROSS ),
    RTF_REMEMBER( "chbgdkhoriz",	RTFidCHBGDKHORIZ ),
    RTF_REMEMBER( "chbgdkvert",		RTFidCHBGDKVERT ),
    RTF_REMEMBER( "chbgdkfdiag",	RTFidCHBGDKFDIAG ),
    RTF_REMEMBER( "chbgdkbdiag",	RTFidCHBGDKBDIAG ),
    RTF_REMEMBER( "chbgdkcross",	RTFidCHBGDKCROSS ),
    RTF_REMEMBER( "chbgdkdcross",	RTFidCHBGDKDCROSS ),

    RTF_BORDER(   "chbrdr",		RTFidCHBRDR ),

    RTF_REMEMBER( "chcbpat",		RTFidCHCBPAT ),
    RTF_REMEMBER( "chcfpat",		RTFidCHCFPAT ),
    RTF_REMEMBER( "chshdng",		RTFidCHSHDNG ),

    RTF_REMEMBER( "faauto",		RTFidFAAUTO ),

				/****************************************/
				/*  Paragraph properties.		*/
				/****************************************/
    RTF_PARA_DEFAULT( "pard" ),

    RTF_PARA_VALUE( "s",		PPpropSTYLE ),

    RTF_PARA_VALUE( "li",		PPpropLEFT_INDENT ),
    RTF_PARA_VALUE( "fi",		PPpropFIRST_INDENT ),
    RTF_PARA_VALUE( "ri",		PPpropRIGHT_INDENT ),

    RTF_PARA_ENUM( "ql",		PPpropALIGNMENT, DOCiaLEFT ),
    RTF_PARA_ENUM( "qr",		PPpropALIGNMENT, DOCiaRIGHT ),
    RTF_PARA_ENUM( "qj",		PPpropALIGNMENT, DOCiaJUSTIFIED ),
    RTF_PARA_ENUM( "qc",		PPpropALIGNMENT, DOCiaCENTERED ),

    RTF_PARA_FLAG( "pagebb",		PPpropNEWPAGE ),

    RTF_PARA_ENUM( "nowidctlpar",	PPpropWIDCTLPAR, 0 ),
    RTF_PARA_ENUM( "widctlpar",		PPpropWIDCTLPAR, 1 ),

    RTF_PARA_FLAG( "keep",		PPpropKEEP ),
    RTF_PARA_FLAG( "keepn",		PPpropKEEPN ),

    RTF_PARA_VALUE( "sb",		PPpropSPACE_BEFORE ),
    RTF_PARA_VALUE( "sa",		PPpropSPACE_AFTER ),
    RTF_PARA_VALUE( "sl",		PPpropLINE_SPACING_DIST ),

    RTF_PARA_FLAG( "slmult",		PPpropLINE_SPACING_MULT ),

    RTF_PARA_VALUE( "outlinelevel",	PPpropOUTLINELEVEL ),

    RTF_PARA_FLAG( "hyphpar",		PPpropHYPHPAR ),

    RTF_PARA_VALUE( "shading",		PPpropSHADING_LEVEL ),

    RTF_PARA_ENUM( "bgbdiag",		PPpropSHADING_PATTERN, DOCspBGBDIAG ),
    RTF_PARA_ENUM( "bgcross",		PPpropSHADING_PATTERN, DOCspBGCROSS ),
    RTF_PARA_ENUM( "bgdcross",		PPpropSHADING_PATTERN, DOCspBGDCROSS ),
    RTF_PARA_ENUM( "bgdkbdiag",		PPpropSHADING_PATTERN, DOCspBGDKBDIAG ),
    RTF_PARA_ENUM( "bgdkcross",		PPpropSHADING_PATTERN, DOCspBGDKCROSS ),
    RTF_PARA_ENUM( "bgdkdcross",	PPpropSHADING_PATTERN, DOCspBGDKDCROSS),
    RTF_PARA_ENUM( "bgdkfdiag",		PPpropSHADING_PATTERN, DOCspBGDKFDIAG ),
    RTF_PARA_ENUM( "bgdkhoriz",		PPpropSHADING_PATTERN, DOCspBGDKHORIZ ),
    RTF_PARA_ENUM( "bgdkvert",		PPpropSHADING_PATTERN, DOCspBGDKVERT ),
    RTF_PARA_ENUM( "bgfdiag",		PPpropSHADING_PATTERN, DOCspBGFDIAG ),
    RTF_PARA_ENUM( "bghoriz",		PPpropSHADING_PATTERN, DOCspBGHORIZ ),
    RTF_PARA_ENUM( "bgvert",		PPpropSHADING_PATTERN, DOCspBGVERT ),

    RTF_PARA_VALUE( "cfpat",		PPpropSHADING_FOREGROUND ),
    RTF_PARA_VALUE( "cbpat",		PPpropSHADING_BACKGROUND ),

    RTF_PARA_BORDER( "brdrt",		PPpropTOP_BORDER ),
    RTF_PARA_BORDER( "brdrb",		PPpropBOTTOM_BORDER ),

					/**/

    RTF_REMEMBER( "intbl",		RTFidINTBL ),

    RTF_REMEMBER( "adjustright",	RTFidADJUSTRIGHT ),
    RTF_REMEMBER( "adjustleft",		RTFidADJUSTLEFT ),

    RTF_REMEMBER( "sbauto",		RTFidSBAUTO ),
    RTF_REMEMBER( "saauto",		RTFidSAAUTO ),

    RTF_REMEMBER( "tql",		RTFidTQL ), /*?*/

    RTF_REMEMBER( "jclisttab",		RTFidJCLISTTAB ),
    RTF_REMEMBER( "nooverflow",		RTFidNOOVERFLOW ),
    RTF_REMEMBER( "faroman",		RTFidFAROMAN ),

    RTF_REMEMBER( "aspalpha",		RTFidASPALPHA ),
    RTF_REMEMBER( "aspnum",		RTFidASPNUM ),

				/****************************************/
				/*  Paragraph Borders.			*/
				/****************************************/
    RTF_BORDER( "brdrl",		RTFidBRDRL ),
    RTF_BORDER( "brdrr",		RTFidBRDRR ),
    RTF_BORDER( "brdrbtw",		RTFidBRDRBTW ),
    RTF_BORDER( "brdrbar",		RTFidBRDRBAR ),
    RTF_BORDER( "box",			RTFidBOX ),

				/****************************************/
				/*  Positioned Objects and Frames:	*/
				/*  ( Paragraph properties. )		*/
				/****************************************/

    RTF_FRAME_VALUE( "absw",		TFPpropABSW ),
    RTF_FRAME_VALUE( "absh",		TFPpropABSH ),

    RTF_FRAME_ENUM( "phmrg",		TFPpropHORIZONTAL_REFERENCE, HFRphMRG ),
    RTF_FRAME_ENUM( "phpg",		TFPpropHORIZONTAL_REFERENCE, HFRphPG ),
    RTF_FRAME_ENUM( "phcol",		TFPpropHORIZONTAL_REFERENCE, HFRphCOL ),

    RTF_FRAME_VALUE( "posx",		TFPpropHORIZONTAL_POSITION ),
    RTF_FRAME_VALUE( "posnegx",		TFPpropHORIZONTAL_POSITION ),

    RTF_FRAME_ENUM( "posxl",		TFPpropHORIZONTAL_ALIGNMENT, HFAposXL ),
    RTF_FRAME_ENUM( "posxr",		TFPpropHORIZONTAL_ALIGNMENT, HFAposXR ),
    RTF_FRAME_ENUM( "posxc",		TFPpropHORIZONTAL_ALIGNMENT, HFAposXC ),
    RTF_FRAME_ENUM( "posxi",		TFPpropHORIZONTAL_ALIGNMENT, HFAposXI ),
    RTF_FRAME_ENUM( "posxo",		TFPpropHORIZONTAL_ALIGNMENT, HFAposXO ),

    RTF_FRAME_ENUM( "pvmrg",		TFPpropVERTICAL_REFERENCE, VFRpvMRG ),
    RTF_FRAME_ENUM( "pvpara",		TFPpropVERTICAL_REFERENCE, VFRpvPARA ),
    RTF_FRAME_ENUM( "pvpg",		TFPpropVERTICAL_REFERENCE, VFRpvPG ),

    RTF_FRAME_VALUE( "posy",		TFPpropVERTICAL_POSITION ),
    RTF_FRAME_VALUE( "posnegy",		TFPpropVERTICAL_POSITION ),

    RTF_FRAME_FLAG( "abslock",		TFPpropLOCKED_TO_PARA ),

    RTF_FRAME_ENUM( "posyb",		TFPpropVERTICAL_ALIGNMENT, VFAposYB ),
    RTF_FRAME_ENUM( "posyc",		TFPpropVERTICAL_ALIGNMENT, VFAposYC ),
    RTF_FRAME_ENUM( "posyil",		TFPpropVERTICAL_ALIGNMENT, VFAposYIL ),
    RTF_FRAME_ENUM( "posyt",		TFPpropVERTICAL_ALIGNMENT, VFAposYT ),

    RTF_FRAME_ENUM( "nowrap",		TFPpropWRAP_STYLE, FWSwrapNOWRAP ),
    RTF_FRAME_ENUM( "overlay",		TFPpropWRAP_STYLE, FWSwrapOVERLAY ),

    RTF_FRAME_VALUE( "dxfrtext",	TFPpropDXFRTEXT ),
    RTF_FRAME_VALUE( "dfrmtxtx",	TFPpropDFRMTXTX ),
    RTF_FRAME_VALUE( "dfrmtxty",	TFPpropDFRMTXTY ),

    /**/
    RTF_REMEMBER( "dropcapli",		RTFidDROPCAPLI ),
    RTF_REMEMBER( "dropcapt",		RTFidDROPCAPT ),
				/****************************************/
				/*  Bullets and Numbering.		*/
				/*  (No destinations)			*/
				/****************************************/
    RTF_REMEMBER( "ls",			RTFidLS ),
    RTF_REMEMBER( "ilvl",		RTFidILVL ),
				/****************************************/
				/*  Revision marks for paragraph	*/
				/*  numbers.				*/
				/****************************************/
    RTF_REMEMBER( "pnrauth",		RTFidPNRAUTH ),
    RTF_REMEMBER( "pnrdate",		RTFidPNRDATE ),
    RTF_REMEMBER( "pnrnfc",		RTFidPNRNFC ),
    RTF_REMEMBER( "pnrpnbr",		RTFidPNRPNBR ),
    RTF_REMEMBER( "pnrrgb",		RTFidPNRRGB ),
    RTF_REMEMBER( "pnrstart",		RTFidPNRSTART ),
    RTF_REMEMBER( "pnrstop",		RTFidPNRSTOP ),
    RTF_REMEMBER( "pnrxst",		RTFidPNRXST ),
				/****************************************/
				/*  Tabs.				*/
				/****************************************/
    RTF_REMEMBER( "tx",			RTFidTX ),
    RTF_REMEMBER( "tb",			RTFidTB ),
    RTF_REMEMBER( "tqc",		RTFidTQC ),
    RTF_REMEMBER( "tqdec",		RTFidTQDEC ),
    RTF_REMEMBER( "tqr",		RTFidTQR ),

    RTF_REMEMBER( "tldot",		RTFidTLDOT ),
    RTF_REMEMBER( "tleq",		RTFidTLEQ ),
    RTF_REMEMBER( "tlhyph",		RTFidTLHYPH ),
    RTF_REMEMBER( "tlth",		RTFidTLTH ),
    RTF_REMEMBER( "tlul",		RTFidTLUL ),
				/****************************************/
				/*  Associated Character Properties	*/
				/****************************************/
    RTF_REMEMBER( "ab",			RTFidAB ),
    RTF_REMEMBER( "acaps",		RTFidACAPS ),
    RTF_REMEMBER( "acf",		RTFidACF ),
    RTF_REMEMBER( "adn",		RTFidADN ),
    RTF_REMEMBER( "aexpnd",		RTFidAEXPND ),
    RTF_REMEMBER( "af",			RTFidAF ),
    RTF_REMEMBER( "afs",		RTFidAFS ),
    RTF_REMEMBER( "ai",			RTFidAI ),
    RTF_REMEMBER( "alang",		RTFidALANG ),
    RTF_REMEMBER( "aoutl",		RTFidAOUTL ),
    RTF_REMEMBER( "ascaps",		RTFidASCAPS ),
    RTF_REMEMBER( "ashad",		RTFidASHAD ),
    RTF_REMEMBER( "astrike",		RTFidASTRIKE ),
    RTF_REMEMBER( "aul",		RTFidAUL ),
    RTF_REMEMBER( "auld",		RTFidAULD ),
    RTF_REMEMBER( "auldb",		RTFidAULDB ),
    RTF_REMEMBER( "aulnone",		RTFidAULNONE ),
    RTF_REMEMBER( "aulw",		RTFidAULW ),
    RTF_REMEMBER( "aup",		RTFidAUP ),
    RTF_REMEMBER( "loch",		RTFidLOCH ),
    RTF_REMEMBER( "hich",		RTFidHICH ),
    RTF_REMEMBER( "dbch",		RTFidDBCH ),
				/****************************************/
				/*  Row, Cell/Column properties.	*/
				/****************************************/
    RTF_REMEMBER( "trowd",		RTFidTROWD ),

    RTF_REMEMBER( "ltrrow",		RTFidLTRROW ),
    RTF_REMEMBER( "rtlrow",		RTFidRTLROW ),

    RTF_REMEMBER( "trgaph",		RTFidTRGAPH ),
    RTF_REMEMBER( "trhdr",		RTFidTRHDR ),
    RTF_REMEMBER( "trkeep",		RTFidTRKEEP ),
    RTF_REMEMBER( "trleft",		RTFidTRLEFT ),
    RTF_REMEMBER( "trqc",		RTFidTRQC ),
    RTF_REMEMBER( "trql",		RTFidTRQL ),
    RTF_REMEMBER( "trqr",		RTFidTRQR ),
    RTF_REMEMBER( "trrh",		RTFidTRRH ),

    RTF_REMEMBER( "trpaddl",		RTFidTRPADDL ),
    RTF_REMEMBER( "trpaddr",		RTFidTRPADDR ),
    RTF_REMEMBER( "trpaddfl",		RTFidTRPADDFL ),
    RTF_REMEMBER( "trpaddfr",		RTFidTRPADDFR ),
    RTF_REMEMBER( "trwWidth",		RTFidTRWWIDTH ),
    RTF_REMEMBER( "trftsWidth",		RTFidTRFTSWIDTH ),

    RTF_REMEMBER( "cellx",		RTFidCELLX ),

    RTF_REMEMBER( "celld",		RTFidCELLD ),
    RTF_REMEMBER( "cltxlrtb",		RTFidCLTXLRTB ),
    RTF_REMEMBER( "cltxtbrl",		RTFidCLTXTBRL ),
    RTF_REMEMBER( "clvertalt",		RTFidCLVERTALT ),
    RTF_REMEMBER( "clvertalc",		RTFidCLVERTALC ),
    RTF_REMEMBER( "clvertalb",		RTFidCLVERTALB ),
    RTF_REMEMBER( "clbgbdiag",		RTFidCLBGBDIAG ),
    RTF_REMEMBER( "clbgcross",		RTFidCLBGCROSS ),
    RTF_REMEMBER( "clbgdcross",		RTFidCLBGDCROSS ),
    RTF_REMEMBER( "clbgdkbdiag",	RTFidCLBGDKBDIAG ),
    RTF_REMEMBER( "clbgdkcross",	RTFidCLBGDKCROSS ),
    RTF_REMEMBER( "clbgdkdcross",	RTFidCLBGDKDCROSS ),
    RTF_REMEMBER( "clbgdkfdiag",	RTFidCLBGDKFDIAG ),
    RTF_REMEMBER( "clbgdkhor",		RTFidCLBGDKHOR ),
    RTF_REMEMBER( "clbgdkvert",		RTFidCLBGDKVERT ),
    RTF_REMEMBER( "clbgfdiag",		RTFidCLBGFDIAG ),
    RTF_REMEMBER( "clbghoriz",		RTFidCLBGHORIZ ),
    RTF_REMEMBER( "clbgvert",		RTFidCLBGVERT ),
    RTF_REMEMBER( "clmgf",		RTFidCLMGF ),
    RTF_REMEMBER( "clmrg",		RTFidCLMRG ),
    RTF_REMEMBER( "clvmgf",		RTFidCLVMGF ),
    RTF_REMEMBER( "clvmrg",		RTFidCLVMRG ),
    RTF_REMEMBER( "clcbpat",		RTFidCLCBPAT ),
    RTF_REMEMBER( "clcfpat",		RTFidCLCFPAT ),
    RTF_REMEMBER( "clshdng",		RTFidCLSHDNG ),

    RTF_REMEMBER( "clwWidth",		RTFidCLWWIDTH ),
    RTF_REMEMBER( "clftsWidth",		RTFidCLFTSWIDTH ),
				/****************************************/
				/*  Row/Cell borders.			*/
				/****************************************/
    RTF_BORDER( "trbrdrb",		RTFidTRBRDRB ),
    RTF_BORDER( "trbrdrh",		RTFidTRBRDRH ),
    RTF_BORDER( "trbrdrl",		RTFidTRBRDRL ),
    RTF_BORDER( "trbrdrr",		RTFidTRBRDRR ),
    RTF_BORDER( "trbrdrt",		RTFidTRBRDRT ),
    RTF_BORDER( "trbrdrv",		RTFidTRBRDRV ),

    RTF_BORDER( "clbrdrb",		RTFidCLBRDRB ),
    RTF_BORDER( "clbrdrl",		RTFidCLBRDRL ),
    RTF_BORDER( "clbrdrr",		RTFidCLBRDRR ),
    RTF_BORDER( "clbrdrt",		RTFidCLBRDRT ),
				/****************************************/
				/*  Document formatting.		*/
				/****************************************/
    RTF_REMEMBER( "pgbrdrhead",		RTFidPGBRDRHEAD ),
    RTF_REMEMBER( "pgbrdrfoot",		RTFidPGBRDRFOOT ),
    RTF_REMEMBER( "pgbrdrt",		RTFidPGBRDRT ),
    RTF_REMEMBER( "pgbrdrb",		RTFidPGBRDRB ),
    RTF_REMEMBER( "pgbrdrl",		RTFidPGBRDRL ),
    RTF_REMEMBER( "pgbrdrr",		RTFidPGBRDRR ),
    RTF_REMEMBER( "brdrart",		RTFidBRDRART ),
    RTF_REMEMBER( "pgbrdropt",		RTFidPGBRDROPT ),
    RTF_REMEMBER( "pgbrdrsnap",		RTFidPGBRDRSNAP ),

    RTF_DOCPROP( "margl",		DGpropLEFT_MARGIN ),
    RTF_DOCPROP( "margr",		DGpropRIGHT_MARGIN ),
    RTF_DOCPROP( "margt",		DGpropTOP_MARGIN ),
    RTF_DOCPROP( "margb",		DGpropBOTTOM_MARGIN ),
    RTF_DOCPROP( "paperh",		DGpropPAGE_HEIGHT ),
    RTF_DOCPROP( "paperw",		DGpropPAGE_WIDTH ),
    RTF_DOCPROP( "gutter",		DGpropGUTTER ),

    RTF_DOCPROP( "widowctrl",		DPpropWIDOWCTRL ),

    RTF_REMEMBER( "psz",		RTFidPSZ ),

    RTF_REMEMBER( "jcompress",		RTFidJCOMPRESS),
    RTF_REMEMBER( "jexpand",		RTFidJEXPAND),

    RTF_REMEMBER( "horzdoc",		RTFidHORZDOC),
    RTF_REMEMBER( "dghspace",		RTFidDGHSPACE),
    RTF_REMEMBER( "dgvspace",		RTFidDGVSPACE),
    RTF_REMEMBER( "dghorigin",		RTFidDGHORIGIN),
    RTF_REMEMBER( "dgvorigin",		RTFidDGVORIGIN),
    RTF_REMEMBER( "dghshow",		RTFidDGHSHOW),
    RTF_REMEMBER( "dgvshow",		RTFidDGVSHOW),
    RTF_REMEMBER( "dgmargin",		RTFidDGMARGIN),

    RTF_DOCPROP( "pgnstart",		DPpropSTART_PAGE ),
    RTF_DOCPROP( "deff",		DPpropDEFF ),
    RTF_DOCPROP( "facingp",		DPpropFACING_PAGES ),
    RTF_DOCPROP( "deftab",		DPpropDEFTAB ),

    RTF_REMEMBER( "truncex",		RTFidTRUNCEX ),
    RTF_REMEMBER( "nolead",		RTFidNOLEAD ),
    RTF_REMEMBER( "msmcap",		RTFidMSMCAP ),
    RTF_REMEMBER( "splytwnine",		RTFidSPLYTWNINE ),
    RTF_REMEMBER( "ftnlytwnine",	RTFidFTNLYTWNINE ),
    RTF_REMEMBER( "htmautsp",		RTFidHTMAUTSP ),
    RTF_REMEMBER( "useltbaln",		RTFidUSELTBALN ),
    RTF_REMEMBER( "alntblind",		RTFidALNTBLIND ),
    RTF_REMEMBER( "lytcalctblwd",	RTFidLYTCALCTBLWD ),
    RTF_REMEMBER( "lyttblrtgr",		RTFidLYTTBLRTGR ),
    RTF_REMEMBER( "lnbrkrule",		RTFidLNBRKRULE ),

    RTF_REMEMBER( "bdrrlswsix",		RTFidBDRRLSWSIX ),
    RTF_REMEMBER( "oldas",		RTFidOLDAS ),

    RTF_REMEMBER( "sprstsm",		RTFidSPRSTSM ), /* ignore */

				/****************************************/
				/*  Document properties.		*/
				/****************************************/
    RTF_REMEMBER( "noxlattoyen",	RTFidNOXLATTOYEN ),
    RTF_REMEMBER( "expshrtn",		RTFidEXPSHRTN ),
    RTF_REMEMBER( "noultrlspc",		RTFidNOULTRLSPC ),
    RTF_REMEMBER( "dntblnsbdb",		RTFidDNTBLNSBDB ),
    RTF_REMEMBER( "nospaceforul",	RTFidNOSPACEFORUL ),

				/****************************************/
				/*  Paragraph Numbering.		*/
				/****************************************/
    RTF_REMEMBER( "pncard",		RTFidPNCARD ),
    RTF_REMEMBER( "pndec",		RTFidPNDEC ),
    RTF_REMEMBER( "pnucltr",		RTFidPNUCLTR ),
    RTF_REMEMBER( "pnucrm",		RTFidPNUCRM ),
    RTF_REMEMBER( "pnlcltr",		RTFidPNLCLTR ),
    RTF_REMEMBER( "pnlcrm",		RTFidPNLCRM ),
    RTF_REMEMBER( "pnord",		RTFidPNORD ),
    RTF_REMEMBER( "pnordt",		RTFidPNORDT ),

    RTF_REMEMBER( "pnstart",		RTFidPNSTART ),
    RTF_REMEMBER( "pnindent",		RTFidPNINDENT ),
    RTF_REMEMBER( "pnhang",		RTFidPNHANG ),

    RTF_REMEMBER( "pnf",		RTFidPNF ),
    RTF_REMEMBER( "pnfs",		RTFidPNFS ),
    RTF_REMEMBER( "pnb",		RTFidPNB ),
    RTF_REMEMBER( "pni",		RTFidPNI ),
    RTF_REMEMBER( "pncaps",		RTFidPNCAPS ),
    RTF_REMEMBER( "pnscaps",		RTFidPNSCAPS ),
    RTF_REMEMBER( "pnul",		RTFidPNUL ),
    RTF_REMEMBER( "pnuld",		RTFidPNULD ),
    RTF_REMEMBER( "pnuldb",		RTFidPNULDB ),
    RTF_REMEMBER( "pnulnone",		RTFidPNULNONE ),
    RTF_REMEMBER( "pnstrike",		RTFidPNSTRIKE ),

				/****************************************/
				/*  Section properties.			*/
				/****************************************/
    RTF_SECT_DEFAULT( "sectd" ),

    RTF_SECT_VALUE( "headery",		DGpropHEADER_POSITION ),
    RTF_SECT_VALUE( "footery",		DGpropFOOTER_POSITION ),
    RTF_SECT_VALUE( "pgwsxn",		DGpropPAGE_WIDTH ),
    RTF_SECT_VALUE( "pghsxn",		DGpropPAGE_HEIGHT ),
    RTF_SECT_VALUE( "marglsxn",		DGpropLEFT_MARGIN ),
    RTF_SECT_VALUE( "margtsxn",		DGpropTOP_MARGIN ),
    RTF_SECT_VALUE( "margrsxn",		DGpropRIGHT_MARGIN ),
    RTF_SECT_VALUE( "margbsxn",		DGpropBOTTOM_MARGIN ),
    RTF_SECT_VALUE( "guttersxn",	DGpropGUTTER ),

    RTF_SECT_VALUE( "ds",		SPpropSTYLE ),

    RTF_SECT_FLAG( "titlepg",		SPpropTITLEPG ),

    RTF_SECT_ENUM( "sbknone",		SPpropBREAK_KIND, DOCsbkNONE ),
    RTF_SECT_ENUM( "sbkcol",		SPpropBREAK_KIND, DOCsbkCOL ),
    RTF_SECT_ENUM( "sbkpage",		SPpropBREAK_KIND, DOCsbkPAGE ),
    RTF_SECT_ENUM( "sbkeven",		SPpropBREAK_KIND, DOCsbkEVEN ),
    RTF_SECT_ENUM( "sbkodd",		SPpropBREAK_KIND, DOCsbkODD ),

    RTF_SECT_ENUM( "pgndec",		SPpropNUMBER_STYLE, DOCpgnDEC ),
    RTF_SECT_ENUM( "pgnucrm",		SPpropNUMBER_STYLE, DOCpgnUCRM ),
    RTF_SECT_ENUM( "pgnlcrm",		SPpropNUMBER_STYLE, DOCpgnLCRM ),
    RTF_SECT_ENUM( "pgnucltr",		SPpropNUMBER_STYLE, DOCpgnUCLTR ),
    RTF_SECT_ENUM( "pgnlcltr",		SPpropNUMBER_STYLE, DOCpgnLCLTR ),

    RTF_SECT_ENUM( "pgncont",		SPpropPAGE_RESTART, 0 ),
    RTF_SECT_ENUM( "pgnrestart",	SPpropPAGE_RESTART, 1 ),

    RTF_SECT_VALUE( "pgnstarts",	SPpropSTART_PAGE ),

    RTF_SECT_VALUE( "cols",		SPpropCOLUMN_COUNT ),
    RTF_SECT_VALUE( "colsx",		SPpropCOLUMN_SPACING ),
    RTF_SECT_VALUE( "colw",		SPpropCOLUMN_WIDTH ),

    /**/

    RTF_REMEMBER( "sectdefaultcl",	RTFidSECTDEFAULTCL ),

    RTF_REMEMBER( "pgnhn",		RTFidPGNHN ),
    RTF_REMEMBER( "pgnhnsh",		RTFidPGNHNSH ),
    RTF_REMEMBER( "pgnhnsp",		RTFidPGNHNSP ),
    RTF_REMEMBER( "pgnhnsc",		RTFidPGNHNSC ),
    RTF_REMEMBER( "pgnhnsm",		RTFidPGNHNSM ),
    RTF_REMEMBER( "pgnhnsn",		RTFidPGNHNSN ),

    RTF_REMEMBER( "pgnx",		RTFidPGNX ),
    RTF_REMEMBER( "pgny",		RTFidPGNY ),

    RTF_REMEMBER( "linex",		RTFidLINEX ),

    RTF_REMEMBER( "colno",		RTFidCOLNO ),
    RTF_REMEMBER( "colsr",		RTFidCOLSR ),

    RTF_REMEMBER( "vertalb",		RTFidVERTALB ),
    RTF_REMEMBER( "vertalc",		RTFidVERTALC ),
    RTF_REMEMBER( "vertalj",		RTFidVERTALJ ),
    RTF_REMEMBER( "vertalt",		RTFidVERTALT ),

    RTF_REMEMBER( "sectlinegrid",	RTFidSECTLINEGRID ),

    RTF_REMEMBER( "endnhere",		RTFidENDNHERE ),
    RTF_REMEMBER( "binfsxn",		RTFidBINFSXN ),
    RTF_REMEMBER( "binsxn",		RTFidBINSXN ),
    RTF_REMEMBER( "linebetcol",		RTFidLINEBETCOL ),
    RTF_REMEMBER( "linecont",		RTFidLINECONT ),
    RTF_REMEMBER( "linemod",		RTFidLINEMOD ),
    RTF_REMEMBER( "lineppage",		RTFidLINEPPAGE ),
    RTF_REMEMBER( "linerestart",	RTFidLINERESTART ),
    RTF_REMEMBER( "linestarts",		RTFidLINESTARTS ),
    RTF_REMEMBER( "lndscpsxn",		RTFidLNDSCPSXN ),
    RTF_REMEMBER( "ltrsect",		RTFidLTRSECT ),
    RTF_REMEMBER( "rtlsect",		RTFidRTLSECT ),
    RTF_REMEMBER( "sectunlocked",	RTFidSECTUNLOCKED ),

				/****************************************/
				/*  Document properties: (Footnotes)	*/
				/****************************************/
    RTF_DOCPROP( "fet",			DPpropNOTETYPES ),

    RTF_DOC_ENUM( "endnotes",		DPpropFOOTNOTE_POSITION,
						    DPftnPOS_SECT_END ),
    RTF_DOC_ENUM( "enddoc",		DPpropFOOTNOTE_POSITION,
						    DPftnPOS_DOC_END ),
    RTF_DOC_ENUM( "ftntj",		DPpropFOOTNOTE_POSITION,
						    DPftnPOS_BELOW_TEXT ),
    RTF_DOC_ENUM( "ftnbj",		DPpropFOOTNOTE_POSITION,
						    DPftnPOS_PAGE_BOTTOM ),

    RTF_DOC_ENUM( "aendnotes",		DPpropENDNOTE_POSITION,
						    DPftnPOS_SECT_END ),
    RTF_DOC_ENUM( "aenddoc",		DPpropENDNOTE_POSITION,
						    DPftnPOS_DOC_END ),
    RTF_DOC_ENUM( "aftntj",		DPpropENDNOTE_POSITION,
						    DPftnPOS_BELOW_TEXT ),
    RTF_DOC_ENUM( "aftnbj",		DPpropENDNOTE_POSITION,
						    DPftnPOS_PAGE_BOTTOM ),

    RTF_DOCPROP( "ftnstart",		DPpropFOOTNOTE_STARTNR ),
    RTF_DOCPROP( "aftnstart",		DPpropENDNOTE_STARTNR ),

    RTF_DOC_ENUM( "ftnrstcont",		DPpropFOOTNOTE_RESTART,
						    DPftnRST_CONTINUOUS ),
    RTF_DOC_ENUM( "ftnrestart",		DPpropFOOTNOTE_RESTART,
						    DPftnRST_PER_SECTION ),
    RTF_DOC_ENUM( "ftnrstpg",		DPpropFOOTNOTE_RESTART,
						    DPftnRST_PER_PAGE ),

    RTF_DOC_ENUM( "aftnrstcont",	DPpropENDNOTE_RESTART,
						    DPftnRST_CONTINUOUS ),
    RTF_DOC_ENUM( "aftnrestart",	DPpropENDNOTE_RESTART,
						    DPftnRST_PER_SECTION ),
    RTF_DOC_ENUM( "aftnrstpg",		DPpropENDNOTE_RESTART,
						    DPftnRST_PER_PAGE ),

    RTF_DOC_ENUM( "ftnnar",		DPpropFOOTNOTE_STYLE, DPftnNAR ),
    RTF_DOC_ENUM( "ftnnalc",		DPpropFOOTNOTE_STYLE, DPftnNALC ),
    RTF_DOC_ENUM( "ftnnauc",		DPpropFOOTNOTE_STYLE, DPftnNAUC ),
    RTF_DOC_ENUM( "ftnnrlc",		DPpropFOOTNOTE_STYLE, DPftnNRLC ),
    RTF_DOC_ENUM( "ftnnruc",		DPpropFOOTNOTE_STYLE, DPftnNRUC ),
    RTF_DOC_ENUM( "ftnnchi",		DPpropFOOTNOTE_STYLE, DPftnNCHI ),

    RTF_DOC_ENUM( "aftnnar",		DPpropENDNOTE_STYLE, DPftnNAR ),
    RTF_DOC_ENUM( "aftnnalc",		DPpropENDNOTE_STYLE, DPftnNALC ),
    RTF_DOC_ENUM( "aftnnauc",		DPpropENDNOTE_STYLE, DPftnNAUC ),
    RTF_DOC_ENUM( "aftnnrlc",		DPpropENDNOTE_STYLE, DPftnNRLC ),
    RTF_DOC_ENUM( "aftnnruc",		DPpropENDNOTE_STYLE, DPftnNRUC ),
    RTF_DOC_ENUM( "aftnnchi",		DPpropENDNOTE_STYLE, DPftnNCHI ),

    RTF_REMEMBER( "ftnalt",		RTFidFTNALT ),
				/****************************************/
				/*  Encoding.				*/
				/****************************************/
    RTF_REMEMBER( "ansi",		RTFidANSI ),
    RTF_REMEMBER( "mac",		RTFidMAC ),
    RTF_REMEMBER( "pc",			RTFidPC ),
    RTF_REMEMBER( "pca",		RTFidPCA ),
    RTF_DOCPROP( "ansicpg",		DPpropANSICPG ),

				/****************************************/
				/*  Unknown				*/
				/****************************************/
    RTF_REMEMBER( "itap",		RTFidITAP ),
    RTF_REMEMBER( "lin",		RTFidLIN ),
    RTF_REMEMBER( "rin",		RTFidRIN ),
    RTF_REMEMBER( "langnp",		RTFidLANGNP ),
    RTF_REMEMBER( "langfe",		RTFidLANGFE ),
    RTF_REMEMBER( "langfenp",		RTFidLANGFENP ),
    RTF_REMEMBER( "nolnhtadjtbl",	RTFidNOLNHTADJTBL ),

				/****************************************/
				/*  Document Formatting properties.	*/
				/****************************************/
    RTF_REMEMBER( "allprot",		RTFidALLPROT ),
    RTF_REMEMBER( "annotprot",		RTFidANNOTPROT ),
    RTF_REMEMBER( "brkfrm",		RTFidBRKFRM ),
    RTF_REMEMBER( "cvmme",		RTFidCVMME ),
    RTF_REMEMBER( "defformat",		RTFidDEFFORMAT ),
    RTF_REMEMBER( "deflang",		RTFidDEFLANG ),
    RTF_REMEMBER( "doctemp",		RTFidDOCTEMP ),
    RTF_REMEMBER( "fldalt",		RTFidFLDALT ),
    RTF_REMEMBER( "formdisp",		RTFidFORMDISP ),
    RTF_REMEMBER( "formprot",		RTFidFORMPROT ),
    RTF_REMEMBER( "formshade",		RTFidFORMSHADE ),
    RTF_REMEMBER( "fracwidth",		RTFidFRACWIDTH ),
    RTF_REMEMBER( "hyphauto",		RTFidHYPHAUTO ),
    RTF_REMEMBER( "hyphcaps",		RTFidHYPHCAPS ),
    RTF_REMEMBER( "hyphconsec",		RTFidHYPHCONSEC ),
    RTF_REMEMBER( "hyphhotz",		RTFidHYPHHOTZ ),
    RTF_REMEMBER( "landscape",		RTFidLANDSCAPE ),
    RTF_REMEMBER( "linestart",		RTFidLINESTART ),
    RTF_REMEMBER( "linkstyles",		RTFidLINKSTYLES ),
    RTF_REMEMBER( "ltrdoc",		RTFidLTRDOC ),
    RTF_REMEMBER( "makebackup",		RTFidMAKEBACKUP ),
    RTF_REMEMBER( "margmirror",		RTFidMARGMIRROR ),
    RTF_REMEMBER( "nocolbal",		RTFidNOCOLBAL ),
    RTF_REMEMBER( "noextrasprl",	RTFidNOEXTRASPRL ),
    RTF_REMEMBER( "notabind",		RTFidNOTABIND ),
    RTF_REMEMBER( "otblrul",		RTFidOTBLRUL ),
    RTF_REMEMBER( "prcolbl",		RTFidPRCOLBL ),
    RTF_REMEMBER( "printdata",		RTFidPRINTDATA ),
    RTF_REMEMBER( "psover",		RTFidPSOVER ),
    RTF_REMEMBER( "revbar",		RTFidREVBAR ),
    RTF_REMEMBER( "revisions",		RTFidREVISIONS ),
    RTF_REMEMBER( "revprop",		RTFidREVPROP ),
    RTF_REMEMBER( "revprot",		RTFidREVPROT ),
    RTF_REMEMBER( "rtldoc",		RTFidRTLDOC ),
    RTF_REMEMBER( "sprslnsp",		RTFidSPRSLNSP ),
    RTF_REMEMBER( "sprsspbf",		RTFidSPRSSPBF ),
    RTF_REMEMBER( "sprstsp",		RTFidSPRSTSP ),
    RTF_REMEMBER( "subfontbysize",	RTFidSUBFONTBYSIZE ),
    RTF_REMEMBER( "swpbdr",		RTFidSWPBDR ),
    RTF_REMEMBER( "transmf",		RTFidTRANSMF ),
    RTF_REMEMBER( "ight",		RTFidIGHT ),
    RTF_REMEMBER( "wraptrsp",		RTFidWRAPTRSP ),
    RTF_REMEMBER( "viewzk",		RTFidVIEWZK ),
    RTF_REMEMBER( "viewkind",		RTFidVIEWKIND ),
    RTF_REMEMBER( "viewscale",		RTFidVIEWSCALE ),
    RTF_REMEMBER( "pgbrdrhead",		RTFidPGBRDRHEAD ),
    RTF_REMEMBER( "pgbrdrfoot",		RTFidPGBRDRFOOT ),
    RTF_REMEMBER( "lytprtmet",		RTFidLYTPRTMET ),
    RTF_REMEMBER( "ogutter",		RTFidOGUTTER ),
    RTF_REMEMBER( "deflangfe",		RTFidDEFLANGFE ),
    RTF_REMEMBER( "uc",			RTFidUC ),
    RTF_REMEMBER( "level",		RTFidLEVEL ),
    RTF_REMEMBER( "ltrpar",		RTFidLTRPAR ),
    RTF_REMEMBER( "noline",		RTFidNOLINE ),
    RTF_REMEMBER( "rtlpar",		RTFidRTLPAR ),
    RTF_REMEMBER( "sbys",		RTFidSBYS ),
    RTF_REMEMBER( "subdocument",	RTFidSUBDOCUMENT ),
				/****************************************/
				/*  Fields in dates and times.		*/
				/****************************************/
    RTF_REMEMBER( "hr",			RTFidHR ),
    RTF_REMEMBER( "min",		RTFidMIN ),
    RTF_REMEMBER( "sec",		RTFidSEC ),
    RTF_REMEMBER( "dy",			RTFidDY ),
    RTF_REMEMBER( "mo",			RTFidMO ),
    RTF_REMEMBER( "yr",			RTFidYR ),
				/****************************************/
				/*  StyleSheet.				*/
				/****************************************/
    RTF_REMEMBER( "additive",		RTFidADDITIVE ),
    RTF_REMEMBER( "sbasedon",		RTFidSBASEDON ),
    RTF_REMEMBER( "snext",		RTFidSNEXT ),
					/*  RTFidS	*/
					/*  RTFidDS	*/
    RTF_REMEMBER( "cs",			RTFidCS ),
    RTF_REMEMBER( "sautoupd",		RTFidSAUTOUPD ),
    RTF_REMEMBER( "shidden",		RTFidSHIDDEN ),

    { 0, 0, 0 }
};

int docRtfRememberProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidDEFLANG:
	case	RTFidDEFLANGFE:
	case	RTFidUC:
	    break;

	case	RTFidDELETED:
	    rrc->rrcInDeletedText= 1;
	    break;

	case	RTFidFTNALT:

	    if  ( rrc->rrcExternalItemKind != DOCinFOOTNOTE )
		{ LLDEB(rrc->rrcExternalItemKind,DOCinFOOTNOTE);	}
	    else{ rrc->rrcExternalItemKind= DOCinENDNOTE;		}

	    break;

	case RTFidTX:
	    rrc->rrcTabStop.tsTwips= arg;
	    if  ( docParaAddTab( &(rrc->rrcParagraphProperties),
						    &(rrc->rrcTabStop) ) < 0 )
		{ LDEB(arg); return -1;	}
	    docInitTabStop( &(rrc->rrcTabStop) );
	    break;

	case RTFidTQR:
	    rrc->rrcTabStop.tsAlignment= DOCtaRIGHT;
	    break;
	case RTFidTQC:	
	    rrc->rrcTabStop.tsAlignment= DOCtaCENTRE;
	    break;
	case RTFidTQDEC:
	    rrc->rrcTabStop.tsAlignment= DOCtaDECIMAL;
	    break;

	case RTFidTLDOT:	rrc->rrcTabStop.tsLeader= DOCtlDOTS;	break;
	case RTFidTLUL:		rrc->rrcTabStop.tsLeader= DOCtlUNDERLINE; break;
	case RTFidTLHYPH:	rrc->rrcTabStop.tsLeader= DOCtlHYPH;	break;
	case RTFidTLTH:		rrc->rrcTabStop.tsLeader= DOCtlTHICK;	break;
	case RTFidTLEQ:		rrc->rrcTabStop.tsLeader= DOCtlEQUAL;	break;

	case RTFidINTBL:
	    rrc->rrcParagraphProperties.ppInTable= arg != 0;
	    if  ( rrc->rrcBd )
		{ rrc->rrcBd->bdProperties.dpContainsTables= 1;	}
	    else{ XDEB(rrc->rrcBd);				}
	    break;

	case RTFidBRDRL:
	    rrc->rrcParagraphProperties.ppLeftBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRR:
	    rrc->rrcParagraphProperties.ppRightBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRBTW:
	    rrc->rrcParagraphProperties.ppBetweenBorder=
						    rrc->rrcBorderProperties;
	    break;

	case RTFidBOX:
	    rrc->rrcParagraphProperties.ppBoxBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidBRDRBAR:
	    rrc->rrcParagraphProperties.ppBar= rrc->rrcBorderProperties;
	    break;

	case RTFidTRGAPH:
	    rrc->rrcRowProperties.rpHalfGapWidthTwips= arg;
	    break;

	case RTFidTRRH:
	    rrc->rrcRowProperties.rpHeightTwips= arg;
	    break;

	case RTFidTRLEFT:
	    rrc->rrcRowProperties.rpLeftIndentTwips= arg;
	    break;

	case RTFidTRHDR:
	    rrc->rrcRowProperties.rpIsTableHeader= ( arg != 0 );
	    break;

	case RTFidTRKEEP:
	    rrc->rrcRowProperties.rpKeepOnOnePage= ( arg != 0 );
	    break;

	case RTFidTRBRDRT:
	    rrc->rrcRowProperties.rpTopBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRL:
	    rrc->rrcRowProperties.rpLeftBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRB:
	    rrc->rrcRowProperties.rpBottomBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRR:
	    rrc->rrcRowProperties.rpRightBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRH:
	    rrc->rrcRowProperties.rpHasHorizontalBorders= ( arg != 0 );
	    rrc->rrcRowProperties.rpHorizontalBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRV:
	    rrc->rrcRowProperties.rpHasVerticalBorders= ( arg != 0 );
	    rrc->rrcRowProperties.rpVerticalBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidTRQL:
	    rrc->rrcRowProperties.rpAlignment= DOCiaLEFT;
	    break;
	case RTFidTRQC:
	    rrc->rrcRowProperties.rpAlignment= DOCiaCENTERED;
	    break;
	case RTFidTRQR:
	    rrc->rrcRowProperties.rpAlignment= DOCiaRIGHT;
	    break;

	case RTFidCLMGF:
	    rrc->rrcCellProperties.cpLeftInMergedRow= ( arg != 0 );
	    break;
	case RTFidCLMRG:
	    rrc->rrcCellProperties.cpMergedWithLeft= ( arg != 0 );
	    break;
	case RTFidCLVMGF:
	    rrc->rrcCellProperties.cpTopInMergedColumn= ( arg != 0 );
	    break;
	case RTFidCLVMRG:
	    rrc->rrcCellProperties.cpMergedWithAbove= ( arg != 0 );
	    break;

	case RTFidCLBRDRT:
	    rrc->rrcCellProperties.cpTopBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRL:
	    rrc->rrcCellProperties.cpLeftBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRR:
	    rrc->rrcCellProperties.cpRightBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRB:
	    rrc->rrcCellProperties.cpBottomBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidCELLD:
	    docCleanCellProperties( &(rrc->rrcCellProperties) );
	    docInitCellProperties( &(rrc->rrcCellProperties) );
	    break;
	case RTFidCLVERTALT:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaTOP;
	    break;
	case RTFidCLVERTALC:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaCENTERED;
	    break;
	case RTFidCLVERTALB:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaBOTTOM;
	    break;
	case RTFidCLTXLRTB:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaVERT_LEFT;
	    break;
	case RTFidCLTXTBRL:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaVERT_RIGHT;
	    break;

	case RTFidCLCFPAT:
	    rrc->rrcCellProperties.cpForegroundColor= arg;
	    break;
	case RTFidCLCBPAT:
	    rrc->rrcCellProperties.cpBackgroundColor= arg;
	    break;

	case RTFidCLSHDNG:
	    rrc->rrcCellProperties.cpShadingLevel= arg;
	    break;

	case RTFidCLBGHORIZ:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGHORIZ;
	    break;
	case RTFidCLBGVERT:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGVERT;
	    break;
	case RTFidCLBGFDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGFDIAG;
	    break;
	case RTFidCLBGBDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGBDIAG;
	    break;
	case RTFidCLBGCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGCROSS;
	    break;
	case RTFidCLBGDCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDCROSS;
	    break;
	case RTFidCLBGDKHOR:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKHORIZ;
	    break;
	case RTFidCLBGDKVERT:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKVERT;
	    break;
	case RTFidCLBGDKFDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKFDIAG;
	    break;
	case RTFidCLBGDKBDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKBDIAG;
	    break;
	case RTFidCLBGDKCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKCROSS;
	    break;
	case RTFidCLBGDKDCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspBGDKDCROSS;
	    break;

	case RTFidADJUSTRIGHT:
	case RTFidADJUSTLEFT:
	    break;
	case	RTFidCF:
	    /*  foreground color: index in color table or 0	*/
	    break;
	case	RTFidFC:
	    /*  Does not appear in the RTF spec.		*/
	    /*  It can however be found in NextStep files.	*/
	    break;
	case	RTFidUP:
	    break;
				/****************************************/
				/*  Set/Unset Text Attributes.		*/
				/****************************************/
	case	RTFidULNONE:
	    rrc->rrcCurrentTextAttribute.taIsUnderlined= 0;
	    break;
				/****************************************/
				/*  Set/Unset Text Attributes.		*/
				/*  In Paragraph Numbering.		*/
				/****************************************/
	case	RTFidPNF:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontNumber= arg;
	    break;
	case	RTFidPNFS:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontSizeHalfPoints= arg;
	    break;
	case	RTFidPNB:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontIsBold= arg != 0;
	    break;
	case	RTFidPNI:
	    rrc->rrcParagraphNumber.pnTextAttribute.taFontIsSlanted= arg != 0;
	    break;
	case	RTFidPNCAPS:
	    rrc->rrcParagraphNumber.pnTextAttribute.taCapitals= arg != 0;
	    break;
	case	RTFidPNSCAPS:
	    rrc->rrcParagraphNumber.pnTextAttribute.taSmallCaps= arg != 0;
	    break;
	case	RTFidPNUL:
	    rrc->rrcParagraphNumber.pnTextAttribute.taIsUnderlined= arg != 0;
	    break;
	case	RTFidPNULNONE:
	    rrc->rrcParagraphNumber.pnTextAttribute.taIsUnderlined= 0;
	    break;
	case	RTFidPNSTRIKE:
	    rrc->rrcParagraphNumber.pnTextAttribute.taHasStrikeThrough=
								    arg != 0;
	    break;

				/****************************************/
	case RTFidCHCFPAT:
	    rrc->rrcTextForeground= arg;
	    break;
	case RTFidCHCBPAT:
	    rrc->rrcTextBackground= arg;
	    break;

	case RTFidCHSHDNG:
	    rrc->rrcTextShadingLevel= arg;
	    break;

	case RTFidCHBGHORIZ:
	    rrc->rrcTextShadingPattern= DOCspBGHORIZ;
	    break;
	case RTFidCHBGVERT:
	    rrc->rrcTextShadingPattern= DOCspBGVERT;
	    break;
	case RTFidCHBGFDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGFDIAG;
	    break;
	case RTFidCHBGBDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGBDIAG;
	    break;
	case RTFidCHBGCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGCROSS;
	    break;
	case RTFidCHBGDCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGDCROSS;
	    break;
	case RTFidCHBGDKHORIZ:
	    rrc->rrcTextShadingPattern= DOCspBGDKHORIZ;
	    break;
	case RTFidCHBGDKVERT:
	    rrc->rrcTextShadingPattern= DOCspBGDKVERT;
	    break;
	case RTFidCHBGDKFDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGDKFDIAG;
	    break;
	case RTFidCHBGDKBDIAG:
	    rrc->rrcTextShadingPattern= DOCspBGDKBDIAG;
	    break;
	case RTFidCHBGDKCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGDKCROSS;
	    break;
	case RTFidCHBGDKDCROSS:
	    rrc->rrcTextShadingPattern= DOCspBGDKDCROSS;
	    break;

	case RTFidCHBRDR:
	    rrc->rrcTextBorderProperties= rrc->rrcBorderProperties;
	    break;

	case	RTFidTROWD:
	    docCleanRowProperties( &(rrc->rrcRowProperties) );
	    docInitRowProperties(  &(rrc->rrcRowProperties) );

	    docInitCellProperties( &(rrc->rrcCellProperties) );
	    break;

	case	RTFidCELLX:
	    rrc->rrcCellProperties.cpRightBoundaryTwips= arg;

	    if  ( docInsertRowColumn( &(rrc->rrcRowProperties), -1,
						&rrc->rrcCellProperties ) )
		{ LDEB(rrc->rrcRowProperties.rpCellCount); return -1; }

	    docInitBorderProperties( &(rrc->rrcCellProperties.cpTopBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpLeftBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpRightBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpBottomBorder) );

	    rrc->rrcCellProperties.cpLeftInMergedRow= 0;
	    rrc->rrcCellProperties.cpMergedWithLeft= 0;
	    rrc->rrcCellProperties.cpTopInMergedColumn= 0;
	    rrc->rrcCellProperties.cpMergedWithAbove= 0;

	    break;

				/****************************************/
				/*  Paragraph Numbering.		*/
				/****************************************/
	case RTFidPNCARD:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnCARD;
	    break;
	case RTFidPNDEC:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnDEC;
	    break;
	case RTFidPNUCLTR:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnUCLTR;
	    break;
	case RTFidPNUCRM:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnUCRM;
	    break;
	case RTFidPNLCLTR:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnLCLTR;
	    break;
	case RTFidPNLCRM:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnLCRM;
	    break;
	case RTFidPNORD:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnORD;
	    break;
	case RTFidPNORDT:
	    rrc->rrcParagraphNumber.pnStyle= DOCpnORDT;
	    break;
	case RTFidPNSTART:
	    rrc->rrcParagraphNumber.pnStartNumber= arg- 1;
	    break;
	case RTFidPNINDENT:
	    rrc->rrcParagraphNumber.pnIndentTwips= arg;
	    break;
	case RTFidPNHANG:
	    rrc->rrcParagraphNumber.pnUseHangingIndent= 1;
	    break;
				/****************************************/
				/*  Encoding.				*/
				/****************************************/
	case RTFidPC:
	    memcpy( rrc->rrcInputMapping, docDOS437_to_ISO1, 256 );
	    break;
	case RTFidPCA:
	    memcpy( rrc->rrcInputMapping, docDOS850_to_ISO1, 256 );
	    break;
	case RTFidMAC:
	    memcpy( rrc->rrcInputMapping, docMAC_to_ISO1, 256 );
	    break;

	case RTFidCGRID:
	case RTFidLANG:
	    break;
				/****************************************/
				/*  Fields in dates and times.		*/
				/****************************************/
	case RTFidHR:
	    rrc->rrcTm.tm_hour= arg;
	    break;
	case RTFidMIN:
	    rrc->rrcTm.tm_min= arg;
	    break;
	case RTFidSEC:
	    rrc->rrcTm.tm_sec= arg;
	    break;
	case RTFidDY:
	    rrc->rrcTm.tm_mday= arg;
	    break;
	case RTFidMO:
	    rrc->rrcTm.tm_mon= arg- 1;
	    break;
	case RTFidYR:
	    rrc->rrcTm.tm_year= arg- 1900;
	    break;
				/****************************************/
				/*  StyleSheet.				*/
				/****************************************/
	case RTFidADDITIVE:
	    rrc->rrcStyle.dsAdditive= 1;
	    break;
	case RTFidSBASEDON:
	    rrc->rrcStyle.dsBasedOn= arg;
	    break;
	case RTFidSNEXT:
	    rrc->rrcStyle.dsNext= arg;
	    break;
	/*  Handeled With the paragraph style.
	case RTFidS:
	    rrc->rrcStyle.dsStyleNumber= arg;
	    rrc->rrcStyle.dsLevel= DOClevPARA;
	    break;
	*/
	/*  Handeled With the section style.
	case RTFidDS:
	    rrc->rrcStyle.dsStyleNumber= arg;
	    rrc->rrcStyle.dsLevel= DOClevSECT;
	    break;
	*/
	case RTFidCS:
	    rrc->rrcStyle.dsStyleNumber= arg;
	    rrc->rrcStyle.dsLevel= DOClevTEXT;
	    break;
	case RTFidSAUTOUPD:
	    rrc->rrcStyle.dsAutoupd= arg;
	    break;
	case RTFidSHIDDEN:
	    rrc->rrcStyle.dsHidden= arg;
	    break;
				/****************************************/
				/*  Unknown				*/
				/****************************************/
	default:
	    /*SLDEB(rcw->rcwWord,rcw->rcwId);*/
	    break;
	}

    return 0;
    }
