/*   FILE: new.c -- A program to create a new file based on a template.
 * AUTHOR: W. Michael Petullo <new@flyn.org>
 *   DATE: 01 MAY 1 999
 *
 * Copyright (c) 1999 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <new/common.h>
#include <new/template.h>
#include <new/pair.h>
#include <libgen.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <limits.h>
#include <libgen.h>

extern char *usage;
void print_usage(const int exitcode, const char *error, const char *more);

/* ============================ settings_t ================================= */
typedef struct settings_t {
    int force;
    int list_only;
    int use_global;
    char *filepath;
    char template_path[PATH_MAX + 1];
    char type[PATH_MAX + 1];
    char mapping_path[PATH_MAX + 1];
    int permissions;
    GList *cmd_line_kv;
} settings_t;

/* ============================ parse_args () =============================== */
void parse_args(int argc, char *argv[], settings_t * settings)
{
    int c;
    int opt_index = 0;
    struct option opts[] = {
	{"help", 0, 0, 'h'},
	{"force", 0, 0, 'f'},
	{"list", 0, 0, 'l'},
	{"global", 0, 0, 'g'},
	{"set", 1, 0, 's'},
	{"template", 1, 0, 't'},
	{"type", 1, 0, 'y'},
	{"permissions", 1, 0, 'p'},
	{"mapping", 1, 0, 'm'},
	{0, 0, 0, 0}
    };
    settings->force = 0;
    settings->list_only = 0;
    settings->use_global = 0;
    settings->filepath = NULL;
    strcpy(settings->template_path, "");
    strcpy(settings->type, "");
    settings->permissions = -1;
    strcpy(settings->mapping_path, "");
    settings->cmd_line_kv = NULL;
    while ((c = getopt_long(argc, argv, "hflgt:s:y:p:m:", opts, &opt_index))
	   >= 0) {
	switch (c) {
	char *key, *val;
	case 'h':
	    print_usage(EXIT_SUCCESS, NULL, NULL);
	case 'f':
	    settings->force = 1;
	    break;
	case 'l':
	    settings->list_only = 1;
	    break;
	case 'g':
	    settings->use_global = 1;
	    break;
	case 't':
	    strncpy(settings->template_path, optarg,
		    sizeof(settings->template_path) - 1);
	    break;
	case 'y':
	    strncpy(settings->type, optarg, sizeof(settings->type) - 1);
	    break;
	case 'p':
	    if (!sscanf(optarg, "%o", &settings->permissions)) {
		fprintf(stderr, "%s\n", usage);
		exit(EXIT_FAILURE);;
	    }
	    break;
	case 's':
	    if (parse_kv(optarg, &key, &val)) {
                pair_t *kv = (pair_t *) malloc (sizeof (pair_t));
		pair_init (kv, strdup(key), strdup(val), free, free);
                if (kv)
                    settings->cmd_line_kv = g_list_append(settings->cmd_line_kv, kv);
            }
	    else {
		fprintf(stderr, "%s\n", usage);
		exit(EXIT_FAILURE);
	    }
	    break;
	case 'm':
	    strncpy(settings->mapping_path, optarg, sizeof(settings->mapping_path - 1));
	    break;
	default:
	    print_usage(EXIT_FAILURE, NULL, NULL);
	}
    }
    if (!(settings->filepath = argv[optind++]))
	print_usage(EXIT_FAILURE, "new", "No filepath specified");
}

/* ============================ main () ===================================== */
int main(int argc, char *argv[], char *env[])
{
    settings_t settings;
    GList *ptr;
    parse_args(argc, argv, &settings);
    template_init();
    if (!template_set_global_dir(GLOBAL_TEMPLATE_DIR)) {
	template_perror("Error setting global template dir");
	exit(EXIT_FAILURE);
    }
    if (!template_set_local_dir(".new/templates")) {
	template_perror("Error setting global template dir");
	exit(EXIT_FAILURE);
    }
    if (settings.list_only) {
	const char *filename = basename(settings.filepath);
	if (!strlen(settings.type))
	    if (!template_set_type(settings.type, filename))
		strcpy(settings.type, filename);
	if (!template_list(settings.type)) {
	    template_perror("Error listing templates");
	    exit(EXIT_FAILURE);
	}
    } else {
	if (!strlen(settings.template_path)) {
	    char template_path[PATH_MAX + 1];
	    char *template_name = argv
		&& argv[optind] ? argv[optind] : "default";
	    const char *filename = basename(settings.filepath);
	    if (!strlen(settings.type))
		if (!template_set_type(settings.type, filename))
		    strcpy(settings.type, filename);
	    if (!template_find (template_path, settings.type, template_name,
		 settings.use_global)) {
		template_perror("error finding template");
		exit(EXIT_FAILURE);
	    }
	    if (!template_write_it_using_map
		(settings.filepath, settings.force, template_path,
		 settings.cmd_line_kv, settings.mapping_path)) {
		 template_perror("error writing");
		 exit(EXIT_FAILURE);
		}
	} else
	    if (!template_write_it_using_map
		(settings.filepath, settings.force, settings.template_path,
		 settings.cmd_line_kv, settings.mapping_path)) {
		 template_perror("error writing");
		 exit(EXIT_FAILURE);
		 }
    }
    if (settings.permissions > 0)
	chmod(settings.filepath, settings.permissions);
    while ((ptr = g_list_first(settings.cmd_line_kv))) { 
        pair_destroy(ptr->data);
        settings.cmd_line_kv = g_list_remove_link(settings.cmd_line_kv, ptr);
    }
    template_destroy();
    exit(EXIT_SUCCESS);
}
