/* Copyright (c) 2003-2004 Ecole centrale de Lyon
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

// Spatial parameters for structure
typedef struct
{
  int nx,ny,nz; // Size of working space
  int block_size; // Max. size for blocks
  int num_pml; // How many cells for each PML
  double delta_x; // Size of base cell
  double min_index; // Minimum index for the whole structure
  double pml_refl; // Theoretical reflection for a PML under normal incidence
  double pml_decay; // Field decay inside the PML's
  double *index; // The index for the whole structure
  double *absor; // The absorption rate for the whole structure
} varSpace;

// Simulation parameters
typedef struct
{
  double delta_t; // Iteration duration
  int it_max;     // Number of iterations
} varSimul;

typedef struct
{
  int x,y,z; // Injection position (we only support dipole injection currently)
  double dir_x,dir_y,dir_z; // What gets injected for each component of E
  double basefreq; // Base frequency in Hz
  double *injection; // Injection over time (a table of length it_max)
} varInj;

enum {
  FIELD_E,
  FIELD_H
};

enum {
  INJ_NONE,
  INJ_FILE,
  INJ_SINUS,
  INJ_GAUSS
};

enum {
  DIR_NONE,
  DIR_X,
  DIR_Y,
  DIR_Z
};

typedef struct
{
  int field;
  int begin;
  int interv;
  int end;
} varOutputCarto;

typedef struct
{
  int field;
  int x,y,z;
  double *valx;
  double *valy;
  double *valz;
} varOutputPonct;

typedef struct
{
  int dir;
  int x1,x2;
  int y1,y2;
  int z1,z2;
  double *val;
} varOutputPoynt;


typedef struct
{
  int nb_cartos;
  varOutputCarto *cartos;
  int nb_poncts;
  varOutputPonct *poncts;
  int nb_poynts;
  int total_loss;
  varOutputPoynt *poynts;
} varOutput;

// The structure containing everything
typedef struct
{
  varSpace *space;
  varSimul *simul;
  varInj *inj;
  varOutput *output;
} SimulDesc;

#define BLOCK_PML_FACE_X 1
#define BLOCK_PML_FACE_Y 2
#define BLOCK_PML_FACE_Z 4
#define BLOCK_PML_NONE 0
#define BLOCK_PML_CORNER 7
#define BLOCK_PML_EDGE_YZ 6
#define BLOCK_PML_EDGE_ZX 5
#define BLOCK_PML_EDGE_XY 3

// Block specials for PML
typedef struct
{
  int block_type;
  double *sumrotEx;
  double *sumrotEy;
  double *sumrotEz;
  double *sumrotHHx;
  double *sumrotHHy;
  double *sumrotHHz;
  double *sumEx;
  double *sumEy;
  double *sumEz;
  double *sumHHx;
  double *sumHHy;
  double *sumHHz;
  double x_decay;
  double y_decay;
  double z_decay;
  double x_rotcoef,y_rotcoef,z_rotcoef;
  double x_sumrotcoef,y_sumrotcoef,z_sumrotcoef;
  double x_sumcoef,y_sumcoef,z_sumcoef;
} varPMLblock;

typedef struct
{
  double *dec; // decay rate for E field due to absorbing/active material
               // Can be common to several blocks: check for double-free()
               // In fact we store c*alpha*delta_t/index
  double *sumEx;
  double *sumEy;
  double *sumEz;
} varAbsBlock;

// A simulation block
typedef struct SvarBlock
{
  int nx,ny,nz; // Size of the block
  int off_x,off_y,off_z; // Offsets indicating the block's position
#ifdef HAVE_MPI
  int mpi_node; // Node on which this block is allocated
#endif
  double *cn; // (cdt/ndx)^2 array for the block
              // Can be common to several blocks: check for double-free()
  double *Ex;
  double *Ey;
  double *Ez;
  double *HHx;
  double *HHy;
  double *HHz;
  struct SvarBlock *block_xm; // The block for the previous x value
  struct SvarBlock *block_xp; // etc.
  struct SvarBlock *block_ym;
  struct SvarBlock *block_yp;
  struct SvarBlock *block_zm;
  struct SvarBlock *block_zp;
  varPMLblock *PML; // If non-NULL, we are in a PML
  varAbsBlock *absor; // If non-NULL, there is absorption
} varBlock;

typedef struct
{
  int n;
  int nx,ny,nz; // Number of blocks in each direction
#ifdef HAVE_MPI
  int mpi_node; // Identity of the current process
  int mpi_nnodes; // Number of processes
  int mpi_nblocks; // Maximum number of blocks per process
  int mpi_startblock; // Block where to start
  int mpi_endblock; // Block where to end
  int mpi_blocks_before; // Number of blocks where we send E (and receive H)
  int mpi_blocks_after; // Number of blocks where we send H and receive E
#endif
  varBlock *blocks;
     // Other things to be freed
  unsigned long *index_crc; // The CRC of each index (cn) block
  unsigned long *decay_crc; // The CRC of each dec block
  varPMLblock *PMLblocks;
  varAbsBlock *AbsBlocks;
  int num_pmlblocks;
  int num_absblocks;
  varBlock *empty_block; // an empty block containing only zero fields
  double *empty; // an zero memory area
  double *garbage; // an allocated memory area which can contain anything
} varBlockList;

#if 0
typedef struct
{
  double *prem; // first plan
  double *dern; // last plan
} OneCteMur;

typedef struct
{
  OneCteMur C1;
  OneCteMur C2;
  OneCteMur C3;
} OneDirMur;

typedef struct
{
  OneDirMur Cx;
  OneDirMur Cy;
  OneDirMur Cz;
} ConstMur;
#endif

#define FILE_BUFFER 4096 // Minimal size for the file reading buffer
#define FILENAME_BUF 32 // Maximal lendth for generated filenames
#define DEFAULT_INTERVAL 100 // Default interval for outputting cartographies
#define CELERITY 2.99792458e8
#define EPSILON_0 8.854187817e-12
#define MU_0 1.25663706144e-6
#define SQRT_3 1.7320508075688772
#define SQRT_2 1.4142135623730951
#define SQUARE(x) ((x)*(x))
#define CUBE(x) ((x)*(x)*(x))
#define MAX(a,b) ((a)>(b)?(a):(b))
#define MIN(a,b) ((a)<(b)?(a):(b))

#ifdef HAVE_MPI
#define TAG_EFIELD 1
#define TAG_HFIELD 2
#define TAG_CARTOOUT 3
#define TAG_PONCTOUT 4
#define TAG_POYNTOUT 5
#define TAG_INIT 6
#endif
