/*  XGDVI
 *  Copyright (C) 1999  Hirotsugu Kakugawa
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif

#include <dvi-2_6.h>
#include <gtk/gtk.h>

#include "defs.h"
#include "xgdvi.h"
#include "gui.h"
#include "buff.h"
#include "path.h"



struct s_dvi_buff {
  DVI_DEVICE    dev;
  char         *file_name;
  DVI           dvi;
  DVI_PROPERTY  prop;
  int           page;
  double        shrink;
  int           paper;
  int           orient;
  double        view_x;
  double        view_y;
  int           ok;
  double        fg_r, fg_g, fg_b; 
};


static int                current_buffer  = -1;
static int                nbuffers        = 0;
static int                nbuffers_opened = 0;
static struct s_dvi_buff  buffer_list[MAX_BUFFS+1];


void
buffer_init(void)
{
  int  i;
  
  dvi      = NULL;
  dvi_file = NULL;
  page     = 1;

  nbuffers = 0;
  nbuffers_opened = 0;
  current_buffer = -1;
  for (i = 0; i < MAX_BUFFS+1; i++){
    buffer_list[i].file_name = NULL;
    buffer_list[i].dvi       = NULL; 
    buffer_list[i].prop      = NULL; 
    buffer_list[i].page      = 0; 
    buffer_list[i].shrink    = 1.0; 
    buffer_list[i].paper     = 0; 
    buffer_list[i].orient    = 0; 
    buffer_list[i].view_x    = 0.0; 
    buffer_list[i].view_y    = 0.0; 
    buffer_list[i].ok        = 0; 
  }
}


int 
buffer_full(void)
{
  return (nbuffers >= MAX_BUFFS);
}


int 
buffer_empty(void)
{
  return (nbuffers <= 0);
}


int 
buffer_size(void)
{
  return nbuffers;
}


int 
buffer_full_opened(void)
{
  return (nbuffers_opened >= MAX_BUFFS_OPENED);
}



char*
buffer_file_name(int i, int *f_page, int  *f_paper, int *f_orient, 
		 double *f_shrink, double *f_view_x, double *f_view_y)
{
  int   j;
  char  *gf;
  
  if ((current_buffer >= 0) && (page > 0))
    buffer_list[current_buffer].page = page;

  gf = gbuff_get_ith(i);
  if (gf == NULL)
    return NULL;
  for (j = 0; j < MAX_BUFFS; j++){
    if ((buffer_list[j].file_name != NULL)
	&& (strcmp(buffer_list[j].file_name, gf) == 0))
      break;
  }
  if (j == MAX_BUFFS)
    return NULL;

  if (f_page != NULL)
    *f_page = buffer_list[j].page;
  if (f_paper != NULL)
    *f_paper = buffer_list[j].paper;
  if (f_orient != NULL)
    *f_orient = buffer_list[j].orient;
  if (f_shrink != NULL)
    *f_shrink = buffer_list[j].shrink;
  if (f_view_x != NULL)
    *f_view_x = buffer_list[j].view_x;
  if (f_view_y != NULL)
    *f_view_y = buffer_list[j].view_y;
  return  buffer_list[j].file_name;
}

#if 0
static int 
buffer_colors(char *f, int op,
	      gint16 *fg_r, gint16 *fg_g, gint16 *fg_b)
{
  int  j;
  
  for (j = 0; j < MAX_BUFFS; j++){
    if ((buffer_list[j].file_name != NULL)
	&& (strcmp(f, buffer_list[j].file_name) == 0)){
      if (op == 0){   /* set */
	if (fg_r != NULL)
	  buffer_list[j].fg_r = *fg_r;
	if (fg_g != NULL)
	  buffer_list[j].fg_g = *fg_g;
	if (fg_b != NULL)
	  buffer_list[j].fg_b = *fg_b;
      } else {        /* get */
	if (fg_r != NULL)
	  *fg_r = buffer_list[j].fg_r;
	if (fg_g != NULL)
	  *fg_g = buffer_list[j].fg_g;
	if (fg_b != NULL)
	  *fg_b = buffer_list[j].fg_b;
      }
      return  0;
    }
  }

  return -1;
}
#endif

int 
buffer_set_colors(char *f, double r, double g, double b)
{
  return 0;
}

int 
buffer_get_colors(char *f, double *r, double *g, double *b) 
{
  return 0;
}



int
buffer_check_in_buffer(char *f)
{
  char *p;
  int   v, j;

  p = path_canonical(f);

  v = -1;
  for (j = 0; j < MAX_BUFFS; j++){
    if ((buffer_list[j].file_name != NULL)
	&& (strcmp(p, buffer_list[j].file_name) == 0)){
      v = 1;  /* opened. in buffer */
      break;
    }
  }

  g_free(p);
  p = NULL;

  return v; 
}


void
buffer_save_current_state(void)
{
  if (current_buffer >= 0){
    buffer_list[current_buffer].page   = page;
    buffer_list[current_buffer].shrink = shrink; 
    buffer_list[current_buffer].paper  = paper; 
    buffer_list[current_buffer].orient = orient; 
    preview_window_get_pos(&buffer_list[current_buffer].view_x,
			   &buffer_list[current_buffer].view_y);
    buffer_list[current_buffer].fg_r = current_color_fg_r;
    buffer_list[current_buffer].fg_g = current_color_fg_g;
    buffer_list[current_buffer].fg_b = current_color_fg_b;
  }
}

int
buffer_add(DVI_DEVICE v, char *file, DVI d, DVI_PROPERTY prop, 
	   int pg, int pid, int oid, int gpos)
{
  int    i, j;
  char  *filepath, *h;

  if (file == NULL)
    return -1;

  if (buffer_full())
    return -1;

  if ((d != NULL) && (buffer_full_opened())){
    for (i = nbuffers-1; i >= 0; i--){
      h = gbuff_get_ith(i);
      for (j = 0; j < MAX_BUFFS; j++){
	if (j == current_buffer)
	  continue;
	if ((buffer_list[j].file_name != NULL)
	    && (strcmp(h, buffer_list[j].file_name) == 0)){
	  break;
	}
      }
      if (buffer_list[j].dvi != NULL){
	DVI_DISPOSE(buffer_list[j].dvi, buffer_list[j].dev);
	buffer_list[j].dvi = NULL;
	nbuffers_opened--;
	break;
      }
    }
  }

  if (pid < 0)
    pid = param_paper;
  if (oid < 0)
    oid = param_orient;

  for (j = 0; j < MAX_BUFFS; j++){
    if (buffer_list[j].file_name == NULL){
      filepath = path_canonical(file);
      if (gpos == 0)
	gbuff_insert_last(filepath, pg);
      else
	gbuff_insert_here(filepath, pg, buffer_list[current_buffer].file_name);
      if (prop == NULL)
	prop = make_dvi_property();
      buffer_list[j].dev       = v;
      buffer_list[j].file_name = filepath;
      buffer_list[j].dvi       = d;
      buffer_list[j].prop      = prop;
      buffer_list[j].page      = pg;
      buffer_list[j].shrink    = param_shrink; 
      buffer_list[j].paper     = pid; 
      buffer_list[j].orient    = oid; 
      buffer_list[j].view_x    = 0.0;
      buffer_list[j].view_y    = 0.0;
      buffer_list[j].fg_r      = default_color_fg_r;
      buffer_list[j].fg_g      = default_color_fg_g;
      buffer_list[j].fg_b      = default_color_fg_b;
      if (d != NULL)
	buffer_list[j].ok      = 1;
      nbuffers++;
      if (d != NULL)
	nbuffers_opened++;
      break;
    }
  }

  return 0;
}


int
buffer_kill(char *f)
{
  int    i, j;
  char  *g;

  if (buffer_check_in_buffer(f) < 0)
    return -1;

  j = -1;
  for (i = 0; i < MAX_BUFFS; i++){
    if ((buffer_list[i].file_name != NULL)
	&& (strcmp(f, buffer_list[i].file_name) == 0)){
      j = i;
      break;
    }
  }
  if (j < 0)
    return -1;

  g = gbuff_get_near(buffer_list[j].file_name);

  gbuff_kill(buffer_list[j].file_name);

  gbuff_select(g);
  buffer_select(g);    /* g can be NULL, but it's ok */

  if (buffer_list[j].dvi != NULL){
    DVI_DISPOSE(buffer_list[j].dvi, buffer_list[j].dev);
    nbuffers_opened--;
  }
  if (buffer_list[j].prop != NULL)
    DVI_PROPERTY_RELEASE(buffer_list[j].prop);

  nbuffers--;
  g_free(buffer_list[j].file_name);
  buffer_list[j].dev       = NULL;
  buffer_list[j].file_name = NULL;
  buffer_list[j].dvi       = NULL;
  buffer_list[j].prop      = NULL;

  return 0;
}


int
buffer_update_dvi(char *f, DVI dvi_new)
{
  int    i, j;

  if (buffer_check_in_buffer(f) < 0)
    return -1;

  j = -1;
  for (i = 0; i < MAX_BUFFS; i++){
    if ((buffer_list[i].file_name != NULL)
	&& (strcmp(f, buffer_list[i].file_name) == 0)){
      j = i;
      break;
    }
  }
  if (j < 0)
    return -1;

  if (buffer_list[j].dvi != NULL){
    DVI_DISPOSE(buffer_list[j].dvi, buffer_list[j].dev);
    nbuffers_opened--;
  }

  buffer_list[j].dvi = dvi_new;
  nbuffers_opened++; 

  return 0;
}


static int
buffer_do_select(int j)
{
  current_buffer = j;
  
  if (buffer_list[j].dvi != NULL){
    if (DVI_FILE_MODIFIED(buffer_list[j].dvi, buffer_list[j].file_name) == 1){
      message(_("DVI file was updated. Reading again..."));
      if (buffer_list[j].dvi == dvi)
	dvi = NULL;
      DVI_DISPOSE(buffer_list[j].dvi, buffer_list[j].dev);
      buffer_list[j].dvi = NULL;
      nbuffers_opened--;
    }
  } else {
    message(_("Reading again..."));
  }

  gbuff_select(buffer_list[j].file_name);
  gpaper_set(buffer_list[j].paper, buffer_list[j].orient);

  buffer_list[j].ok = 1;
  if (buffer_list[j].dvi == NULL){
    buffer_list[j].dvi = load_dvi(buffer_list[j].file_name, 
				  buffer_list[j].dev, 
				  buffer_list[j].prop,
				  0);
  }

  if (buffer_list[j].dvi != NULL)
    nbuffers_opened++;

  dvi_file = buffer_list[j].file_name;
  dvi      = buffer_list[j].dvi;
  page     = buffer_list[j].page;
  on_buffer_switch_shrink = buffer_list[j].shrink;
  on_buffer_switch_paper  = buffer_list[j].paper;
  on_buffer_switch_orient = buffer_list[j].orient;
  on_buffer_switch_view_x = buffer_list[j].view_x;
  on_buffer_switch_view_y = buffer_list[j].view_y;

  if (buffer_list[j].dvi == NULL){
    buffer_list[j].ok = 0;
    return -1;
  }

  return 0;
}

int
buffer_select(char *f)
{
  int   j, v;

  if (f != NULL){
    for (j = 0; j < MAX_BUFFS; j++){
      if ((buffer_list[j].file_name != NULL) 
	  && (strcmp(f, buffer_list[j].file_name) == 0)){
        v = buffer_do_select(j); 
	return v;
      }
    }
  }

  dvi_file = NULL;
  dvi      = NULL;
  page     = 0;

  return -1;
}


int
buffer_select_any_other(char *f)
{
  int    j;
  char  *g;

  if (f != NULL){
    if ((g = gbuff_get_near(f)) != NULL){
      for (j = 0; j < MAX_BUFFS; j++){
	if ((buffer_list[j].file_name != NULL)
	    && ((g != NULL) && (strcmp(g, buffer_list[j].file_name) == 0))){
	  return  buffer_do_select(j);
	}
      }
    }
  }

  dvi_file = NULL;
  dvi      = NULL;
  page     = 0;

  return -1;
}


void
buffer_select_next(int next)
{
  int   j;
  char  *g, *f;

  if (current_buffer >= 0){
    f = buffer_list[current_buffer].file_name;
    g = gbuff_get_next(f, next);
    if (g != NULL){
      for (j = 0; j < MAX_BUFFS; j++){
	if ((buffer_list[j].file_name != NULL) 
	    && (strcmp(g, buffer_list[j].file_name) == 0)){
          buffer_do_select(j);
	  break;
	}
      }
    }
  }
}



int
buffer_dump(void)
{
  int    j;

  for (j = 0; j < MAX_BUFFS; j++){
    if (buffer_list[j].file_name != NULL){
      printf("%02d %s\n", j, buffer_list[j].file_name);
    }
  }
  return 0;
}
