#ifndef INDII_CLUSTER_PEARSONDISTANCE_HPP
#define INDII_CLUSTER_PEARSONDISTANCE_HPP

#include "ClusterVector.hpp"

namespace indii {
  namespace cluster {
/**
 * Pearson distance.
 *
 * @tparam T Datum element type.
 */
template<class T = float>
class PearsonDistance {
public:
  /**
   * Preprocess point.
   *
   * @param x Point.
   */
  static void prepare(typename ClusterVector<T>::type& x);

  /**
   * Calculate distance.
   *
   * @param x1 First point.
   * @param x2 Second point, should have same size as first.
   *
   * @return %Distance between the two points.
   */
  static T distance(const typename ClusterVector<T>::type& x1,
      const typename ClusterVector<T>::type& x2);

};

  }
}

#include <cassert>

template<class T>
inline void indii::cluster::PearsonDistance<T>::prepare(
    typename ClusterVector<T>::type& x) {
  /* pre-condition */
  assert (x.size() == ClusterVector<T>::N);

  const unsigned int N = ClusterVector<T>::N;
  T mu;
  
  mu = boost::numeric::ublas::sum(x) / N;
  x -= boost::numeric::ublas::scalar_vector<T>(N,mu);
  x /= boost::numeric::ublas::norm_2(x);
}

template<class T>
inline T indii::cluster::PearsonDistance<T>::distance(
    const typename ClusterVector<T>::type& x1,
    const typename ClusterVector<T>::type& x2) {
  /* pre-condition */
  assert (x1.size() == x2.size());

  return static_cast<T>(1.0) - boost::numeric::ublas::inner_prod(x1,x2);
}
 
#endif

