#ifndef INDII_TINT_COLOURSPACE_HPP
#define INDII_TINT_COLOURSPACE_HPP

#include "boost/numeric/ublas/vector.hpp"

namespace indii {
  namespace tint {
/**
 * Colour space.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * @version $Rev: 75 $
 * @date $Date: 2009-03-15 23:18:20 +0900 (Sun, 15 Mar 2009) $
 */
class ColourSpace {
public:
  /*
   * Types of colour tuplets.
   */
  typedef boost::numeric::ublas::vector<unsigned char,
      boost::numeric::ublas::bounded_array<unsigned char,3> > rgb_t;
  typedef boost::numeric::ublas::vector<float,
      boost::numeric::ublas::bounded_array<float,3> > hsl_t;

  /**
   * Constructor.
   */
  ColourSpace();

  /**
   * Get red channel lightness proportion.
   *
   * @return Red channel proportion.
   */
  float getRedMix();

  /**
   * Get green channel lightness proportion.
   *
   * @return Green channel proportion.
   */
  float getGreenMix();

  /**
   * Get blue channel lightness proportion.
   *
   * @return Blue channel proportion.
   */
  float getBlueMix();

  /**
   * Set channel proportions for lightness.
   * 
   * @param r Red channel proportion.
   * @param g Green channel proportion.
   * @param b Blue channel proportion.
   */
  void setLightness(const float r, const float g, const float b);

  /**
   * Convert RGB to HSL coordinates using greyscale parameters.
   *
   * @param in Input RGB values.
   * @param[out] out Output HSL values.
   */
  void rgb2hsl(const rgb_t& rgb, hsl_t& hsl);

  /**
   * Convert RGB to HSL coordinates using greyscale parameters.
   *
   * @param in Input HSL values.
   * @param[out] out Output RGB values.
   */
  void hsl2rgb(const hsl_t& hsl, rgb_t& rgb);

  /**
   * Round and cast float to unsigned char.
   */
  static unsigned char uround(const float val);

private:
  /**
   * Lightness proportions.
   */
  float greyR, greyG, greyB;

};

  }
}

#define BASE_R 6.0f
#define BASE_G 2.0f
#define BASE_B 4.0f

inline float indii::tint::ColourSpace::getRedMix() {
  return greyR;
}

inline float indii::tint::ColourSpace::getGreenMix() {
  return greyG;
}

inline float indii::tint::ColourSpace::getBlueMix() {
  return greyB;
}

inline void indii::tint::ColourSpace::rgb2hsl(const rgb_t& rgb, hsl_t& hsl) {
  float& h = hsl(0);
  float& s = hsl(1);
  float& l = hsl(2);

  const unsigned char& r = rgb(0);
  const unsigned char& g = rgb(1);
  const unsigned char& b = rgb(2);
  float k1, k2, k3;

  /* lightness */
  l = greyR*r + greyG*g + greyB*b;

  /* hue & saturation */
  if (r == g && g == b) {
    // special case for greys
    h = 0.0f;
    s = 0.0f;
  } else if (r >= g) {
    if (g >= b) {
      // r > g > b
      k1 = static_cast<float>(g - b);
      k2 = static_cast<float>(r - b);
      k3 = static_cast<float>(r + b);
      h = k1/k2;
      s = k2/k3;
    } else if (r >= b) {
      // r > b > g
      k1 = static_cast<float>(b - g);
      k2 = static_cast<float>(r - g);
      k3 = static_cast<float>(r + g);
      h = BASE_R - k1/k2;
      s = k2/k3;
    } else {
      // b > r > g
      k1 = static_cast<float>(r - g);
      k2 = static_cast<float>(b - g);
      k3 = static_cast<float>(b + g);
      h = BASE_B + k1/k2;
      s = k2/k3;
    }
  } else {
    if (r >= b) {
      // g > r > b
      k1 = static_cast<float>(r - b);
      k2 = static_cast<float>(g - b);
      k3 = static_cast<float>(g + b);
      h = BASE_G - k1/k2;
      s = k2/k3;
    } else if (g >= b) {
      // g > b > r
      k1 = static_cast<float>(b - r);
      k2 = static_cast<float>(g - r);
      k3 = static_cast<float>(g + r);
      h = BASE_G + k1/k2;
      s = k2/k3;
    } else {
      // b > g > r
      k1 = static_cast<float>(g - r);
      k2 = static_cast<float>(b - r);
      k3 = static_cast<float>(b + r);
      h = BASE_B - k1/k2;
      s = k2/k3;
    }
  }

  /* post-condition */
  assert (h >= 0.0f && h < BASE_R);
  assert (s >= 0.0f && s <= 1.0f);
  assert (l >= 0.0f && l <= 255.0f);
}

inline void indii::tint::ColourSpace::hsl2rgb(const hsl_t& hsl, rgb_t& rgb) {
  float r, g, b;
  float k1, k2, k3;

  const float& h = hsl(0);
  const float& s = hsl(1);
  const float& l = hsl(2);

  assert (h >= 0.0f && h < BASE_R);
  assert (s >= 0.0f && s <= 1.0f);
  assert (l >= 0.0f && l <= 255.0f);

  if (s == 0.0f) {
    // special case for greys
    r = l;
    g = l;
    b = l;
  } else {
    k2 = (s - 1.0f)/(s + 1.0f);
    switch ((unsigned char)h) {
      case 0:
        // r > g > b
        k1 = h;
        k3 = k1 + k2*(k1 - 1.0f);
        r = l/(greyR + greyG*k3 - greyB*k2);
        g = k3*r;
        b = -k2*r;
        break;
      case 1:
        // g > r > b
        k1 = h - BASE_G;
        k3 = -k1 - k2*(k1 + 1.0f);
        g = l/(greyG + greyR*k3 - greyB*k2);
        r = k3*g;
        b = -k2*g;
        break;
      case 2:
        // g > b > r
        k1 = h - BASE_G;
        k3 = k1 + k2*(k1 - 1.0f);  
        g = l/(greyG + greyB*k3 - greyR*k2);
        b = k3*g;
        r = -k2*g;
        break;
      case 3:
        // b > g > r
        k1 = h - BASE_B;
        k3 = -k1 - k2*(k1 + 1.0f);  
        b = l/(greyB + greyG*k3 - greyR*k2);
        g = k3*b;
        r = -k2*b;
        break;
      case 4:
        // b > r > g
        k1 = h - BASE_B;
        k3 = k1 + k2*(k1 - 1.0f);  
        b = l/(greyB + greyR*k3 - greyG*k2);
        r = k3*b;
        g = -k2*b;
        break;
      case 5:
        // r > b > g
        k1 = h - BASE_R;
        k3 = -k1 - k2*(k1 + 1.0f);  
        r = l/(greyR + greyB*k3 - greyG*k2);
        b = k3*r;
        g = -k2*r;
        break;
      default:
        assert(false);
    }
  }

  rgb(0) = uround(r);
  rgb(1) = uround(g);
  rgb(2) = uround(b);
}

inline unsigned char indii::tint::ColourSpace::uround(const float val) {
  /* pre-condition */
  assert (val >= 0.0);
  if (val >= 255.0f) {
    return 255;
  } else {
    return static_cast<unsigned char>(val+0.5);
  }
}

#endif

