#include "PostProcessPane.hpp"

#include "wx/wx.h"
#include "wx/gbsizer.h"

using namespace indii::tint;
using namespace indii::tint::gui;

PostProcessPane::PostProcessPane(wxWindow* parent, ClusterModel* model) :
    TintPane(parent), model(model) {
  static const int PADDING = 4;
  watch(model);

  wxGridBagSizer* sizer = new wxGridBagSizer();
  sizer->SetFlexibleDirection(wxHORIZONTAL);
  sizer->AddGrowableCol(1);

  slideSatDecay = new wxSlider(this, ID_SLIDE_SAT_DECAY,
      ColourSpace::uround(model->getSaturationDecay()*255.0f), 0, 255);
  slideSatSoft = new wxSlider(this, ID_SLIDE_SAT_SOFT,
      ColourSpace::uround(model->getSaturationSoftness()*255.0f), 0, 255);
  slideCenDecay = new wxSlider(this, ID_SLIDE_CEN_DECAY,
      ColourSpace::uround(model->getCentroidDecay()*255.0f), 0, 255);
  slideCenSoft = new wxSlider(this, ID_SLIDE_CEN_SOFT,
      ColourSpace::uround(model->getCentroidSoftness()*255.0f), 0, 255);

  wxStaticText *labelSatDecay, *labelCenDecay, *labelSatSoft, *labelCenSoft;
  labelSatDecay = new wxStaticText(this, -1, _("Sat decay"));
  labelSatSoft = new wxStaticText(this, -1, _("Sat edge"));
  labelCenDecay = new wxStaticText(this, -1, _("Hue decay"));
  labelCenSoft = new wxStaticText(this, -1, _("Hue edge"));

  sizer->Add(labelSatDecay, wxGBPosition(0,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelSatSoft, wxGBPosition(1,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelCenDecay, wxGBPosition(2,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT, PADDING);
  sizer->Add(labelCenSoft, wxGBPosition(3,0), wxDefaultSpan,
      wxALIGN_CENTRE_VERTICAL|wxTOP|wxLEFT|wxRIGHT|wxBOTTOM, PADDING);

  sizer->Add(slideSatDecay, wxGBPosition(0,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideSatSoft, wxGBPosition(1,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideCenDecay, wxGBPosition(2,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxEXPAND|wxTOP|wxRIGHT, PADDING);
  sizer->Add(slideCenSoft, wxGBPosition(3,1), wxDefaultSpan,
      wxALIGN_RIGHT|wxEXPAND|wxTOP|wxBOTTOM|wxRIGHT, PADDING);

  SetSizerAndFit(sizer);
}

void PostProcessPane::notifySaturationDecayChange() {
  int val = ColourSpace::uround(model->getSaturationDecay()*255.0f);
  if (slideSatDecay->GetValue() != val) {
    slideSatDecay->SetValue(val);
  }
}

void PostProcessPane::notifyCentroidDecayChange() {
  int val = ColourSpace::uround(model->getCentroidDecay()*255.0f);
  if (slideCenDecay->GetValue() != val) {
    slideCenDecay->SetValue(val);
  }
}

void PostProcessPane::notifySaturationSoftnessChange() {
  int val = ColourSpace::uround(model->getSaturationSoftness()*255.0f);
  if (slideSatSoft->GetValue() != val) {
    slideSatSoft->SetValue(val);
  }
}

void PostProcessPane::notifyCentroidSoftnessChange() {
  int val = ColourSpace::uround(model->getCentroidSoftness()*255.0f);
  if (slideCenSoft->GetValue() != val) {
    slideCenSoft->SetValue(val);
  }
}

void PostProcessPane::OnSlideSatDecay(wxScrollEvent& evt) {
  model->setSaturationDecay(slideSatDecay->GetValue()/255.0f);
}

void PostProcessPane::OnSlideCenDecay(wxScrollEvent& evt) {
  model->setCentroidDecay(slideCenDecay->GetValue()/255.0f);
}

void PostProcessPane::OnSlideSatSoft(wxScrollEvent& evt) {
  model->setSaturationSoftness(slideSatSoft->GetValue()/255.0f);
}

void PostProcessPane::OnSlideCenSoft(wxScrollEvent& evt) {
  model->setCentroidSoftness(slideCenSoft->GetValue()/255.0f);
}

BEGIN_EVENT_TABLE(PostProcessPane, wxPanel)

EVT_COMMAND_SCROLL_THUMBRELEASE(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_THUMBRELEASE(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_THUMBRELEASE(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_THUMBRELEASE(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

EVT_COMMAND_SCROLL_TOP(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_TOP(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_TOP(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_TOP(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

EVT_COMMAND_SCROLL_BOTTOM(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_BOTTOM(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_BOTTOM(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_BOTTOM(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

EVT_COMMAND_SCROLL_PAGEUP(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_PAGEUP(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_PAGEUP(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_PAGEUP(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

EVT_COMMAND_SCROLL_PAGEDOWN(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_PAGEDOWN(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_PAGEDOWN(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_PAGEDOWN(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

EVT_COMMAND_SCROLL_LINEUP(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_LINEUP(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_LINEUP(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_LINEUP(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

EVT_COMMAND_SCROLL_LINEDOWN(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_LINEDOWN(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_LINEDOWN(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_LINEDOWN(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

EVT_COMMAND_SCROLL_THUMBTRACK(ID_SLIDE_SAT_DECAY, PostProcessPane::OnSlideSatDecay)
EVT_COMMAND_SCROLL_THUMBTRACK(ID_SLIDE_CEN_DECAY, PostProcessPane::OnSlideCenDecay)
EVT_COMMAND_SCROLL_THUMBTRACK(ID_SLIDE_SAT_SOFT, PostProcessPane::OnSlideSatSoft)
EVT_COMMAND_SCROLL_THUMBTRACK(ID_SLIDE_CEN_SOFT, PostProcessPane::OnSlideCenSoft)

END_EVENT_TABLE()

