#include "ThumbImage.hpp"

#include "../ImageManipulation.hpp"

#include "wx/dcmemory.h"
#include "wx/dcbuffer.h"

#define ICON_SIZE 100
#define THUMB_SIZE 80

using namespace indii::tint;
using namespace indii::tint::gui;

ThumbImage::ThumbImage(wxWindow *parent,
    ImageResource* res, ClusterModel* model, const unsigned int cluster) :
    wxControl(parent, -1, wxDefaultPosition, wxDefaultSize, wxBORDER_NONE),
    res(res), model(model), cluster(cluster),
    isActive(cluster < model->getNumClusters() && model->isShown(cluster)),
	fg(ICON_SIZE,ICON_SIZE), bg(ICON_SIZE,ICON_SIZE) {
  watch(model);

  SetBackgroundStyle(wxBG_STYLE_CUSTOM);
  SetSize(ICON_SIZE,ICON_SIZE);
  SetMinSize(wxSize(ICON_SIZE,ICON_SIZE));

  prepareBackground();
  fgPrepared = false;
}

ThumbImage::~ThumbImage() {
  //
}

void ThumbImage::notifyNumClustersChange() {
  if (cluster < model->getNumClusters()) {
    isActive = model->isShown(cluster);
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifySaturationDecayChange() {
  if (cluster < model->getNumClusters()) {
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifyCentroidDecayChange() {
  if (cluster < model->getNumClusters()) {
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifySaturationSoftnessChange() {
  if (cluster < model->getNumClusters()) {
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifyCentroidSoftnessChange() {
  if (cluster < model->getNumClusters()) {
    fgPrepared = false;
    Refresh();
  }
}

void ThumbImage::notifyClusterChange(const unsigned int i) {
  //
}

void ThumbImage::OnPaint(wxPaintEvent& evt) {
  if (cluster < model->getNumClusters()) {
    if (!fgPrepared) {
      prepareForeground();
    }

    int x, y;
    x = (ICON_SIZE - THUMB_SIZE) / 2 + 1;
    y = (ICON_SIZE - THUMB_SIZE) / 2 + 1;

	wxPaintDC dc(this);
	dc.SetBackground(wxBrush(wxColour(69,69,69)));
	dc.Clear();
    if (isActive) {
      dc.SetBrush(wxBrush(wxColour(128,128,128)));
      dc.SetPen(wxPen(model->getColour(cluster), 3));
      dc.DrawRoundedRectangle(2, 2, ICON_SIZE - 4, ICON_SIZE - 4, 10);
    }
    dc.DrawBitmap(bg, x, y);
    dc.DrawBitmap(fg, x, y);
  }
}

void ThumbImage::OnLeftDown(wxMouseEvent& evt) {
  isActive = !isActive;
  
  Refresh();
}

void ThumbImage::OnLeftUp(wxMouseEvent& evt) {
  isActive = !model->isShown(cluster);
  Refresh();
  Update();
  model->show(cluster, isActive);
}

void ThumbImage::OnLeave(wxMouseEvent& evt) {
  if (cluster < model->getNumClusters()) {
    bool isShown = model->isShown(cluster);
    if (isActive != isShown) {
      isActive = isShown;  
      Refresh();
    }
  }
}

void ThumbImage::prepareBackground() {
  int thumbWidth, thumbHeight;

  /* sizes, dimensions */
  res->fitOutside(THUMB_SIZE, THUMB_SIZE, &thumbWidth, &thumbHeight);

  /* crop */
  wxImage* image = res->get(thumbWidth, thumbHeight, true);
  if (thumbWidth > thumbHeight) {
    /* crop width */
    wxRect crop((thumbWidth - THUMB_SIZE) / 2, 0, THUMB_SIZE, THUMB_SIZE);
    bg = wxBitmap(image->GetSubImage(crop).ConvertToGreyscale());
  } else if (thumbHeight > thumbWidth) {
    /* crop height */
    wxRect crop(0, (thumbHeight - THUMB_SIZE) / 2, THUMB_SIZE, THUMB_SIZE);
    bg = wxBitmap(image->GetSubImage(crop).ConvertToGreyscale());
  } else {
    bg = wxBitmap(image->ConvertToGreyscale());
  }
}

void ThumbImage::prepareForeground() {
  int thumbWidth, thumbHeight;

  /* sizes, dimensions */
  res->fitOutside(THUMB_SIZE, THUMB_SIZE, &thumbWidth, &thumbHeight);

  /* rectangle for crop */
  wxRect crop;
  if (thumbWidth > thumbHeight) {
    /* crop width */
    crop = wxRect((thumbWidth - THUMB_SIZE) / 2, 0, THUMB_SIZE, THUMB_SIZE);
  } else if (thumbHeight > thumbWidth) {
    /* crop height */
    crop = wxRect(0, (thumbHeight - THUMB_SIZE) / 2, THUMB_SIZE, THUMB_SIZE);
  } else {
    /* no crop */
    crop = wxRect(0, 0, THUMB_SIZE, THUMB_SIZE);
  }

  /* draw "blob" foreground */
  wxBitmap fg1(crop.width, crop.height); // ,32?
  wxBitmap fg2(crop.width, crop.height); // ,32?

  wxMemoryDC dc1(fg1);
  wxMemoryDC dc2(fg2);
  
  wxColour c1(model->getColour(cluster));
  wxColour c2;
  wxImage::RGBValue rgb(c1.Red(), c1.Green(), c1.Blue());
  wxImage::HSVValue hsv;
  
  hsv = wxImage::RGBtoHSV(rgb);
  hsv.value *= 0.9;
  rgb = wxImage::HSVtoRGB(hsv);
  c2.Set(rgb.red, rgb.green, rgb.blue);
  
  dc1.SetBackground(wxBrush(c1));
  dc2.SetBackground(wxBrush(c2));
  dc1.Clear();  
  dc2.Clear();

  wxImage img1(fg1.ConvertToImage());
  sparse_mask mask(model->calcMask(cluster, crop, thumbWidth, thumbHeight));
  hide(img1);
  show(img1, mask);
  
  dc2.DrawBitmap(img1, 0, 0, true);
  
  wxImage img2(fg2.ConvertToImage());
  a.resize(crop.height, crop.width);
  model->calcAlpha(cluster, crop, thumbWidth, thumbHeight, a);
  applyAlpha(img2, a);
  fg = wxBitmap(img2.Blur(2));
  
  fgPrepared = true;
}

BEGIN_EVENT_TABLE(ThumbImage, wxWindow)
EVT_PAINT(ThumbImage::OnPaint)
//EVT_LEFT_DOWN(ThumbImage::OnLeftDown)
EVT_LEFT_UP(ThumbImage::OnLeftUp)
//EVT_LEAVE_WINDOW(ThumbImage::OnLeave)
END_EVENT_TABLE()

