#ifndef INDII_TINT_GUI_TINTWINDOW_HPP
#define INDII_TINT_GUI_TINTWINDOW_HPP

#include "PreviewPane.hpp"
#include "ThumbPane.hpp"
#include "HSLPane.hpp"
#include "PreProcessPane.hpp"
#include "PostProcessPane.hpp"
#include "ChannelMixerPane.hpp"
#include "TintWindowObserver.hpp"

#include "../ImageResource.hpp"
#include "../ClusterModel.hpp"
#include "../ClusterModelObserver.hpp"

#include "wx/frame.h"
#include "wx/aui/aui.h"

namespace indii {
  namespace tint {
    namespace gui {
/**
 * Main window.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * @version $Rev: 124 $
 * @date $Date: 2010-01-24 10:54:26 +0800 (Sun, 24 Jan 2010) $
 */
class TintWindow : public wxFrame, public ClusterModelObserver {
public:
  /**
   * Modes.
   */
  enum Mode {
    /**
     * Standalone mode. Includes Save and Save As buttons.
     */
    STANDALONE,
    
    /**
     * Embedded mode. Includes OK button.
     */
    EMBED
  };

  /**
   * Constructor.
   *
   * @param parent Parent.
   * @param res Image resource.
   * @param model Cluster model.
   * @param mode Mode of operation.
   * @param obs Enclosing application, if not embedded mode.
   */
  TintWindow(wxWindow* parent, ImageResource* res, ClusterModel* model,
      const Mode mode = EMBED, TintWindowObserver* obs = NULL);

  /**
   * Constructor.
   *
   * @param parent Parent.
   * @param path Path of image to open.
   * @param mode Mode of operation.
   * @param obs Enclosing application, if not embedded mode.
   */
  TintWindow(wxWindow* parent, const wxString& path, const Mode = STANDALONE,
      TintWindowObserver* obs = NULL);

  /**
   * Destructor.
   */
  ~TintWindow();

  /**
   * Is image saved?
   */
  bool IsSaved();
  
  /**
   * Get return code.
   */
  int GetReturnCode();

  /**
   * Set return code.
   */
  void SetReturnCode(const int returnCode);

  /**
   * Close event.
   */
  void OnSystemClose(wxCloseEvent& evt);

  /**
   * Drag and drop permission event.
   */
  void OnAllowNotebookDnD(wxAuiNotebookEvent& evt);

  /**
   * Save button event.
   */
  void OnButtonSave(wxCommandEvent& evt);

  /**
   * Save As button event.
   */
  void OnButtonSaveAs(wxCommandEvent& evt);

  /**
   * Close button event.
   */
  void OnButtonClose(wxCommandEvent& evt);

  /**
   * OK button event.
   */
  void OnButtonOk(wxCommandEvent& evt);

  /**
   * Cancel button event.
   */
  void OnButtonCancel(wxCommandEvent& evt);

  /**
   * About button event.
   */
  void OnButtonAbout(wxCommandEvent& evt);

  /**
   * Default button event.
   */
  void OnButtonDefault(wxCommandEvent& evt);

  /**
   * Handle zoom out button press.
   */
  void OnButZoomOut(wxCommandEvent& evt);
  
  /**
   * Handle zoom in button press.
   */
  void OnButZoomIn(wxCommandEvent& evt);
  
  /**
   * Handle zoom 1:1 button press.
   */
  void OnButZoomNormal(wxCommandEvent& evt);
  
  /**
   * Handle zoom fit button press.
   */
  void OnButZoomFit(wxCommandEvent& evt);
  
  virtual void notifyClusterChange(const unsigned int i);

  virtual void notifyNumClustersChange();

  virtual void notifyNumRepetitionsChange();

  virtual void notifySaturationThresholdChange();

  virtual void notifyMaxPixelsChange();

  virtual void notifySaturationDecayChange();

  virtual void notifyCentroidDecayChange();

  virtual void notifySaturationSoftnessChange();

  virtual void notifyCentroidSoftnessChange();

  virtual void notifyGreyscaleChange();

private:
  /**
   * Image resources.
   */
  ImageResource* res;

  /**
   * Cluster model.
   */
  ClusterModel* model;

  /**
   * Enclosing application.
   */
  TintWindowObserver* obs;
  
  /**
   * Do we own the res and model?
   */
  bool own;

  /**
   * Mode of operation.
   */
  Mode mode;

  /**
   * Return code.
   */
  int returnCode;

  /**
   * Is the image, in its current state, saved?
   */
  bool isSaved;
  
  /**
   * Do we have a path for saving the image?
   */
  bool havePath;

  /**
   * Path to saved image.
   */
  wxString path;

  /**
   * AUI manager.
   */
  wxAuiManager* manager;

  /**
   * Preview pane.
   */
  PreviewPane* previewPane;
  
  /**
   * Thumbs pane.
   */
  ThumbPane* thumbPane;
  
  /**
   * Pre-processing pane.
   */
  PreProcessPane* prePane;
  
  /**
   * Post-processing pane.
   */
  PostProcessPane* postPane;

  /**
   * Channel mixer pane.
   */
  ChannelMixerPane* mixerPane;

  /**
   * Hue-saturation-luminance pane.
   */
  HSLPane* hslPane;

  /**
   * Initialise.
   *
   * @param mode Mode of operation.
   */
  void init(const Mode mode);

  /**
   * Save image.
   */
  void save();

  /*
   * Control ids.
   */
  enum ID {
    ID_BUT_REGISTER,
    ID_BUT_SPLASH,
    ID_BUT_DEFAULT,
    ID_BUT_ZOOM_OUT,
    ID_BUT_ZOOM_IN,
    ID_BUT_ZOOM_NORMAL,
    ID_BUT_ZOOM_FIT
  };

  DECLARE_EVENT_TABLE()

};

    }
  }
}

#endif
