/*
 *  ISEM - Instructional Sparc EMulator and tkisem
 *  Copyright (C) 1993, 1994, 1995, 1996
 *	 Department of Computer Science,
 *       The University of New Mexico
 *
 *  Please send questions, comments, and bug reports to: isem@cs.unm.edu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#if __GNUC__
#define UNUSED __attribute__ ((unused)) 
#else
#define UNUSED
#endif

static char rcsid[] UNUSED = "$Id: divide.cpp 1.1 Fri, 25 Oct 1996 18:04:04 -0600 maccabe $";

//--------------------------------------------------------------------------
// Divide Instructions - see p176 of SPARC Architecture Manual, Version 8
//    this code does NOT store the remainder in the Y register
//--------------------------------------------------------------------------

#include "Assert.h"
#include "sizedefs.h"
#include "Instruct.h"
#include "RegBlock.h"
#include "IU.h"


inline void RShiftComb( UInt32& hi, UInt32& lo, UInt32 cnt ) {
    // treat  hi  and  lo  as a 64-bit vlaue and perform a right shift
    UInt32 mask = ~(~(UInt32)0<<cnt);
    lo = ((hi&mask)<<(32-cnt)) | lo>>cnt;
    hi = hi >> cnt;
}

inline UInt32 GetMid( UInt32 hi, UInt32 lo, UInt32 cnt ) {
    // treat  hi  and  lo  as a 64-bit value and extract the 32 bits
    // starting  cnt  bits from the left
    UInt32 mask = ~(~(UInt32)0<<cnt);
    return ((hi&mask)<<(32-cnt)) | lo>>cnt;
}

// divide_unsigned - 64-bit / 32-bit 
//
static void
divide_unsigned(UInt32 Y, UInt32& result, UInt32 op1, UInt32 op2, int& overflow) {

    UInt32 divisor_hi = op2, divisor_lo = 0;
    UInt32 shift_cnt = 32;    // we've already shifted the divisor by 32 bits
    UInt32 dividend = op1;

    result = 0;
    overflow = 0;
    
    // First, we deal with possible overflow
    if( Y >= divisor_hi ) {
	overflow = 1;
	result = ~0;
    } else {
	// next, we deal with the most significant 32 bits of the dividend
	while ( divisor_hi > Y ) {
	    shift_cnt--;
	    RShiftComb( divisor_hi, divisor_lo, 1 );
	}
	while ( Y != 0 ) {
	    while ( op2 > GetMid( Y, dividend, shift_cnt ) ) {
		shift_cnt--;
		RShiftComb( divisor_hi, divisor_lo, 1 );
	    }
	    
	    if ( divisor_lo > dividend ) {
//		dividend += (((UInt32)1<<31) - divisor_lo);
		dividend += (~(UInt32)0 - divisor_lo) + 1;
		Y -= 1;
	    } else {
		dividend -= divisor_lo;
	    }
	    Y -= divisor_hi;
	    result += 1<<shift_cnt;
	}

	result += dividend / op2;
    }
}


static void
divide_signed(UInt32 Y, UInt32& result, UInt32 op1, UInt32 op2, int& overflow) {
    int sign1 = 1, sign2 = 1;

    if ( op2 & (1<<31) ) {
	op2 = ~op2 + 1;
	sign1 = -1;
    }

    if ( Y & (1<<31) ) {

	// invert both halves of the 64-bit number
	op1 = ~op1;
	Y = ~Y;

	// add 1 to the 64-bit number
	if( op1 == ~0U ) {
	    op1 = 0;
	    Y++;
	} else {
	    op1++;
	}
	sign2 = -1;
    }

    divide_unsigned(Y, result, op1, op2, overflow);

    overflow |= result >> 31;

    if( overflow ) {
	if( sign1 == sign2 ) {
	    result = 0x7fffffff;
	} else {
	    result = 0x80000000;
	}
    } else {
	if ( sign1 != sign2 ) {
	    result = ~result + 1;
	}
    }
}

void IntegerUnit::divide(const Instruction& inst){
    UInt32 operand2 = (inst.i() == 0) ? reg[inst.rs2()] : inst.simm13();
    int op3 = inst.op3();
    int overflow;

    // next
    if( operand2 == 0 ) {
	trap = 1;
	trapFlag[division_by_zero] = 1;
	return;
    } else {
	UInt32 result;   

	if( op3==Instruction::UDIV || op3==Instruction::UDIVcc ) {
	    divide_unsigned(IU_Y, result, reg[inst.rs1()], operand2, overflow);
	} else if( op3==Instruction::SDIV || op3==Instruction::SDIVcc ) {
	    divide_signed(IU_Y, result, reg[inst.rs1()], operand2, overflow);
	} else {
	    Assert(0, "Unknown op3: PLEASE EMAIL isem@cs.unm.edu with this error!");
	}
	// next

	// the division operations set the result correctly on overflow.

	reg[inst.rd()] = result;

	if( op3 == Instruction::UDIVcc || op3==Instruction::SDIVcc ) {
	    IU_N = (result >> 31);         
	    IU_Z = (result == 0) ? 1 : 0;
	    IU_V = overflow;
	    IU_C = 0;
	}
    }
}
