namespace eval xargs {
    variable xargs

    # Define the xargs array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set xargs {
	list.reset ""
	list.clear ""
	entry.file ""
	dialog ""
	max_lines 1
	max_chars 20000
	max_procs 1
	show_help ""
	show_version ""
    }
}

# xargs::create --
#
#   Method to create the dialog box for the xargs command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc xargs::create { } {
    global tkWorld
    variable xargs

    # Put the focus on the xargs dialog if it is already open.
    if [winfo exists $xargs(dialog)] {
	switch -- [wm state $xargs(dialog)] {
	    normal {
		raise $xargs(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $xargs(dialog)
	    }
	}
	focus $xargs(dialog)
	return
    } else {
	set xargs(dialog) [dialog::create .xargs Xargs]
    }

    # The first tab has file and regexp information, along with
    # the standard command line options.
    set tab1 [tabnotebook::page [dialog::interior $xargs(dialog)] "Options"]

    # Now build the individual checkbutton options available to
    # the user.
    set f0 [frame $tab1.f0 \
	    -class TabnotebookFrame]
    checkbutton $f0.show_help \
	    -text "Display Help" \
	    -variable xargs::xargs(show_help) \
	    -onvalue "-help" \
	    -offvalue ""
    checkbutton $f0.show_version \
	    -text "Show Version" \
	    -variable xargs::xargs(show_version) \
	    -onvalue "-version" \
	    -offvalue ""
    grid $f0.show_help $f0.show_version \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    message $f1.help_message \
	    -width 350 \
	    -text "Use xargs to control the input to a command from a\
	    pipe as in:\n\nls *.tcl | xargs wc -l\n\nwill produce a\
	    line count listing of all the tcl files in a directory."
    grid $f1.help_message 

    # Build the first tab.
    pack $f0 $f1 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # The second tab provides access to some really advanced options
    # that the user should never have to change unless they really
    # understand the xargs command.
    set tab2 [tabnotebook::page [dialog::interior $xargs(dialog)] "Advanced"]

    # Build the frame for the advanced options.
    set f2 [frame $tab2.f2 \
	    -class TabnotebookFrame]
    checkbutton $f2.null \
	    -text "Null" \
	    -variable xargs::xargs(null) \
	    -onvalue "0" \
	    -offvalue ""
    checkbutton $f2.no_run \
	    -text "No Run" \
	    -variable xargs::xargs(no_run) \
	    -onvalue "r" \
	    -offvalue ""
    checkbutton $f2.verbose \
	    -text "Verbose" \
	    -variable xargs::xargs(verbose) \
	    -onvalue "t" \
	    -offvalue ""
    checkbutton $f2.exit \
	    -text "Exit" \
	    -variable xargs::xargs(exit) \
	    -onvalue "x" \
	    -offvalue ""
    grid $f2.null $f2.no_run \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f2.verbose $f2.exit \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Now get all of the command line options that have entries.
    set f3 [frame $tab2.f3 \
	    -class TabnotebookFrame]
    message $f3.label_eof_str \
	    -text "End of File String" \
	    -width 200 \
	    -anchor e
    set xargs(entry.eof_str) [entry $f3.entry_eof_str \
	    -width 10 \
	    -textvariable xargs::xargs(eof_str)]
    message $f3.label_replace_str \
	    -text "Replace String" \
	    -width 200 \
	    -anchor e
    set xargs(entry.replace_str) [entry $f3.entry_replace_str \
	    -width 10 \
	    -textvariable xargs::xargs(replace_str)]
    message $f3.label_max_lines \
	    -text "Maximum Lines of Input per Command Line" \
	    -width 200 \
	    -anchor e
    set xargs(entry.max_lines) [entry $f3.entry_max_lines \
	    -width 5 \
	    -textvariable xargs::xargs(max_lines)]
    message $f3.label_max_chars \
	    -text "Maximum Characters of ..." \
	    -width 200 \
	    -anchor e
    set xargs(entry.max_chars) [entry $f3.entry_max_chars \
	    -width 7 \
	    -textvariable xargs::xargs(max_chars)]
    message $f3.label_max_procs \
	    -text "Maximum Number of Procedures" \
	    -width 200 \
	    -anchor e
    set xargs(entry.max_procs) [entry $f3.entry_max_procs \
	    -width 5 \
	    -textvariable xargs::xargs(max_procs)]
    grid $f3.label_eof_str $f3.entry_eof_str \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f3.label_replace_str $f3.entry_replace_str \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f3.label_max_lines $f3.entry_max_lines \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f3.label_max_chars $f3.entry_max_chars \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f3.label_max_procs $f3.entry_max_procs \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Build a frame for the message to warn the user to not change
    # this properties unless they really know what they are doing.
    set f4 [frame $tab2.f4 \
	    -class TabnotebookFrame]
    message $f4.message_help \
	    -text "Do not change these settings unless you really\
	    understand the effect on the command.  Consult the man\
	    page for more information on their meaning." \
	    -anchor w \
	    -width 400
    grid $f4.message_help \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Build the first tab.
    pack $f2 $f3 $f4 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $xargs(entry.eof_str)

    # Define the lists for the reset and clear methods
    set xargs(list.reset) "null no_run verbose exit show_help \
	    show_version"
    set xargs(list.clear) "eof_str replace_str max_lines max_chars \
	    max_procs"
}

# xargs::ok --
#
#   Method to insert the command the user has created into the CC
#   as a Tcl list.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc xargs::ok { } {
    global tkWorld
    variable xargs

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    $tkWorld(cmd_center) insert insert "xargs "

    # Build the command line argument.
    set cmd_arg ""
    foreach x $xargs(list.reset) {
	if [string length $xargs($x)] {
	    switch -- $x {
		show_help -
		show_version {
		    $tkWorld(cmd_center) insert insert "-$xargs($x) "
		}
		default {
		    append cmd_arg $xargs($x)
		}
	    }
	}
    }
    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert "-$cmd_arg "
    }

    foreach x $xargs(list.clear) {
	if [string length $xargs($x)] {
	    switch -- $x {
		"eof_str" {
		    $tkWorld(cmd_center) insert insert "-e$xargs($x) "
		}
		"replace_str" {
		    $tkWorld(cmd_center) insert insert "-i$xargs($x) "
		}
		"max_lines" {
		    if ![regexp {[0-9].*} $xargs($x)] {
			$xargs(entry.max_lines) configure \
				-bg red -fg white
			stderr::log 2002
		    } else {
			if {$xargs($x) != "1"} {
			    $tkWorld(cmd_center) insert insert "-l$xargs($x) "
			}
		    }
		}
		"max_chars" {
		    if ![regexp {[0-9].*} $xargs($x)] {
			$xargs(entry.max_chars) configure \
				-bg red -fg white
			stderr::log 2002
		    } else {
			if {$xargs($x) != "20000"} {
			    $tkWorld(cmd_center) insert insert "-l$xargs($x) "
			}
		    }
		}
		"max_procs" {
		    if ![regexp {[0-9].*} $xargs($x)] {
			$xargs(entry.max_procs) configure \
				-bg red -fg white
			stderr::log 2002
		    } else {
			if {$xargs($x) != "1"} {
			    $tkWorld(cmd_center) insert insert "-l$xargs($x) "
			}
		    }
		}
	    }
	}
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# xargs::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc xargs::reset { } {
    variable xargs

    # It is easier to list the elements to exclude rather than list
    # all of the values to reset.
    foreach x $xargs(list.reset) {
	set xargs($x) ""
    }
}

# xargs::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc xargs::clear { } {
    variable xargs

    # Reset the data structure elements and bg/fg.
    foreach x $xargs(list.clear) {
	switch -- $x {
	    "max_lines" -
	    "max_procs" {
		set xargs($x) 1
	    }
	    "max_chars" {
		set xargs($x) 20000
	    }
	    default {
		set xargs($x) ""
	    }
	}
	$xargs(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $xargs(entry.eof_str)
}

# xargs::help --
#
#   Method to invoke the Xargs Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc xargs::help { } {
    global tkWorld

    help::create "help/xargs.html" "Xargs Command Help"
}

# xargs::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc xargs::close { } {
    variable xargs
    
    balloonhelp::cancel
    destroy $xargs(dialog)
}

# xargs::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc xargs::open { } {
    variable xargs

    # Insert the file list and move the cursor.
    $xargs(entry.file) insert insert [file::select]
    $xargs(entry.file) icursor end

    # Set the focus on the entry with the file list in it.
    focus $xargs(entry.file)
}
