/***********************************************************************
 * 
 * File: timer.c
 *
 * Author: Martin Fluch
 *
 *     Copyright (c) 1999 Martin Fluch, All rights reserved
 *     
 *     This program is free software; you can redistribute it and/or 
 *     modify it under the terms of the GNU General Public License as 
 *     published by the Free Software Foundation; either version 2 of 
 *     the License, or (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *     GNU General Public License for more details.
 * 
 *     To receive a copy of the GNU General Public License, please write
 *     to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *     Boston, MA 02111-1307 USA
 *
 ***********************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include <linux/types.h>
#include <stdio.h>
#include <string.h>

#include <curses.h>
#include <menu.h>

#include "ntpctl.h"


#define TIMER_ALL_MODE		0x0000
#define TIMER_MANUAL_AC		0x0100
#define TIMER_MANUAL_BATTERY	0x0200
#define TIMER_HIGH		0xf300
#define TIMER_AUTO		0xf400

#define TIMER_HIB		0xff00

#define RTC_MODE_TIME		0
#define RTC_MODE_DATE		1

/***********************************************************************
 *
 * get_timer()
 * 
 * returns -1 on error, else it returns the timer value in minutes
 *
 * see TP 770 Technical Reference page 117 to 120
 *
 ***********************************************************************/

int get_timer(int timer, word *cap)
{
  smapi_outparm_t *result;

  if((result=do_smapi(0x32,(timer!=TIMER_HIB) ? 0x02 : 0x06,
		      (timer!=TIMER_HIB) ? timer : 0x00,0,0,0,0,FALSE))==NULL)
    return -1;

  if(cap) *cap=result->wParm1 & 0x0100;

  return result->wParm2 & 0x00ff;
}

/***********************************************************************
 *
 * set_timer()
 *
 * returns 1 on error
 *
 * see TP 770 Technical Reference page 117 to 120
 *
 ***********************************************************************/

int set_timer(int timer, word value)
{
  switch(timer)
    {
    case TIMER_HIB:
      if(do_smapi(0x32,0x07,(value & 0x00ff),0,0,0,0,TRUE)==NULL) return 1;
      break;

    default:
      if(do_smapi(0x32,0x03,timer | (value & 0x00ff),0,0,0,0,TRUE)==NULL)
	return 1;
      break;
    }

  return 0;
}

/***********************************************************************
 *
 * handle_timer()
 *
 ***********************************************************************/

int handle_timer(struct menuinfo *mi, int action, int timer)
{
  int value=0;
  int chconfig=FALSE;

  if((value=get_timer(timer,NULL))==-1) return 1;

  switch(action)
    {
    case ACTION_NEXT:
      value=value+1;
      chconfig=TRUE;
      break;

    case ACTION_PREV:
      if(value>0) 
	{
	  value=value-1;
	  chconfig=TRUE;
	}
      break;

    case ACTION_KEY_RET:
      mvaddstr(LINES-2,3,"Enter time in minutes: ");
      clrtoeol();
      echo();
      if(scanw("%d",&value)==1) chconfig=(value>=0) ? TRUE : FALSE;
      noecho();
      break;

    case ACTION_KEY_D:
      value=0;
      chconfig=TRUE;
      break;
    }
  
  if(chconfig) if(set_timer(timer,value)) return 1;
  if((value=get_timer(timer,NULL))==-1) return 1;

  if(value) snprintf(mi->desc,MAXDESCSIZE,"%d min    ",value);
  else strncpy(mi->desc,"disabled",MAXDESCSIZE);

  return 0;
}

/***********************************************************************
 *
 * tm_system_manual_ac()
 *
 ***********************************************************************/

int tm_system_manual_ac(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer delay--manual ac mode",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_timer];
    }

  return handle_timer(mi,action,TIMER_MANUAL_AC);
}

/***********************************************************************
 *
 * tm_system_manual_battery()
 *
 ***********************************************************************/

int tm_system_manual_battery(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer delay--manual battery mode",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_timer];
    }

  return handle_timer(mi,action,TIMER_MANUAL_BATTERY);
}

/***********************************************************************
 *
 * tm_system_high()
 *
 ***********************************************************************/

int tm_system_high(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer delay--high mode",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_timer];
    }

  return handle_timer(mi,action,TIMER_HIGH);
}

/***********************************************************************
 *
 * tm_system_auto()
 *
 ***********************************************************************/

int tm_system_auto(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer delay--auto mode",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_timer];
    }

  return handle_timer(mi,action,TIMER_AUTO);
}

/***********************************************************************
 *
 * tm_hibernation()
 *
 ***********************************************************************/

int tm_hibernation(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer delay--hibernate from suspend",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_timer];
    }

  return handle_timer(mi,action,TIMER_HIB);
}

/***********************************************************************
 *
 * handle_timer_mode()
 *
 ***********************************************************************/

int handle_timer_control(struct menuinfo *mi, int action, int bitnr)
{
  smapi_outparm_t *result;
  int tmp;
  int chconfig=FALSE;

  if((result=do_smapi(0x22,0x02,0,0,0,0,0,FALSE))==NULL) return -1;
  
  if((action==ACTION_PREV) || (action==ACTION_NEXT))
    {
      if((result->wParm2 >> bitnr) & 0x0001) action=ACTION_KEY_N;
      else				     action=ACTION_KEY_Y;
    }

  switch(action)
    {
    case ACTION_KEY_E:
    case ACTION_KEY_Y:
      tmp=(result->wParm2 & 0x00ff) |  (0x0001 << bitnr);
      chconfig=TRUE;
      break;

    case ACTION_KEY_D:
    case ACTION_KEY_N:
      tmp=(result->wParm2 & 0x00ff) & ~(0x0001 << bitnr);
      chconfig=TRUE;
      break;
    }

  if(chconfig) if(do_smapi(0x22,0x03,tmp,0,0,0,0,TRUE)==NULL) return 1; 
  if((result=do_smapi(0x22,0x02,0,0,0,0,0,FALSE))==NULL) return 1;

  tmp=(result->wParm2 >> bitnr) & 0x0101;
  snprintf(mi->desc,MAXDESCSIZE,"%s       %s",
	   (tmp & 0x0001) ? "yes" : "no ",
	   (tmp & 0x0100) ? "(capable)    " : "(not capable)");
 
  return 0;
}

/***********************************************************************
 *
 * tm_ctrl_hib_susp()
 *
 ***********************************************************************/

int tm_ctrl_hib_susp(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer control hibernate/suspend",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_ed];      
    }

  return handle_timer_control(mi, action, 0);
}

/***********************************************************************
 *
 * tm_ctrl_standby()
 *
 ***********************************************************************/

int tm_ctrl_standby(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer control standby",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_ed];      
    }

  return handle_timer_control(mi, action, 1);
}

/***********************************************************************
 *
 * tm_ctrl_LCD_off()
 *
 ***********************************************************************/

int tm_ctrl_LCD_off(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer control LCD off",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_ed];      
    }

  return handle_timer_control(mi, action, 2);
}

/***********************************************************************
 *
 * tm_ctrl_HDD_off()
 *
 ***********************************************************************/

int tm_ctrl_HDD_off(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"Timer control HDD off",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_ed];      
    }

  return handle_timer_control(mi, action, 3);
}

/***********************************************************************
 *
 * checkRTCrange()
 *
 ***********************************************************************/

int checkRTCrange(dword *p4, dword *p5)
{
  dword op4,op5;

  op4=*p4;
  op5=*p5;

  if((((*p4 >>  0) & 0xff) > 0x59) || (((*p4 >>  0) & 0x0f) > 0x09) || 
     (((*p4 >>  0) & 0xf0) > 0x90)) *p4 &= 0xffffff00L;

  if((((*p4 >>  8) & 0xff) > 0x59) || (((*p4 >>  8) & 0x0f) > 0x09) || 
     (((*p4 >>  8) & 0xf0) > 0x90)) *p4 &= 0xffff00ffL;

  if((((*p4 >> 16) & 0xff) > 0x23) || (((*p4 >> 16) & 0x0f) > 0x09) || 
     (((*p4 >> 16) & 0xf0) > 0x90)) *p4 &= 0xff00ffffL;

  if((((*p5 >>  0) & 0xff) > 0x31) || (((*p5 >>  0) & 0x0f) > 0x09) || 
     (((*p5 >>  0) & 0xf0) > 0x90) || (((*p5 >>  0) & 0xff) < 0x01)) 
    {*p5 &= 0xffffff00;*p5|=0x00000001L;}

  if((((*p5 >>  8) & 0xff) > 0x12) || (((*p5 >>  8) & 0x0f) > 0x09) || 
     (((*p5 >>  8) & 0xf0) > 0x90) || (((*p5 >>  8) & 0xff) < 0x01)) 
    {*p5 &= 0xffff00ff;*p5|=0x00000100L;}

  if((((*p5 >> 16) & 0xff) > 0x99) || (((*p5 >> 16) & 0x0f) > 0x09) || 
     (((*p5 >> 16) & 0xf0) > 0x90)) *p5 &= 0xff00ffffL;

  return ((op4==*p4) && (op5==*p5)) ? 0 : 1;
}

/***********************************************************************
 *
 * handle_RTC()
 *
 ***********************************************************************/

int handle_RTC(struct menuinfo *mi, int action, int RTCmode)
{
  smapi_outparm_t *result;
  int chconfig=FALSE;
  dword tmp;

  if((result=do_smapi(0x34,0x02,0,0,0,0,0,FALSE))==NULL) return 1;

  if(action==ACTION_KEY_RET)
    {
      mvaddstr(LINES-2,3,"Enter new value ");
      addstr((RTCmode==RTC_MODE_TIME) ? "(hhmmss): " : "(yymmdd): ");
      clrtoeol();
      echo();
      if(scanw("%lx",&tmp)==1) chconfig=TRUE;
      noecho();
    }

  if(chconfig)
    {
      if(RTCmode==RTC_MODE_TIME)
	{
	  result->dwParm4=tmp & 0x00ffffffL;
	}
      else
	{
	  result->dwParm5&=0x80000000L;
	  result->dwParm5|=(tmp&0x00ffffffL);
	}

      checkRTCrange(&(result->dwParm4),&(result->dwParm5));

      if(do_smapi(0x34,0x03,0,0,0,result->dwParm4,result->dwParm5,TRUE)==NULL)
	return 1;
      if((result=do_smapi(0x34,0x02,0,0,0,0,0,FALSE))==NULL) return 1;
    }

  checkRTCrange(&(result->dwParm4),&(result->dwParm5));

  switch(RTCmode)
    {
    case RTC_MODE_TIME:
      tmp=result->dwParm4;
      snprintf(mi->desc,MAXDESCSIZE,"%.2x:%.2x:%.2x",
	       (int)((tmp >> 16) & 0xff),
	       (int)((tmp >>  8) & 0xff),
	       (int)((tmp >>  0) & 0xff));
      break;

    case RTC_MODE_DATE:
      tmp=result->dwParm5;
      snprintf(mi->desc,MAXDESCSIZE,"%.2x-%.2x-%.2x",
	       (int)((tmp >> 16) & 0xff),
	       (int)((tmp >>  8) & 0xff),
	       (int)((tmp >>  0) & 0xff));
      break;
    }
  
  return 0;
}

/***********************************************************************
 *
 * rt_time()
 *
 ***********************************************************************/

int rt_time(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"RTC appointment time (hh:mm:ss)",MAXNAMESIZE);
      strncpy(mi->desc,"__:__:__",MAXDESCSIZE);
      mi->help=helptext[HELP_value];
    }

  return handle_RTC(mi, action, RTC_MODE_TIME);
}

/***********************************************************************
 *
 * rt_date()
 *
 ***********************************************************************/

int rt_date(struct menuinfo *mi, int action)
{
  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"RTC appointment date (yy-mm-dd)",MAXNAMESIZE);
      strncpy(mi->desc,"__-__-__",MAXDESCSIZE);
      mi->help=helptext[HELP_value];
    }

  return handle_RTC(mi, action, RTC_MODE_DATE);
}

/***********************************************************************
 *
 * rt_daily()
 *
 ***********************************************************************/

int rt_daily(struct menuinfo *mi, int action)
{
  static struct menuinfo *dailymi=NULL;
  smapi_outparm_t *result;
  int chconfig=FALSE;

  if(mi==NULL)
    {
      if(dailymi==NULL) return 1;
      mi=dailymi;
    }
  else dailymi=mi;

  if(action==ACTION_INIT)
    {
      strncpy(mi->name,"RTC appointment daily",MAXNAMESIZE);
      strncpy(mi->desc,"__",MAXDESCSIZE);
      mi->help=helptext[HELP_yn];
    }

  if((result=do_smapi(0x34,0x02,0,0,0,0,0,FALSE))==NULL) return 1;

  if((action==ACTION_NEXT) || (action==ACTION_PREV))
    {
      action = (result->dwParm5 & 0x80000000) ? ACTION_KEY_N : ACTION_KEY_Y;
    }

  switch(action)
    {
    case ACTION_KEY_E:
    case ACTION_KEY_Y:
      result->dwParm5 |= 0x80000000;
      chconfig=TRUE;
      break;

    case ACTION_KEY_D:
    case ACTION_KEY_N:
      result->dwParm5 &= ~0x80000000;
      chconfig=TRUE;
      break;
    }

  checkRTCrange(&(result->dwParm4),&(result->dwParm5));

  if(chconfig)
    {
      if(!do_smapi(0x34,0x03,0,0,0,result->dwParm4,result->dwParm5,TRUE))
	return 1;
      if(action!=ACTION_INIT) if(pm_resume_RTC(NULL, ACTION_UPDATE)) return 1;
      if((result=do_smapi(0x34,0x02,0,0,0,0,0,FALSE))==NULL) return 1;
    }

  if(result->dwParm5 & 0x80000000) strncpy(mi->desc,"yes",MAXDESCSIZE);
  else strncpy(mi->desc,"no ",MAXDESCSIZE);

  return 0;
}
