// timer.c - timer routines are provided for the PDP emulator
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <sys/time.h>
#include <signal.h>

#include "emu/defs.h"

#ifdef HAVE_SIGACTION
struct sigaction sigTimer;
#endif /* HAVE_SIGACTION */

static struct itimerval startTimer =
{ 
	{ 0, 10000 },
	{ 0, 10000 }
};

static struct itimerval stopTimer =
{ 
	{ 0, 0 },
	{ 0, 0 }
};

static struct itimerval oldTimer;

#define  NOQUEUE_WAIT 10000

CLK_QUEUE *ts10_SimClock  = NULL; // Simulation Clock Queue
CLK_QUEUE *ts10_RealClock = NULL; // Real Clock Queue
int32     ts10_ClkInterval = 0;

void ts10_SetRealTimer(CLK_QUEUE *qptr)
{
	// Enter a timer entry into real clock queue.
	qptr->Next     = ts10_RealClock;
	ts10_RealClock = qptr;
}

void ts10_CancelRealTimer(CLK_QUEUE *qptr)
{
	CLK_QUEUE *cptr;

	// Remove that entry from the real timer queue.
	if (ts10_RealClock == qptr) {
		ts10_RealClock = qptr->Next;
		return;
	}
	
	for (cptr = ts10_RealClock; cptr; cptr = cptr->Next) {
		if (cptr->Next == qptr) {
			cptr->Next = qptr->Next;
			return;
		}
	}
}

void ts10_TickRealTimer(int sig)
{
	CLK_QUEUE *qptr, *pptr;

	if (qptr = ts10_RealClock) {
		do {
			if ((qptr->Flags & CLK_ENABLE) &&
			    ((qptr->outTimer -= CLK_TICK) <= 0)) {
				// Update clock queue
				if (qptr->Flags & CLK_REACTIVE) {
					qptr->outTimer = qptr->nxtTimer;
				} else {
					if (qptr == ts10_RealClock)
						ts10_RealClock = qptr->Next;
					else
						pptr = qptr->Next;
				}
				// Execute a tick routine.
				qptr->Execute(qptr->Device);
			}
			pptr = qptr;
		} while (qptr = qptr->Next);
	}
}

void ts10_SetTimer(CLK_QUEUE *qptr)
{
	CLK_QUEUE *cptr, *pptr = NULL;
	int adjTimer = 0;

	// First, check if this unit already is activated.
	if (qptr->Flags & CLK_PENDING) {
#ifdef DEBUG
		if (dbg_Check(DBG_TIMER))
			dbg_Printf("TIMER: Entry: %s - Already Pending.\n",
				qptr->Name ? qptr->Name : "(None)");
#endif /* DEBUG */
		return;
	}

	if ((qptr->outTimer < 0) || (qptr->nxtTimer < 0)) {
#ifdef DEBUG
		if (dbg_Check(DBG_TIMER)) {
			dbg_Printf("TIMER: Entry: %s - Negative Countdown.\n",
				qptr->Name ? qptr->Name : "(None)");
			dbg_Printf("TIMER:    Current %d   Next %d\n",
				qptr->outTimer, qptr->nxtTimer);
		}
#endif /* DEBUG */
		return;
	}

	if (ts10_ClkInterval < 0) {
#ifdef DEBUG
		dbg_Printf("TIMER: *** Negative Clock Interval: %d\n",
			ts10_ClkInterval);
#endif /* DEBUG */
		ts10_ClkInterval = 0;
	}

	qptr->outTimer =  qptr->nxtTimer;
	qptr->Flags    |= CLK_PENDING;

	if (ts10_SimClock == NULL) {
		// Put that entry on the empty clock queue.
		qptr->Next    = NULL;
		ts10_SimClock = qptr;
	} else {
		// Update first entry for latest interval count.
		ts10_SimClock->outTimer = ts10_ClkInterval;

		// Find that entry between their timer by ascending.
		for (cptr = ts10_SimClock; cptr; cptr = cptr->Next) {
			if (qptr->outTimer < (cptr->outTimer + adjTimer))
				break;
			adjTimer += cptr->outTimer;
			pptr = cptr;
		}

		// Insert that entry on elsewhere within the clock queue.
		qptr->outTimer -= adjTimer;
		if (cptr == ts10_SimClock) {
			qptr->Next = ts10_SimClock;
			ts10_SimClock = qptr;
		} else {
			qptr->Next =  pptr->Next;
			pptr->Next =  qptr;
		}
	}

	// Set latest next interval count
	ts10_ClkInterval = ts10_SimClock->outTimer;

#ifdef DEBUG
	if (dbg_Check(DBG_TIMER)) {
		CLK_QUEUE *nptr = ts10_SimClock;
		dbg_Printf("TIMER: That Entry: %s  Interval Count: %d\n",
			qptr->Name ? qptr->Name : "(None)", qptr->outTimer + adjTimer);
		if (qptr != nptr) {
			dbg_Printf("TIMER: Next Entry: %s  Interval Count: %d\n",
				nptr->Name ? nptr->Name : "(None)", nptr->outTimer);
		}
	}
#endif /* DEBUG */
}

void ts10_CancelTimer(CLK_QUEUE *qptr)
{
	CLK_QUEUE *cptr, *nptr = NULL;

	// Remove that entry from the timer queue.
	if (ts10_SimClock == qptr) {
		nptr = ts10_SimClock = qptr->Next;
	} else {
		for (cptr = ts10_SimClock; cptr; cptr = cptr->Next) {
			if (cptr->Next == qptr) {
				nptr = cptr->Next = qptr->Next;
				break;
			}
		}
	}

	// Update timer alarm.
	qptr->Flags &= ~CLK_PENDING;
	if (nptr != NULL)
		nptr->outTimer += qptr->outTimer;
	ts10_ClkInterval = ts10_SimClock
		? ts10_SimClock->outTimer : NOQUEUE_WAIT;
	qptr->Next = NULL;
}

void ts10_ExecuteTimer(void)
{
	CLK_QUEUE *qptr;

	if (ts10_SimClock == NULL) {
		ts10_ClkInterval = NOQUEUE_WAIT;
	} else {
		do {
			// Remove that entry from the queue.
			qptr = ts10_SimClock;
			ts10_SimClock = qptr->Next;

#ifdef DEBUG
			if (dbg_Check(DBG_TIMER)) {
				CLK_QUEUE *nptr;
				dbg_Printf("TIMER: Entry: %s - Now Executing.\n",
					qptr->Name ? qptr->Name : "(None)");
				if (nptr = qptr->Next) {
					dbg_Printf("TIMER: Next Entry: %s  Interval Count: %d\n",
						nptr->Name ? nptr->Name : "(None)", nptr->outTimer);
				}
			}
#endif /* DEBUG */

			// Reset that entry or put it back to the queue.
			qptr->Flags &= ~CLK_PENDING;
			if (qptr->Flags & CLK_REACTIVE)
				ts10_SetTimer(qptr);
			else
				qptr->Next = NULL;

			// Update timer alarm.
			ts10_ClkInterval = ts10_SimClock
				? ts10_SimClock->outTimer : NOQUEUE_WAIT;

			// Now execute that entry.
			if (qptr->Execute != NULL)
				qptr->Execute(qptr->Device);
		} while (ts10_ClkInterval == 0);
	}
}

int ts10_StartTimer(void)
{
	setitimer(ITIMER_REAL, &startTimer, &oldTimer);
}

int ts10_StopTimer(void)
{
	setitimer(ITIMER_REAL, &stopTimer, &oldTimer);
}

int ts10_SetAlarm(void (*handler)(int))
{
#ifdef HAVE_SIGACTION
	sigTimer.sa_handler = handler;
	sigTimer.sa_flags   = 0;
	sigaction(SIGALRM, &sigTimer, NULL);
#else
	signal(SIGALRM, handler);
#endif /* HAVE_SIGACTION */
	return EMU_OK;
}

