/*
 * @(#)Arts.cpp 3.00 12 October 2000
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "tse3/plt/Arts.h"

#include "tse3/Error.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <kde/arts/artsmidi.h>

#include <vector>
#include <utility>
#include <iostream>

using namespace TSE3;
using namespace TSE3::Plt;

namespace
{
    /**
     * this is a singleton shared by all aRts midi schedulers (if there might
     * be many?) to prevent the MCOP protocol from being initialized more than
     * once
     */
    class ArtsCommon
    {
        public:
            Arts::Dispatcher dispatcher;
            Arts::MidiManager manager;
            Arts::MidiClient client;
            Arts::MidiPort port;
            int offsetSec;

            bool alive;

            ArtsCommon()
            {
                alive = false;
                manager = Arts::Reference("global:Arts_MidiManager");

                if (manager.isNull())
                {
                    std::cerr << "TSE3: Arts MidiManager isNull\n";
                    throw TSE3::MidiSchedulerError(MidiSchedulerCreateErr);
                }

                std::string title = "TSE3";
                client            = manager.addClient(Arts::mcdPlay,
                                                      Arts::mctApplication,
                                                      title,"tse3");
                port              = client.addOutputPort();
                offsetSec         = port.time().sec;

                alive             = true;
            }
    };

    ArtsCommon *artsCommon     = 0;
    int         artsCommonInit = 0;
};


ArtsMidiScheduler::ArtsMidiScheduler()
{
    if(!artsCommonInit) { assert(!artsCommon); artsCommon = new ArtsCommon(); }
    artsCommonInit++;
}

ArtsMidiScheduler::~ArtsMidiScheduler()
{
    artsCommonInit--;
    if(!artsCommonInit) { delete artsCommon; artsCommon = 0; }
}

const char *ArtsMidiScheduler::implementationName()
{
    return "ArtsMidiScheduler version 1.00";
}

size_t ArtsMidiScheduler::ports() const
{
    return 1;
}

const char *ArtsMidiScheduler::portName(size_t /*port*/) const
{
    return "Single aRts MIDI output port";
}

const char *ArtsMidiScheduler::portType(size_t /*port*/) const
{
    return "aRts MIDI port";
}

bool ArtsMidiScheduler::portReadable(size_t /*port*/) const
{
    return false;
}

bool ArtsMidiScheduler::portWriteable(size_t /*port*/) const
{
    return true;
}

void ArtsMidiScheduler::tx(MidiCommand mc)
{
    if(!artsCommon->alive) return;

    Arts::MidiCommand command(mc.status << 4 | mc.channel, mc.data1, mc.data2);
    artsCommon->port.processCommand(command);
}

void ArtsMidiScheduler::start(Clock /*startTime*/)
{
    notify(&MidiSchedulerListener::MidiScheduler_Started);
}

void ArtsMidiScheduler::stop(Clock /*stopTime*/)
{
    notify(&MidiSchedulerListener::MidiScheduler_Stopped);
}

void ArtsMidiScheduler::moveTo(Clock /*moveTime*/, Clock /*newTime*/)
{
    notify(&MidiSchedulerListener::MidiScheduler_Moved);
}

Clock ArtsMidiScheduler::clock()
{
    return msToClock(msecs());
}

int ArtsMidiScheduler::msecs()
{
    Arts::TimeStamp time = artsCommon->port.time();
    time.sec -= artsCommon->offsetSec; /* to reduce the risk of overflows */

    return time.sec*1000 + time.usec/1000;
}

void ArtsMidiScheduler::setTempo(int /*newTempo*/, Clock /*changeTime*/)
{
}

bool ArtsMidiScheduler::eventWaiting()
{
    return false;
}

MidiEvent ArtsMidiScheduler::rx()
{
    /*if(!eventWaiting()) <- always true */
        return MidiEvent();
}

void ArtsMidiScheduler::txSysEx(const unsigned char * /*data*/, size_t /*size*/)
{
}

void ArtsMidiScheduler::tx(MidiEvent e)
{
    if(!artsCommon->alive) return;

    Arts::TimeStamp time(clockToMs(e.time)/1000, (clockToMs(e.time)%1000)*1000);
    MidiCommand& mc = e.data;
    Arts::MidiCommand command(mc.status << 4 | mc.channel, mc.data1, mc.data2);
    artsCommon->port.processEvent(Arts::MidiEvent(time,command));
}

ArtsMidiSchedulerFactory::ArtsMidiSchedulerFactory()
{
}

ArtsMidiSchedulerFactory::~ArtsMidiSchedulerFactory()
{
}


ArtsMidiScheduler *ArtsMidiSchedulerFactory::createScheduler()
{
    try
    {
        ArtsMidiScheduler *ms = new ArtsMidiScheduler();
        return ms;
    }
    catch (...)
    {
        std::cerr << "TSE3: Failed to create an ArtsMidiScheduler\n";
        throw;
    }
}

