/*
 * @(#)plt/Unix.h 3.00 15 October 1999
 *
 * Copyright (c) 2000 Pete Goodliffe (pete.goodliffe@pace.co.uk)
 *
 * This file is part of TSE3 - the Trax Sequencer Engine version 3.00.
 *
 * This library is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifndef TSE3_PLT_UNIX_H
#define TSE3_PLT_UNIX_H

#include "tse3/MidiScheduler.h"

namespace TSE3
{
    /**
     * The Plt namespace contains classes that provide particular
     * platform @ref TSE3::MidiScheduler classes for different hardware
     * or software platforms.
     *
     * @short   @ref TSE3::MidiScheduler platform implementations
     * @author  Pete Goodliffe
     * @version 3.00
     * @see     TSE3
     */
    namespace Plt
    {
        class OSSMidiScheduler;
        class AlsaMidiScheduler;
        class ArtsMidiScheduler;

        /**
         * This is the Unix platform version of the @ref MidiSchedulerFactory.
         * It makes appropriate decisions to deliver you a @ref MidiScheduler
         * that will run on your Unix platform.
         *
         * This will (depending on availablity and compile-time options) be
         * either:
         *   @li An @ref OSSMidiScheduler
         *   @li An @ref AlsaMidiScheduler
         *   @li An @ref ArtsMidiScheduler
         *
         * @short   Unix platform MidiScheduler class factory
         * @author  Pete Goodliffe
         * @version 1.00
         */
        class UnixMidiSchedulerFactory : public MidiSchedulerFactory
        {
            public:

                /**
                 * Creates a unixMidiSchedulerFactory object.
                 *
                 * You may specify whether or not this factory can return
                 * a @ref TSE3::Util::NullMidiScheduler or not. (They might
                 * be returned if no other Unix-specific scheduler can
                 * be created.)
                 *
                 * @param canReturnNull If true, then this object can
                 *                      return a @ref NullMidiScheduler
                 */
                UnixMidiSchedulerFactory(bool canReturnNull = false);
                virtual ~UnixMidiSchedulerFactory();

                /**
                 * enum type describing a type of Unix @ref MidiScheduler
                 * class.
                 */
                enum UnixPlatform
                {
                    UnixPlatform_OSS,
                    UnixPlatform_Alsa,
                    UnixPlatform_Arts,
                    UnixPlatform_Null
                };

                /**
                 * Returns the currently set preferred platform.
                 *
                 * The default is UnixPlatform_Alsa.
                 *
                 * @return Preferred platform
                 * @see    setPreferredPlatform
                 */
                UnixPlatform preferredPlatform() const { return plt; }

                /**
                 * Sets the preferred platform. You will need to call
                 * this prior to the @ref createScheduler method being
                 * called if you require behaviour different from the default.
                 *
                 * The default is UnixPlatform_Alsa.
                 *
                 * @param p Preferred platform type
                 * @see   preferredPlatform
                 */
                void setPreferredPlatform(UnixPlatform p) { plt = p; }

                /**
                 * @reimplemented
                 *
                 * If the preferred type of platform is available then
                 * that will be created. Otherwise, another type will be
                 * attempted.
                 *
                 * If no Unix platform schedulers can be created, a
                 * @ref TSE3::Util::NullMidiSchedler will be created.
                 */
                virtual TSE3::MidiScheduler *createScheduler();

                /**
                 * Returns the type of platform that has been created.
                 *
                 * The result is only valid after @ref createScheduler has been
                 * called.
                 */
                UnixPlatform createdPlatform() const { return cplt; }

            private:

                /**
                 * Try to create an @ref OSSMidiScheduler. Catch any
                 * exceptions thrown, and return zero if didn't create.
                 */
                MidiScheduler *createOSS();

                /**
                 * Try to create an @ref AlsaMidiScheduler. Catch any
                 * exceptions thrown, and return zero if didn't create.
                 */
                MidiScheduler *createAlsa();

                /**
                 * Try to create an @ref ArtsMidiScheduler. Catch any
                 * exceptions thrown, and return zero if didn't create.
                 */
                MidiScheduler *createArts();

                UnixPlatform plt;
                UnixPlatform cplt;
                bool         canReturnNull;
        };
    }
}

#endif
