/*
 *        Lexer for TTCN-3
 *
 * Copyright (C) 2001 Institute for Telematics
 *
 *    Roman Koch <rkoch@itm.mu-luebeck.de>
 *    Michael Schmitt <schmitt@itm.mu-luebeck.de>
 *
 *    Medical University of Luebeck,
 *    Ratzeburger Allee 160,
 *    23538 Luebeck,
 *    Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: TTCNLexer.g,v 1.41 2001/08/03 15:42:48 schmitt Exp $
 *
 * This file is intended to be used with ANTLR 2.7.1 or higher. ANTLR is a parser generator that
 * is based on 'linear approximate' lookahead analysis (a kind of weak LL(k) analysis).
 * ANTLR is free software provided by jGuru.com (MageLang Institute). For further information
 * please refer to http://www.antlr.org. My special thanks go to Terence Parr from jGuru.com.
 *
 * If you make use of this TTCN-3 grammar specification, please send a little email in which
 * you tell us what you are using it for and whether you consider it useful. Any grammar
 * corrections and extensions (e.g. for semantics checks and abstract syntax tree transformation)
 * are welcome. For the latest release of all TTCN-3 grammar files please have a look at
 * http://www.itm.mu-luebeck.de/english/research/specification/ttcn3parser. We also invite you to
 * visit the WWW pages of the Institute for Telematics at http://www.itm.mu-luebeck.de.
 *
 * Unsolved problems:
 *
 *   - 'FreeText' and 'ExtendedAlphaNum' are not supported
 *
 */

header "pre_include_hpp"
{
  // $Id: TTCNLexer.g,v 1.41 2001/08/03 15:42:48 schmitt Exp $

  #include <string>

  using std::string;
}


header "post_include_cpp"
{
  #include <iostream>

  using std::cerr;
  using std::endl;

  namespace TTCNGrammar
  {
    void TTCNLexer::reportError( const string &errorMessage )
    {
      reportMessage( errorMessage );
      noOfErrors++;
    }

    void TTCNLexer::reportMessage( const string &message ) const
    {
      if ( getFilename().length() > 0 )
        cerr << getFilename();
      else
        cerr << "<stdin>";

      cerr << ":" << getLine() << ":" << getColumn() << ": error: " << message << endl;
    }

    unsigned int TTCNLexer::numberOfErrors() const
    {
      return noOfErrors;
    }

    void TTCNLexer::resetErrors()
    {
      noOfErrors = 0;
    }

    const string &TTCNLexer::versionInfo()
    {
      static string   info = "$Id: TTCNLexer.g,v 1.41 2001/08/03 15:42:48 schmitt Exp $";

      return info;
    }
  }
}


options
{
  language  = "Cpp";
  namespace = "TTCNGrammar";
  namespaceStd = "std";
  namespaceAntlr = "antlr";
  genHashLines = true;
}


class TTCNLexer extends Lexer;

options
{
  k = 2;
  exportVocab = TTCNLexer;
  testLiterals = false;
  caseSensitive = true;
}

tokens
{
  ACTION = "action";
  ACTIVATE = "activate";
  ADDRESS = "address";
  ALL = "all";
  ALT = "alt";
  AND = "and";
  AND4B = "and4b";
  ANY = "any";
  BITSTRING = "bitstring";
  BOOLEAN = "boolean";
  CALL = "call";
  CATCH = "catch";
  CHAR = "char";
  CHARSTRING = "charstring";
  CHECK = "check";
  CLEAR = "clear";
  COMPLEMENT = "complement";
  COMPONENT = "component";
  CONNECT = "connect";
  CONST = "const";
  CONTROL = "control";
  CREATE = "create";
  DEACTIVATE = "deactivate";
  DEFAULT = "default";
  DISCONNECT = "disconnect";
  DISPLAY = "display";
  DO = "do";
  DONE = "done";
  ELSE = "else";
  ENCODE = "encode";
  ENUMERATED = "enumerated";
  ERROR = "error";
  EXCEPTION = "exception";
  EXECUTE = "execute";
  EXTENSION = "extension";
  EXTERNAL = "external";
  FAIL = "fail";
  FALSE = "false";
  FLOAT = "float";
  FOR = "for";
  FROM = "from";
  FUNCTION = "function";
  GET = "get";
  GETCALL = "getcall";
  GETREPLY = "getreply";
  GOTO = "goto";
  GROUP = "group";
  HEXSTRING = "hexstring";
  IF = "if";
  IFPRESENT = "ifpresent";
  IMPORT = "import";
  IN = "in";
  INCONC = "inconc";
  INFINITY = "infinity";
  INOUT = "inout";
  INTEGER = "integer";
  INTERLEAVE = "interleave";
  LABEL = "label";
  LANGUAGE = "language";
  LENGTH = "length";
  LOG = "log";
  MAP = "map";
  MATCH = "match";
  MESSAGE = "message";
  MIXED = "mixed";
  MOD = "mod";
  MODIFIES = "modifies";
  MODULE = "module";
  MTC = "mtc";
  NOBLOCK = "noblock";
  NONE = "none";
  NONRECURSIVE = "nonrecursive";
  NOT = "not";
  NOT4B = "not4b";
  NOWAIT = "nowait";
  NULL_ = "null";
  OBJID = "objid";
  OCTETSTRING = "octetstring";
  OF = "of";
  OMIT = "omit";
  ON = "on";
  OPTIONAL = "optional";
  OR = "or";
  OR4B = "or4b";
  OUT = "out";
  OVERRIDE = "override";
  PARAM = "param";
  PASS = "pass";
  PATTERN = "pattern";
  PORT = "port";
  PROCEDURE = "procedure";
  RAISE = "raise";
  READ = "read";
  RECEIVE = "receive";
  RECORD = "record";
  REM = "rem";
  REPEAT = "repeat";
  REPLY = "reply";
  RETURN = "return";
  RUNNING = "running";
  RUNS = "runs";
  SELF = "self";
  SEND = "send";
  SENDER = "sender";
  SET = "set";
  SIGNATURE = "signature";
  START = "start";
  STOP = "stop";
  SUT = "sut";
  SYSTEM = "system";
  TEMPLATE = "template";
  TESTCASE = "testcase";
  TESTSTEP = "teststep";
  TIMEOUT = "timeout";
  TIMER = "timer";
  TO = "to";
  TRIGGER = "trigger";
  TRUE = "true";
  TYPE = "type";
  UNION = "union";
  UNIVERSAL = "universal";
  UNMAP = "unmap";
  VALUE = "value";
  VALUEOF = "valueof";
  VAR = "var";
  VERDICT = "verdict";
  VERDICTTYPE = "verdicttype";
  WHILE = "while";
  WITH = "with";
  XOR = "xor";
  XOR4B = "xor4b";

  FloatValue;
  BString;
  HString;
  OString;
  BStringMatch;
  HStringMatch;
  OStringMatch;

  BeginChar = "{";
  EndChar = "}";
  LParenthesis = "(";
  RParenthesis = ")";
  LBracket = "[";
  RBracket = "]";

  Dot = ".";
  Dash = "-";
  SemiColon = ";";
  Colon = ":";
  Comma = ",";
  AssignmentChar = ":=";
  PortRedirectSymbol = "->";
  Range = "..";
  QuestionMark = "?";
  Asterisk = "*";
  Divide = "/";
  Plus = "+";
  Ampersand = "&";

  Equals = "==";
  NotEquals = "!=";
  Less = "<";
  Greater = ">";
  GreaterOrEqual = ">=";
  LessOrEqual = "<=";

  ShiftLeft = "<<";
  ShiftRight = ">>";
  RotateLeft = "<@";
  RotateRight = "@>";
}

{
  public:

    unsigned int numberOfErrors() const;
    void resetErrors();

    static const string &versionInfo();

  protected:

    void reportError( const string &errorMessage );
    void reportMessage( const string &message ) const;

  private:

    unsigned int   noOfErrors;
}

/***** SECTION A.1.6.5 - Value *****/

AnyNumber :

  Number
  (   ( '.' ~'.' | 'E' ( '-' )? Num ) =>
      ( '.' DecimalNumber )? ( 'E' ( '-' )? Number )?
      { $setType( FloatValue ); }
    | { $setType( Number ); }
  )

;


protected FloatValue options { paraphrase = "a floating point constant"; } : /* unused rule */ ;


protected Number options { paraphrase = "an integer constant"; } :

    NonZeroNum ( Num )*
  | '0'

;


protected NonZeroNum :

  '1' .. '9'

;


protected DecimalNumber :

  ( Num )*

;


protected Num :

    '0'
  | NonZeroNum

;


AnyString { bool isBin = true, isMatch = false; int count = 0; } :

  "'"!
  (   ( '0' | '1' )
        { count++; }
    | ( '2' .. '9' | 'A' .. 'F' | 'a' .. 'f' )
        { isBin = false; count++; }
    | ( '?' | '*' )
        { isMatch = true; count++; }
  )*
  "'"!
  (   { isBin }? 'B'!
      { if ( isMatch ) $setType( BStringMatch ); else $setType( BString ); }
    | { ( count & 0x01 ) == 0 }? 'O'!
      { if ( isMatch ) $setType( OStringMatch ); else $setType( OString ); }
    | 'H'!
      { if ( isMatch ) $setType( HStringMatch ); else $setType( HString ); }
  )

;


protected BStringMatch
  options { paraphrase = "a binary string constant with wildcards"; } : /* unused rule */ ;
protected OStringMatch
  options { paraphrase = "an octet string constant with wildcards"; } : /* unused rule */ ;
protected HStringMatch
  options { paraphrase = "a hex string constant with wildcards"; } : /* unused rule */ ;
protected BMatch
  options { paraphrase = "a binary string"; } : /* unused rule */ ;
protected OMatch
  options { paraphrase = "an octet string"; } : /* unused rule */ ;
protected HMatch
  options { paraphrase = "a hex string"; } : /* unused rule */ ;


CString options { paraphrase = "a character string"; } :

  "\""! ( Char | "\""! "\"" )* "\""!

;


protected Char :

    '\040'
  | '\041'
  | '\043' .. '\377'

;


Identifier options { paraphrase = "an identifier"; testLiterals = true; } :

  Alpha ( AlphaNum | '_' )*

;


protected Alpha :

    UpperAlpha
  | LowerAlpha

;


protected AlphaNum :

    Alpha
  | Num

;


protected UpperAlpha :

  'A' .. 'Z'

;


protected LowerAlpha :

  'a' .. 'z'

;


/***** SECTIONS A.1.6.5 - Value; A.1.6.9 - Basic Statements; A.1.6.10 - Miscellaneous productions *****/

Miscellaneous options { testLiterals = true; } :

    "{"   | "}"   | "("  | ")"  | "["  | "]"
  | "."   | "-"   | ";"  | ":"  | ","  | ":=" | "->"
  | ".."  | "?"   | "*"  | "/"  | "+"  | "&"
  | "=="  | "!="  | "<"  | ">"  | ">=" | "<="
  | "<<"  | ">>"  | "<@" | "@>" 
  // Error handling
  | "="
    {
      reportError( "'=' is not a valid operator in TTCN-3." );
      reportMessage( "It will be replaced by ':=' for further processing." );
      $setText( ":=" );
    }

;


/***** Comments *****/

Comment options { paraphrase = "a C-style comment"; } :

  "/*"
  (  options {
       generateAmbigWarnings=false;
     } :
        { LA( 2 ) != '/' }? '*'
     | '\r' '\n'  { newline(); }
     | '\r'       { newline(); }
     | '\n'       { newline(); }
     | ~'*'
  )*
  "*/"

  { $setType( antlr::Token::SKIP ); }

;


SingleLineComment options { paraphrase = "a single-line comment"; } :

  "//" ( ~( '\r' | '\n' ) )*

  { $setType( antlr::Token::SKIP ); }

;


/***** Whitespaces *****/

WhiteSpaces options { paraphrase = "whitespaces/line breaks"; } :

  ( Space )+

  { $setType(antlr::Token::SKIP); }

;


protected Space :

    '\000' .. '\011'      // Control characters
  | '\n' { newline(); }
  | '\013' .. '\037'      // Control characters
  | ' '

;
