/*
	TuxPuck - tuxpuck.c
	Copyright (C) 2001-2002 Jacob Kroon

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

	Jacob Kroon - d00jkr@efd.lth.se
*/

/* includes */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <SDL.h>
#include "scoreboard.h"
#include "video.h"
#include "audio.h"
#include "board.h"
#include "player.h"
#include "timer.h"

/* defines */
#ifdef windows
#define SETTINGSFILE "tuxpuck.ini"
#else
#define SETTINGSFILE _settings_file
#endif

/* externals */
extern time_t time(time_t *);
extern void run_intro(void);
extern AIPlayer *tux_create(Pad *, Puck *);
extern AIPlayer *arcana_create(Pad *, Puck *);

/* structs */
typedef struct
{
  Uint8 sound;
  Uint8 fullscreen;
  Uint8 mouse_speed;
}
Settings;

/* statics */
static Settings *_settings = NULL;
#ifndef windows
static char _settings_file[200];
#endif

/* functions */
static void _play_match(Uint8 opponent)
{
  SDL_Event event;
  Uint8 loop = 1, scorer = 0;
  Uint32 elapsed_time = 0;
  char buffer[50];
  HumanPlayer *p1 = NULL;
  AIPlayer *p2 = NULL;
  Timer *timer = NULL;
  float alpha = 0.0;

  memset(buffer, 0, 50);
  board_init();
  scoreboard_init();
  video_save();
  p1 = human_create(board_get_pad(1), "Human");
  p2 = arcana_create(board_get_pad(2), board_get_puck());
  timer = timer_create();
  timer_reset(timer);
  while (loop)
    {
      while (SDL_PollEvent(&event))
	if (event.type == SDL_MOUSEBUTTONDOWN)
	  {
	    loop = 0;
	    alpha = 1.0;
	  }
      SDL_Delay(_SLEEP);
      timer_update(timer);
      timer_reset(timer);
      elapsed_time = timer_elapsed(timer);
      alpha += elapsed_time * 0.001;
      if (alpha > 1.0)
	{
	  loop = 0;
	  alpha = 1.0;
	}
      board_clean_up();
      scoreboard_erase();
      aiplayer_erase(p2);
      aiplayer_set_alpha(p2, (Uint8) (alpha * 255));
      scoreboard_set_alpha((Uint8) (alpha * 255));
      entity_set_alpha((Entity *) board_get_puck(), (Uint8) (alpha * 255));
      entity_set_alpha(board_get_pad(1), (Uint8) (alpha * 255.0 / 2.0));
      entity_set_alpha(board_get_pad(2), (Uint8) (alpha * 255.0 / 2.0));
      aiplayer_blit(p2);
      board_reblit();
      scoreboard_blit();
      video_update();
    }
  loop = 1;
  board_clean_up();
  aiplayer_blit(p2);
  video_save();
  board_reblit();
  video_update();
  SDL_PumpEvents();
  SDL_GetRelativeMouseState(NULL, NULL);
#ifndef _DEBUG
  SDL_WM_GrabInput(SDL_GRAB_ON);
#endif
  human_set_speed(p1, _settings->mouse_speed);
  timer_reset(timer);
  while (loop)
    {
      while (SDL_PollEvent(&event))
	switch (event.type)
	  {
	  case SDL_KEYDOWN:
	    switch (event.key.keysym.sym)
	      {
	      case SDLK_ESCAPE:
		loop = 0;
		timer_reset(timer);
		break;
	      case SDLK_F1:
		if (_settings->sound)
		  {
		    _settings->sound = 0;
		    audio_set_mute(1);
		  }
		else
		  {
		    _settings->sound = 1;
		    audio_set_mute(0);
		  }
		break;
	      case SDLK_F5:
		if (_settings->mouse_speed > 1)
		  _settings->mouse_speed--;
		human_set_speed(p1, _settings->mouse_speed);
		scoreboard_set_mousebar(_settings->mouse_speed);
		break;
	      case SDLK_F6:
		if (_settings->mouse_speed < 10)
		  _settings->mouse_speed++;
		human_set_speed(p1, _settings->mouse_speed);
		scoreboard_set_mousebar(_settings->mouse_speed);
		break;
	      case SDLK_f:
		if (_settings->fullscreen)
		  _settings->fullscreen = 0;
		else
		  _settings->fullscreen = 1;
		video_toggle_fullscreen();
		break;
	      default:
		break;
	      }
	    break;
	  case SDL_QUIT:
	    loop = 0;
	    break;
	  }
      SDL_Delay(_SLEEP);
      timer_update(timer);
      timer_reset(timer);
      elapsed_time = timer_elapsed(timer);
      human_update(p1, elapsed_time);
      aiplayer_update(p2, elapsed_time);
      scoreboard_update(elapsed_time);
      if ((scorer = board_update(elapsed_time)) != 0)
	{
	  scoreboard_add_point(scorer);
	  if (scorer == 1)
	    {
	      human_give_point(p1);
	      p2->set_state(p2, PLAYER_STATE_LOOSE_POINT);
	    }
	  else
	    {
	      p2->points++;
	      p2->set_state(p2, PLAYER_STATE_WIN_POINT);
	    }
	  if (human_get_points(p1) >= 15 || p2->points >= 15)
	    {
	      if (human_get_points(p1) == 15)
		p2->set_state(p2, PLAYER_STATE_LOOSE_GAME);
	      else
		p2->set_state(p2, PLAYER_STATE_WIN_GAME);
	    }
	}
      board_clean_up();
      scoreboard_clean_up();
      scoreboard_reblit();
      if (p2->reblit(p2, elapsed_time) == 0)
	loop = 0;
      board_reblit();
      video_update();
    }
#ifndef _DEBUG
  SDL_WM_GrabInput(SDL_GRAB_OFF);
#endif
  timer_free(timer);
  human_free(p1);
  p2->free(p2);
  board_deinit();
  scoreboard_deinit();
}

static void _read_settings(void)
{
  FILE *file = NULL;
  char buffer[100], buffer2[100];
  Uint32 uint32 = 0;

  if ((file = fopen(SETTINGSFILE, "r")) == NULL)
    return;
  while (fgets(buffer, 100, file) != 0)
    {
      if (sscanf(buffer, "%s %d\n", buffer2, &uint32) != 2)
	{
	  fclose(file);
	  return;
	}
      else if (strcmp(buffer2, "SOUND") == 0)
	_settings->sound = (Uint8) uint32;
      else if (strcmp(buffer2, "FULLSCREEN") == 0)
	_settings->fullscreen = (Uint8) uint32;
      else if (strcmp(buffer2, "MOUSESPEED") == 0)
	_settings->mouse_speed = (Uint8) uint32;
    }
  fclose(file);
}

static void _save_settings(void)
{
  FILE *file = NULL;

  if ((file = fopen(SETTINGSFILE, "w")) == NULL)
    return;
  fprintf(file, "SOUND %d\n", _settings->sound);
  fprintf(file, "FULLSCREEN %d\n", _settings->fullscreen);
  fprintf(file, "MOUSESPEED %d\n", _settings->mouse_speed);
  fclose(file);
}

void tuxpuck_init(void)
{
#ifndef windows
  char *homeDir = NULL;
#endif
  srand(time(NULL));
  SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO);
  audio_init();
  video_init();

  video_save();
  _settings = (Settings *) malloc(sizeof(Settings));
  memset(_settings, 0, sizeof(Settings));
  _settings->sound = 1;
  _settings->fullscreen = 0;
  _settings->mouse_speed = 5;
#ifndef windows
  homeDir = getenv("HOME");
  sprintf(_settings_file, "%s/.tuxpuckrc", homeDir);
#endif
  _read_settings();
  audio_set_mute(!_settings->sound);
  if (_settings->fullscreen)
    video_toggle_fullscreen();

  run_intro();
  video_save();
}

void tuxpuck_deinit(void)
{
  audio_deinit();
  video_deinit();
  SDL_Quit();
  _save_settings();
  free(_settings);
}

void tuxpuck_start(void)
{
  _play_match(0);
}

int main(int argc, char **argv)
{
  tuxpuck_init();
  tuxpuck_start();
  tuxpuck_deinit();
  return 0;
}
