/* ==================================================== ======== ======= *
 *
 *  unat.hh
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:01] ======= *
 * ==================================================== ======== ======= */

#ifndef _unat_hh
#define	_unat_hh
//pragma ident	"@(#)unat.hh	ubit:b1.11.4"
#include <X11/Intrinsic.h>
#include <uappli.hh>

//package_private: ====[internal implementation]===========================
//NOTE: this header is PLATFORM DEPENDENT: X-Window version (X11R5/X11R6)

typedef union _XEvent* UX_Event;        //also in uevent.hh
typedef struct _WidgetRec* UX_Widget;
typedef XtAppContext    UX_Context;
typedef Display*	UX_Display;
typedef Screen*		UX_Screen;
typedef Visual*		UX_Visual;
typedef XVisualInfo*	UX_VisualInfo;
typedef Colormap	UX_Colormap;
typedef Window		UX_Window;
typedef GC		UX_GC;
typedef XFontStruct*	UX_Font;
typedef Cursor		UX_Cursor;
typedef Pixel		UX_Pixel;
typedef Pixmap		UX_Pixmap;
typedef XImage*		UX_Image;
typedef XRectangle	UX_Rectangle;
typedef Atom		UX_Atom;

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// sert a catcher les X Errors safely
// usage (att: les { } sont essentiels!)
// {
//    UXtry xtry;
//    xDoTrucQuiPlante();
//    if (!xtry.status()) doOnError();
// }

struct UXtry {
  //NB: pas vraiment thread-safe car xerror doit etre static
  //(mais pas tres grave, le but 1er etant d'eviter des exit intempestifs)
  static int xerror;
  XErrorHandler xhandler;
  static int quietErrorHandler(Display *, XErrorEvent *xer);

  UXtry() {xerror = Success; xhandler = XSetErrorHandler(quietErrorHandler);}
  ~UXtry(){XSetErrorHandler(xhandler);}
  u_bool status() {return xerror == Success;}
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
//!ATTENTION: les NatGraph sont PARTAGES!
//(en fait il n'y a generalement qu'un seul natgraph pour toute l'appli)

class UNatGraph {
  friend class UGraph;
  friend class UWinGraph;
  UGraph     *graph_lock;
  UX_Display xdisplay;
  UX_GC      gc;
  UX_Pixel   color, bgcolor;
  UX_Font    font;
  u_bool     xorMode;
  short      thickness;		     // !!a revoir avec UPen !!
  u_pos      x_win, y_win;	     // (x,y)offset from 'dest' origin 
  UX_Rectangle cliprect;
  // 'dest' is a Window or a Pixmap (when double buffering is on)
  class UNatWin *dest;                   // where graphics are drawn
public:
  UNatGraph(class UNatDisp*);
  ~UNatGraph();

  void lock(UGraph*);
  void unlock();
  u_bool isLocked()     {return (graph_lock != null);}
  void setDest(UNatWin *d) {dest = d;}
  void setDest(UNatWin *d, u_pos x_origin, u_pos y_origin);
  void setClip(const URegion&);
  void setClip(u_pos x, u_pos y, u_dim width, u_dim height);

  // !Implementation dependent
  UX_GC    getGC()      {return gc;}
  UX_Pixel getPixel()   {return color;}
  UX_Pixel getBgpixel() {return bgcolor;}
  UX_Font  getFont()    {return font;}
  u_dim getClipWidth()  {return cliprect.width;}
  u_dim getClipHeight() {return cliprect.height;}
};

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
// Notes:
// -- the background of the pixmap is "transparent" (i.e. shaped)
//    if transp_bg = true (this feature is only available if supported 
//    by the image file format)
// -- return value: one of UnatIma::Reader* constants
//    (ReaderSuccess == 0 if the image could be read)

int UXpmFileReader(UNatDisp *nd, const char *fpath,
		   Pixel fg, Pixel bg, u_bool transp_bg,
		   UX_Image &ima, UX_Image &shapeima,
		   u_dim &width, u_dim &height);

int UXpmDataReader(UNatDisp *nd, const char *xpmdata,
		   Pixel fg, Pixel bg, u_bool transp_bg,
		   UX_Image &ima, UX_Image &shapeima,
		   u_dim &width, u_dim &height);

int UGifFileReader(UNatDisp *nd, const char *fpath,
		   Pixel fg, Pixel bg, u_bool transp_bg,
		   UX_Image &ima, UX_Image &shapeima,
		   u_dim &width, u_dim &height);

class UNatIma {
  friend class UGraph;
  friend class UNatGraph;
  friend class UIma;
  friend class UPix;
public:
  UNatDisp *natdisp;
  UX_Image xima, ximashape;
  // extensions used by class UIma
  int      lscale;   //LOGICAL scale
  UNatIma  *next;

public:
  typedef int (*Reader) (class UNatDisp *nd, const char *fpath,
			 Pixel fg, Pixel bg, u_bool transp_bg,
			 UX_Image &ima, UX_Image &shapeima,
			 u_dim &width, u_dim &height);

  // ximashape == null ==> opaque image
  // ximashape != null ==> transparent background
  // (!!ATT: la transparence ne fonctionne que pour les UPix/UNatPix !!)

  UNatIma(UNatDisp*, UX_Image xima, UX_Image ximashape);
  ~UNatIma();

  u_dim  getWidth()   const;
  u_dim  getHeight()  const;
  u_bool isRealized() const;
  u_bool isShaped()  const {return ximashape != null;}
  const UX_Image getXImage()      const {return xima;}
  const UX_Image getXImageShape() const {return ximashape;}

  // Creates a scaled clone image from this image
  UNatIma* clone(UNatDisp*, float xscale, float yscale);

  // Creates a scaled image from a source image
  static UNatIma* createScaledIma(UNatDisp*, 
				  UX_Image xima, UX_Image ximashape, 
				  float xscale, float yscale);

  // returns a new XImage or NULL
  static UX_Image createScaled_XImage(UNatDisp*, UX_Image xima, 
				     float xscale, float yscale);
  // returns xima1 or xima2 or NULL
  static UX_Image blend_XImages(UNatDisp*, UX_Image xima1, UX_Image xima2, 
				     float alpha);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UNatPix {
  friend class UGraph;
  friend class UNatGraph;
  friend class UPix;
public:
  UNatDisp *natdisp;
  UX_Pixmap xpix, xpixshape;
  u_dim width, height;
  int   depth, lscale;   //lscale = LOGICAL scale (used by UPix)

public:
  // xpixshape == None ==> opaque image
  // xpixshape != None ==> transparent background
  UNatPix(UNatDisp*, UX_Pixmap xpix, UX_Pixmap xpixshape, 
	  u_dim width, u_dim height, int depth);

  // creates the pixmaps from the images
  // ximashape == null ==> opaque image
  // ximashape != null ==> transparent background
  UNatPix(UNatDisp*, UX_Image xima, UX_Image ximashape);

  UNatPix(UNatDisp*, UNatIma*);
  ~UNatPix();

  void set(UX_Image xima, UX_Image ximashape);

  u_dim  getWidth()   const {return width;}
  u_dim  getHeight()  const {return height;}
  u_bool isRealized() const {return xpix != None;}
  u_bool isShaped()   const {return xpixshape != None;}
  const UX_Pixmap getXPixmap()      const {return xpix;}
  const UX_Pixmap getXPixmapShape() const {return xpixshape;}
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UNatTimer {
public:
  class UTimer* timer;
  //UX_Context xappc;
  //XtIntervalId xtid;

  UNatTimer(class UNatAppli*, class UTimer*);
  u_bool run(int duration);
  void stop();
  static void XTimeOut(XtPointer timer, XtIntervalId*);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UNatWin {
  friend class UGraph;
  friend class UWinGraph;
  friend class UNatGraph;
protected:
  UX_Widget xwidget;	// corresponding Xt Widget
  UX_Window xwin;	// corresponding X Window
public:
  UNatWin();
  ~UNatWin();

  u_bool isRealized() const {return (xwin != None);}

  void move(class UNatDisp*, u_pos x, u_pos y);
  void resize(class UNatDisp*, u_dim w, u_dim h);
  void show(class UNatDisp*, u_bool);

  // get coordinates of the windown in the screen
  void where(class UNatDisp*, u_pos &screen_x, u_pos &screen_y);
  void setTitle(class UNatDisp*, const char*);

  UX_Widget getXWidget() {return xwidget;}  // corresponding Xt Widget
  UX_Window getXWindow() {return xwin;}     // corresponding X Window

  static u_bool realizeMenu(class UNatDisp*,     UNatWin*, UWin*);
  static u_bool realizeDialog(class UNatDisp*,   UNatWin*, UWin*);
  static u_bool realizeFrame(class UNatDisp*,    UNatWin*, UWin*);
  static u_bool realizeMainFrame(class UNatDisp*,UNatWin*, UWin*);
  static u_bool realizeIncrust(class UNatDisp*,  UNatWin*, UWin*);
  void reshapeCB(class UNatDisp*, class UView *winview);

  //void setEventHandlers(class UWin *win);
  static void manageEvents(class UNatDisp*, UNatWin*, UWin*);

  static void on_any_event(UAppli*, UWin*, UView*, UX_Event);
  static void on_message(UAppli*, UWin*, UView*, UX_Event);
  static void on_selection(UAppli*, UWin*, UView*, UX_Event);
  static void on_misc(UAppli*, UWin*, UView*, UX_Event);
  static void on_expose(UAppli*, UWin*, UView*, UX_Event);
  static void on_configure(UAppli*, UWin*, UView*, UX_Event);
  static void on_mpress(UAppli*, UWin*, UView*, UX_Event);
  static void on_mrelease(UAppli*, UWin*, UView*, UX_Event);
  static void on_mmove(UAppli*, UWin*, UView*, UX_Event);
  static void on_kpress(UAppli*, UWin*, UView*, UX_Event);
  static void on_krelease(UAppli*, UWin*, UView*, UX_Event);
  static void on_enter(UAppli*, UWin*, UView*, UX_Event);
  static void on_leave(UAppli*, UWin*, UView*, UX_Event);
  static void on_focus(UAppli*, UWin*, UView*, UX_Event);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UNatDisp {
  friend class UWinGraph;
  friend class UGraph;
protected:
  UDisp       *disp;   // corresponding abstract Display counterpart
  UX_Display  xdisplay;
  UX_Screen   xscreen;
  UX_Visual   xvisual;
  UX_Colormap xcmap;
  UX_Window   xwin;    // used for getting depth, etc for drawing pixmaps...
  UX_Widget   xwidget; // Display Shell (usually the TopLevel Shell)
  int         depth;

  UX_Pixel *virtualColormap;
  //EN FAIT IL FAUDRAIT UN COMPTAGE GLOBAL A TOUS LES UDisp
  int virtualColormapSize;

  UX_Font **fontFamilyMaps;
  //EN FAIT IL FAUDRAIT UN COMPTAGE GLOBAL A TOUS LES UDisp
  int fontFamilyCount;
  
  //default Pixmap with same depth as UNatDisp
  UX_Pixmap defaultPix;

  UX_Cursor *cursorMap;
  int cursorMapSize;

  // used by UWinGraph objects for standard toolkit drawing
  UNatGraph *sharedGraph;
  // used by UGraph objects for appli. drawing
  UNatGraph *clientGraph;

public:
  UNatDisp(UDisp*);
  virtual ~UNatDisp();

  // initializes actual X data
  virtual void realize();
  virtual u_bool isRealized() const {return (xwin != None);}

  // Searchs if Screen 'screen_no' supports this depth with this visual_class
  // -- usually 'screen_no' = 0 (except if you have several screens on your display)
  // -- 'visual_class' = PseudoColor, TrueColor, etc.
  // -- if 'depth_hint' is not supported with this 'visual_class', the function
  //    tries to to find the closest depth that is compatible with this 'visual_class'
  //    and it returns it
  // -- returned value:
  //    depth_hint if supported, the closest supported depth otherwise
  //    and 0 if this visual_class is not supported at all by this screen

  int matchVisualProps(int visual_class, int depth_hint, int screen_no = 0);

  // this variant fills the VisualInfo struct. that is given as an argument
  int matchVisualProps(UX_VisualInfo, 
		       int visual_class, int depth_hint, int screen_no = 0);

  // Set the requested visual props if they are supported on this screen
  // -- this fct. calls matchVisualProps() first and it works in the same way
  // -- the Colormap arg. is optional and its is automatically created if == None
  // -- this fct. must be called before realizing X windows 
  //    (ie. before appli->add(), appli->realize() and mainLoop())

  int setVisualProps(int visual_class, int depth_hint, int screen_no = 0,
		     UX_Colormap = None);

  //  Set the requested visual props according to the VisualInfo struct.
  int setVisualProps(UX_VisualInfo, UX_Colormap = None);

  // returns the corresponding abstract Display counterpart
  UDisp*      getDisp()      const {return disp;}
  UX_Display  getXDisplay()  const {return xdisplay;}
  UX_Screen   getXScreen()   const {return xscreen;}
  UX_Visual   getXVisual()   const {return xvisual;}
  UX_Colormap getXColormap() const {return xcmap;}
  UX_Window   getXRootWindow() const;
  UX_Widget   getXWidget()   const {return xwidget;}
  UX_Window   getXWindow()   const {return xwin;}

  struct {
    UX_Atom PRIMARY_SELECTION, SECONDARY_SELECTION, UBIT_SELECTION,
      WM_PROTOCOLS, WM_DELETE_WINDOW, WM_TAKE_FOCUS;
  } atoms;

  // depth of *this* UNatDisp
  int getDepth() const {return depth;}

  // default depth of the Screen
  int getScreenDefaultDepth() const;

  // size of the Screen
  int getScreenWidth()  const;
  int getScreenHeight() const;

  //default Pixmap with same depth as UNatDisp (for XCreateWindow)
  UX_Pixmap getDefaultPix() {return defaultPix;}

  // returns the native color corresponding to this UColor
  // -- Note: the color is implicitely realized if necessary
  UX_Pixel getXPixel(const UColor*);

  // this fct is used for changing the virtual colormap dynamically
  void setXPixel(const UColor*, UX_Pixel);

  // Allocates physical resources for this UColor on this UDisp
  // -- returns true if the color could be allocated.
  // -- returns false and sets a default color otherwise
  u_bool realizeColor(const UColor*);

  // returns the native cursor corresponding to this UCursor
  // -- Note: the cursor is implicitely realized if necessary
  UX_Cursor getXCursor(const UCursor*);
  //
  // Allocates physical resources for this UCursor on this UDisp
  // -- returns true if the cursor was found and false otherwise
  u_bool realizeCursor(const UCursor*);

  // returns the native font corresponding to this UFont
  // -- Note: the font is implicitely realized if necessary
  UX_Font getXFont(const class UFontDesc*);
  
  // Allocates physical resources for this UFont/UFontDesc on this UDisp
  // -- returns true if the font was found and false otherwise
  u_bool realizeFont(const class UFont*);
  u_bool realizeFont(const class UFontDesc*);
  u_bool realizeFontFamily(const class UFontFamily*);
  UX_Font loadNatFont(const class UFontFamily*, int styles, int lsize);

  // NOTE: clients should not access the following data directly
  // (these functions should only be used internally by the Ubit lib)

  // used by UWinGraph objects for standard toolkit drawing
  UNatGraph *getSharedGraph() {return sharedGraph;}

  // used by UGraph objects for appli. drawing
  UNatGraph *getClientGraph() {return clientGraph;}

  // dispatches X Events to Windows (function called by the main loop)
  void dispatchEvents(UX_Event);
};

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

class UNatAppli : public UNatDisp {
  friend class UNatDisp;
protected:
  UX_Context xcontext;
  int status;
public:
  UNatAppli(UAppli*, int *argc, char *argv[]);
  //obs: UNatAppli(UAppli*);
  virtual ~UNatAppli();

  int mainLoop();	        // starts the main loop
  void quit(int status);	// quits the main loop

  // initializes actual X data (NB: argc, argv can be null)
  virtual void realize();
  virtual void realize(int *argc, char *argv[]);
  virtual u_bool isRealized() const;

  UAppli*    getAppli()    {return (UAppli*)disp;}
  UX_Context getXContext() {return xcontext;}
  //obs: void setXContext(UX_Context);
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
