/* ==================================================== ======== ======= *
 *
 *  uuprop.cc
 *  Ubit Project  [Elc][beta1][2001]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2001 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:O1] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuprop.cc	ubit:b1.11.7"
#include <udefs.hh>
#include <ubrick.hh>
#include <uprop.hh>
#include <uctrl.hh>
#include <ucontext.hh>
#include <uview.hh>
#include <ubox.hh>
#include <upix.hh>
#include <ucolor.hh>
#include <ucall.hh>
#include <uevent.hh>
#include <uobs.hh>  //ufloating()

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

const UClass UProp::uclass("UProp");

void UProp::changed(u_bool update_now) {
  if (update_now) update();

  if (cache) {
    UEvent e(UEvent::change, null, NULL);
    e.setPropSource(this);
 
    fire(e, UOn::change);    
    // ensuite on lance les callbacks des parents
    parents.fireParents(e, UOn::propChange);
  }
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

const UClass UScale::uclass("UScale");

UScale::UScale(int val, u_modes m) : UProp(m) {
  value = val;
}

void UScale::set(int val, u_bool upd) {
  if (value == val) return;
  value = val;
  changed(upd);
}

void UScale::incr(int delta) {
  value += delta;
  changed(true);
}

void UScale::update() {
  //parents.updateParents(UUpdate::win);
  parents.updateParents(UUpdate::layout);
}

void UScale::putProp(UContext *props, UCtrl *state) {
  //NB: lscale = LOGICAL scale
  props->lscale += value;	// += c'est en relatif !
  props->fontdesc.incrScale(value);
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

const UClass UAlpha::uclass("UAlpha");

UAlpha::UAlpha(float val, u_modes m) : UProp(m) {
  value = val;
}

void UAlpha::set(float val, u_bool upd) {
  if (value == val) return;
  value = val;
  changed(upd);
}

void UAlpha::update() {
  parents.updateParents(UUpdate::paint);
}

void UAlpha::putProp(UContext *props, UCtrl *state) {
  props->local.alpha = value;
}

UAlpha& ualpha(float val) {return *(new UAlpha(val));}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

const UClass UBackground::uclass("UBackground");

UBackground& ubackground(const UColor &bgcolor) {
  return *(new UBackground(bgcolor));
}

UBackground& ubackground(const UIma &bgima) {
  return *(new UBackground(bgima));
}

UBackground::UBackground(u_modes m) : UProp(m) {
  bgima   = null;
  bgcolor = null;
  halign  = null;
  valign  = null;
}

UBackground::UBackground(const UIma &_bgima, u_modes m) : UProp(m) {
  bgima   = &_bgima;
  bgcolor = null;
  halign  = null;
  valign  = null;
}
UBackground::UBackground(const UColor &_bgcolor, u_modes m) : UProp(m) {
  bgima   = null;
  bgcolor = &_bgcolor;
  halign  = null;
  valign  = null;
}

void UBackground::set(const UIma *_bgima, const UColor *_bgcolor) {
  if (bgima == _bgima && bgcolor == _bgcolor) return;
  bgima   = _bgima;
  bgcolor = _bgcolor;
  changed(true);
}

void UBackground::set(const UIma *_bgima) {
  set(_bgima, null);
}
void UBackground::set(const UIma &_bgima) {
  set(&_bgima, null);
}
void UBackground::set(const UColor *_bgcolor) {
  set(null, _bgcolor);
}
void UBackground::set(const UColor &_bgcolor) {
  set(null, &_bgcolor);
}

void UBackground::setLayout(const UHalign *ha, const UValign *va) {
  halign = ha;
  valign = va;
}

void UBackground::update(){
  parents.updateParents(UUpdate::paint);
}
void UBackground::putProp(UContext *props, UCtrl*){
  props->local.background = this;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

const UClass UShape::uclass("UShape");

UShape& ushape(UPix &pix) {
  return *(new UShape(pix));
}

UShape::UShape(const UPix &_pix, u_modes m) : UProp(m){
  pix = &_pix;
}

void UShape::set(const UPix *_pix) {
  if (pix == _pix) return;
  pix = _pix;
  changed(true);
}
void UShape::set(const UPix &_pix) {set(&_pix);}

void UShape::update(){
  parents.updateParents(UUpdate::layout);
}
void UShape::putProp(UContext *props, UCtrl*){
  props->local.shape = this;
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */

const UClass UOrient::uclass("UOrient");

UOrient UOrient::vertical("vertical",   UView::VERTICAL, UMode::UCONST);
UOrient UOrient::horizontal("horizontal", UView::HORIZONTAL, UMode::UCONST);


UOrient& uorient(const UOrient& _o) {
  return *(new UOrient(_o));
}
UOrient::UOrient(const UOrient &_o, u_modes m) : UProp(m) {
  value = _o.value;
}

// ce constructeur est reserve a l'usage interne
UOrient::UOrient(const char *iname, char val, u_modes bm) : UProp(bm) {
  value = val;
}

void UOrient::set(const UOrient &o) {
  if (value == o.value) return;
  value = o.value; //if (upd) update();
  changed(true);
}

void UOrient::update() {
  //parents.updateParents(UUpdate::win);
  UUpdate upd(UUpdate::SHOW);
  parents.updateParents(upd);    //!!PBM eventuel?
}

void UOrient::putProp(UContext *props, UCtrl *state) {
  props->local.orient = value;
}

/* ==================================================== [Elc:98] ======= */
/* ==================================================== ======== ======= */

const UClass UValign::uclass("UValign");

UValign UValign::top("top", UView::TOP, UMode::UCONST);
UValign UValign::bottom("bottom", UView::BOTTOM, UMode::UCONST);
UValign UValign::center("center", UView::VCENTER, UMode::UCONST);
UValign UValign::flex("flex", UView::VFLEX, UMode::UCONST);

UValign& uvalign(const UValign &o) {
  return *(new UValign(o));
}

// NB: must be UProp because flex is propagated to direct children
UValign::UValign(const UValign &o, u_modes m) : UProp(m) {
  value = o.value;
}

// ce constructeur est reserve a l'usage interne
UValign::UValign(const char *iname, char val, u_modes bm): UProp(bm){
  value = val;
}

void UValign::set(const UValign &obj) {
  if (value == obj.value) return;
  value = obj.value;
  changed(true);
}

void UValign::update() {
  // box size won't change but children layout must be updated
  parents.updateParents(UUpdate::layout);
}

void UValign::putProp(UContext *props, UCtrl *state) {
  props->local.valign = value;
}

/* ==================================================== [Elc:98] ======= */
/* ==================================================== ======== ======= */

const UClass UHalign::uclass("UHalign");

UHalign UHalign::left("left", UView::LEFT, UMode::UCONST);
UHalign UHalign::right("right", UView::RIGHT, UMode::UCONST);
UHalign UHalign::center("center", UView::HCENTER, UMode::UCONST);
UHalign UHalign::flex("flex", UView::HFLEX, UMode::UCONST);

UHalign& uhalign(const UHalign &o) {
  return *(new UHalign(o));
}

// NB: must be UProp because flex is propagated to the direct child
UHalign::UHalign(const UHalign &o, u_modes m) : UProp(m) {
  value = o.value;
}

// ce constructeur est reserve a l'usage interne
UHalign::UHalign(const char *iname, char val, u_modes bm): UProp(bm){
  value = val;
}

void UHalign::set(const UHalign &o) {
  if (value == o.value) return;
  value = o.value;
  changed(true);
}

void UHalign::update() {
  // box size won't change but children layout must be updated
  parents.updateParents(UUpdate::layout);
}

void UHalign::putProp(UContext *props, UCtrl *state) {
  props->local.halign = value;
}

/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */

const UClass UHmargin::uclass("UHmargin");
//UHmargin UHmargin::none(0);
//UHmargin UHmargin::one(1);
//UHmargin UHmargin::two(2);

UHmargin& uhmargin(int val) {
  return *(new UHmargin(val));
}

UHmargin::UHmargin(int val, u_modes bm): UProp(bm) {
  value = val;
}

void UHmargin::set(int val) {
  if (value == val) return;
  value = val; 
  changed(true);
}

void UHmargin::update() {
  parents.updateParents(UUpdate::layout);
}

void UHmargin::putProp(UContext *props, UCtrl *state) {
  //props->local.hmargin = value;
  props->local.padding.left = value;
  props->local.padding.right = value;
}

/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */

const UClass UVmargin::uclass("UVmargin");
//UVmargin UVmargin::none(0);
//UVmargin UVmargin::one(1);
//UVmargin UVmargin::two(2);

UVmargin& uvmargin(int val) {
  return *(new UVmargin(val));
}

UVmargin::UVmargin(int val, u_modes bm) : UProp(bm) {
  value = val;
}

void UVmargin::set(int val) {
  if (value == val) return;
  value = val;
  changed(true);
}

void UVmargin::update() {
  parents.updateParents(UUpdate::layout);
}

void UVmargin::putProp(UContext *props, UCtrl *state) {
  //props->local.vmargin = value;
  props->local.padding.top = value;
  props->local.padding.bottom = value;
}

/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */

const UClass UHspacing::uclass("UHspacing");
//UHspacing UHspacing::none(0);
//UHspacing UHspacing::one(1);
//UHspacing UHspacing::two(2);

UHspacing& uhspacing(int val) {
  return *new UHspacing(val);
}

UHspacing::UHspacing(int val, u_modes bm) : UProp(bm) {
  value = val;
}

void UHspacing::set(int val) {
  if (value == val) return;
  value = val;
  changed(true);
}

void UHspacing::update() {
  parents.updateParents(UUpdate::layout);
}

void UHspacing::putProp(UContext *props, UCtrl *state) {
  props->local.hspacing = value;
}

/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */

const UClass UVspacing::uclass("UVspacing");
//UVspacing UVspacing::none(0);
//UVspacing UVspacing::one(1);
//UVspacing UVspacing::two(2);

UVspacing& uvspacing(int val) {
  return *(new UVspacing(val));
}

UVspacing::UVspacing(int val, u_modes bm) : UProp(bm) {
  value = val;
}

void UVspacing::set(int val) {
  if (value == val) return;
  value = val;
  changed(true);
}

void UVspacing::update() {
  parents.updateParents(UUpdate::layout);
}

void UVspacing::putProp(UContext *props, UCtrl *state) {
  props->local.vspacing = value;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
// Usage:
// -- uwidth(int) with int >= 0  : size fixed by argument.
//    horizontal size won't change except if the box is uhflex()
// -- uwidth(UWidth::keepSize)   : keep initial size.
//    size is computed automatically the first time, then does not change.
// -- uwidth(UWidth::autoResize) : size automatically adapts
//    size is re-computed automatically each time

const UClass UWidth::uclass("UWidth");
const u_dim UWidth::autoResize  = -2;
const u_dim UWidth::keepSize = -1;

UWidth& uwidth(u_dim val) {
  return *(new UWidth(val));
}

UWidth::UWidth(u_dim val, u_modes m) : UProp(m) {
  value = val;
}

void UWidth::set(u_dim val) {
  if (value == val) return;
  value = val;
  changed(true);
}

void UWidth::update() {
  parents.updateParents(UUpdate::layout);
}

void UWidth::putProp(UContext *props, UCtrl*) {
  props->local.width = value;
}

/* ==================================================== [Elc:99] ======= */
/* ==================================================== ======== ======= */
// Usage:
// -- same as UWidth (see above)

const UClass UHeight::uclass("UHeight");
const u_dim UHeight::autoResize  = -2;
const u_dim UHeight::keepSize = -1;

UHeight& uheight(u_dim val) {
  return *(new UHeight(val));
}

UHeight::UHeight(u_dim val, u_modes m) : UProp(m) {
  value = val;
}

void UHeight::set(u_dim val) {
  if (value == val) return;
  value = val;
  changed(true);
}

void UHeight::update() {
  parents.updateParents(UUpdate::layout);
}

void UHeight::putProp(UContext *props, UCtrl*) {
  props->local.height = value;
}

/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
///TEMPORARIREMENT EN Uprop

const UClass UComment::uclass("UComment");

UComment& ucomment(const char*str) {
  return *(new UComment(str));
}

// can resize if val >= 0 / can't resize if val < 0
UComment::UComment(const char* str, u_modes m) : UProp(m) {
  value = strdup(str);
}

// can resize if val >= 0 / can't resize if val < 0
void UComment::set(const char* str) {
  if (value == str) return;
  if (value) free(value);
  value = strdup(str);
  changed(false);
}

void UComment::update() {
  //parents.updateParents(UUpdate::layout);
}

void UComment::putProp(UContext *props, UCtrl*) {
  //props->local.wrap = value;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

const UClass UPos::uclass("UPos");

UPos::UPos(u_pos _x, u_pos _y, u_modes bm) : UProp(bm) {
  x = _x; y = _y;
}

UPos::UPos() : UProp(0) {
  x = 0; y = 0;
}

void UPos::set(u_pos _x, u_pos _y, u_bool update_now)  {
  if (x == _x && y == _y) return;
  UUpdate upd(UUpdate::MOVE);
  upd.setOffset(_x - x, _y - y);
  x = _x;
  y = _y;
  if (update_now) {
    //NEW: double buffering fait dans UBox::updateView cas MOVE
    //UNIQUEMENT si la view est TRANSPARENTE
    //upd.doubleBuffering(true);

    // il faut d'ABORD mettre le champ OBSOLETE_COORDS a true
    // sur TOUTES les vues PUIS faire updateParent
    // *Raison:
    // il est necessaire de savoir QUAND les coords d'une view
    // sont a jour et qunad elles ne le sont pas, d'ou ces deux phases:
    // -1) setModesOfParentViews active le flag OBSOLETE_COORDS
    //     de toutes les vues concernees pour indiquer que leurs
    //     coords ne sont plus a jour
    // -2) UView::doUpdate/setCoords remet les coords a jour 
    //     et annule ce flag
    // *Note:
    // ce flag est en particulier indispensable pour UBox::updateView
    // dans le cas MOVE, pour savoir si view.x/y sont les anciennes
    // ou nouvelles coords

    parents.setModesOfParentViews(UView::OBSOLETE_COORDS, true);
    parents.updateParents(upd);
  }
  changed(false); //upd deja fait si requis
}

void UPos::update()  {
  //hum, c'est pas le layout qui change mais la position...!
  parents.updateParents(UUpdate::layout);
}

void UPos::putProp(UContext *props, UCtrl *state) {
  props->local.xpos = x;
  props->local.ypos = y;
}

void UPos::addingTo(ULink *selflink,  UGroup *parent) {
  UProp::addingTo(selflink, parent);
  // Rendre parent sensitif aux events ad hoc
  parent->setCmodes(UMode::FLOATING, true);
}

void UPos::removingFrom(ULink *selflink, UGroup *parent) {
  //int count = getChildClassCount<UGroup*, UPos*>(parent);
  int count = parent->getTypedChildCount<UPos>();
  if (count <= 1)   parent->setCmodes(UMode::FLOATING, false);
  UProp::removingFrom(selflink, parent);
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:00] ======= */
/* ==================================================== ======== ======= */
//OBSOLETE : REMPLACE PAR UPos

UFloating::UFloating(UBox *bx) : UBoxLink(bx) {
  rx = ry = 0;
}
UFloating::UFloating(int x, int y, UBox *bx) : UBoxLink(bx) {
  rx = x; ry = y;
}

void UFloating::init(UBox *bx) {
  if (!b) b = bx;
}
void UFloating::init(UBox &bx) {
  if (!b) b = &bx;
}

u_pos UFloating::getX() {return rx;}
u_pos UFloating::getY() {return ry;}
void UFloating::where(u_pos *xloc, u_pos *yloc) {
  *xloc = rx; *yloc = ry;
}

// changes the location of the UFloating relatively to its PARENT
//
void UFloating::move(u_pos xloc, u_pos yloc) {
  UUpdate upd(UUpdate::MOVE);
  upd.setOffset(xloc - rx, yloc - ry);
  upd.doubleBuffering(true);
  rx = xloc;
  ry = yloc;
  //if (boxCast()) boxCast()->update(upd);
  UBox *bx = dynamic_cast<UBox*>(brick());
  if (bx) bx->update(upd);
}

void UFloating::setX(u_pos xloc)  {move(xloc, ry);}
void UFloating::setY(u_pos yloc)  {move(rx, yloc);}

//don't update graphics now !!!
//
void UFloating::_setX(int xloc)  {rx = xloc;}
void UFloating::_setY(int yloc)  {ry = yloc;}

/* ==================================================== ======== ======= */

UFloating& ufloating(UBox &b) {
  return *(new UFloating(&b));
}

UFloating& ufloating(UBox *b) {
  return *(new UFloating(b));
}

UFloating& ufloating(int relx, int rely, UBox &b) {
  return *(new UFloating(relx, rely, &b));
}

UFloating& ufloating(int relx, int rely, UBox *b) {
  return *(new UFloating(relx, rely, b));
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:01] ======= */
