/* ==================================================== ======== ======= *
 *
 *  uuconf.cpp : configuration of the UAppli
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuconf.cpp	ubit:03.06.04"
#include <locale.h>
#include <iostream>
#include <config.h>
#include <ubit.hpp>
using namespace std;

std::vector<UConf::Profile> UConf::profiles;
bool UConf::init = UConf::initDefaultProfiles();

/* ==================================================== ======== ======= */

UConf::UConf(int& _argc, char** _argv, UOption* _client_options) :
  filebox_width(*new UWidth()),
  filebox_height(*new UHeight())
{
  disp_name = null;
  app_name = null;
  saveArgs(_argc, _argv);

  //locale = "iso_8859_1"; 
  locale           = "";        // locale is given by the environment
  truecolor_depth  = 24;        // default color depth (this is a hint)
  softwins         = false;     // group and GL modes set this mode to true
  telepointers     = false;     // group mode sets this mode to true
  linear_gamma     = true;      // linear gamma visual for Solaris
  transp_scrollbars= true;      // false reduces CPU usage
  sync             = false;     // synchronous mode (for debug)
  menu_grab        = true;	// false deactivates menu_grabs (for debug)
  force_winfocus   = true;	// should be true except in special cases
  scrollbar_grab   = false;	// not impl
  iconic           = false;	// not impl

#ifdef WITH_GL
  double_buffering = true;      // hard doublebuff (managed by GL)
#else
  double_buffering = false;     // soft doublebuff (simulated by Ubit)
#endif

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

  defaultStyle = new UStyle();	// initializes the default style

  app_lscale          = 0;	// 0 is the default
  app_xyscale         = 1.;     // set by Appli::UAppli
  max_app_width      = -1;	// -1 means any size
  max_app_height     = -1;	// -1 means any size

  filebox_width        = 400;	// MUST be defined
  filebox_height       = 230;	// idem
  filebox_line_count   = 10;	// idem

  scrollpane_bar_size  = 10;     // not impl
  unknown_image = &UPix::question; // MUST be defined

  mouse_click_radius   = 7;	// MUST be defined
  mouse_click_delay    = 500;	// millisec
  auto_open_menu_delay = 300;	// millisec

  default_selection_color  = null; // none if pointer is null
  default_selection_bgcolor = &UBgcolor::lightyellow; // idem
  default_selection_font    = &UFont::fill; // idem

  UFontFamily::helvetica.init("helvetica","medium","bold","o","normal","1");
  UFontFamily::courier.init("courier", "medium", "bold", "o", "normal","1");
  UFontFamily::times.init("times", "medium", "bold", "i", "normal","1");
  UFontFamily::fixed.init("fixed", "medium", "bold", "o", "*", "1");
  UFontFamily::standard = UFontFamily::helvetica;
  UFontFamily::any = UFontFamily::standard;
  
  parseOptions(_argc, _argv, _client_options);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UConf::saveArgs(int _argc, char** _argv) {
  if (_argc && _argv) {
    app_argc = _argc;
    app_argv = new char*[_argc];
    for (int k = 0; k < _argc; k++) app_argv[k] = CStr::strdup(_argv[k]);
  }
  else {
    app_argc = 0;
    app_argv = null;
  }

  UStr cname((_argc >=1) ? _argv[0] : null);
  cname = cname.getFileName();
  app_name = CStr::strdup(cname.chars());
}

/* ==================================================== ======== ======= */

static const char* testOption(const char* arg, const char* name,
                              const char* ending, const char*& stat) {
  const char *pa = arg, *pn;
  stat = "true";
    
  if (pa[0] == 'n' && pa[1] == 'o' && pa[2] == '-') {
    stat = null;
    pa = arg+3;
  }
  
  for (pn = name; *pn != 0; pn++) {
    if (*pa == *pn) pa++;
    else return null;
  }
  
  for (pn = ending; *pn != 0; pn++) {
    if (*pa == *pn) pa++;
    else if (*pa == 0) return pa;
    else return null;
  }

  return pa;
}

static void removeOption(int& card, char** tab, int k) {
  card--;
  for (int j = k; j < card; j++) tab[j] = tab[j+1];
}

static UOption* match(const char* arg,
                      UOption* client_options, UOption* ubit_options,
                      const char*& stat)
{
  if (client_options) {
    for (UOption* p = client_options; p->begname != null; p++) {
      if (testOption(arg, p->begname, p->endname, stat))
        return p;
    }
  }
  
  for (UOption* p = ubit_options; p->begname != null; p++) {
    if (testOption(arg, p->begname, p->endname, stat))
      return p;
  }

  return null;  // not found
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

struct UOptionArg {
  virtual int getArgCount() = 0;
  virtual void set(const char*) = 0;
};

struct UOptionBoolArg : public UOptionArg {
  bool& val;
  UOptionBoolArg(bool& _val) : val(_val) {}
  virtual int getArgCount() {return 0;}
  virtual void set(const char* _val) {val = (_val ? true : false);}
};

struct UOptionIntArg : public UOptionArg {
  int& val;
  UOptionIntArg(int& _val) : val(_val) {}
  virtual int getArgCount() {return 1;}
  virtual void set(const char* _val) {val = (_val ? atoi(_val) : 0);}
};

struct UOptionCStrArg : public UOptionArg {
  char*& val;
  UOptionCStrArg(char*& _val) : val(_val) {}
  virtual int getArgCount() {return 1;}
  virtual void set(const char* _val) {val = (_val ? CStr::strdup(_val) : null);}
};

struct UOptionCCStrArg : public UOptionArg {
  const char*& val;
  UOptionCCStrArg(const char*& _val) : val(_val) {}
  virtual int getArgCount() {return 1;}
  virtual void set(const char* _val) {val = (_val ? CStr::strdup(_val) : null);}
};

/* ==================================================== ======== ======= */

class UOptionArg* UOption::Arg(bool& _val) {
  return new UOptionBoolArg(_val);
}

class UOptionArg* UOption::Arg(int& _val) {
  return new UOptionIntArg(_val);
}

class UOptionArg* UOption::Arg(char*& _val) {
  return new UOptionCStrArg(_val);
}

class UOptionArg* UOption::Arg(const char*& _val) {
  return new UOptionCCStrArg(_val);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UConf::parseOptions(int& argc, char** argv, UOption* client_options) {
  bool group = false;
  bool help = false;
  const char* conf_file = null;
  
  UOption ubit_options[] = {
    {"disp","lay",         UOption::Arg(disp_name)},
    {"scale","",           UOption::Arg(app_lscale)},
    {"doubleb","uffering", UOption::Arg(double_buffering)},
    {"dbf","",             UOption::Arg(double_buffering)},
    {"transps","crollbars",UOption::Arg(transp_scrollbars)},
    {"tsb","",             UOption::Arg(transp_scrollbars)},
    {"softw","ins",        UOption::Arg(softwins)},
    {"tele","pointers",    UOption::Arg(telepointers)},
    {"group","ware",       UOption::Arg(group)},
    {"grab","",            UOption::Arg(menu_grab)},
    {"sync","hronous",     UOption::Arg(sync)},
    {"conf","",            UOption::Arg(conf_file)},
    {"help-x","",          UOption::Arg(help)},
    {"help-u","bit",       UOption::Arg(help)},
    {"locale","",          UOption::Arg(locale)},
    {"bpp","",             UOption::Arg(truecolor_depth)},
    {null, null, null}
  };

  int k = 1;
  while (k < argc) { 

    const char *p = argv[k];
    //const char *ending = null;
    const char* stat = null;
    
    if (*p != '-') {k++; continue;}     // un - obligatoire
    else p++;
    if (*p == '-') p++;                 // un - optionnel
    if (*p == 0) {k++; continue;}

    UOption* opt = null;
    
    if ((opt = match(p, client_options, ubit_options, stat))) {
      
      removeOption(argc, argv, k);

      if (opt->arg->getArgCount() == 0) {
        opt->arg->set(stat);
      }
      
      else if (opt->arg->getArgCount() > 0) {
        if (k < argc) {
          opt->arg->set(argv[k]);
          removeOption(argc, argv, k);
        }
        else {
          //UError::error("warning@UConf::parseOptions",
          //              "this option requires an argument");
          std::string optname;
          if (opt->begname) {
            optname = " --";
            optname += opt->begname;
          }
          if (opt->endname) {
            optname += "[";
            optname += opt->endname;
            optname += "]";
          }
          cerr << "Warning: option" << optname <<  " requires an argument" << endl;
        }
      }
    }
    else k++;   // autre options: a traiter ailleurs
  }

  if (group) {
    softwins = true;
    telepointers = true;
  }

  if (conf_file) setProfile(conf_file);

  if (help) {
    cout
    << "Ubit options: "
    << "\n  --help-ubit, --help-x      : prints this message"
    << "\n  --conf <profile>           : sets configuration profile"
    << "\n  --disp[lay] <displayname>  : uses X-server 'displayname'"
    << "\n  --locale <string>          : uses this locale"
    << "\n  --scale <int>              : sets global scale (0=default, > 0 larger, < 0 smaller)"
    << "\n  --bpp <int>                : #bits per pixel (hint)"
    << "\n  --[no-]doubleb[uffering], --[no-]dbf : double buffering mode (slower but avoids flickering)"
    << "\n  --[no-]softw[ins]          : soft window mode"
    << "\n  --[no-]tele[pointers]      : tele pointer mode"
    << "\n  --[no-]group[ware]         : groupware mode"
    << "\n  --[no-]transps[crollbars], --[no-]tsb: transparent scrollbar mode"
    << "\n  --[no-]grab                : disables mouse grabs (for debugging)"
    << "\n  --[no-]sync[hronous]       : synchronous mode (for debugging)"
    << endl << endl;
    exit(0);
  }

  /*
   else if (testOption(p, "GL")) {
     graphics_engine = GL;
     removeOption(argc, argv, k);
   }
   else if (testOption(p, "iconic")) {
     iconic = true;
     removeOption(argc, argv, k);
   }
   */
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UConf::Profile::Profile(const char* _name, void (*_apply)(UConf&)) {
  name = _name; 
  apply= _apply;
}

void UConf::addProfile(const char* name, void (*apply)(UConf&)) {
  Profile prof(name, apply);
  // chercher s'il existe pas deja !... !!!
  profiles.push_back(prof);
}

bool UConf::setProfile(const char* name) {
  for (unsigned int k=1; k < profiles.size(); k++)
    if (strcasecmp(profiles[k].name, name) == 0) {
      (profiles[k].apply)(*this);
      return true;
    }

  //UError::error("warning@UConf::setProfile","unknown configuration profile:",name);
  cerr << "Warning: --prof " << name << " : unknown configuration profile" << endl; 
  return false;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static void apply_IPAQ(UConf& conf) {
  conf.scrollpane_mode = conf.scrollpane_mode & ~UScrollbar::TRANSP;
  conf.app_lscale   = -1;
  conf.max_app_width  = 240;
  conf.max_app_height = 320;
}

static void apply_VGA(UConf& conf) {
  conf.max_app_width  = 640;
  conf.max_app_height = 480;
}

static void apply_SVGA(UConf& conf) {
  conf.max_app_width  = 800;
  conf.max_app_height = 600;
}

static void apply_XGA(UConf& conf) {
  conf.max_app_width  = 1024;
  conf.max_app_height = 768;
}

bool UConf::initDefaultProfiles() {
  //addProfile("DEFAULT", apply_UNCONSTRAINED);
  //addProfile("UNCONSTRAINED", apply_UNCONSTRAINED);
  addProfile("IPAQ", apply_IPAQ);
  addProfile("I", apply_IPAQ);
  addProfile("VGA",  apply_VGA);
  addProfile("SVGA", apply_SVGA);
  addProfile("XGA",  apply_XGA);
  return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/*
  void setLocale(const char*);
  void setMouseClickDelay(u_time);
  void setMouseClickRadius(u_dim);
  void setAutoOpenMenuDelay(u_time);
  ///< configuration (can be called at any time).

  void setSelectionColor(const UColor*);
  void setSelectionBgcolor(const UColor*);
  void setSelectionFont(const UFont*);
void UAppli::setLocale(const char* loc)           {setlocale(LC_ALL, loc);}

void UAppli::setMouseClickDelay(u_time t)         {MOUSE_CLICK_DELAY = t;}
void UAppli::setMouseClickRadius(u_dim r)         {MOUSE_CLICK_RADIUS = r;}
void UAppli::setAutoOpenMenuDelay(u_time t)       {AUTO_OPEN_MENU_DELAY = t;}

void UAppli::setSelectionColor(const UColor* c)   {SELECTION_COLOR = c;}
void UAppli::setSelectionBgcolor(const UColor* c) {SELECTION_BGCOLOR = c;}
void UAppli::setSelectionFont(const UFont* f)     {SELECTION_FONT = f;}
*/

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
