/* ==================================================== ======== ======= *
 *
 *  umsd.cpp : Ubit Multiple Mouse,Message Server
 *
 *  Ubit Project [Elc::2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)umsd.cpp	ubit:03.06.04"
#include <stdlib.h>
#include <signal.h>        
#include "umserver.hpp"
#include "umsflow.hpp"
#include "drivers/serial_mouse.hpp"
#include "drivers/mimio.hpp"
using namespace std;

// default serial ports
static const char* SOLARIS_SERIAL_PORT   = "/dev/ttya";
static const char* LINUX_SERIAL_PORT     = "/dev/ttyS0";   // (= COM1)
static const char* LINUX_USBSERIAL_PORT = "/dev/ttyUSB0";
static const char* MACOSX_USBSERIAL_PORT = "/dev/tty.USA19QW1P1.1";

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

class UMSD {
public:
  UMSD();
  ~UMSD();

  void usage();
  void parseOptions(int argc, char *argv[]);
  void start();

private:
  struct Pointer {
    int ptr_no;
    const char *fg, *bg;
    Pointer() {ptr_no = 1; fg = null; bg = null;}
    Pointer(int _no, const char* _fg, const char* _bg) {
      ptr_no = _no; fg = _fg; bg = _bg;}
  };

  struct Device {
    enum Type {MOUSE, MIMIO} type; 
    const char* port;
    int ptr_no;
    Device(Type t) {type = t; port = null; ptr_no = 1;}
  };

  struct Neighbor {
    const char* pos;
    const char* host;
    Neighbor() {pos = null; host = null;}
  };

  // these fct can possibly do something useful (see: setActionKey())
  void key_pressed(UMServer&, XEvent& e) {
    cerr << "action key " << e.xkey.keycode << " was pressed" << endl;
  }
  void key_released(UMServer&, XEvent& e) {
    cerr << "action key " << e.xkey.keycode << " was released" << endl;
  }

  const char* testOption(int& k, const char* name, const char* ending = "");
  const char* getPort(int& k);
  int getUInt(int& k);
  const char* getStr(int& k);

  int argc; char** argv;
  const char* display_name;
  bool emulate_buttons;
  int ums_port;
  UMServer* ums;
  vector <Device> devices;
  vector <Pointer> pointers;
  vector <Neighbor> neighbors;
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

// NB: desctructor always called at exit (closes sockets and devices) 
static UMSD umsd;

int main(int argc, char *argv[]) {
  umsd.parseOptions(argc, argv);
  umsd.start();
}

UMSD::UMSD() {
  argc = 0; 
  argv = NULL;
  display_name = ":0";
  emulate_buttons = true;
  ums_port = 0;    // will use UMS_DEFAULT_PORT
  ums = null;

  signal(SIGSEGV, exit);
  signal(SIGTERM, exit);
  signal(SIGQUIT, exit);
  signal(SIGINT,  exit);
  signal(SIGHUP,  exit);
  //atexit(cleanup);
}

UMSD::~UMSD() {
  cerr << "*** UMS cleanup ***" << endl << endl;
  if (ums) delete ums;  // closes devices and sockets
  ums = null;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMSD::start() {
  cout << "*** UMS (Ubit Multiple Mouse/Message Server) ***" << endl;

  ums = new UMServer(ums_port, display_name);
  if (!ums->isInit()) exit(1);

  // =====================================================================
  // create the event flows 
  //  - several event flows can be created
  //  - their IDs MUST be != 0 (and have different values)
  //  - the native event flow (that corresponds to the standard X mouse)
  //    is created implicitely by the UMServer. Its ID is always 0.
  //  - NOTE that X clients can send events to any valid event flow
  //    (including the native X event flow)

  for (u_int p = 0; p < pointers.size(); p++) {
    UMSmouseFlow* mflow = ums->getOrCreateMouseFlow(pointers[p].ptr_no);
    if (!mflow)
      cerr << "invalid pointer number: " << pointers[p].ptr_no << endl;
    else if (pointers[p].ptr_no > 0) {
      mflow->setPointer(pointers[p].fg, pointers[p].bg);
      cout << " - Pointer #" << pointers[p].ptr_no << " created" << endl;
    }
  }

  // =====================================================================
  // open the sources and connect them to the event flows
  // NB: several sources can be connected to the same event flow 
    
  for (u_int d = 0; d < devices.size(); d++) {

    UMSmouseFlow* mflow = ums->getOrCreateMouseFlow(devices[d].ptr_no);
    if (!mflow)  {
      cerr << "invalid pointer number: " << devices[d].ptr_no << endl;
      continue;
    }

    UMSeventSource* so = null;

    if (devices[d].type == Device::MIMIO) {
      so = new UMSmimioSource(mflow, devices[d].port);
      if (so) {
	// default mappings that can be changed here
	/*
         // attention: il n'y a que 5 Buttons avec X (4 & 5 servent au scroll)
         setButtonMapping(1, 0, Button1, 0);	        // black pen
         setButtonMapping(2, 0, Button1, ShiftMask);	// blue pen
         setButtonMapping(3, 0, Button3, 0); 	        // green pen
         setButtonMapping(4, 0, Button1, ShiftMask|ControlMask); // red pen
         setButtonMapping(9, 0, Button2, ControlMask);	// big eraser
         setButtonMapping(10,0, Button2, 0);		// small eraser
         
	// boutons de controle du MIMIO (values: 10 + (1, 2, 4, 8, 16))
	so->setButtonMapping(11, 0, Button3, 0);	  // b1 (top)
	so->setButtonMapping(12, 0, Button3, ControlMask);// b2
	so->setButtonMapping(14, 0, Button3, ShiftMask);  // b3
	so->setButtonMapping(18, 0, Button3, Mod1Mask);	  // b4
	so->setButtonMapping(26, 0, Button3, Mod2Mask);	  // b5 (bottom)
	*/
      } 
    }

    else {  //serial mouse
      so = new UMSserialMouseSource(mflow, devices[d].port);
      if (so) {
 	// default mappings that can be changed here
	/*
	so->setButtonMapping(Button1, 0, Button1, 0);
	so->setButtonMapping(Button2, 0, Button2, 0);
	so->setButtonMapping(Button3, 0, Button3, 0);
	so->setButtonMapping(Button4, 0, Button4, 0);
	so->setButtonMapping(Button5, 0, Button5, 0);
	*/
      }
    }

    if (so->is_open()) {
      cout << " - Device opened on port: "<< devices[d].port << endl;
      ums->addEventSource(so);
    } 
    else  {
      cerr << "! can't open device on port: " << devices[d].port << endl;
      delete so;
      exit(2);
    }
  }

  // =====================================================================
  // emulation of mouse button 2 and 3 events (for machines that do not
  // have 3 buttons such as the IPAQ):

  if (emulate_buttons) {
    cout << " - Mouse button emulation:" << endl;

    cout << "   + Ctrl-Btn1 emulates Btn3" << endl;
    ums->setNativeButtonMapping(Button1, ControlMask, Button3, 0);
    
    cout << "   + Mod1-Btn1 emulates Btn2" << endl;
    //ums->setNativeButtonMapping(Button1, Mod4Mask, Button2, 0);
    ums->setNativeButtonMapping(Button1, Mod1Mask, Button2, 0);
 
    //cout << "   + Ctrl-Btn2 emulates ScrollUp" << endl;
    //ums->setNativeButtonMapping(Button2, ControlMask, Button4, 0);

    //cout << "   + Alt-Btn2 emulates ScrollDown" << endl;
    //ums->setNativeButtonMapping(Button2, Mod4Mask, Button5, 0);
    //ums->setNativeButtonMapping(Button2, Mod1Mask, Button5, 0);
  }
  
  // =====================================================================
  // pre-defined keys that perform some action (such as opening
  // menus or whatever)
  // ums->setActionKey(110, 0, key_pressed, key_released);

  // =====================================================================
  // defines the topology of UMS servers

  for (u_int n = 0; n < neighbors.size(); n++) {
    if (ums->addNeighbor(neighbors[n].pos, neighbors[n].host))
      cout << " - Topology: adding neighbor: " 
           << neighbors[n].pos << " " << neighbors[n].host << endl;
    else
      cerr << "! invalid neighbor in topology: " 
           << neighbors[n].pos << " " << neighbors[n].host << endl;
  }

  // =====================================================================
  // start reading sources, UMS server requests and X server

  // TEST
  // ums->startCalibration("MIMIO Calibration", false);

  ums->start();
  exit(0);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMSD::usage() {
  cerr
  << "usage: umsd [options]" << endl
  << "options: "<< endl
  << "   -mo[use] port ptr_id   // port = sun,linux,usblinux,usbmac,/dev/tty**" << endl
  << "   -mi[mio] port ptr_id   // ptr_id  = 0 (X pointer) | 1,2,3... (alternate pointers)" << endl
  << "   -p[tr] ptr_id [fg_color] [bg_color]" << endl
  << "   -e[mulate] [0|1]    // emulates mouse buttons" << endl
  << "   -t[opology] pos1 host1 [pos2 host2] ...  // pos = n, s, e, w, ne, nw" << endl
  << "   -disp[lay] display-name" << endl
  << "   -ums[port] port-number" << endl
  << endl;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMSD::parseOptions(int _argc, char *_argv[]) {
  argc = _argc;
  argv = _argv;

  Pointer p1(1, "red", null);
  pointers.push_back(p1);

  Pointer p2(2, "green", null);
  pointers.push_back(p2);

  Pointer p3(3, "blue", null);
  pointers.push_back(p3);
  
  int last_default_ptr = 3;

  int k = 1;
  while (k < argc) { 

    if (testOption(k,"-p","tr") || testOption(k,"-p","ointer")) {
        Pointer p;
        p.ptr_no = getUInt(k);
        if (p.ptr_no <= 0) p.ptr_no = 1;  //defaut
        p.fg = getStr(k);
        p.bg = getStr(k);
        if (p.ptr_no <= last_default_ptr) {
          cerr << "pointer " << p.ptr_no
          << " exists already; No changed to "
          << p.ptr_no + last_default_ptr << endl;
          p.ptr_no += last_default_ptr;
        }
        pointers.push_back(p);
    }

    else if (testOption(k,"-mo","use")) {
      if (k < argc) {
        Device d(Device::MOUSE);
	d.port = getPort(k);
  	d.ptr_no = getUInt(k);
        if (d.ptr_no < 0) d.ptr_no = 0;  // default
        devices.push_back(d);
      }
      else {cerr << "-mouse requires an argument" << endl; exit(1);}
    }

    else if (testOption(k,"-mi","mio")) {
      if (k < argc) {
        Device d(Device::MIMIO);
	d.port = getPort(k);
	d.ptr_no = getUInt(k);
        if (d.ptr_no < 0) d.ptr_no = 0;  // default
        devices.push_back(d);
      }
      else {cerr << "-mimio requires an argument" << endl; exit(1);}
    }

    else if (testOption(k,"-t","opology")) {
      if (k < argc) {
        while(true) {
          Neighbor n;
	  n.pos  = getStr(k);
	  n.host = getStr(k);
          if (n.pos) neighbors.push_back(n);
          else break;
        } 
      }
      else {cerr << "-topology requires an argument" << endl; exit(1);}
    }

    else if (testOption(k,"-e","mulate")) {
      emulate_buttons = true;
      if (k < argc) {
        emulate_buttons = getUInt(k);
        /*
         const char* s = getStr(k);  //sw or b23
	if (strcmp(s,"sw")==0) emulate_scroll_wheel = getUInt(k);
	else if (strcmp(s,"b23")==0) emulate_button_2_and_3 = getUInt(k);
         */
      }
    }

    else if (testOption(k,"-disp","lay")) {
      if (k < argc) display_name = getStr(k);
      else {cerr << "-display requires an argument" << endl; exit(1);}
    }

    else if (testOption(k,"-ums","port")) {
      if (k < argc) ums_port = getUInt(k);
      else {cerr << "-umsport requires an argument" << endl; exit(1);}
    }

    else if (testOption(k,"-h","elp") || testOption(k,"--h","elp")) {
      usage();
      exit(1);
    }

    else {
      cerr << "unrecognized option" << endl;
      usage();
      exit(1);
    }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

const char* UMSD::testOption(int& k, const char* name, const char* ending) {
  const char* pa = argv[k];
  const char* pn;

  for (pn = name; *pn != 0; pn++) {
    if (*pa == *pn) pa++;
    else return null;
  }
  for (pn = ending; *pn != 0; pn++) {
    if (*pa == *pn) pa++;
    else if (*pa == 0) {k++; return pa;}
    else return null;
  }

  k++; return pa;
}

const char* UMSD::getPort(int& k) {
  const char* p = argv[k];

  if (p[0] == '/') ;

  else if (strncmp(p,"lin",3)==0)
    p = LINUX_SERIAL_PORT;

  else if (strncmp(p,"usblin",6)==0)
    p = LINUX_USBSERIAL_PORT;

  else if (strncmp(p,"usbmac",6)==0)
    p = MACOSX_USBSERIAL_PORT;
  
  else if (strncmp(p,"sun",3)==0)
    p = SOLARIS_SERIAL_PORT;

  else if (strncmp(p,"sol",3)==0)
    p = SOLARIS_SERIAL_PORT;
  else {
    string* s = new string("/dev/");
    *s += p;
    p = s->c_str();
  }

  k++;
  return p;
}

int UMSD::getUInt(int& k) {
  if (k < argc && argv[k][0] >= '0' && argv[k][0] <= '9') {
    return atoi(argv[k++]);
  }
  return -1;   // invalid
}

const char* UMSD::getStr(int& k) {
  if (k < argc && argv[k][0] != '-') {
    return argv[k++];
  }
  return null;  // default
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
