/* $Id: hw_agp.c,v 1.5 2000/03/29 00:53:44 gareth Exp $ */

/*
 * AGP memory handling.
 *
 * The current interface allows blocks of AGP memory to be allocated and
 * deallocated as required.  There is no upper limit other than the size
 * of the aperture, and blocks will be obtained from the agpgart module
 * on demand.
 *
 * There is scope to change this allocation scheme to a growable-buffer
 * approach, but the typical use of AGP memory shouldn't require this.
 * That is, DMA buffers that reside in AGP space will be allocated on
 * startup, and AGP textures should be allocated on program startup and
 * freed on program termination.  There will be a performance hit if the
 * application creates and deletes a lot of textures.  glTexSubImage*()
 * is the preferred way to update textures.
 *
 * There will also be space lost when allocating small buffers, as only
 * multiples of 4k pages can be obtained.
 *
 * Gareth Hughes <gareth@precisioninsight.com>
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>

#ifdef HAVE_LINUX_NEWAGP
/* AGP kernel module interface */
#include <linux/agpgart.h>
#include <sys/ioctl.h>
#endif

#include "hwlog.h"
#include "glx_symbols.h"

#include "hw_mtrr.h"
#include "hw_agp.h"

#define PAGE_SHIFT			12
#define	PAGE_SIZE			(1<<PAGE_SHIFT)

#ifdef HAVE_LINUX_NEWAGP
static int		gartfd = -1;
static void		*gartbuf;

#define GLX_AGPDEVICE			AGP_DEVICE
#define GLX_AGP_INFO_IOCTL		AGPIOC_INFO

static agp_info		info;
#endif

memHeap_t		*heap;


#ifdef HAVE_LINUX_NEWAGP

/*
 * Initialize the agpgart module and AGP aperture.  Map the agpgart device
 * into memory.  Set AGP 1X, 2X or 4X mode as required.  This function must
 * be called before any other access to AGP memory is made.  If limit is
 * non-zero, set the upper limit of available AGP memory to this amount (in
 * megabytes).
 *
 * If any stage of the initialization fails, close the device filehandle and
 * return an error code.
 */
int hwInitAGPMem( int mode, int limit )
{
	size_t		size;

	/* open the device */
	gartfd = open( GLX_AGPDEVICE, O_RDWR );
	if ( gartfd == -1 ) {
		hwMsg( 1, "unable to open " GLX_AGPDEVICE ": %s\n", sys_errlist[errno] );
		return -1;
	}

	/* acquire the module */
	if ( ioctl( gartfd, AGPIOC_ACQUIRE ) != 0 ) {
		hwMsg( 1, "error acquiring AGP module: %s\n", sys_errlist[errno] );
		close( gartfd );
		return -1;
	}

	/* get the current module info */
	if ( ioctl( gartfd, AGPIOC_INFO, &info ) != 0 ) {
		hwMsg( 1, "error doing AGP info ioctl: %s\n", sys_errlist[errno] );
		hwMsg( 1, "first attempt\n" );
		close( gartfd );
		return -1;
	}

	/* put an upper limit on the available memory */
	if ( ( limit == 0 ) || ( info.aper_size <= limit ) ) {
		size = info.aper_size;
	} else {
		size = limit;
	}

	hwMsg( 10, "InitAGPMem: using AGP %dX, %dmb of %dmb aperture\n", mode, size, info.aper_size );

	/* map the device into our memory space */
	gartbuf = mmap( NULL, info.aper_size * 0x100000, PROT_READ | PROT_WRITE, MAP_SHARED, gartfd, 0 );
	if ( gartbuf == MAP_FAILED ) {
		hwMsg( 1, "mmap() on " GLX_AGPDEVICE " failed: %s\n", sys_errlist[errno] );
		close( gartfd );
		return -1;
	}

	if ( __glx_is_server ) {
		agp_setup	setup;

		/* set up the new agp mode */
		setup.agp_mode = (info.agp_mode & ~7) | (info.agp_mode & mode);

		if( ioctl( gartfd, AGPIOC_SETUP, &setup ) != 0 ) {
			hwMsg(1, "error initializing AGP point to point connection\n");
			close( gartfd );
			return -1;
		}

		/* get the current module info, with the mode change */
		if ( ioctl( gartfd, AGPIOC_INFO, &info ) != 0 ) {
			hwMsg( 1, "error doing AGP info ioctl: %s\n", sys_errlist[errno] );
			hwMsg( 1, "second attempt\n" );
			close( gartfd );
			return -1;
		}

		/* make the aperture write-combining */
		SetWriteCombining( info.aper_base, info.aper_size * 0x100000 );
	}

	/* initialize the agp memory heap manager (with size limit) */
	heap = mmInit( 0, size * 0x100000 );

	/* release the module */
	if ( ioctl( gartfd, AGPIOC_RELEASE ) != 0 ) {
		hwMsg( 1, "error releasing AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	return 0;
}


/*
 * Get the current status of the AGP module.  Useful for obtaining the size
 * and base of the aperture for hardware-specific initialization, among
 * other things.  This function takes a void pointer that will be cast to an
 * agp_info pointer when the agpgart module is present, preventing dependence
 * on the agpgart headers when the module is not present.
 */
int hwGetAGPInfo( void *info )
{
	/* ensure the device has been opened */
	if ( gartfd == - 1 ) {
		hwMsg( 1, "GetAGPAperture: " GLX_AGPDEVICE " not opened\n" );
		return -1;
	}

	/* acquire the module */
	if ( ioctl( gartfd, AGPIOC_ACQUIRE ) != 0 ) {
		hwMsg( 1, "error acquiring AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	/* get the current module info */
	if ( ioctl( gartfd, AGPIOC_INFO, (agp_info *)info ) != 0 ) {
		hwMsg( 1, "error doing AGP info ioctl: %s\n", sys_errlist[errno] );
		return -1;
	}

	/* release the module */
	if ( ioctl( gartfd, AGPIOC_RELEASE ) != 0 ) {
		hwMsg( 1, "error releasing AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	return 1;
}


/*
 * Allocate and deallocate blocks of memory in the AGP aperture.  Pointers to
 * the virtual memory mapping of the block will be set up, as well as offsets
 * from the base of the aperture for programming hardware access to the block.
 */
int hwAllocAGPMem( agp_mem_t *b, size_t size )
{
	agp_allocate	*entry = &(b->entry);
	agp_bind	bind;
	int		pages;

	hwMsg( 15, "AllocAGPMem: allocating %d bytes\n", size );

	/* ensure the device has been opened */
	if ( gartfd == - 1 ) {
		hwMsg( 1, "AllocAGPMem: " GLX_AGPDEVICE " not opened\n" );
		return -1;
	}

	/* grab the block from the heap manager */
        if ( ( b->memBlock = mmAllocMem( heap, size, PAGE_SHIFT, 0 ) ) == 0 ) {
		hwMsg( 15, "couldn't allocate buffer\n" );
		b->size = 0;
		return 0;
        }

	/* acquire the module */
	if ( ioctl( gartfd, AGPIOC_ACQUIRE ) != 0 ) {
		hwMsg( 1, "error acquiring AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	pages = (size + PAGE_SIZE - 1) / PAGE_SIZE;

	memset( entry, 0, sizeof(agp_allocate) );

	/* allocate the agp memory */
	entry->pg_count = pages;
	entry->type = 0;

	if ( ioctl( gartfd, AGPIOC_ALLOCATE, entry ) ) {
		hwMsg( 1, "AllocAGPMem: allocation of %i pages failed\n", pages );
		return -1;
	}

	/* bind the agp memory */
	bind.key = entry->key;
	bind.pg_start = b->memBlock->ofs >> PAGE_SHIFT;

	if ( ioctl( gartfd, AGPIOC_BIND, &bind ) != 0 ) {
		hwMsg( 1, "AllocAGPMem: bind of %i pages failed\n", pages );
		return -1;
	}

	/* release the module */
	if ( ioctl( gartfd, AGPIOC_RELEASE ) != 0 ) {
		hwMsg( 1, "error releasing AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	/* set up the newly-allocated virtual memory block */
	b->buffer = (unsigned char *)gartbuf + b->memBlock->ofs;
	b->size = size;
	b->age = -1;

	return 1;
}

int hwFreeAGPMem( agp_mem_t *b )
{
	hwMsg( 15, "FreeAGPMem: freeing %d bytes\n", b->size );

	/* ensure the device has been opened */
	if ( gartfd == - 1 ) {
		hwMsg( 1, "FreeAGPMem: " GLX_AGPDEVICE " not opened\n" );
		return -1;
	}

	/* acquire the module */
	if ( ioctl( gartfd, AGPIOC_ACQUIRE ) != 0 ) {
		hwMsg( 1, "error acquiring AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	/* deallocate the given block */
	if ( ioctl( gartfd, AGPIOC_DEALLOCATE, b->entry.key ) != 0 ) {
		hwMsg( 1, "FreeAGPMem: deallocation of %i pages failed\n", b->entry.pg_count );
		return -1;
	}

	/* clean up after the deallocation */
	mmFreeMem( b->memBlock );
	memset( &(b->entry), 0, sizeof(agp_allocate) );
	b->buffer = NULL;
	b->size = 0;
	b->age = -1;

	/* release the module */
	if ( ioctl( gartfd, AGPIOC_RELEASE ) != 0 ) {
		hwMsg( 1, "error releasing AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	return 1;
}


/*
 * Clean up the agpgart module - direct clients must call this upon exit.
 *
 * Unmap the agpgart device and close the device filehandle.  Currently, any
 * memory allocated must be explicitly deallocated and unbound using
 * hwFreeAGPMem(), but this may change.
 */
int hwReleaseAGPMem( void )
{
	hwMsg( 10, "ReleaseAGPMem: unmapping AGP aperture\n" );

	/* ensure the device has been opened */
	if ( gartfd == - 1 ) {
		hwMsg( 1, "ReleaseAGPMem: " GLX_AGPDEVICE " not opened\n" );
		return -1;
	}

	/* acquire the module */
	if ( ioctl( gartfd, AGPIOC_ACQUIRE ) != 0 ) {
		hwMsg( 1, "error acquiring AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	/* get the current module info */
	if ( ioctl( gartfd, AGPIOC_INFO, &info ) != 0 ) {
		hwMsg( 1, "error doing AGP info ioctl: %s\n", sys_errlist[errno] );
		return -1;
	}

	/* unmap the device */
	if ( munmap( gartbuf, info.aper_size * 0x100000 ) != 0 ) {
		hwMsg( 1, "munmap() on " GLX_AGPDEVICE " failed: %s\n", sys_errlist[errno] );
		close( gartfd );
		return -1;
	}

	/* release the module */
	if ( ioctl( gartfd, AGPIOC_RELEASE ) != 0 ) {
		hwMsg( 1, "error releasing AGP module: %s\n", sys_errlist[errno] );
		return -1;
	}

	/* close the device */
	if ( close( gartfd ) != 0 ) {
		hwMsg( 1, "unable to close " GLX_AGPDEVICE ": %s\n", sys_errlist[errno] );
		return -1;
	}

	/* dump info about the agp heap */
	if ( hwGetLogLevel() >= 10 ) {
		mmDumpMemInfo( heap );
	}

	return 1;
}


/*
 * Display information about the AGP heap, including all blocks of memory
 * currently allocated and bound.
 */
void hwDumpAGPMemInfo( void )
{
	mmDumpMemInfo( heap );
}


#else /* HAVE_LINUX_NEWAGP */

/* dummy functions if agpgart isn't available */

int hwInitAGPMem( int mode, int limit )
{
	hwMsg( 1, "InitAGP: AGP support not available\n" );
	return -1;
}

int hwGetAGPInfo( void *info )
{
	hwMsg( 1, "GetAGPInfo: AGP support not available\n" );
	return 0;
}

int hwAllocAGPMem( agp_mem_t *b, size_t size )
{
	hwMsg( 1, "AllocAGPMem: AGP support not available\n" );
	return 1;
}

int hwFreeAGPMem( agp_mem_t *b )
{
	hwMsg( 1, "FreeAGPMem: AGP support not available\n" );
	return 1;
}

int hwReleaseAGPMem( void )
{
	hwMsg( 1, "ReleaseAGPMem: AGP support not available\n" );
	return 1;
}

void hwDumpAGPMemInfo( void )
{
	hwMsg( 1, "DumpAGPMemInfo: AGP support not available\n" );
}

#endif /* HAVE_LINUX_NEWAGP */


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
