/*
 * GLX Hardware Device Driver for Intel 810
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on matrox driver by
 *    Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 */

/* $Id: i810buf.c,v 1.3 2000/01/28 05:00:44 keithw Exp $ */
#include <stdlib.h>
#include "i810lib.h"
#include "simple_list.h"
#include "i810_3d_reg.h"


struct { int pitch; CARD32 bits; } i810_dest_pitch[] = {
   { 512, 0x0 },
   { 1024, 0x1 },
   { 2048, 0x2 },
   { 4096, 0x3 }
};

struct i810_dest_buffer *i810CreateDestBuffer(int Format, int Width, 
					      int Height)
{
   struct i810_dest_buffer *buf;
   int ofs, i;
   GLuint pitch_bits;

   buf = CALLOC_STRUCT(i810_dest_buffer);
   if (!buf) return 0;

   buf->Format = Format;

   switch (Format) {
   case DV_PF_INDEX: 
      buf->BytesPerPixel = 1;
      FatalError("wrong");
      break;
   case DV_PF_565:
      buf->BytesPerPixel = 2; 
      break;
   case DV_PF_555:
      buf->BytesPerPixel = 2; 
      FatalError("wrong");
      break;
   default:
      FatalError("wrong");
      return 0;
   }

   for (i = 0 ; i < Elements(i810_dest_pitch) ; i++) 
      if (Width * buf->BytesPerPixel < i810_dest_pitch[i].pitch) {
	 pitch_bits = i810_dest_pitch[i].bits;
	 buf->Pitch = i810_dest_pitch[i].pitch;
	 break;
      }

   if (i == Elements(i810_dest_pitch)) {
      free(buf);
      return 0;
   }

   buf->Height = Height;
   buf->Width = Width;
   buf->Drawable = 1;		/* wtf */

   buf->MemBlock = mmAllocMem( i810glx.sysmemHeap, 
			       buf->Pitch*buf->Height,
			       12, 0);

   if (!buf->MemBlock) {
      free(buf);
      return 0;
   }

   ofs = buf->MemBlock->ofs;

   buf->Setup[I810_DESTREG_DI0] = CMD_OP_DESTBUFFER_INFO;
   buf->Setup[I810_DESTREG_DI1] = (ofs & DB1_BASE_ADDR_MASK) | pitch_bits;
   buf->Setup[I810_DESTREG_DV0] = GFX_OP_DESTBUFFER_VARS;
   buf->Setup[I810_DESTREG_DV1] = (DV_HORG_BIAS_OGL |
				   DV_VORG_BIAS_OGL |
				   Format);

   buf->Setup[I810_DESTREG_DR0] = GFX_OP_DRAWRECT_INFO;
   buf->Setup[I810_DESTREG_DR1] = ( 0 
				    | DR1_RECT_CLIP_ENABLE
				    ); 
   buf->Setup[I810_DESTREG_DR2] = 0;
   buf->Setup[I810_DESTREG_DR3] = ( 0 
				    |  ((Height-1)<<16)|(Width-1)
                                  ); 
   buf->Setup[I810_DESTREG_DR4] = 0;

   buf->Setup[I810_DESTREG_ZB0] = 0;
   buf->Setup[I810_DESTREG_ZB1] = 0;


   buf->BufAddr = GLXSYM(vgaLinearBase) + ofs;
/*     mmMarkReserved(buf->MemBlock); */
   return buf;
}


void i810DestroyDestBuffer(struct i810_dest_buffer *buf)
{
   if (buf->ZBuffer) {
      i810FreeZBuffer( buf->ZBuffer );
      buf->ZBuffer = 0;
   }

   if (mmFreeMem(buf->MemBlock) == -1) {
      ErrorF("Could not free dest buffer %08x\n",buf->MemBlock->ofs);
      FatalError("memory problem\n");
   }

   free(buf);
}



/* Create and attach a zbuffer to the parent buffer.  Put setup instructions
 * into the parent->Setup field.
 */
struct i810_z_buffer *i810CreateZBuffer(struct i810_dest_buffer *parent)
{
   struct i810_z_buffer *zb;
   int bytesPerPixel = 2;
   int i, pitch_bits, size;

   /* Catch this earlier */
   if (parent->ZBuffer) {
      i810FreeZBuffer(parent->ZBuffer);
      parent->ZBuffer = 0;
   }

   zb = CALLOC_STRUCT(i810_z_buffer);
   if (!zb)
      return 0;

   for (i = 0 ; i < Elements(i810_dest_pitch) ; i++) 
      if (parent->Width * bytesPerPixel < i810_dest_pitch[i].pitch) {
	 pitch_bits = i810_dest_pitch[i].bits;
	 zb->Pitch = i810_dest_pitch[i].pitch;
	 break;
      }

   if (i == Elements(i810_dest_pitch)) {
      free(zb);
      return 0;
   }

   size = zb->Pitch * parent->Height;


   /* Try to allocate it in dcache if possible, else use system memory.
    */
   if ( (zb->MemBlock = mmAllocMem( i810glx.cardHeap, size, 12, 0 )) != 0 ) {
      fprintf(stderr, "Allocated Z buffer in dcache memory\n");
   } else {
      zb->MemBlock = mmAllocMem( i810glx.sysmemHeap, size, 12, 0 );
      if (!zb->MemBlock) {
	 free(zb);
	 return 0;
      }
      fprintf(stderr, "Allocated Z buffer in system memory\n");
   }

   zb->BufAddr = i810glx.sysmemVirtual + zb->MemBlock->ofs;

   parent->Setup[I810_DESTREG_ZB0] = CMD_OP_Z_BUFFER_INFO;
   parent->Setup[I810_DESTREG_ZB1] = (((zb->MemBlock->ofs)&ZB_BASE_ADDR_MASK) |
				      pitch_bits);

   
   if (parent == i810DB)
     i810DmaExecute( i810DB->Setup, I810_DEST_SETUP_SIZE );

   parent->ZBuffer = zb;

   return zb;
}

void i810FreeZBuffer(struct i810_z_buffer *buf)
{
   if (mmFreeMem(buf->MemBlock) == -1) {
      ErrorF("Could not free z buffer %08x\n",buf->MemBlock->ofs);
      FatalError("memory problem\n");
   }

   free(buf);
}



