/*
 * GLX Hardware Device Driver for Intel i810
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

/* $Id: i810dd.c,v 1.6 2000/02/29 10:40:30 giles Exp $ */

#include <stdlib.h>
#include <stdio.h>

#include "xsmesaP.h"
#include "mesaglx/types.h"
#include "vbrender.h"

#include "glx_log.h"

#include "vga.h"
#include "vgaPCI.h"

#include "mm.h"
#include "i810lib.h"
#include "i810glx.h"
#include "i810clear.h"
#include "i810dd.h"
#include "i810direct.h"
#include "i810log.h"
#include "i810state.h"
#include "i810tex.h"
#include "i810tris.h"
#include "i810vb.h"
#include "extensions.h"
#include "vb.h"
#include "dd.h"
#include "hw_buffer.h"

extern int I810ydstorg;
extern int I810maccess;
extern void xsmesa_setup_DD_pointers( GLcontext *ctx );


/***************************************
 * Mesa's Driver Functions
 ***************************************/


const GLubyte *i810DDGetString( GLcontext *ctx, GLenum name )
{
   switch (name) {
   case GL_VENDOR:
      return "Keith Whitwell, Precision Insight Inc.";
   case GL_RENDERER:
      return "GLX-I810";
   default:
      return 0;
   }
}


void i810DDExtensionsInit( GLcontext *ctx )
{
   /* CVA only available for direct contexts.
    */
   if (__glx_is_server) 
      gl_extensions_disable( ctx, "GL_EXT_compiled_vertex_array" );

   /* paletted_textures currently doesn't work.
    */
   gl_extensions_disable( ctx, "GL_EXT_shared_texture_palette" );
   gl_extensions_disable( ctx, "GL_EXT_paletted_texture" );

   /* we don't support point parameters in hardware.
    */
   gl_extensions_disable( ctx, "GL_EXT_point_parameters" );

   /* The imaging subset of 1.2 isn't supported by any mesa driver.
    */
   gl_extensions_disable( ctx, "ARB_imaging" );
   gl_extensions_disable( ctx, "GL_EXT_blend_minmax" );
   gl_extensions_disable( ctx, "GL_EXT_blend_logic_op" );
   gl_extensions_disable( ctx, "GL_EXT_blend_subtract" );
   gl_extensions_disable( ctx, "GL_INGR_blend_func_separate" );


   /* We do support tex_env_add, however
    */
   gl_extensions_enable( ctx, "GL_EXT_texture_env_add" );
}


static GLint i810GetParameteri(const GLcontext *ctx, GLint param)
{
   switch (param) {
   case DD_HAVE_HARDWARE_FOG:  
      return 1;
   default:
      return 0; 
   }
}


GLboolean i810IsTextureResident( GLcontext *ctx, struct gl_texture_object *t )
{
   i810TextureObjectPtr mt = (i810TextureObjectPtr)t->DriverData;
   return mt && mt->MemBlock;
}


static int i810CanUseHardware(const GLcontext *ctx)
{
   XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
   XSMesaBuffer xsmbuf = xsmesa->xsm_buffer;
   CHECK_CONTEXT( return 0; );

   if ( !i810DB || !xsmbuf->buffer == XIMAGE ) {
      return 0;
   }
  
   return 1;
}


/*
 * if we can't hardware accelerate the context now, but could
 * before (the window was sized up to the point a buffer alloc failed)
 * then just have software mesa do everything.
 */
void i810_setup_DD_pointers_no_accel( GLcontext *ctx ) {
    /* disable hardware accel. */
    i810Msg(5,"no hw accel.\n");

    ctx->Driver.RenderStart  = NULL; 
    ctx->Driver.RenderFinish = NULL; 
    ctx->Driver.NearFar = NULL;
    ctx->Driver.GetParameteri = NULL;
    ctx->Driver.RasterSetup = NULL;    
    ctx->Driver.TexEnv = NULL;
    ctx->Driver.TexImage = NULL;
    ctx->Driver.BindTexture = NULL;
    ctx->Driver.DeleteTexture = NULL;
    ctx->Driver.TexParameter = NULL;
    
    xsmesa_setup_DD_pointers(ctx);
    ctx->Driver.UpdateState = i810_setup_DD_pointers;
}



void i810_setup_DD_pointers( GLcontext *ctx ) {

   if ( !i810CanUseHardware(ctx) ) {
      i810_setup_DD_pointers_no_accel( ctx );
      return;
   }

   xsmesa_setup_DD_pointers(ctx);

   i810DDInitStatePointers(ctx);
   
   ctx->Driver.GetString = i810DDGetString;
   ctx->Driver.UpdateState = i810DDUpdateState;
   ctx->Driver.RegisterVB = i810DDRegisterVB;
   ctx->Driver.UnregisterVB = i810DDUnregisterVB;
   ctx->Driver.Clear = i810Clear;
   ctx->Driver.GetParameteri = i810GetParameteri;
   ctx->Driver.TexEnv = i810TexEnv;
   ctx->Driver.TexImage = i810TexImage;
   ctx->Driver.TexSubImage = i810TexSubImage;
   ctx->Driver.BindTexture = i810BindTexture;
   ctx->Driver.DeleteTexture = i810DeleteTexture;
   ctx->Driver.TexParameter = i810TexParameter;
   ctx->Driver.UpdateTexturePalette = i810UpdateTexturePalette;
   ctx->Driver.IsTextureResident = i810IsTextureResident;

   ctx->Driver.TriangleCaps = (DD_TRI_CULL|
			       DD_TRI_LIGHT_TWOSIDE|
			       DD_TRI_OFFSET);

	SetDriverBufferFunctions( ctx, (void(*)(void))i810WaitDrawingEngine,
		i810DB->BufAddr, i810DB->Pitch, i810DB->Height, CB_16BIT,
		i810DB->ZBuffer ? i810DB->ZBuffer->BufAddr : NULL, i810DB->Pitch, i810DB->Height, DB_16BIT );

   /* Also do the normal GL state-change checks.
    */
   i810DDUpdateState( ctx );
}
