/* $Id: mach64direct.c,v 1.17 2000/10/03 11:42:23 svartalf Exp $ */

/*
 * GLX Hardware Device Driver for mach64
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * John Carmack <johnc@idsoftware.com>
 * based on mach64direct.c by Keith Whitwell <keithw@precisioninsight.com>
 */

#include <stdlib.h>
#include <signal.h>

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "mach64glx.h"

#include "glx_clients.h"
#include "glx_symbols.h"
#include "glx_init.h"

#include "xf86_OSproc.h"

#include "hw_agp.h"


/*

To perform direct rendering, we need to get all configuration information
from the server and mmap the framebuffer and dma buffers,

Direct rendering will only be allowed if there are no other contexts active.

Because it is only possible on the same machine, we don't have to worry about
byte swapping any of our communication structures, we can just copy them
into message buffers.



shouldn't we have a glxReleaseDirect instead of void (**release_hook)( void )?

*/


/*
 */
#define GLX_START                X_GLXDirectPrivateStart
// X_GLXDirectGoDirect:
// X_GLXDirectRelease:
#define X_GLXDirectSwapBuffers   (GLX_START+2)
#define X_GLXDirectDmaFlush      (GLX_START+3)


extern int __glXNumClients();


static int (*send_vendor_private)( int opcode,
				   char *vp,
				   size_t vp_sz,
				   xReply *reply,
				   char *data,
				   size_t data_sz ) = 0;


/* =======================================================================
 * Initialization
 */


/* Nice thing about direct rendering is that the server *is* the
 * client, in the sense that they should both be the same executable
 * image 'glx.so'.  This means that most of the protocol structs can
 * be private to this file.
 */
typedef struct {
	char		init_fn[20];	/* The first element is always the
					 * name of the function for the client
					 * to call to initialize glx.so.  */

	mach64Glx_t	mach64glx;
	int		glx_first_visual;
} mach64DirectHWInfo;


static void mach64ClientReleaseHardware( void )
{
	if ( mach64glx.linearBase == (pointer) -1 ) {
 	  	return;
  	}
#ifndef __PPC__
	xf86UnMapVidMem( 0,
			 LINEAR_REGION,
			 (pointer) mach64glx.linearBase,
			 mach64glx.videoRam*1024 );
#else
	// fbdev server gets video memory from /dev/fb
#endif

	if ( mach64glx.dmaDriver >= 3 ) {
		hwReleaseAGPMem();
	}

	mach64glx.linearBase = (pointer) -1;
}



ClientPtr mach64GLXClientInit( mach64DirectHWInfo *hw,
			       void *display,
			       int screen,
			       int (*send_func)(),
			       void (**release_hook)( void ) )
{
	if ( ! glxInstallLocalMach64Symbols() ) {
		return 0;
	}

	// this is how we send messages to the X server
	send_vendor_private = send_func;

	// this is all the configuration info from the server,
	// including screen res, dma buffer location, etc
	mach64glx = hw->mach64glx;

	// clean up pointers in the struct
	mach64glx.logFile = NULL;

	// memory map the framebuffer
#ifndef __PPC__
   	mach64glx.linearBase = xf86MapVidMem( 0 /* screenNum */,
					      LINEAR_REGION,
					      (pointer) mach64glx.linearPhysical,
					      mach64glx.videoRam*1024 );
#else
	// fbdev server gets video memory from /dev/fb
#endif

	if ( mach64glx.linearBase == (pointer) -1 ) {
		hwError( "failed to map vga linear region" );
		mach64ClientReleaseHardware();
		return 0;
	}

	// memory map the dma memory

	// GLX will call this when the client drops.
	// I think this should be a separate entry point...
	*release_hook = mach64ClientReleaseHardware;

        return glxInitDirectClient( display, screen, hw->glx_first_visual,
				    XF_UNKNOWN, ATI_RAGEPRO );
}

static int mach64GLXGoDirect( ClientPtr client )
{
	xGLXGetStringReply	reply;
	mach64DirectHWInfo	*hw;
	int			l;

	/* Allow only one direct client, which must be the only active
	 * client, and must be on the local server.
	 */
	if ( direct_client || !__glx_is_server || 
	        __glXNumClients() != 1 || !LocalClient( client ) ) {
		return BadAccess;
	}

	if ( mach64glx.dmaDriver < 1 ) {
		hwError( "Direct clients only allowed with real dma" );
		return BadMatch;
	}

	mach64DmaFlush();

	direct_client = client;
	hw = (mach64DirectHWInfo *)malloc( sizeof(*hw) );

	/* Name of function to call to revive the struct.  Client will
	 * receive buf in reply to its go_direct request.  Client will map
	 * glx.so, and use buf as a string argument to dlsym(), to retreive
	 * a function to call, namely mach64GLXClientInit.  It will then call
	 * that function passing hw as its argument.
	 *
	 * hw should contain all the information necessary to get glx.so
	 * talking to hardware from the client.
	 */
	strcpy( hw->init_fn, "mach64GLXClientInit" );
	hw->mach64glx = mach64glx;

	hw->glx_first_visual = __glx_first_visual;

	reply.type = X_Reply;
	reply.sequenceNumber = client->sequence;
	reply.length = 0;
	reply.n = l = (sizeof(*hw)+3)/4;

	WriteToClient( client, sizeof(xGLXGetStringReply), (char*)&reply );
	WriteToClient( client, sizeof(int)*l, (char*)hw );

	return client->noClientException;
}



static int mach64GLXReleaseDirect( ClientPtr client )
{
	if ( __glx_is_server && direct_client && direct_client == client ) {
		direct_client = 0;
		return Success;
	}

	return BadAccess;
}




/* =======================================================================
 * Dma flush
 */
typedef struct  {
	CARD32		dma_bufferDwords;
	CARD32		dma_buffer_id;		/* validation only */
	CARD32		dma_sync;
} DmaFlushReq;


typedef struct {
	BYTE		type;
	CARD8		xpad;
	CARD16		sequenceNumber;
	CARD32		length;
	CARD32		dma_buffer_id;		/* new active id */
} DmaFlushReply;



static int mach64GLXDirectDMAFlush( ClientPtr client, xGLXVendorPrivateReq *stuff )
{
	DmaFlushReq	*vp = (DmaFlushReq *)(stuff + 1);
	xReply		reply;
	DmaFlushReply	*rep = (DmaFlushReply *)&reply;
	extern void mach64ServerDmaFlush( int );


	if ( client != direct_client ) {
		return BadAccess;
	}

	if ( mach64glx.activeDmaBuffer != vp->dma_buffer_id ) {
		hwError( "someone's been playing with dma on the server\n" );
		return BadImplementation;
	}

	mach64glx.dma_buffer->bufferDwords = vp->dma_bufferDwords;

	mach64ServerDmaFlush( vp->dma_sync );

	rep->type = X_Reply;
	rep->length = 0;
	rep->sequenceNumber = client->sequence;
	rep->dma_buffer_id = mach64glx.activeDmaBuffer;

	WriteToClient( client, sizeof(xReply), (char *) &reply );

	return client->noClientException;
}


/*
 * mach64ClientDMAFlush
 * When direct rendering, this is called instead of starting
 * the dma from user space.  This only happens for overflows,
 * explicit flush/finish, and software rendering.  The swapbuffers
 * interface handles flushing in normal cases.
 */
void mach64DirectClientDMAFlush( int wait )
{
	DmaFlushReq	vp;
	xReply		reply;

	vp.dma_buffer_id = mach64glx.activeDmaBuffer;
	vp.dma_bufferDwords = mach64glx.dma_buffer->bufferDwords;
	vp.dma_sync = wait;

	if ( send_vendor_private( X_GLXDirectDmaFlush,
				  (char *)&vp, sizeof(vp),
				  &reply, 0, 0 ) )
	{
		DmaFlushReply	*rep = (DmaFlushReply *)&reply;
		mach64glx.activeDmaBuffer = rep->dma_buffer_id;
	}

	mach64DmaResetBuffer();
}

/* ===========================================================================
 * Buffer swap
 *   - Combines dma flush and swapbuffers
 */

/* Need to issue the swap commands from the server as the client
 * doesn't have the clip rects.  This isn't so bad as we would want to
 * do a dma flush immediately afterwards anyhow.
 */
typedef struct {
	mach64Buffer	backBuffer;
	TMemBlock	backBufferBlock;
	CARD32		front_drawable;
	CARD32		dma_bufferDwords;
	CARD32		dma_buffer_id;		/* validation only */
	CARD32		attrib;
	CARD32		flag;
} SwapBufferReq;

typedef struct {
	BYTE		type;
	CARD8		xpad;
	CARD16		sequenceNumber;
	CARD32		length;
	CARD32		dma_buffer_id;		/* new active id */
	CARD16		width;
	CARD16		height;
	CARD16		hardwareWentIdle;
} SwapBufferReply;


/*
 * this happens on the X server as a result of a direct rendering
 * client attempting to swapbuffers
 */
static int mach64GLXDirectSwapBuffers( ClientPtr client,
				       xGLXVendorPrivateReq *stuff )
{
	SwapBufferReq		*vp = (SwapBufferReq *)(stuff+1);
	xReply			reply;
	SwapBufferReply		*rep = (SwapBufferReply *)&reply;
	DrawablePtr		frontbuf;

	if ( client != direct_client ) {
		return BadAccess;
	}

	if ( mach64glx.activeDmaBuffer != vp->dma_buffer_id ) {
		fprintf( stderr, "somebody's been playing with dma on the server %d %ld\n",
			 mach64glx.activeDmaBuffer, vp->dma_buffer_id );
		return BadImplementation;
	}

	mach64glx.dma_buffer->bufferDwords = vp->dma_bufferDwords;

	// find the window we are going to blit to
	frontbuf = LookupIDByClass( vp->front_drawable, RC_DRAWABLE );

	if ( !frontbuf ) {
		return GLXBadDrawable;
	}

	mach64DB = &vp->backBuffer;
	mach64DB->backBufferBlock = &vp->backBufferBlock;

	/* hardware accelerated back to front blit */
	mach64BackToFront( frontbuf, &vp->backBuffer );

	/* make sure all dma is going to get executed */
	/* if everything has gone well, this will be the only
	   flush each frame */
	mach64DmaFlush();

	rep->type = X_Reply;
	rep->length = 0;
	rep->sequenceNumber = client->sequence;
	rep->dma_buffer_id = mach64glx.activeDmaBuffer;
	rep->width = frontbuf->width;
	rep->height = frontbuf->height;
	rep->hardwareWentIdle = mach64glx.hardwareWentIdle;

	WriteToClient( client, sizeof(xReply), (char *) &reply );

	return client->noClientException;
}

/*
 * This will be called instead of performing the buffer swap
 * if we are a direct rendering client.
 */
void mach64DirectClientSwapBuffers( XSMesaBuffer b )
{
	SwapBufferReq		vp;
	xReply			reply;
	mach64BufferPtr		buf;

	if ( !b->db_state ) {
		hwMsg( 10, "client swap buffers: only single buffered!\n" );
	}

	if ( !b->backimage || !(buf = (mach64BufferPtr)b->backimage->devPriv) ) {
		fprintf( stderr, "client swap buffers: wtf???\n" );
		return;
	}
#if 0
	/* FIXME: double buffering of single-buffered apps breaks this */
	if ( mach64Ctx && mach64Ctx->gl_ctx && b->db_state /* surely this is broken... */ ) {
		FLUSH_VB( mach64Ctx->gl_ctx, "mach64 client swap buffers" );
	}
#endif
	mach64glx.swapBuffersCount++;

	vp.front_drawable = b->frontbuffer->id;
	vp.backBuffer = *buf;
	vp.backBufferBlock = *buf->backBufferBlock;

	vp.dma_buffer_id = mach64glx.activeDmaBuffer;
	vp.dma_bufferDwords = mach64glx.dma_buffer->bufferDwords;
	vp.flag = 0;

	if ( send_vendor_private( X_GLXDirectSwapBuffers,
				  (char *)&vp, sizeof(vp), &reply, 0, 0 ) )
	{
		SwapBufferReply		*rep = (SwapBufferReply *)&reply;

		b->frontbuffer->width = rep->width;
		b->frontbuffer->height = rep->height;

		mach64glx.activeDmaBuffer = rep->dma_buffer_id;
		mach64glx.hardwareWentIdle = rep->hardwareWentIdle;
		mach64DmaResetBuffer();
	} else {
		FatalError( "clientSwapBuffers failed" );
	}
}




/* =======================================================================
 * GLX procs
 */


int mach64GLXVendorPrivate( ClientPtr client,
			    XSMesaContext ctx,
			    xGLXVendorPrivateReq *stuff )
{
	if ( !__glx_is_server ) {
		return GLXUnsupportedPrivateRequest;
	}

	switch ( stuff->opcode ) {
	case X_GLXDirectGoDirect:
		return mach64GLXGoDirect( client );
	case X_GLXDirectRelease:
		return mach64GLXReleaseDirect( client );
	case X_GLXDirectSwapBuffers:
		return mach64GLXDirectSwapBuffers( client, stuff );
	case X_GLXDirectDmaFlush:
		return mach64GLXDirectDMAFlush( client, stuff );
	default:
		hwError( "not-handled case in mach64GLXVendorPrivate" );
		return GLXUnsupportedPrivateRequest;
	}
}


int mach64GLXAllowDirect( ClientPtr client )
{
	if ( mach64glx.dmaDriver < 1 ) {
		hwError("Don't allow direct clients with pseudo-dma");
		return 0;
	}

	return 1;
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
