/* $Id: mach64pixel.c,v 1.3 2000/03/27 07:27:03 gareth Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *    Gareth Hughes <gareth@precisioninsight.com>
 */


#include <stdlib.h>

#include "context.h"
#include "macros.h"
#include "types.h"

#include "glx_log.h"
#include "glx_symbols.h"

#include "mach64glx.h"


/*
 * mach64ConvertPixels
 * Converts a block of pixels to the apropriate hardware format via a host
 * data blit.
 */
static void mach64ConvertPixels( const GLvoid *pixels, GLenum format, int pixelBytes,
				 int x, int y, int width, int height )
{
	int		i, j;
	hwUI8		*src;
	int		dwords;
	DMALOCALS;

	/* account for the actual space needed by the hostdata stream */
	/* this must be the same as in the check for recursive subdivision */
	dwords =  16 * ( ( width * height * pixelBytes + 59 ) / 60 );

	/* this should never overflow, because then the blit would get */
	/* the context register restore data instead of pixels... */
	DMAGETPTR( dwords );

	switch ( pixelBytes ) {
	case 1:
		switch ( format ) {
		case GL_COLOR_INDEX:
		case GL_INTENSITY:
		case GL_LUMINANCE:
		case GL_ALPHA:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x );
				for ( j = width >> 2 ; j ; j-- ) {
					int	pix;

					pix = src[0] | ( src[1] << 8 ) | ( src[2] << 16 ) | ( src[3] << 24 );
					DMAOUTHOSTDATA( pix );
					src += 4;
				}
			}
			break;
		default:
			goto format_error;
		}
		break;
	case 2:
		switch ( format ) {
		case GL_RGB:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x ) * 3;
				for ( j = width >> 1 ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR565( src[0], src[1], src[2] ) |
					    ( MACH64PACKCOLOR565( src[3], src[4], src[5] ) << 16 );
					DMAOUTHOSTDATA( pix );
					src += 6;
				}
			}
			break;
		case GL_RGBA:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x ) * 4;
				for ( j = width >> 1 ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR4444( src[0], src[1], src[2], src[3] ) |
					    ( MACH64PACKCOLOR4444( src[4], src[5], src[6], src[7] ) << 16 );
					DMAOUTHOSTDATA( pix );
					src += 8;
				}
			}
			break;
		case GL_LUMINANCE:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x );
				for ( j = width >> 1 ; j ; j-- ) {
					int	pix;
					/* FIXME: should probably use 555 texture to get true grey */
					pix = MACH64PACKCOLOR565( src[0], src[0], src[0] ) |
					    ( MACH64PACKCOLOR565( src[1], src[1], src[1] ) << 16 );
					DMAOUTHOSTDATA( pix );
					src += 2;
				}
			}
			break;
		case GL_INTENSITY:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x );
				for ( j = width >> 1 ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR4444( src[0], src[0], src[0], src[0] ) |
					    ( MACH64PACKCOLOR4444( src[1], src[1], src[1], src[1] ) << 16 );
					DMAOUTHOSTDATA( pix );
					src += 2;
				}
			}
			break;
		case GL_ALPHA:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x );
				for ( j = width >> 1 ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR4444( 255, 255, 255, src[0] ) |
					    ( MACH64PACKCOLOR4444( 255, 255, 255, src[1] ) << 16 );
					DMAOUTHOSTDATA( pix );
					src += 2;
				}
			}
			break;
		case GL_LUMINANCE_ALPHA:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x ) * 2;
				for ( j = width >> 1 ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR4444( src[0], src[0], src[0], src[1] ) |
					    ( MACH64PACKCOLOR4444( src[2], src[2], src[2], src[3] ) << 16 );
					DMAOUTHOSTDATA( pix );
					src += 4;
				}
			}
			break;
		default:
			goto format_error;
		}
		break;
	case 4:
		switch ( format ) {
		case GL_RGB:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x ) * 3;
				for ( j = width ; j ; j-- ) {
					int	pix;

					pix = ( 255 << 24 ) | MACH64PACKCOLOR888( src[0], src[1], src[2] );
					DMAOUTHOSTDATA( pix );
					src += 3;
				}
			}
			break;
		case GL_RGBA:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x ) * 4;
				for ( j = width  ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR8888( src[0], src[1], src[2], src[3] );
					DMAOUTHOSTDATA( pix );
					src += 4;
				}
			}
			break;
		case GL_LUMINANCE:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x );
				for ( j = width ; j ; j-- ) {
					int	pix;

					pix = ( 255 << 24 ) | MACH64PACKCOLOR888( src[0], src[0], src[0] );
					DMAOUTHOSTDATA( pix );
					src += 1;
				}
			}
		case GL_INTENSITY:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x );
				for ( j = width ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR8888( src[0], src[0], src[0], src[0] );
					DMAOUTHOSTDATA( pix );
					src += 1;
				}
			}
			break;
		case GL_ALPHA:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x );
				for ( j = width ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR8888( 255, 255, 255, src[0] );
					DMAOUTHOSTDATA( pix );
					src += 1;
				}
			}
			break;
		case GL_LUMINANCE_ALPHA:
			for ( i = 0 ; i < height ; i++ ) {
				src = (hwUI8 *)pixels + ( ( y + i ) * width + x ) * 2;
				for ( j = width ; j ; j-- ) {
					int	pix;

					pix = MACH64PACKCOLOR8888( src[0], src[0], src[0], src[1] );
					DMAOUTHOSTDATA( pix );
					src += 2;
				}
			}
		default:
			goto format_error;
		}
		break;
	default:
		goto format_error;
	}

	DMAHOSTDATAEND();
	DMAADVANCE();

	return;

 format_error:
	hwError( "Unsupported pixelBytes %i, format %i\n", pixelBytes, format );
}


/*
 * mach64UploadPixels
 * Set up the host data blit for the pixel upload.
 */
static void mach64UploadPixels( const GLvoid *pixels, GLenum format,
				GLint x, GLint y, GLsizei width, GLsizei height )
{
	int		x2;
	int		dwords;
	int		pixelFormat, pixelBytes, pixelsPerDword;
	int		imageWidth, imageHeight;
	int		blitX, blitY;
	DMALOCALS;

	switch( mach64glx.depth ) {
	case 15: pixelFormat = 3; pixelBytes = 2; pixelsPerDword = 2; break;
	case 16: pixelFormat = 4; pixelBytes = 2; pixelsPerDword = 2; break;
	case 32: pixelFormat = 6; pixelBytes = 4; pixelsPerDword = 1; break;
	}

	imageWidth = mach64DB->pitch;
	imageHeight = mach64DB->height;

	/* pad the size out to dwords.  The image is a pointer to the entire image,
	   so we can safely reference outside the x,y,width,height bounds if we need to */
	x2 = x + width;
	x2 = ( x2 + ( pixelsPerDword - 1 ) ) & ~(pixelsPerDword-1);

	x = ( x + ( pixelsPerDword - 1 ) ) & ~(pixelsPerDword-1);
	width = x2 - x;

	/* we may not be able to upload the entire texture in one batch due
	   to register limits or dma buffer limits.  Recursively split it up. */
	while ( 1 ) {
		/* account for the actual space needed by the hostdata stream */
		/* this must be the same as in the DMAGETPTR when uploading */
		dwords =  16 * ( ( width * height * pixelBytes + 59 ) / 60 );

		if ( dwords + 32 <= mach64glx.dma_buffer->overflowBufferDwords ) {
			break;
		}
		hwMsg( 10, "mach64UploadLocalSubImage: recursively subdividing\n" );

		mach64UploadPixels( pixels, format, x, y, width, height >> 1 );

		/* update actual and blit dimensions */
		y += ( height >> 1 );
		height -= ( height >> 1 );
	}

	/* transform window coords to blit coords */
	blitX = x;
	blitY = mach64DB->height - y - 1;

	/* make sure we overflow here instead of in convertTexture */
	DMAGETPTR( 32 + dwords );

	hwMsg( 10, "mach64UploadPixels: %i,%i of %i,%i at %i,%i\n", width, height, imageWidth, imageHeight, x, y );

	/* blit via the host data registers */

	/* do we really need all this? */
	DMAOUTREG( MACH64_Z_CNTL, 0 );
	DMAOUTREG( MACH64_SCALE_3D_CNTL, 0 );

	DMAOUTREG( MACH64_SC_LEFT_RIGHT, 0 | ( 8191 << 16 ) );	/* no scissor */
	DMAOUTREG( MACH64_SC_TOP_BOTTOM, 0 | ( 16383 << 16 )  );

	DMAOUTREG( MACH64_CLR_CMP_CNTL, 0 );			/* disable */
	DMAOUTREG( MACH64_GUI_TRAJ_CNTL, DST_X_LEFT_TO_RIGHT | DST_Y_BOTTOM_TO_TOP );

	DMAOUTREG( MACH64_DP_PIX_WIDTH,
		   ( pixelFormat << 0 )				/* dst pix width */
		   | ( pixelFormat << 4 )			/* composite pix width */
		   | ( pixelFormat << 8 )			/* src pix width */
		   | ( pixelFormat << 16 )			/* host data pix width */
		   | ( pixelFormat << 28 )			/* scaler/3D pix width */
		   );

	DMAOUTREG( MACH64_DP_WRITE_MASK, 0xffffffff );		/* enable all planes */
	DMAOUTREG( MACH64_DP_MIX, BKGD_MIX_D | FRGD_MIX_S );
	DMAOUTREG( MACH64_DP_SRC, BKGD_SRC_BKGD_CLR | FRGD_SRC_HOST | MONO_SRC_ONE );

	DMAOUTREG( MACH64_DST_OFF_PITCH, ( (mach64DB->pitch/8) << 22 ) | (mach64DB->backBufferBlock->ofs>>3) ) ;
	DMAOUTREG( MACH64_DST_X_Y, (blitY << 16) | blitX );
	DMAOUTREG( MACH64_DST_WIDTH_HEIGHT, (height << 16) | width );

	DMAADVANCE();

	/* blit via the host data registers the properly converted pixels from the mesa buffer */
	mach64ConvertPixels( pixels, format, pixelBytes, 0, 0, width, height );
}


/*
 * mach64DDDrawPixels
 * Mesa device driver function for hardware-accelerated glDrawPixels().
 */
GLboolean mach64DDDrawPixels( GLcontext *ctx,
			      GLint x, GLint y, GLsizei width, GLsizei height,
			      GLenum format, GLenum type,
			      const struct gl_pixelstore_attrib *unpack,
			      const GLvoid *pixels )
{
  	hwMsg( 10, "mach64DDDrawPixels:  %d / %d  %d / %d\n", x, y, width, height );

	/* cases we can't handle */
	if ( unpack->Alignment != 1 || unpack->SwapBytes || unpack->LsbFirst ) {
		hwMsg( 10, "mach64DDDrawPixels: can't handle pixelstore attribs\n" );
		return GL_FALSE;
	}

	mach64UploadPixels( pixels, format, x, y, width, height );

	return GL_TRUE;
}


/*
 * mach64DDBitmap
 * Mesa device driver function for hardware-accelerated glBitmap().
 */
GLboolean mach64DDBitmap( GLcontext *ctx,
			  GLint x, GLint y, GLsizei width, GLsizei height,
			  const struct gl_pixelstore_attrib *unpack,
			  const GLubyte *bitmap )
{
  	hwMsg( 10, "mach64DDBitmap:  %d / %d  %d / %d\n", x, y, width, height );
	hwMsg( 10, "mach64DDBitmap: not implemented yet...\n" );
	return GL_FALSE;

	/* cases we can't handle */
	if ( unpack->Alignment != 1 || unpack->SwapBytes || unpack->LsbFirst ) {
		hwMsg( 10, "mach64DDBitmap: can't handle pixelstore attribs\n" );
		return GL_FALSE;
	}

	mach64UploadPixels( bitmap, GL_COLOR_INDEX, x, y, width, height );

	return GL_TRUE;
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
