/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 2000 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * based on mach64direct.c by Keith Whitwell <keithw@precisioninsight.com>
 *    Jim Duchek <jimduchek@linuxpimps.com>
 */

#include <stdlib.h>
#include <signal.h>

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "sis6326glx.h"

#include "glx_clients.h"
#include "glx_symbols.h"
#include "glx_init.h"

#include "xf86_OSproc.h"


/*

To perform direct rendering, we need to get all configuration information
from the server and mmap the framebuffer and dma buffers,

Direct rendering will only be allowed if there are no other contexts active.

Because it is only possible on the same machine, we don't have to worry about
byte swapping any of our communication structures, we can just copy them
into message buffers.



shouldn't we have a glxReleaseDirect instead of void (**release_hook)( void )?

*/


/*
 */
#define GLX_START                X_GLXDirectPrivateStart
// X_GLXDirectGoDirect:
// X_GLXDirectRelease:
#define X_GLXDirectSwapBuffers   (GLX_START+2)
#define X_GLXDirectDmaFlush      (GLX_START+3)


extern int __glXNumClients();


static int (*send_vendor_private)( int opcode,
				   char *vp,
				   size_t vp_sz,
				   xReply *reply,
				   char *data,
				   size_t data_sz ) = 0;


/* =======================================================================
 * Initialization
 */


/* Nice thing about direct rendering is that the server *is* the
 * client, in the sense that they should both be the same executable
 * image 'glx.so'.  This means that most of the protocol structs can
 * be private to this file.
 */
typedef struct {
	char		init_fn[20];	/* The first element is always the
					 * name of the function for the client
					 * to call to initialize glx.so.  */

	sis6326Glx_t	sis6326glx;
	int		glx_first_visual;
} sis6326DirectHWInfo;


static void sis6326ClientReleaseHardware( void )
{
	if ( sis6326glx.linearBase == (pointer) -1) {
 	  	return;
  	}
	xf86UnMapVidMem( 0,
			 LINEAR_REGION,
			 (pointer) sis6326glx.linearBase,
			 sis6326glx.videoRam*1024 );

	sis6326glx.linearBase = (pointer) -1;
}



ClientPtr sis6326GLXClientInit( sis6326DirectHWInfo *hw,
			       void *display,
			       int screen,
			       int (*send_func)(),
			       void (**release_hook)( void ) )
{
	if ( ! glxInstallLocalMach64Symbols() ) {
		return 0;
	}

	// this is how we send messages to the X server
	send_vendor_private = send_func;

	// this is all the configuration info from the server,
	// including screen res, dma buffer location, etc
	sis6326glx = hw->sis6326glx;

	// clean up pointers in the struct
	sis6326glx.logFile = NULL;

	// memory map the framebuffer
   	sis6326glx.linearBase = xf86MapVidMem( 0 /* screenNum */,
					      LINEAR_REGION,
					      (pointer) sis6326glx.linearPhysical,
					      sis6326glx.videoRam*1024 );

	if ( sis6326glx.linearBase == (pointer) -1 ) {
		hwError( "failed to map vga linear region" );
		sis6326ClientReleaseHardware();
		return 0;
	}


	// GLX will call this when the client drops.
	// I think this should be a separate entry point...
	*release_hook = sis6326ClientReleaseHardware;

        return glxInitDirectClient( display, screen, hw->glx_first_visual,
				    XF_UNKNOWN, ATI_RAGEPRO );
}

static int sis6326GLXGoDirect( ClientPtr client )
{
	xGLXGetStringReply	reply;
	sis6326DirectHWInfo	*hw;
	int			l;

	/* Allow only one direct client, which must be the only active
	 * client, and must be on the local server.
	 */
	if ( direct_client || !__glx_is_server ||
	     __glXNumClients() != 1 || !LocalClient( client ) ) {
		return BadAccess;
	}


	direct_client = client;
	hw = (sis6326DirectHWInfo *)malloc( sizeof(*hw) );

	/* Name of function to call to revive the struct.  Client will
	 * receive buf in reply to its go_direct request.  Client will map
	 * glx.so, and use buf as a string argument to dlsym(), to retreive
	 * a function to call, namely sis6326GLXClientInit.  It will then call
	 * that function passing hw as its argument.
	 *
	 * hw should contain all the information necessary to get glx.so
	 * talking to hardware from the client.
	 */
	strcpy( hw->init_fn, "sis6326GLXClientInit" );
	hw->sis6326glx = sis6326glx;

	hw->glx_first_visual = __glx_first_visual;

	reply.type = X_Reply;
	reply.sequenceNumber = client->sequence;
	reply.length = 0;
	reply.n = l = (sizeof(*hw)+3)/4;

	WriteToClient( client, sizeof(xGLXGetStringReply), (char*)&reply );
	WriteToClient( client, sizeof(int)*l, (char*)hw );

	return client->noClientException;
}



static int sis6326GLXReleaseDirect( ClientPtr client )
{
	if ( __glx_is_server && direct_client && direct_client == client ) {
		direct_client = 0;
		return Success;
	}

	return BadAccess;
}




/* ===========================================================================
 * Buffer swap
 *   - Combines dma flush and swapbuffers
 */

/* Need to issue the swap commands from the server as the client
 * doesn't have the clip rects.  This isn't so bad as we would want to
 * do a dma flush immediately afterwards anyhow.
 */
typedef struct {
	sis6326Buffer	backBuffer;
	TMemBlock	backBufferBlock;
	CARD32		front_drawable;
	CARD32		attrib;
	CARD32		flag;
} SwapBufferReq;

typedef struct {
	BYTE		type;
	CARD8		xpad;
	CARD16		sequenceNumber;
	CARD32		length;
	CARD16		width;
	CARD16		height;
} SwapBufferReply;

#define FLG_TEX_SWAP 0x1

/*
 * this happens on the X server as a result of a direct rendering
 * client attempting to swapbuffers
 */
static int sis6326GLXDirectSwapBuffers( ClientPtr client,
				       xGLXVendorPrivateReq *stuff )
{
	SwapBufferReq		*vp = (SwapBufferReq *)(stuff+1);
	xReply			reply;
	SwapBufferReply		*rep = (SwapBufferReply *)&reply;
	DrawablePtr		frontbuf;

	if ( client != direct_client ) {
		return BadAccess;
	}


	// find the window we are going to blit to
	frontbuf = LookupIDByClass( vp->front_drawable, RC_DRAWABLE );

	if ( !frontbuf ) {
		return GLXBadDrawable;
	}

	sis6326DB = &vp->backBuffer;
	sis6326DB->backBufferBlock = &vp->backBufferBlock;

	/* diagnostic drawing tools */
	sis6326PerformanceBoxes( 1 );

	/* hardware accelerated back to front blit */
	sis6326BackToFront( frontbuf, &vp->backBuffer );


	rep->type = X_Reply;
	rep->length = 0;
	rep->sequenceNumber = client->sequence;
	rep->width = frontbuf->width;
	rep->height = frontbuf->height;

	WriteToClient( client, sizeof(xReply), (char *) &reply );

	return client->noClientException;
}

/*
 * This will be called instead of performing the buffer swap
 * if we are a direct rendering client.
 */
void sis6326DirectClientSwapBuffers( XSMesaBuffer b )
{
	SwapBufferReq		vp;
	xReply			reply;
	sis6326BufferPtr		buf;

	if ( !b->db_state ) {
		hwMsg( 10, "client swap buffers: only single buffered!\n" );
	}

	if ( !b->backimage || !(buf = (sis6326BufferPtr)b->backimage->devPriv) ) {
		fprintf( stderr, "client swap buffers: wtf???\n" );
		return;
	}

	/* FIXME: double buffering of single-buffered apps breaks this */
	if ( sis6326Ctx && sis6326Ctx->gl_ctx && b->db_state /* surely this is broken... */ ) {
		FLUSH_VB( sis6326Ctx->gl_ctx, "sis6326 client swap buffers" );
	}

	sis6326glx.swapBuffersCount++;

	vp.front_drawable = b->frontbuffer->id;
	vp.backBuffer = *buf;
	vp.backBufferBlock = *buf->backBufferBlock;

	vp.flag = 0;

	if ( sis6326glx.c_textureSwaps ) {
		vp.flag |= FLG_TEX_SWAP;
	}

	if ( send_vendor_private( X_GLXDirectSwapBuffers,
				  (char *)&vp, sizeof(vp), &reply, 0, 0 ) )
	{
		SwapBufferReply		*rep = (SwapBufferReply *)&reply;

		b->frontbuffer->width = rep->width;
		b->frontbuffer->height = rep->height;

	} else {
		FatalError( "clientSwapBuffers failed" );
	}
}




/* =======================================================================
 * GLX procs
 */


int sis6326GLXVendorPrivate( ClientPtr client,
			    XSMesaContext ctx,
			    xGLXVendorPrivateReq *stuff )
{
	if ( !__glx_is_server ) {
		return GLXUnsupportedPrivateRequest;
	}

	switch ( stuff->opcode ) {
	case X_GLXDirectGoDirect:
		return sis6326GLXGoDirect( client );
	case X_GLXDirectRelease:
		return sis6326GLXReleaseDirect( client );
	case X_GLXDirectSwapBuffers:
		return sis6326GLXDirectSwapBuffers( client, stuff );
	default:
		hwError( "not-handled case in sis6326GLXVendorPrivate" );
		return GLXUnsupportedPrivateRequest;
	}
}


int sis6326GLXAllowDirect( ClientPtr client )
{

	return 1;
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
