/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64glx.h
 *
 *    Jim Duchek <jim@linuxpimps.com>
 */

#ifndef __SIS6326_GLX_H__
#define __SIS6326_GLX_H__

#include <stdio.h>
#include "compiler.h"
#include "types.h"		/* MESA types */
#ifndef MESA31
#error "The SiS 6326 driver now requires Mesa 3.1 or higher"
#endif

#include "xsmesaP.h"	/* GLX header */
extern void xsmesa_setup_DD_pointers( GLcontext *ctx );

#include "hwlog.h"
#include "hwtypes.h"
#include "mm.h"
#include "sis6326regs.h"

/*
 * main buffer structure for drawing destination
 */
#define sis6326BufferMagic		0x84e7a8b2
#define VALID_SIS6326_BUFFER( b )	( (b) && ( (b)->magic == sis6326BufferMagic ) )

struct sis6326_buffer_t
{
	hwUI32		magic;
	struct sis6326_buffer_t *next;

	void		*backBuffer;	/* pointer to buffer in VM */
	PMemBlock	backBufferBlock; /* location in card's space, NULL if software only */

	void		*depthBuffer;
	PMemBlock	depthBufferBlock;

	int		width, height;	/* in pixels */
	int		pitch;		/* in pixels */

	XSMesaBuffer	xsBuf;		/* needed to get window for front buffer swaps */
};
typedef struct sis6326_buffer_t sis6326Buffer;
typedef struct sis6326_buffer_t *sis6326BufferPtr;

/*
 * main hardware context structure
 */
#define sis6326ContextMagic		0xae8d4c84
#define VALID_SIS6326_CONTEXT( ctx )	( (ctx) && ( (ctx)->magic == sis6326ContextMagic ) )

/* for sis6326Ctx->new_state (when we do it...) */
#define SIS6326_NEW_DEPTH	0x1
#define SIS6326_NEW_ALPHA	0x2
#define SIS6326_NEW_FOG		0x4
#define SIS6326_NEW_CLIP	0x8
#define SIS6326_NEW_MASK	0x10
#define SIS6326_NEW_TEXTURE	0x20
#define SIS6326_NEW_CULL	0x40
#define SIS6326_NEW_CONTEXT	0x100

struct sis6326_context_t
{
	hwUI32		magic;
	int		refcount;

	struct sis6426_buffer_t *DB;
	GLcontext	*gl_ctx;

	/* map GL texture units onto hardware. */
	int		multitex;
	int		tmu_source[2];

	int		frontBufferRendering;	/* generate auto-swapbuffers */
};

#define SIS6326_CONTEXT(ctx)	((sis6326ContextPtr)(((XSMesaContext)(ctx)->DriverCtx)->hw_ctx))


struct sis6326_context_t;
typedef struct sis6326_context_t sis6326Context;
typedef struct sis6326_context_t *sis6326ContextPtr;


/*
 * texture object
 */
#define SIS6326_TEXTURE_OBJECT_MAGIC 0x4ab872f1

typedef struct sis6326_texture_object_s {
	hwUI32		magic;
	struct sis6326_texture_object_s *next;	/* driver global chain of textures */
	struct gl_texture_object *tObj;		/* mesa texture */
	sis6326ContextPtr ctx;

	PMemBlock	memBlock;		/* buffer for all miplevels */
	PMemBlock	memBlocks[2];		/* double buffer for agp texturing */
	int		activeMemBlock;

	hwUI32		totalSize;		/* for adding to c_textureUtilization */
	int		texelBytes;
	
	hwUI32		offsets[10];

	int		widthLog2;
	int		heightLog2;
	int		maxLog2;

	int		hasAlpha;		/* for SCALE_3D_CNTL */
	int		textureFormat;		/* see page 5-47 in docs */

	hwUI32 		age;   			/* set to mgaglx.swapBufferCount when bound */
} sis6326TextureObject_t;

typedef sis6326TextureObject_t *sis6326TextureObjectPtr;



/*
 * main driver variables structure
 */
typedef struct
{
	sis6326Buffer		*bufferList;
	sis6326TextureObject_t	*textureList;
	sis6326TextureObject_t	*currentTexture;

	/* vertex reuse stuff */
	GLuint		setupVertices[3];	/* vertices currently in setup engine */
	int		setupMax;		/* maximum vertex buffer size for clipped
						   vertex handling */
						   
	int 		lastPrim;		/* to TEND right */
	int		sendSpec;
	int		sendTex;

	/* logging stuff */
	hwUI32		logLevel;
	FILE		*logFile;

	char		*dmaMemory;		/* virtual memory location */
	char		memoryFileName[256];	/* shared file for memory mapping */
	int		dmaMemoryFd;		/* file descriptor for mapped memory */
	int		*memoryRemapping;	/* physical address of each page in dmaMemory */

	/* agp texturing stuff */
        int             agpTextures;		/* use agp memory for texture heap */
	hwUI32		agpSize;		/* in megs */
	unsigned char	*agpMemory;		/* virtual memory location */
	hwUI32		agpPhysical;		/* physical agp aperture */

	/* options */
	hwUI32		nullprims;		/* skip all primitive generation */
	hwUI32		boxes;			/* draw performance boxes */
	hwUI32		noFallback;		/* don't fall back to software, do best-effort rendering */
	hwUI32		skipDma;		/* don't send anything to the hardware */
	hwUI32		enforceFinish;		/* force glFinish to work by spec, hurting many
						   application's performance */

	/* performance counters */
	hwUI32		swapBuffersCount;	/* also used for texture thrash checking */

	hwUI32		c_textureUtilization;
	hwUI32		c_textureSwaps;
	hwUI32		c_setupPointers;
	hwUI32		c_clears;
	hwUI32		c_triangles;		/* also used to see if a front buffer swap is needed */
	hwUI32		c_drawWaits;
	hwUI32		c_dmaFlush;
	hwUI32		c_overflows;
	hwUI32		c_numCmds;

	hwUI32		hardwareWentIdle;	/* cleared each swapbuffers, set if
						   a waitfordmacompletion ever exited
						   without having to wait */

	hwUI32		timeTemp;		/* for time counting in logfiles */

	/* stuff we get from the apropriate fbdev/sis6326info struct */
	int		depth;			/* should be 15, 16, or 24 */
	int		virtualX;
	int		virtualY;
	int		displayWidth;
	int		videoRam;		/* in K */
	char		*linearBase;		/* in virtual memory */
	char		*MMIOBase;		/* in virtual memory */
	hwUI32		linearPhysical;		/* physical location of framebuffer */
	int		bytesPerPixel;		/* should be 2 or 4 */

	hwUI32		registers[0x200];	/* read at startup */
	hwUI32		apertureOffset;		/* for dma descriptor entries */
	
	hwUI32		texChange;	 	/* For fun. */
} sis6326Glx_t;


extern XSMesaContext	XSMesa;
extern sis6326Glx_t	sis6326glx;
extern sis6326BufferPtr	sis6326DB;
extern sis6326ContextPtr	sis6326Ctx;


#define SIS6326PACKCOLOR555( r, g, b, a )				\
    ((((r) & 0xf8) << 7) | (((g) & 0xf8) << 2) | (((b) & 0xf8) >> 3) |	\
     ((a) ? 0x8000 : 0))

#define SIS6326PACKCOLOR565( r, g, b )					\
    ((((r) & 0xf8) << 8) | (((g) & 0xfc) << 3) | (((b) & 0xf8) >> 3))

#define SIS6326PACKCOLOR888( r, g, b )					\
    (((r) << 16) | ((g) << 8) | (b))

#define SIS6326PACKCOLOR8888( r, g, b, a )				\
    (((a) << 24) | ((r) << 16) | ((g) << 8) | (b))

#define SIS6326PACKCOLOR4444( r, g, b, a )				\
    ((((a) & 0xf0) << 8) | (((r) & 0xf0) << 4) | ((g) & 0xf0) | ((b) >> 4))

static __inline__ hwUI32 sis6326PackColor( hwUI8 r, hwUI8 g,
					  hwUI8 b, hwUI8 a )
{
	switch ( sis6326glx.depth )
	{
	case 15:
		return SIS6326PACKCOLOR555( r, g, b, a );
	case 16:
		return SIS6326PACKCOLOR565( r, g, b );
	case 24:
		return SIS6326PACKCOLOR888( r, g, b );
	case 32:
		return SIS6326PACKCOLOR8888( r, g, b, a );
	default:
		return 0;
	}
}

/* ===== sis6326glx.c ===== */

sis6326ContextPtr sis6326CreateContext( GLcontext *ctx );
int sis6326DestroyContext( sis6326ContextPtr ctx );

int usec( void );

int sis6326MakeCurrent( sis6326ContextPtr ctx, sis6326BufferPtr buf );


void sis6326GLXCreateDepthBuffer( GLcontext* ctx );
void sis6326GLXDestroyImage( GLXImage* image );
GLXImage* sis6326GLXCreateImage( WindowPtr pwindow, int depth,
				int width, int height, GLXImage *old_image );
GLboolean sis6326GLXMakeCurrent( XSMesaContext c );
GLboolean sis6326GLXBindBuffer( XSMesaContext c, XSMesaBuffer b );
XSMesaContext sis6326GLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list );
void sis6326GLXDestroyContext( XSMesaContext c );


/* the main initialization of the entire mach64 hardware driver */
GLboolean sis6326InitGLX( void );

void sis6326Finish ( void );


/* reset the engine when dma transfers go wrong */
void sis6326ResetEngine( void );
void sis6326DumpEngineState( void );

/* cardHeap is the 8 / 16 / 32 megs of memory on the video card */
extern	memHeap_t	*cardHeap;

/* textureHeap will point to either cardHeap or agpHeap */
extern	memHeap_t	*textureHeap;


/* ===== sis6326clear.c ===== */

GLbitfield sis6326Clear( GLcontext *ctx, GLbitfield mask, GLboolean all,
			GLint x, GLint y, GLint width, GLint height );


/* ===== sis6326swap.c ===== */

void sis6326GLXSwapBuffers( XSMesaBuffer b );
void sis6326GLXSwapBuffersWithoutFlush( XSMesaBuffer b );

void sis6326PerformanceBoxes( int is_direct );
void sis6326BackToFront( DrawablePtr drawable, sis6326BufferPtr buf );


/* ===== sis6326dd.c ===== */

void sis6326_setup_DD_pointers( GLcontext *ctx );

void sis6326DDExtensionsInit( GLcontext *ctx );


/* ===== sis6326state.c ===== */

void sis6326DDUpdateState( GLcontext *ctx );

/* ===== sis6326tri.c ===== */

void sis6326Triangle( GLcontext *ctx, GLuint e0, GLuint e1, GLuint e2, GLuint pv );
void sis6326Quad( GLcontext *ctx, GLuint e0, GLuint e1, GLuint e2, GLuint e3, GLuint pv );

/* ===== sis6326line.c ===== */

void sis6326Line( GLcontext *ctx, GLuint e0, GLuint e1, GLuint pv );


/* ===== sis6326tex.c ===== */

void sis6326CreateTexObj( sis6326ContextPtr ctx, struct gl_texture_object *tObj );
GLboolean sis6326IsTextureResident( GLcontext *ctx, struct gl_texture_object *tObj );
void sis6326DeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj );
void sis6326TexSubImage( GLcontext *ctx, GLenum target,
			struct gl_texture_object *tObj, GLint level,
			GLint xoffset, GLint yoffset,
			GLsizei width, GLsizei height,
			GLint internalFormat,
			const struct gl_texture_image *image );
void sis6326TexImage( GLcontext *ctx, GLenum target,
		     struct gl_texture_object *tObj, GLint level,
		     GLint internalFormat,
		     const struct gl_texture_image *image );



/* ===== sis6326direct.c ===== */

int sis6326GLXVendorPrivate( ClientPtr client,
			    XSMesaContext ctx,
			    xGLXVendorPrivateReq *stuff );


int sis6326GLXAllowDirect( ClientPtr client );

void sis6326DirectClientSwapBuffers( XSMesaBuffer b );

/* ===== sis6326pipeline.c ===== */

GLuint sis6326DDRegisterPipelineStages( struct gl_pipeline_stage *out,
				       const struct gl_pipeline_stage *in,
				       GLuint nr );

#endif


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */

