/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gtk/gtkalignment.h>
#include <string.h>

#include "gtk_toolPanelWidget.h"

#include <support.h>
#include <visu_tools.h>
#include <gtk_main.h>

/**
 * SECTION:toolpanel
 * @short_description: Defines a widget that hold a set of V_Sim panel.
 * 
 * <para>This widget is a complex association of a #GtkComboBox and a
 * #GtkNotebook with V_Sim panels. It can have its own window or be
 * attached into the main command panel.</para>
 *
 * <para>This widget also has a built-in menu to exchange V_Sim panel
 * between different instances of a #ToolPanel. A #ToolPanel is
 * refered by its name and has a position and a size. These values can
 * be stored in the parameter files.</para>
 */

enum {
  NOTEBOOK_ENTERED,
  LAST_SIGNAL
};

enum
  {
    TOOL_LIST_ICON,            /* The icon shown */
    TOOL_LIST_NAME,            /* The label shown */
    TOOL_LIST_POINTER_TO_DATA, /* Pointer to the tool panel. */
    TOOL_LIST_N_COLUMNS
  };

/* Local structures. */
struct _DockWindow
{
  /* The name of the dock window. */
  gchar *name;
  /* The window holding the notebook of tools.
     Maybe NULL if the DockWindow is integrated
     in something else.*/
  GtkWidget *window;
  /* The top container widget for the DockWindow. */
  GtkWidget *vbox;
  /* The header line with the combobox. */
  GtkWidget *hbox;
  /* The notebook holding the tools. */
  GtkWidget *notebook;
  /* The combo list of accessible tool panels. */
  GtkWidget *combo;
  /* The list store associated to the combo. */
  GtkListStore *list;
  /* Handler on the change signals. */
  gulong notebookChanged, comboChanged;
  /* TRUE if the dock is visible. */
  gboolean show;
};

/* Object structures. */
struct _ToolPanel
{
  GtkAlignment alignment;

  /* Set if the toolPanel can be separate from the main
     window and be hosted in its own window. */
  gboolean dockable;

  /* An id, untranslated, used to identify the panel
     in a configuration file. It should not contain
     any space character. */
  gchar *id;

  /* Value that is shown in the combo box, which
     is a "long" label. */
  gchar *comboLabel;

  /* Value that appears in te tab of the
     GtkNoteBook */
  gchar *tabLabel;

  /* An image representing the tool. */
  GtkWidget *icon;

  /* The hosting container. */
  DockWindow *container;

  /* Internal widgets. */
  GtkWidget *headerWidget;

  /* Memory gestion. */
  gboolean dispose_has_run;
};

struct _ToolPanelClass
{
  GtkAlignmentClass parent_class;

  void (*toolPanel) (ToolPanel *tool);

  /* This list holds pointer on active hosting windows. */
  GList* hostingWindows;

  /* This list holds tools than have no containers. */
  GList* orphanToolPanel;

  /* This is the DockWindow of the command panel. */
  DockWindow *commandPanel;

  /* Give a quick access from id of tool panels to pointers. */
  GHashTable *allToolPanels;

  /* Pointer on current handled VisuData.
     No reference is hold. */
  VisuData *dataObj;

  /* If TRUE, the labels are always shown in the tabs. */
  gboolean showHeader;
};

G_DEFINE_TYPE(ToolPanel, toolPanel, GTK_TYPE_ALIGNMENT)

#define MAIN_PANEL_NAME _("Command panel")

/* Local variables. */
static ToolPanelClass *local_class = NULL;
static GQuark CURRENT_TOOLPANEL_POINTER;
static guint toolPanel_signals[LAST_SIGNAL] = { 0 };

/* Local methods. */
static void toolPanel_class_init(ToolPanelClass *klass);
static void toolPanel_init(ToolPanel *toolPanel);
static void toolPanel_dispose(GObject *toolPanel);
static void toolPanel_finalize(GObject *obj);
static GtkWidget* buildDockMenu(ToolPanel *toolPanel, GList *listOfDocks);
static GtkWidget* buildMainMenu(DockWindow *window);
static DockWindow* buildDockWindow(gchar *name, gboolean withWindow);

/* Local callbacks. */
static void onDockButtonClicked(ToolPanel *toolPanel, gpointer data);
static void onMainButtonClicked(GtkButton *button, gpointer data);
static void onDockMenuClicked(GtkMenuItem *menuitem, gpointer user_data);
static void onDockMenuNewClicked(GtkMenuItem *menuitem, gpointer user_data);
static void onDockMenuHideClicked(GtkMenuItem *menuitem, gpointer user_data);
static void onDockMenuSelected(GtkMenuShell *menushell, gpointer user_data);
static void onMainMenuClicked(GtkMenuItem *menuitem, gpointer user_data);
static void onMainMenuShowClicked(GtkMenuItem *menuitem, gpointer user_data);
static void onMainMenuHideClicked(GtkMenuItem *menuitem, gpointer user_data);
static void onMainMenuSelected(GtkMenuShell *menushell, gpointer user_data);
static gboolean onKillDockWindowEvent(GtkWidget *widget, GdkEvent *event, gpointer user_data);
static void onPageChanged(GtkNotebook *book, GtkNotebookPage *page,
			  guint num, gpointer data);
static void onComboChanged(GtkComboBox *combo, gpointer data);

static void toolPanel_class_init(ToolPanelClass *klass)
{
  DBG_fprintf(stderr, "Gtk ToolPanel : creating the class of the widget.\n");

  local_class  = klass;

  DBG_fprintf(stderr, "                     - adding new signals ;\n");
  /**
   * ToolPanel::page-entered:
   * @panel: the #ToolPanel that emits the signal.
   *
   * This signal is emitted when a page of the #GtkNotebook with all
   * the V_Sim panels is entered.
   *
   * Since: 3.3
   */
  toolPanel_signals[NOTEBOOK_ENTERED] =
    g_signal_new ("page-entered",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET(ToolPanelClass, toolPanel),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  DBG_fprintf(stderr, "                     - initializing the list of hosting windows.\n");
  klass->hostingWindows = (GList*)0;
  klass->orphanToolPanel = (GList*)0;
  klass->commandPanel = (DockWindow*)0;
  klass->dataObj = (VisuData*)0;
  klass->allToolPanels = g_hash_table_new_full(g_str_hash, g_str_equal,
					       NULL, NULL);
  klass->showHeader = FALSE;

  CURRENT_TOOLPANEL_POINTER =
    g_quark_from_static_string("ToolPanel_currentToolPanel");

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose = toolPanel_dispose;
  G_OBJECT_CLASS(klass)->finalize = toolPanel_finalize;
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void toolPanel_dispose(GObject *toolPanel)
{
  DBG_fprintf(stderr, "Gtk ToolPanel : dispose object %p (%s).\n",
	      (gpointer)toolPanel, TOOL_PANEL(toolPanel)->id);

  if (TOOL_PANEL(toolPanel)->dispose_has_run)
    return;

  TOOL_PANEL(toolPanel)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(toolPanel_parent_class)->dispose(toolPanel);
}
/* This method is called once only. */
static void toolPanel_finalize(GObject *obj)
{
  ToolPanel *toolPanel;

  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Gtk ToolPanel : finalize object %p.\n", (gpointer)obj);

  toolPanel = TOOL_PANEL(obj);

  /* Remove me from the list of toolPanels. */
  g_hash_table_remove(local_class->allToolPanels, toolPanel->id);

  if (toolPanel->comboLabel)
    g_free(toolPanel->comboLabel);
  if (toolPanel->tabLabel)
    g_free(toolPanel->tabLabel);
  if (toolPanel->id)
    g_free(toolPanel->id);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(toolPanel_parent_class)->finalize(obj);

  DBG_fprintf(stderr, "Gtk ToolPanel : freeing ... OK.\n");
}


static void toolPanel_init(ToolPanel *toolPanel)
{
  DBG_fprintf(stderr, "Gtk ToolPanel : initializing new object (%p).\n",
	      (gpointer)toolPanel);

  toolPanel->dispose_has_run = FALSE;

  toolPanel->dockable     = FALSE;
  toolPanel->id           = (gchar*)0;
  toolPanel->comboLabel   = (gchar*)0;
  toolPanel->tabLabel     = (gchar*)0;
  toolPanel->icon         = (GtkWidget*)0;
  toolPanel->headerWidget = (GtkWidget*)0;
  toolPanel->container    = (DockWindow*)0;
}

GtkWidget* toolPanelNew(gchar *id, gchar* name, gchar *tabName)
{
  ToolPanel *toolPanel;

  g_return_val_if_fail(id && id[0] && !strstr(id, " "), (GtkWidget*)0);
  g_return_val_if_fail(name && name[0], (GtkWidget*)0);
  g_return_val_if_fail(tabName && tabName[0], (GtkWidget*)0);

  toolPanel = TOOL_PANEL(g_object_new(toolPanel_get_type(), NULL));

  DBG_fprintf(stderr, "Gtk ToolPanel : creating new object '%s' : %p.\n",
	      name, (gpointer)toolPanel);

  toolPanel->id         = g_strdup(id);
  toolPanel->comboLabel = g_strdup(name);
  toolPanel->tabLabel   = g_strdup(tabName);

  /* Add me to the list of toolPanels. */
  g_hash_table_insert(local_class->allToolPanels, toolPanel->id,
		      (gpointer)toolPanel);

  return GTK_WIDGET(toolPanel);
}

GtkWidget* toolPanelNew_withIconFromPath(gchar *id, gchar* name, gchar *tabName,
					 const gchar* iconPath)
{
  ToolPanel *toolPanel;

  toolPanel = TOOL_PANEL(toolPanelNew(id, name, tabName));
  if (!toolPanel)
    return (GtkWidget*)0;

  /* TODO : mettre un truc qui limite la taille des images
     effiches  20x20 par exemple en chargeant de GdkImage et en
     regardant sa taille. */
  toolPanel->icon = create_pixmap((GtkWidget*)0, iconPath);
  return GTK_WIDGET(toolPanel);
}

GtkWidget* toolPanelNew_withIconFromStock(gchar *id, gchar* name, gchar *tabName,
					  const gchar* stock)
{
  ToolPanel *toolPanel;

  toolPanel = TOOL_PANEL(toolPanelNew(id, name, tabName));
  if (!toolPanel)
    return (GtkWidget*)0;

  toolPanel->icon = gtk_image_new_from_stock(stock, GTK_ICON_SIZE_MENU);
  return GTK_WIDGET(toolPanel);
}

GtkWidget* toolPanelGet_headerWidget(ToolPanel *toolPanel)
{
  GtkWidget *label, *wd, *image, *align;
  GtkTooltips *tooltips;

  g_return_val_if_fail(toolPanel, (GtkWidget*)0);

  /* If the header has never been created,
     we do it now. */
  if (!toolPanel->headerWidget)
    {
      tooltips = gtk_tooltips_new ();

      toolPanel->headerWidget = gtk_hbox_new(FALSE, 0);
      wd = gtk_event_box_new();
      gtk_event_box_set_visible_window(GTK_EVENT_BOX(wd), FALSE);
      gtk_box_pack_start(GTK_BOX(toolPanel->headerWidget),
			 wd, FALSE, FALSE, 0);
      gtk_tooltips_set_tip(tooltips, wd, toolPanel->comboLabel, NULL);
      if (toolPanel->icon)
	gtk_container_add(GTK_CONTAINER(wd), toolPanel->icon);
      else
	gtk_container_add(GTK_CONTAINER(wd),
			  gtk_image_new_from_stock(GTK_STOCK_MISSING_IMAGE,
						   GTK_ICON_SIZE_MENU));
      label = gtk_label_new(toolPanel->tabLabel);
      gtk_misc_set_padding(GTK_MISC(label), 2, 0);
      gtk_box_pack_start(GTK_BOX(toolPanel->headerWidget), label, FALSE, FALSE, 0);

      if (toolPanel->dockable)
	{
	  wd = gtk_button_new();
	  gtk_button_set_relief(GTK_BUTTON(wd), GTK_RELIEF_NONE);
	  g_signal_connect_swapped(G_OBJECT(wd), "clicked",
				   G_CALLBACK(onDockButtonClicked), (gpointer)toolPanel);
	  align = gtk_alignment_new(1.0, 0.5, 0., 0.);
	  gtk_box_pack_start(GTK_BOX(toolPanel->headerWidget), align, TRUE, TRUE, 0);
	  image = create_pixmap((GtkWidget*)0, "stock-menu-detach.png");
	  gtk_container_add(GTK_CONTAINER(wd), image);
	  gtk_container_add(GTK_CONTAINER(align), wd);

	  gtk_tooltips_set_tip(tooltips, wd,
			       _("Manage this subpanel: attach/detach"
				 " or hide it."), NULL);
	}
      else
	align = (GtkWidget*)0;

      g_object_ref(G_OBJECT(toolPanel->headerWidget));
      gtk_widget_show_all(toolPanel->headerWidget);
      gtk_widget_hide(label);
      if (align) gtk_widget_hide(align);
    }

  return toolPanel->headerWidget;
}

const gchar* toolPanelGet_label(ToolPanel *toolPanel)
{
  g_return_val_if_fail(toolPanel, (gchar*)0);

  return toolPanel->comboLabel;
}

const gchar* toolPanelGet_id(ToolPanel *toolPanel)
{
  g_return_val_if_fail(toolPanel, (gchar*)0);

  return toolPanel->id;
}

void toolPanelSet_dockable(ToolPanel *toolPanel, gboolean value)
{
  g_return_if_fail(toolPanel);

  toolPanel->dockable = value;
}

DockWindow* toolPanelGet_container(ToolPanel *toolPanel)
{
  g_return_val_if_fail(IS_TOOL_PANEL(toolPanel), (DockWindow*)0);

  return toolPanel->container;
}
GtkWindow* toolPanelGet_containerWindow(ToolPanel *toolPanel)
{
  g_return_val_if_fail(IS_TOOL_PANEL(toolPanel), (GtkWindow*)0);

  if (toolPanel->container)
    {
      if (toolPanel->container->window)
	return GTK_WINDOW(toolPanel->container->window);
      else
	return GTK_WINDOW(gtkMainClassGet_currentPanel());
    }
  else
    return (GtkWindow*)0;
}
void toolPanelSet_container(ToolPanel *toolPanel, DockWindow *container)
{
  g_return_if_fail(IS_TOOL_PANEL(toolPanel));

  if (container == toolPanel->container)
    return;

  /* If the panel is already attached, we detach it. */
  if (toolPanel->container)
    {
      DBG_fprintf(stderr, "Gtk ToolPanel : detaching panel '%s' (%p).\n",
		  toolPanel->tabLabel, (gpointer)toolPanel);
      g_object_ref(G_OBJECT(toolPanel));
      toolPanelDetach(toolPanel);
    }

  /* If a new container is given, we attach it. */
  if (container)
    {
      DBG_fprintf(stderr, "Gtk ToolPanel : attaching panel '%s' (%p) to %p.\n",
		  toolPanel->tabLabel, (gpointer)toolPanel, (gpointer)container);
      toolPanelAttach(toolPanel, container);
      g_object_unref(G_OBJECT(toolPanel));
    }
}

const gchar* toolPanelGet_containerId(ToolPanel *toolPanel)
{
  g_return_val_if_fail(IS_TOOL_PANEL(toolPanel), (gchar*)0);

  if (!toolPanel->container)
    return "None";

  if (toolPanel->container == TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel))->commandPanel)
    return "Main";

  return toolPanel->container->name;
}
void toolPanelSet_containerId(ToolPanel *toolPanel, const gchar *id)
{
  g_return_if_fail(IS_TOOL_PANEL(toolPanel) && id && id[0]);

  toolPanelSet_container(toolPanel, toolPanelClassGet_container(id));
}
VisuData* toolPanelGet_visuData(ToolPanel *toolPanel)
{
  g_return_val_if_fail(IS_TOOL_PANEL(toolPanel), (VisuData*)0);
  
  return TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel))->dataObj;
}
gboolean toolPanelGet_visible(ToolPanel *toolPanel)
{
  g_return_val_if_fail(IS_TOOL_PANEL(toolPanel), FALSE);

  /* If the tool panel is not shown, we return fFALSE. */
  if (!toolPanel->container || !toolPanel->container->show)
    return FALSE;

  /* If the tool panel is attached to a visible dock, we
     test if the current page of the notebook is our tool panel. */
  return (toolPanel == TOOL_PANEL(gtk_notebook_get_nth_page
	  (GTK_NOTEBOOK(toolPanel->container->notebook),
	   gtk_notebook_get_current_page
	   (GTK_NOTEBOOK(toolPanel->container->notebook)))));
}


/*****************/
/* Menu gestion. */
/*****************/
static void onDockButtonClicked(ToolPanel *toolPanel, gpointer data _U_)
{
  GtkWidget *wd;

  g_return_if_fail(IS_TOOL_PANEL(toolPanel));

  wd = buildDockMenu(toolPanel, TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel))->hostingWindows);
  g_signal_connect(G_OBJECT(wd), "selection-done",
		   G_CALLBACK(onDockMenuSelected), (gpointer)0);

  gtk_widget_show_all(wd);
  gtk_menu_popup(GTK_MENU(wd), NULL, NULL, NULL, NULL, 
		 1, gtk_get_current_event_time());
}

static GtkWidget* buildDockMenu(ToolPanel *toolPanel, GList *listOfDocks)
{
  GtkWidget *menu, *item;
  DockWindow *window;

  menu = gtk_menu_new();
  
  /* All dock windows. */
  while (listOfDocks)
    {
      window = (DockWindow*)listOfDocks->data;
      if (window->show)
	{
	  item = gtk_menu_item_new_with_label(window->name);
	  g_signal_connect(G_OBJECT(item), "activate",
			   G_CALLBACK(onDockMenuClicked), listOfDocks->data);
	  g_object_set_qdata_full(G_OBJECT(item), CURRENT_TOOLPANEL_POINTER,
				  (gpointer)toolPanel, NULL);
	  gtk_widget_set_sensitive(item, (toolPanel->container != window));
	  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	}
      listOfDocks = g_list_next(listOfDocks);
    }
  /* Separator. */
  item = gtk_separator_menu_item_new();
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  /* Create a new dock window. */
  item = gtk_menu_item_new_with_label(_("New dock"));
  g_signal_connect(G_OBJECT(item), "activate",
		   G_CALLBACK(onDockMenuNewClicked), (gpointer)toolPanel);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  /* Remove the current tool panel. */
  item = gtk_menu_item_new_with_label(_("Hide tool"));
  g_signal_connect(G_OBJECT(item), "activate",
		   G_CALLBACK(onDockMenuHideClicked), (gpointer)toolPanel);
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  
  DBG_fprintf(stderr, "Gtk ToolPanel : create the dock menu %p.\n", (gpointer)menu);

  return menu;
}

static void onDockMenuClicked(GtkMenuItem *menuitem, gpointer user_data)
{
  ToolPanel *toolPanel;

  toolPanel = (ToolPanel*)0;
  toolPanel = TOOL_PANEL(g_object_get_qdata(G_OBJECT(menuitem),
					    CURRENT_TOOLPANEL_POINTER));
  g_return_if_fail(IS_TOOL_PANEL(toolPanel));

  toolPanelSet_container(toolPanel, (DockWindow*)user_data);
}
static void onDockMenuNewClicked(GtkMenuItem *menuitem _U_, gpointer user_data)
{
  GList **lst;
  ToolPanel *toolPanel;
  DockWindow *window;
  guint nb;
  gchar *name;

  lst = &local_class->hostingWindows;
  nb = g_list_length(*lst);
  name = g_strdup_printf(_("Dock window (%d)"), nb);
  window = buildDockWindow(name, TRUE);
  *lst = g_list_append(*lst, (gpointer)window);
  gtk_widget_show_all(window->window);

  toolPanel = TOOL_PANEL(user_data);
  if (toolPanel)
    toolPanelSet_container(toolPanel, window);
}
static void onDockMenuHideClicked(GtkMenuItem *menuitem _U_, gpointer user_data)
{
  DBG_fprintf(stderr, "Gtk ToolPanel : hide the panel %p.\n", (gpointer)user_data);
  toolPanelSet_container(TOOL_PANEL(user_data), (DockWindow*)0);
}

static void onDockMenuSelected(GtkMenuShell *menushell, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Gtk ToolPanel : destroy the dock menu %p.\n", (gpointer)menushell);
  gtk_widget_destroy(GTK_WIDGET(menushell));
}

static DockWindow* buildDockWindow(gchar *name, gboolean withWindow)
{
  DockWindow *window;
  GtkWidget *wd, *image, *align;
  GtkTooltips *tooltips;
  GtkCellRenderer *renderer;

  window = g_malloc(sizeof(DockWindow));

  DBG_fprintf(stderr, "Gtk ToolPanel : create a new dock window %p (%s).\n",
	      (gpointer)window, name);

  window->name = name;

  window->show = !strcmp(name, MAIN_PANEL_NAME);

  window->vbox = gtk_vbox_new(FALSE, 0);

  if (withWindow)
    {
      window->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
      gtk_window_set_title(GTK_WINDOW(window->window), window->name);
      gtk_window_set_default_size(GTK_WINDOW(window->window), 350, 350);
      gtk_window_set_type_hint(GTK_WINDOW(window->window), GDK_WINDOW_TYPE_HINT_UTILITY);
      gtk_window_set_skip_pager_hint(GTK_WINDOW(window->window), TRUE);
      g_signal_connect(G_OBJECT(window->window), "delete-event",
		       G_CALLBACK(onKillDockWindowEvent), (gpointer)window);
      g_signal_connect(G_OBJECT(window->window), "destroy-event",
		       G_CALLBACK(onKillDockWindowEvent), (gpointer)window);

      gtk_container_set_border_width(GTK_CONTAINER(window->window), 3);
      gtk_container_add(GTK_CONTAINER(window->window), window->vbox);
/*       gtk_widget_show(window->window); */
    }
  else
    window->window = (GtkWidget*)0;

  align = gtk_alignment_new(0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 5, 0, 0, 0);
  gtk_box_pack_end(GTK_BOX(window->vbox), align, TRUE, TRUE, 0);
  
  window->notebook = gtk_notebook_new();
  gtk_notebook_set_tab_border(GTK_NOTEBOOK(window->notebook), 1);
  gtk_notebook_set_scrollable(GTK_NOTEBOOK(window->notebook), TRUE);
  gtk_container_add(GTK_CONTAINER(align), window->notebook);
  window->notebookChanged =
    g_signal_connect(G_OBJECT(window->notebook), "switch-page",
		     G_CALLBACK(onPageChanged), (gpointer)window);

  window->hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(window->vbox), window->hbox, FALSE, FALSE, 0);

  tooltips = gtk_tooltips_new ();

  /* Add a help tooltips. */
  wd = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(wd), FALSE);
  gtk_tooltips_set_tip(tooltips, wd,
		       _("Positions, sizes, names, contains... of dock" \
			 " windows are stored in the parameters file, " \
			 "see the 'Config. files' button on the command panel."), NULL);
  gtk_box_pack_end(GTK_BOX(window->hbox), wd, FALSE, FALSE, 2);
  image = gtk_image_new_from_stock(GTK_STOCK_HELP, GTK_ICON_SIZE_MENU);
  gtk_container_add(GTK_CONTAINER(wd), image);

  /* Add a menu button. */
  wd = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(wd), GTK_RELIEF_NONE);
  gtk_box_pack_end(GTK_BOX(window->hbox), wd, FALSE, FALSE, 2);
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onMainButtonClicked), (gpointer)window);
  image = create_pixmap((GtkWidget*)0, "stock-menu-detach.png");
  gtk_container_add(GTK_CONTAINER(wd), image);
  gtk_tooltips_set_tip(tooltips, wd,
		       _("Manage hidden subpanels and dock windows."), NULL);

  /* Little image. */
  image = create_pixmap((GtkWidget*)0, "icone-tool.png");
  gtk_box_pack_start(GTK_BOX(window->hbox), image, FALSE, FALSE, 0);

  /* The Label to introduce the combo list. */
  wd = gtk_label_new(_("Tool: "));
  gtk_box_pack_start(GTK_BOX(window->hbox), wd, FALSE, FALSE, 2);

  /* We create the list store. */
  window->list = gtk_list_store_new(TOOL_LIST_N_COLUMNS,
				    GDK_TYPE_PIXBUF,
				    G_TYPE_STRING, G_TYPE_POINTER);
  /* The combo list. */
  window->combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(window->list));
  renderer = gtk_cell_renderer_pixbuf_new();
  gtk_cell_renderer_set_fixed_size(renderer, 22, -1);
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(window->combo), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(window->combo),
				renderer, "pixbuf", TOOL_LIST_ICON);
  renderer = gtk_cell_renderer_text_new();
  gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(window->combo), renderer, FALSE);
  gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(window->combo),
				renderer, "text", TOOL_LIST_NAME);
  gtk_box_pack_start(GTK_BOX(window->hbox), window->combo, TRUE, TRUE, 0);
  window->comboChanged = g_signal_connect(G_OBJECT(window->combo), "changed",
					  G_CALLBACK(onComboChanged),
					  (gpointer)window);

  gtk_widget_show_all(window->vbox);
  return window;
}

static gboolean emitPageEnter(gpointer data)
{
  g_return_val_if_fail(IS_TOOL_PANEL(data), FALSE);

  DBG_fprintf(stderr, "Gtk ToolPanel : '%p' emits 'page-entered' signal.\n", data);
  g_signal_emit(G_OBJECT(data), toolPanel_signals[NOTEBOOK_ENTERED],
		0, NULL);
  
  return FALSE;
}

static void changeHeaderVisibility(ToolPanel *panel, gboolean status)
{
  GList *list, *tmpLst;
  
  g_return_if_fail(IS_TOOL_PANEL(panel) && panel->headerWidget);

  list = gtk_container_get_children(GTK_CONTAINER(panel->headerWidget));
  tmpLst = list->next;
  while(tmpLst)
    {
      if (status)
	gtk_widget_show(GTK_WIDGET(tmpLst->data));
      else
	gtk_widget_hide(GTK_WIDGET(tmpLst->data));
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(list);
}

static void onPageChanged(GtkNotebook *book, GtkNotebookPage *page _U_,
			  guint num, gpointer data)
{
  GtkWidget *wd;
  DockWindow *window;
  int numOld;

  wd = gtk_notebook_get_nth_page(book, num);
  g_return_if_fail(IS_TOOL_PANEL(wd));

  window = (DockWindow*)data;
  g_return_if_fail(window);

  DBG_fprintf(stderr, "Gtk ToolPanel : caught 'page-switch' signal, jump to %d.\n", num);
  /* Change the visibility of the header labels. */
  if (!local_class->showHeader)
    {
      numOld = gtk_combo_box_get_active(GTK_COMBO_BOX(window->combo));
      changeHeaderVisibility
	(TOOL_PANEL(gtk_notebook_get_nth_page(book, numOld)), FALSE);
      changeHeaderVisibility
	(TOOL_PANEL(gtk_notebook_get_nth_page(book, num)), TRUE);
    }
  
  /* Change the combo to the right one. */
  g_signal_handler_block(G_OBJECT(window->combo), window->comboChanged);
  gtk_combo_box_set_active(GTK_COMBO_BOX(window->combo), num);
  g_signal_handler_unblock(G_OBJECT(window->combo), window->comboChanged);


  g_idle_add(emitPageEnter, wd);
}

static void onComboChanged(GtkComboBox *combo, gpointer data)
{
  int index, numOld;
  DockWindow *window;
  GtkNotebook *book;
  
  window = (DockWindow*)data;
  g_return_if_fail(window);

  index = gtk_combo_box_get_active(combo);

  /* Change the visibility of the header labels. */
  if (!local_class->showHeader)
    {
      book = GTK_NOTEBOOK(window->notebook);
      numOld = gtk_notebook_get_current_page(book);
      changeHeaderVisibility
	(TOOL_PANEL(gtk_notebook_get_nth_page(book, numOld)), FALSE);
      changeHeaderVisibility
	(TOOL_PANEL(gtk_notebook_get_nth_page(book, index)), TRUE);
    }

  /* Change the page to the right one. */
  gtk_notebook_set_current_page(GTK_NOTEBOOK(window->notebook), index);
}


static gboolean onKillDockWindowEvent(GtkWidget *widget, GdkEvent *event _U_,
				      gpointer user_data)
{
  g_return_val_if_fail(user_data, TRUE);

  DBG_fprintf(stderr, "Gtk ToolPanel : delete or destroy event cancelled.\n");
  gtk_widget_hide(widget);
  ((DockWindow*)user_data)->show = FALSE;
  return TRUE;
}

void toolPanelAttach(ToolPanel *toolPanel, DockWindow *dock)
{
  GtkTreeIter iter;
  ToolPanelClass *klass;
  GdkPixbuf *pixbuf;

  DBG_fprintf(stderr, "Gtk ToolPanel : attach a toolPanel (%p) to a dock"
	      " window (%s).\n", (gpointer)toolPanel, dock->name);

  g_return_if_fail(dock && toolPanel);

  if (gtk_image_get_storage_type(GTK_IMAGE(toolPanel->icon)) == GTK_IMAGE_PIXBUF)
    pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(toolPanel->icon));
  else
    pixbuf = (GdkPixbuf*)0;

  /* We add the window in the list store. */
  gtk_list_store_append(dock->list, &iter);
  gtk_list_store_set(dock->list, &iter,
		     TOOL_LIST_ICON, pixbuf,
		     TOOL_LIST_NAME, toolPanel->comboLabel,
		     TOOL_LIST_POINTER_TO_DATA, toolPanel,
		     -1);

  gtk_notebook_append_page(GTK_NOTEBOOK(dock->notebook), GTK_WIDGET(toolPanel),
			   toolPanelGet_headerWidget(toolPanel));
  gtk_widget_show(GTK_WIDGET(toolPanel));
  toolPanel->container = dock;

  if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(dock->notebook)) > 1)
    gtk_widget_show(dock->hbox);
  else
    gtk_widget_hide(dock->hbox);

  klass = TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel));
  klass->orphanToolPanel = g_list_remove(klass->orphanToolPanel,
					 (gpointer)toolPanel);

}

void toolPanelDetach(ToolPanel *toolPanel)
{
  gint page;
  DockWindow *window;
  GtkTreeIter iter;
  GtkTreePath* path;
  gboolean valid;
  ToolPanelClass *klass;

  g_return_if_fail(toolPanel && toolPanel->container);

  window = toolPanel->container;

  /* Get the id. */
  page = gtk_notebook_page_num(GTK_NOTEBOOK(window->notebook), GTK_WIDGET(toolPanel));

  DBG_fprintf(stderr, "Gtk ToolPanel : detach a toolPanel (%s) from page %d of dock"
	      " (%s).\n", toolPanel->tabLabel, page, window->name);

  /* We remove the page for the notebook. */
  gtk_notebook_remove_page(GTK_NOTEBOOK(window->notebook), page);

  if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(window->notebook)) < 2)
    gtk_widget_hide(window->hbox);

  /* We remove the tool from the combo list. */
  path = gtk_tree_path_new_from_indices(page, -1);
  valid = gtk_tree_model_get_iter(GTK_TREE_MODEL(window->list), &iter, path);
  if (valid)
    gtk_list_store_remove(window->list, &iter);
  gtk_tree_path_free(path);

  /* If there is no more tool in this window, we delete it. */
  if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(window->notebook)) == 0 &&
      window != TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel))->commandPanel)
    {
      DBG_fprintf(stderr, "Gtk ToolPanel : destroying dock window %p (%s).\n",
		  (gpointer)window, window->name);
      gtk_widget_destroy(window->window);
      TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel))->hostingWindows =
	g_list_remove(TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel))->hostingWindows,
		      (gconstpointer)window);
      g_free(window->name);
      g_object_unref(window->list);
      g_free(window);
    }

  toolPanel->container = (DockWindow*)0;
  klass = TOOL_PANEL_CLASS(G_OBJECT_GET_CLASS(toolPanel));
  klass->orphanToolPanel = g_list_prepend(klass->orphanToolPanel,
					  (gpointer)toolPanel);
  DBG_fprintf(stderr, "Gtk ToolPanel : number of orphan tool panels %d.\n",
	      g_list_length(klass->orphanToolPanel));
}

static void onMainButtonClicked(GtkButton *button _U_, gpointer data)
{
  GtkWidget *wd;

  wd = buildMainMenu((DockWindow*)data);
  g_signal_connect(G_OBJECT(wd), "selection-done",
		   G_CALLBACK(onMainMenuSelected), (gpointer)0);

  gtk_widget_show_all(wd);
  gtk_menu_popup(GTK_MENU(wd), NULL, NULL, NULL, NULL, 
		 1, gtk_get_current_event_time());
}

static GtkWidget* buildMainMenu(DockWindow *window)
{
  GtkWidget *menu, *item;
  GList* tmpLst;
  gchar *label;
  ToolPanel *panel;
  DockWindow *dock;
  gboolean haveHiddenDock;

  menu = gtk_menu_new();
  
  tmpLst = local_class->orphanToolPanel;
  /* All dock windows. */
  while (tmpLst)
    {
      panel = (ToolPanel*)tmpLst->data;
      label = g_strdup_printf(_("Show '%s'"), panel->comboLabel);
      item = gtk_menu_item_new_with_label(label);
      g_free(label);
      g_signal_connect(G_OBJECT(item), "activate",
		       G_CALLBACK(onMainMenuClicked), window);
      g_object_set_qdata_full(G_OBJECT(item), CURRENT_TOOLPANEL_POINTER,
			      (gpointer)panel, NULL);
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
      tmpLst = g_list_next(tmpLst);
    }
  if (!local_class->orphanToolPanel)
    {
      item = gtk_menu_item_new_with_label(_("No hidden tool"));
      gtk_widget_set_sensitive(item, FALSE);
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    }
  /* Separator. */
  item = gtk_separator_menu_item_new();
  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
  /* Hidden dock. */
  haveHiddenDock = FALSE;
  tmpLst = local_class->hostingWindows;
  while (tmpLst)
    {
      dock = (DockWindow*)tmpLst->data;
      if (!dock->show)
	{
	  haveHiddenDock = TRUE;
	  label = g_strdup_printf(_("Show '%s'"), dock->name);
	  item = gtk_menu_item_new_with_label(label);
	  g_free(label);
	  g_signal_connect(G_OBJECT(item), "activate",
			   G_CALLBACK(onMainMenuShowClicked), dock);
	  gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
	}
      tmpLst = g_list_next(tmpLst);
    }
  if (!haveHiddenDock)
    {
      item = gtk_menu_item_new_with_label(_("No hidden dock"));
      gtk_widget_set_sensitive(item, FALSE);
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    }
  if (window->window)
    {
      /* Separator. */
      item = gtk_separator_menu_item_new();
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
      /* Hide action */
      item = gtk_menu_item_new_with_label(_("Hide dock"));
      g_signal_connect(G_OBJECT(item), "activate",
		       G_CALLBACK(onMainMenuHideClicked), window);
      gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    }
  
  DBG_fprintf(stderr, "Gtk ToolPanel : create the main menu %p.\n", (gpointer)menu);

  return menu;
}
static void onMainMenuClicked(GtkMenuItem *menuitem, gpointer user_data)
{
  ToolPanel *toolPanel;
  DockWindow *window;

  toolPanel = (ToolPanel*)0;
  toolPanel = TOOL_PANEL(g_object_get_qdata(G_OBJECT(menuitem),
					    CURRENT_TOOLPANEL_POINTER));
  g_return_if_fail(IS_TOOL_PANEL(toolPanel));

  window = (DockWindow*)user_data;
  
  DBG_fprintf(stderr, "Gtk ToolPanel : attaching panel '%s' (%p) to %p.\n",
	      toolPanel->tabLabel, (gpointer)toolPanel, (gpointer)window);
  toolPanelAttach(toolPanel, window);
  g_object_unref(G_OBJECT(toolPanel));
}
static void onMainMenuShowClicked(GtkMenuItem *menuitem _U_, gpointer user_data)
{
  DockWindow *dock;

  g_return_if_fail(user_data);
  
  dock = (DockWindow*)user_data;
  if (dock->window)
    gtk_widget_show(dock->window);
  dock->show = TRUE;
}
static void onMainMenuHideClicked(GtkMenuItem *menuitem _U_, gpointer user_data)
{
  DockWindow *dock;

  g_return_if_fail(user_data);
  
  dock = (DockWindow*)user_data;
  if (dock->window)
    gtk_widget_hide(dock->window);
  dock->show = FALSE;
}
static void onMainMenuSelected(GtkMenuShell *menushell, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Gtk ToolPanel : destroy the main menu %p.\n", (gpointer)menushell);
  gtk_widget_destroy(GTK_WIDGET(menushell));
}


/******************/
/* Class methods. */
/******************/

DockWindow* toolPanelClassGet_commandPanel()
{
  GObject *dummy;

  if (!local_class)
    {
      dummy = g_object_new(toolPanel_get_type(), NULL);
      g_object_ref(dummy);
      g_object_unref(dummy);
    }

  if (!local_class->commandPanel)
    {
      local_class->commandPanel = buildDockWindow(MAIN_PANEL_NAME, FALSE);
      local_class->hostingWindows = g_list_prepend(local_class->hostingWindows,
						   local_class->commandPanel);
    }

  return local_class->commandPanel;
}
void toolPanelClassSet_visuData(VisuData *dataObj)
{
  GObject *dummy;

  if (!local_class)
    {
      dummy = g_object_new(toolPanel_get_type(), NULL);
      g_object_ref(dummy);
      g_object_unref(dummy);
    }

  local_class->dataObj = dataObj;
}
void toolPanelClassSet_windowSize(const gchar *id, guint width, guint height)
{
  DockWindow *dock;
  GdkScreen *screen;
  gint widthScreen, heightScreen;
  gint widthSet, heightSet;

  g_return_if_fail(strcmp(id, "Main"));

  /* Get the dock window associated to id, or create a new one. */
  dock = toolPanelClassGet_container(id);
  g_return_if_fail(dock->window);

  /* Before setting (width, height), we check that it is coherent with
     the screen size. */
  if (GDK_IS_DRAWABLE(dock->window->window))
    screen = gdk_drawable_get_screen(GDK_DRAWABLE(dock->window->window));
  else
    screen = gdk_screen_get_default();
  widthScreen = gdk_screen_get_width(screen);
  heightScreen = gdk_screen_get_height(screen);

  widthSet = ((gint)width > widthScreen)?widthScreen:(gint)width;
  heightSet = ((gint)height > heightScreen)?heightScreen:(gint)height;
  if (widthSet < 0) widthSet = 50;
  if (heightSet < 0) heightSet = 50;
  DBG_fprintf(stderr, "Gtk ToolPanel: set window (%s) size %dx%d.\n",
	      id, widthSet, heightSet);
  gtk_window_resize(GTK_WINDOW(dock->window), widthSet, heightSet);
}
void toolPanelClassSet_windowPosition(const gchar *id, guint x, guint y)
{
  DockWindow *dock;
  GdkScreen *screen;
  gint xScreen, yScreen;
  gint xPosition, yPosition;
  gint width, height;

  g_return_if_fail(strcmp(id, "Main"));

  /* Get the dock window associated to id, or create a new one. */
  dock = toolPanelClassGet_container(id);
  g_return_if_fail(dock->window);

  /* Before setting (x, y), we check that it is coherent with
     the screen definition. */
  if (GDK_IS_DRAWABLE(dock->window->window))
    screen = gdk_drawable_get_screen(GDK_DRAWABLE(dock->window->window));
  else
    screen = gdk_screen_get_default();
  xScreen = gdk_screen_get_width(screen);
  yScreen = gdk_screen_get_height(screen);
  gtk_window_get_size(GTK_WINDOW(dock->window), &width, &height);

  xPosition = ((gint)x + width > xScreen)?xScreen - width:(gint)x;
  yPosition = ((gint)y + height > yScreen)?yScreen - height:(gint)y;
  if (xPosition < 0) xPosition = 0;
  if (yPosition < 0) yPosition = 0;
  DBG_fprintf(stderr, "Gtk ToolPanel: set window (%s) position %dx%d.\n",
	      id, xPosition, yPosition);
  gtk_window_move(GTK_WINDOW(dock->window), xPosition, yPosition);
}
void toolPanelClassSet_windowVisibility(const gchar *id, gboolean visible)
{
  DockWindow *dock;

  g_return_if_fail(strcmp(id, "Main"));

  /* Get the dock window associated to id, or create a new one. */
  dock = toolPanelClassGet_container(id);
  g_return_if_fail(dock->window);

  dock->show = visible;
  if (!visible)
    gtk_widget_hide(dock->window);
  else
    gtk_widget_show(dock->window);
}
ToolPanel* toolPanelClassGet_toolPanelById(const gchar *id)
{
  GObject *dummy;
  gpointer panel;

  if (!local_class)
    {
      dummy = g_object_new(toolPanel_get_type(), NULL);
      g_object_ref(dummy);
      g_object_unref(dummy);
    }

  DBG_fprintf(stderr, "Gtk ToolPanel : search panel '%s'.\n", id);
  panel = g_hash_table_lookup(local_class->allToolPanels, id);
  if (!panel)
    return (ToolPanel*)0;
  else
    return TOOL_PANEL(panel);
}
DockWindow* toolPanelClassGet_container(const gchar *id)
{
  GObject *dummy;
  DockWindow *window;
  GList *tmplst;

  if (!local_class)
    {
      dummy = g_object_new(toolPanel_get_type(), NULL);
      g_object_ref(dummy);
      g_object_unref(dummy);
    }

  if (!strcmp(id, "None"))
    return (DockWindow*)0;

  if (!strcmp(id, "Main"))
    return local_class->commandPanel;
  
  tmplst = local_class->hostingWindows;
  while (tmplst)
    {
      if (!strcmp(((DockWindow*)tmplst->data)->name, id))
	return (DockWindow*)tmplst->data;
      tmplst = g_list_next(tmplst);
    };
  /* No matching name found, we create a new window. */
  window = buildDockWindow(g_strdup(id), TRUE);
  local_class->hostingWindows = g_list_prepend(local_class->hostingWindows,
					      (gpointer)window);

  return window;
}
GList* toolPanelClassGet_allToolPanels()
{
  GObject *dummy;
  GList *tmplst, *returnlst;
  gboolean valid;
  GtkTreeIter iter;
  ToolPanel *tool;
  GtkTreeModel *treeModel;

  if (!local_class)
    {
      dummy = g_object_new(toolPanel_get_type(), NULL);
      g_object_ref(dummy);
      g_object_unref(dummy);
    }
  
  DBG_fprintf(stderr, "Gtk ToolPanel : create list of all panels.\n");
  returnlst = (GList*)0;
  tmplst = local_class->hostingWindows;
  while (tmplst)
    {
      treeModel = GTK_TREE_MODEL(((DockWindow*)tmplst->data)->list);
      valid = gtk_tree_model_get_iter_first(treeModel, &iter);
      while (valid)
	{
	  gtk_tree_model_get(treeModel, &iter, TOOL_LIST_POINTER_TO_DATA, &tool, -1);
	  DBG_fprintf(stderr, " | add '%s'\n", tool->id);
	  returnlst = g_list_prepend(returnlst, (gpointer)tool);
	  valid = gtk_tree_model_iter_next(treeModel, &iter);
	}
      tmplst = g_list_next(tmplst);
    }
  returnlst = g_list_concat(returnlst, g_list_copy(local_class->orphanToolPanel));
  return returnlst;
}
GList* toolPanelClassGet_allWindows()
{
  GObject *dummy;
  GList *returnlst;

  if (!local_class)
    {
      dummy = g_object_new(toolPanel_get_type(), NULL);
      g_object_ref(dummy);
      g_object_unref(dummy);
    }
  
  DBG_fprintf(stderr, "Gtk ToolPanel : create list of all windows.\n");
  returnlst = g_list_copy(local_class->hostingWindows);
  returnlst = g_list_remove(returnlst, local_class->commandPanel);
  return returnlst;
}
void toolPanelClassSet_headerVisibility(gboolean status)
{
  GObject *dummy;
  GList *tmpLst;
  int num, i;
  GtkNotebook *notebook;

  if (!local_class)
    {
      dummy = g_object_new(toolPanel_get_type(), NULL);
      g_object_ref(dummy);
      g_object_unref(dummy);
    }
  
  DBG_fprintf(stderr, "Gtk ToolPanel : set the header visibility.\n");
  if (local_class->showHeader == status)
    return;

  tmpLst = local_class->hostingWindows;
  while(tmpLst)
    {
      notebook = GTK_NOTEBOOK(((DockWindow*)tmpLst->data)->notebook);
      num = gtk_notebook_get_current_page(notebook);
      for (i = 0; i < gtk_notebook_get_n_pages(notebook); i++)
	changeHeaderVisibility(TOOL_PANEL(gtk_notebook_get_nth_page(notebook, i)),
			       status || (i == num));
      tmpLst = g_list_next(tmpLst);
    }
  local_class->showHeader = status;
}
void toolPanelClassGet_windowCharacteristics(DockWindow *dock, gchar **id, 
					     gboolean *visibility, 
					     gint *x, gint *y, 
					     gint *width, gint *height)
{
  g_return_if_fail(dock);

  *id = dock->name;
  *visibility = dock->show;
  gtk_window_get_position(GTK_WINDOW(dock->window), x, y);
  gtk_window_get_size(GTK_WINDOW(dock->window), width, height);
}
GtkWidget* dockWindowGet_container(DockWindow *dock)
{
  g_return_val_if_fail(dock, (GtkWidget*)0);
  
  return dock->vbox;
}
GtkWidget* dockWindowGet_notebook(DockWindow *dock)
{
  g_return_val_if_fail(dock, (GtkWidget*)0);
  
  return dock->notebook;
}
GtkWidget* dockWindowGet_window(DockWindow *dock)
{
  g_return_val_if_fail(dock, (GtkWidget*)0);
  
  if (dock->window)
    return dock->window;
  else
    return GTK_WIDGET(gtkMainClassGet_currentPanel());
}
