"""
Contains a loader to load configuration from a file using a
``ConfigParser``.
"""

import ConfigParser

class ConfigFileLoader(object):
    def __init__(self, path, section, optional=False):
        """
        Initializes a ``ConfigFileLoader``, which reads configuration from
        a file in the format made for Python's built-in ``ConfigParser``
        (similar to ini file format).

        :Parameters:
          - `path`: Path to the file to read. This path should contain
            an interpolation variable for ``environment`` so that Choices
            knows how to get the environment-specific path for this
            configuration. More information below.
          - `section`: The top-level section to load configuration from.
          - `optional`: If True, then a ValueError is not raised if the file does not exist

        The ``path`` variable allows for interpolation with any defined
        setting, and will use the values that are loaded up to that point
        to determine the path. For example, if the file is::

            /etc/project/%(environment)s.config.cfg

        And it was loaded like so::

            c.add_loader(DictLoader({"environment":"staging"}))
            c.add_loader(ConfigFileLoader("/etc/project/%(environment)s.config.cfg"))
            c.load()

        Then the final load path would be::

            /etc/project/staging.config.cfg

        """
        self.path = path
        self.section = section
        self.optional = optional

    def load(self, options, settings, **kwargs):
        """
        Loads the configuration. If the file doesn't exist to load the
        configuration, a ``ValueError`` is raised. If the proper section
        doesn't exist within the configuration, a ``ValueError`` is raised.
        """
        # Use the current settings to build the path
        path = self.path % settings

        config = ConfigParser.RawConfigParser()
        if config.read([path]) != [path] and not self.optional:
            raise ValueError, "Failed to parse configuration file: %s" % self.path

        if config.has_section(self.section):
            return dict(config.items(self.section))
        elif not self.optional:
            raise ValueError, "Configuration file does not have proper top-level section: %s" % self.section
        return {}

class PythonFileLoader(object):
    def __init__(self, path, variable=None, optional=False):
        """
        A python file loader will load the Python file at the given path,
        and use the local variables created to setup the configuration.

        :Parameters:
          - `path`: Path of the file to load.
          - `variable` (optional): A local variable that will contain the
            dictionary of settings. If not given, the entire locals dictionary
            generated by evaluating the file will be used.
          - `optional` (optional): If ``True``, then an error will not be
            raised if the file given doesn't exist.

        The ``path`` variable also allows for interpolation on any defined
        settings. For example, if the ``environment`` setting exists and
        is set, then the value of that setting will be interpolated into
        the path variable in this case:

            /etc/project/config.%(environment)s.py

        .. warning::

           This is **not safe** for untrusted files. The Python file will
           be fully evaluated, so only use this for files you absolutely
           trust.
        """
        self.path = path
        self.variable = variable
        self.optional = optional

    def load(self, options, settings, **kwargs):
        """
        Loads the configuration.
        """
        path = self.path % settings
        new_locals = {}

        try:
            # Evaluate the file. Pass in no globals but pass in a custom
            # dictionary for the locals so that we can read from it.
            execfile(path, {}, new_locals)
        except IOError:
            # This error occurs if the path doesn't exist. If its optional,
            # then it is okay.
            if self.optional:
                return {}

            raise

        if self.variable is not None:
            if self.variable not in new_locals:
                raise KeyError, "'%s' not exported from configuration file: %s" % \
                    (self.variable, path)

            new_locals = new_locals[self.variable]

        return new_locals

class PythonModuleLoader(object):
    def __init__(self, path, optional=False):
        """
        Initializes a ``PythonModuleLoader``, which reads an exported variable
        from a python module and expects that to be a dictionary. This
        file loader allows you to use calculations to generate configuration
        dynamically.

        :Parameters:
          - `path`: Path of the module name plus the variable name to load
            files from. For example ``foo.bar.settings`` would load the
            ``foo.bar`` module and read the ``settings`` variable exported
            from that module.
          - `optional` (optional): If True, then if the module or attribute
            is missing, this will be ignored.

        The ``path`` variable allows for interpolation on any defined settings
        For example, for the following path::

            project.config.%(environment)s.settings

        Then if you load the settings with the "staging" environment then the
        following actual path will be loaded:

            project.config.staging.settings

        .. warning::

           This is **not safe** for anything but trusted configuration files.
           The Python module is loaded directly into this memory space, and is
           free to do anything.
        """
        self.path = path
        self.optional = optional

    def load(self, options, settings, **kwargs):
        """
        Loads the configuration.
        """
        # Interpolate the settings into the path
        path = self.path % settings

        module_path, _sep, variable = path.rpartition(".")
        try:
            module = __import__(module_path, globals(), locals(), [variable], -1)
            result = getattr(module, variable)
            return result
        except (ImportError, AttributeError):
            # If this is marked as optional then we ignore import and
            # attribute errors.
            if self.optional:
                return {}

            raise
