/*
 * ===========================
 * VDK Visual Development Kit
 * Version 0.4
 * October 1998
 * ===========================
 *
 * Copyright (C) 1998, Mario Motta
 * Developed by Mario Motta <mmotta@guest.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef  TEXT_H
#define  TEXT_H

#include <vdk/vdkobj.h>
#include <vdk/vdkprops.h>
/*!
  \class VDKTextFont
  \brief provides a font wrapper for VDKText widget
 */
class VDKTextFont
{
 public:
  GdkColor *foreground, *background;
  VDKFont* font;
  VDKTextFont(): foreground((GdkColor*) NULL),
    background((GdkColor*) NULL),
    font((VDKFont*) NULL) {}
  ~VDKTextFont() {}
};

/*!
  \class VDKText
  \brief Provides a gtk+ gtktext wrapper
  \par TIP
  Any updating operation on this widget should be done after widget
  has been "realized". A good practice is to connect with
  realize_signal and do these operations on the response method.
  \code
  DEFINE_SIGNAL_MAP(TextWindow,VDKForm)
    ON_SIGNAL(text,realize_signal,OnRealizeText) 
  END_SIGNAL_MAP
  //....
  bool
  TextWindow::OnRealizeText(VDKObject* sender)
  {
  VDKText* t = dynamic_cast<VDKText*>(sender);
  g_return_val_if_fail(t != NULL, true);
  // use default
  t->TextInsert(line1);
  // chage font and colors
  t->TextFont.foreground = *navy;
  t->TextFont.font = new VDKFont(this,courier12);
  // ...
  return true;
  }
  \endcode
  \par EXAMPLES
  Into ./testvdk/textwindow.cc
 */
class VDKText: public VDKObject
{
 protected:
 static int KeyEvent(GtkWidget *wid, GdkEventKey* ev,gpointer gp);
 protected:
  GtkWidget *text,*scrollbar;
 public:
  // properties
  /*!
    Use this to set text font, foreground and background, setting any
    property to NULL restores defaults.
   */
  VDKTextFont TextFont;
  /*!
    Sets/gets word wrapping at line end.
   */
  VDKReadWriteValueProp<VDKText,bool>  WordWrap;
  /*!
    Sets/gets editable status
   */
  VDKReadWriteValueProp<VDKText,bool>  Editable;
  /*!
    Sets/gets text pointer position.
   */
  VDKReadWriteValueProp<VDKText,int>   Pointer;
  /*!
    Gets text length
   */
  VDKReadOnlyValueProp<VDKText,unsigned int> Length;
  /*!
    Set to true if text was changed, should be explicitely set to false
    by user.
   */
  bool Changed;
  /*!
    Constructor
    \param owner
    \param editable
    \param override (users don't care about this, leave it always false)
   */
  VDKText(VDKForm* owner,bool editable = false, bool override = false);
  /*!
    Destructor
   */
  virtual ~VDKText();
  /*!
    Load text widget with a file
    \param filename
   */
  int LoadFromFile(char* filename);
  /*!
    Save text into a file
    \param filename
   */
  int SaveToFile(char* filename);

  virtual void SetBackground(VDKRgb color, 
			     GtkStateType state = GTK_STATE_NORMAL);
  virtual void SetFont(VDKFont* font);
  /*!
    Insert text at pointer position
    \param text
    \param nchars number of chars to be inserted, leaving to -1 causes
    all text to be inserted.
   */
  void TextInsert(char* text, int nchars = -1);
  /*!
    Freezes widget for long time updating operations avoid flicking
   */
  void Freeze() { gtk_text_freeze(GTK_TEXT(text)); }
  /*!
    Unfreeze
   */
  void Thaw() { gtk_text_thaw(GTK_TEXT(text)); }
  /*!
    Inserts a new line
   */
  void Eol() { TextInsert("\n"); }
  /*!
    Delete backward
    \param nchars number of chars to be deleted
   */
  int  BackwardDelete(int nchars)
    { 
      Changed = true;
      return gtk_text_backward_delete(GTK_TEXT(text),nchars); 
    }
  /*!
    Delete forward
    \param nchars number of chars to be deleted
   */
  int  ForwardDelete(int nchars)
    { 
      Changed = true;
      return gtk_text_forward_delete(GTK_TEXT(text),nchars); 
    }
  /*!
    Gets chars from text widget
    \param start from this position
    \param end to this position, leaving it to -1 gets all chars
    from <start> to the ned.
    \par TIP
    Received address should be freed by the user
    \code
    char* p = text->GetChars(start,end);
    if(p)
     {
     // use p ..
     g_free(p);
     }
    \endcode
   */
  char* GetChars(int start, int end = -1)
    { return gtk_editable_get_chars(GTK_EDITABLE(text),start,end); }
  int GetPointer() { return gtk_text_get_point(GTK_TEXT(text)); }
  void SetPointer(int pos) { gtk_text_set_point(GTK_TEXT(text),pos); }
  bool GetWordWrap() { return WordWrap; }
  void SetWordWrap(bool flag)
    { gtk_text_set_word_wrap(GTK_TEXT(text),flag); }
  bool GetEditable () { return Editable; }
  void SetEditable(bool flag)
    { gtk_text_set_editable(GTK_TEXT(text),flag); }
  /*!
    Clears text widget
   */
  void Clear();
  unsigned int GetLength() 
    { return gtk_text_get_length(GTK_TEXT(text)); }
#ifdef USE_SIGCPLUSPLUS 
  /*!
    Extended LS signal system:
    Received when editable status changes
    \param bool editable
  */
  VDKSignal1<void, bool> OnEditableChanged;
  VDKSignal3<void, const char*, int, int> OnTextInsert;
  VDKSignal2<void, int, int> OnTextDelete;
  VDKSignal2<void, int , int> OnMoveCursor;
  VDKSignal1<void, int> OnMoveWord; // Richtung?
#endif /* USE_SIGCPLUSPLUS */
};
#endif
