#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/types.h>

#include <gtk/gtk.h>
#include <gtkgl/gtkglarea.h>

#include "msglist.h"
#include "v3dtex.h"

#include "texbrowser.h"
#include "texbrowsercb.h"
#include "texbrowserpv.h"

#include "guiutils.h"
#include "editor.h"
#include "editorselect.h"
#include "editortexture.h"

#include "vmacfg.h"
#include "vmacfglist.h"
#include "vma.h"

#include "messages.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


#include "texbrowser.xpm"

#include "images/icon_browse_20x20.xpm"
#include "images/icon_add_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_properties_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_close_20x20.xpm"


void TexBrowserSetBusy(ma_texture_browser_struct *tb);
void TexBrowserSetReady(ma_texture_browser_struct *tb);
int TexBrowserBuildMenuBar(
        ma_texture_browser_struct *tb,
        GtkWidget *parent
);
int TexBrowserBuildToolRibbon(
        ma_texture_browser_struct *tb,
        GtkWidget *parent
);
int TexBrowserBuildBaseDirRibbon(
        ma_texture_browser_struct *tb,
        GtkWidget *parent
);
int TexBrowserBuildSearchRibbon(
        ma_texture_browser_struct *tb,
        GtkWidget *parent
);
int TexBrowserCreate(
	ma_texture_browser_struct *tb,
	void *editor
);
int TexBrowserPreviewEnableContext(ma_texture_browser_struct *tb);
void TexBrowserUpdateMenus(ma_texture_browser_struct *tb);
void TexBrowserReset(ma_texture_browser_struct *tb, gbool unmap);
void TexBrowserDestroy(ma_texture_browser_struct *tb);

void TexBrowserMap(ma_texture_browser_struct *tb);
void TexBrowserUnmap(ma_texture_browser_struct *tb);


#define TITLE	": Texture Browser"


/*
 *	Blocks input and sets texture browser as marked busy.
 */
void TexBrowserSetBusy(ma_texture_browser_struct *tb)
{
        GtkWidget *w;
        GdkCursor *cur;

        if(tb == NULL)
            return;
 
        w = tb->toplevel;
        if(w == NULL)
            return;

        cur = tb->busy_cur;  
        if(cur == NULL)
            return;

        if(GTK_WIDGET_NO_WINDOW(w))
            return;

        gdk_window_set_cursor(w->window, cur);
	gdk_flush();

        return;
}

/*
 *      Allows input and sets texture browser as ready for input.
 */
void TexBrowserSetReady(ma_texture_browser_struct *tb)
{
        GtkWidget *w;

        if(tb == NULL)
            return;

        w = tb->toplevel;
        if(w == NULL)
            return;

        if(GTK_WIDGET_NO_WINDOW(w))
            return;

        gdk_window_set_cursor(w->window, NULL);
	gdk_flush();

        return;
}



/*
 *	Builds the menus for the texture browser.
 *
 *	Parent assumed to be a handle bar.
 */
int TexBrowserBuildMenuBar(
	ma_texture_browser_struct *tb,
	GtkWidget *parent
)
{
        GtkWidget *menu_bar, *menu, *w, *fw;
        int accel_key;
        void *accel_group; 
        unsigned int accel_mods;
        void *client_data = tb;
        u_int8_t **icon;
        char *label = NULL;
        void (*func_cb)(GtkWidget *w, void *);


        /* Create menu bar. */
        menu_bar = (GtkWidget *)GUIMenuBarCreate(&accel_group);
	tb->menu_bar = menu_bar;
        gtk_container_add(GTK_CONTAINER(parent), menu_bar);
        gtk_widget_show(menu_bar);

#define DO_ADD_MENU_ITEM_LABEL	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  client_data, func_cb \
 ); \
}

#define DO_ADD_MENU_ITEM_CHECK	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_CHECK, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  client_data, func_cb \
 ); \
}

#define DO_ADD_MENU_SEP \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

        /* Texture menu. */
        menu = GUIMenuCreate();
        if(menu != NULL)
        {
            icon = (u_int8_t **)icon_select_20x20_xpm;
            label = "Select";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserSelectTextureCB;
            DO_ADD_MENU_ITEM_LABEL
	    tb->select_mi = w;

	    DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_add_20x20_xpm;
            label = "Add...";
            accel_key = 0; 
            accel_mods = 0;
            func_cb = TexBrowserAddTextureCB;
            DO_ADD_MENU_ITEM_LABEL
            tb->add_mi = w;

            icon = (u_int8_t **)icon_remove_20x20_xpm;
            label = "Remove";
            accel_key = 0; 
            accel_mods = 0;
            func_cb = TexBrowserRemoveTextureCB;
            DO_ADD_MENU_ITEM_LABEL
            tb->remove_mi = w;

	    DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_close_20x20_xpm;
            label = "Close";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserCloseMCB;
            DO_ADD_MENU_ITEM_LABEL
	}
        GUIMenuAddToMenuBar(
            menu_bar, menu,
            "Texture",
            GUI_MENU_BAR_ALIGN_LEFT
        );

        /* Edit. */
        menu = GUIMenuCreate();
        if(menu != NULL)
        {
            icon = NULL;
            label = "Select All";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserSelectAllCB;
            DO_ADD_MENU_ITEM_LABEL
            tb->select_all_mi = w;

            icon = NULL;
            label = "Unselect All";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserUnselectAllCB;
            DO_ADD_MENU_ITEM_LABEL
            tb->unselect_all_mi = w;

            DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_properties_20x20_xpm;
            label = "Texture Properties...";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserEditPropertiesCB;
            DO_ADD_MENU_ITEM_LABEL
	    tb->properties_mi = w;
        }
        GUIMenuAddToMenuBar(
            menu_bar, menu, 
            "Edit",
            GUI_MENU_BAR_ALIGN_LEFT
        );

        /* View. */
        menu = GUIMenuCreate();
        if(menu != NULL)
        {
            icon = NULL;
            label = "by Creation";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserSortMCB;
            DO_ADD_MENU_ITEM_CHECK
	    tb->sort_by_index_micheck = fw;
	    if(fw != NULL)
		gtk_check_menu_item_set_active(
		    GTK_CHECK_MENU_ITEM(fw), TRUE
		);

            icon = NULL;
            label = "by Name";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserSortMCB;
            DO_ADD_MENU_ITEM_CHECK
            tb->sort_by_name_micheck = fw;
            if(fw != NULL)
                gtk_check_menu_item_set_active(
                    GTK_CHECK_MENU_ITEM(fw), FALSE
                );

            icon = NULL;
            label = "by Path";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserSortMCB;
            DO_ADD_MENU_ITEM_CHECK
            tb->sort_by_path_micheck = fw;
            if(fw != NULL)
                gtk_check_menu_item_set_active(
                    GTK_CHECK_MENU_ITEM(fw), FALSE
                );

            icon = NULL;
            label = "by Priority";
            accel_key = 0;
            accel_mods = 0;
            func_cb = TexBrowserSortMCB;
            DO_ADD_MENU_ITEM_CHECK
            tb->sort_by_priority_micheck = fw;
            if(fw != NULL)
                gtk_check_menu_item_set_active(
                    GTK_CHECK_MENU_ITEM(fw), FALSE
                );

            DO_ADD_MENU_SEP

            icon = (u_int8_t **)icon_reload_20x20_xpm;
            label = "Reload"; 
            accel_key = 0;  
            accel_mods = 0; 
            func_cb = TexBrowserReloadCB;
            DO_ADD_MENU_ITEM_LABEL
        }
        GUIMenuAddToMenuBar(
            menu_bar, menu, 
            "View",
            GUI_MENU_BAR_ALIGN_LEFT
        );


#undef DO_ADD_MENU_SEP
#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_CHECK

        /* Attach accel group to toplevel window. */
        if((tb->toplevel != NULL) &&
           (accel_group != NULL)
        )
        {
            gtk_window_add_accel_group(
                GTK_WINDOW(tb->toplevel),
                (GtkAccelGroup *)accel_group
            );
        }

	return(0);
}

/*
 *	Builds the tool ribbon for the texture browser.
 *
 *	Parent assumed to be a handle box.
 */
int TexBrowserBuildToolRibbon(
	ma_texture_browser_struct *tb,
        GtkWidget *parent
)
{
	const gchar *msglist[] = VMA_MSGLIST_TEXBROWSER_TOOLTIPS;
	gint border_major = 5;
        GtkWidget *w, *parent2, *parent3;
        gint bw = 60, bh = 50;
        gint sw = 5, sh = -1;


	/* Main hbox. */          
        w = gtk_hbox_new(FALSE, 0);
	tb->general_tool_ribbon = w;
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_container_border_width(GTK_CONTAINER(w), 2);
        gtk_widget_show(w); 
        parent2 = w;

#define DO_ADD_SEPARATOR        \
{ \
 w = gtk_vbox_new(TRUE, 0); \
 gtk_widget_set_usize(w, sw, sh); \
 gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0); \
 gtk_widget_show(w); \
 parent3 = w; \
 w = gtk_vseparator_new(); \
 gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, border_major); \
 gtk_widget_show(w); \
}

	/* Begin creating buttons. */

        /* Add texture referance. */
        w = GUIButtonPixmapLabelV(
            (u_int8_t **)icon_add_20x20_xpm, "Add", NULL
        );
	tb->add_btn = w;
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_set_usize(w, bw, bh);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "clicked",
            GTK_SIGNAL_FUNC(TexBrowserAddTextureCB),
            tb
        );
	GUISetWidgetTip(
	    w,
	    MsgListMatchCaseMessage(
		msglist, VMA_MSGNAME_TEXBROWSER_TEXTURE_ADD
	    )
	);
        gtk_widget_show(w);

	/* Remove texture referance. */
        w = GUIButtonPixmapLabelV(
            (u_int8_t **)icon_remove_20x20_xpm, "Remove", NULL
        );
	tb->remove_btn = w;
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_set_usize(w, bw, bh);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "clicked",
            GTK_SIGNAL_FUNC(TexBrowserRemoveTextureCB),
            tb
        );
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_TEXBROWSER_TEXTURE_REMOVE
            )
        );
        gtk_widget_show(w);

	DO_ADD_SEPARATOR

        /* Edit texture referance properties. */
        w = GUIButtonPixmapLabelV(
            (u_int8_t **)icon_properties_20x20_xpm, "Properties", NULL
        );
        tb->properties_btn = w;
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_set_usize(w, bw, bh);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "clicked",
            GTK_SIGNAL_FUNC(TexBrowserEditPropertiesCB),
            tb
        );
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_TEXBROWSER_TEXTURE_PROPERTIES
            )
        );
        gtk_widget_show(w);

	DO_ADD_SEPARATOR

        /* Reload textures listing. */
        w = GUIButtonPixmapLabelV(  
            (u_int8_t **)icon_reload_20x20_xpm, "Reload", NULL
        );
        tb->reload_btn = w;
        gtk_button_set_relief(GTK_BUTTON(w), GTK_RELIEF_NONE);
        gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
        gtk_widget_set_usize(w, bw, bh);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "clicked",
            GTK_SIGNAL_FUNC(TexBrowserReloadCB),
            tb
        );
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_TEXBROWSER_TEXTURES_RELOAD
            )
        );
        gtk_widget_show(w);

#undef DO_ADD_SEPARATOR

	return(0);
}

/*
 *      Builds the base directory ribbon for the texture browser.
 *
 *      Parent assumed to be a handle box.
 */
int TexBrowserBuildBaseDirRibbon(
        ma_texture_browser_struct *tb,
        GtkWidget *parent
)
{
	const char *msglist[] = VMA_MSGLIST_TEXBROWSER_TOOLTIPS;
        GtkWidget *w, *parent2;


        /* Main table. */
        w = gtk_table_new(1, 4, FALSE);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent2 = w;

	/* Label. */
        w = gtk_label_new("Base Directory:");
        gtk_table_attach(GTK_TABLE(parent2), w,
            0, 1,
            0, 1,
            0,
            0,
            2, 2
        );
        gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
        gtk_widget_show(w);

	/* Text entry. */
        w = gtk_entry_new_with_max_length(PATH_MAX + NAME_MAX);
        tb->base_dir_entry = w;
        gtk_signal_connect(
            GTK_OBJECT(w),
            "activate",
            GTK_SIGNAL_FUNC(TexBrowserEnterCB),
            tb
        );
        gtk_table_attach(GTK_TABLE(parent2), w,
            1, 2,
            0, 1,  
            GTK_EXPAND | GTK_FILL | GTK_SHRINK,
            0,
            2, 2
        );
        gtk_entry_set_text(GTK_ENTRY(w), "");
        gtk_entry_set_editable(GTK_ENTRY(w), tb->base_dir_override);
        gtk_widget_show(w);

        /* Texture base directory browse button. */
        w = GUIButtonPixmap((u_int8_t **)icon_browse_20x20_xpm);
        tb->base_dir_browse_btn = w;
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(TexBrowserBrowseBaseDirCB),
            tb   
        );
        gtk_table_attach(GTK_TABLE(parent2), w,
            2, 3,
            0, 1,
            0,
            0,
            2, 2
        );
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(
                msglist, VMA_MSGNAME_TEXBROWSER_BASE_DIR_BROWSE
            )
        );
        gtk_widget_show(w);

        /* Override texture base dir check. */ 
        w = gtk_check_button_new_with_label("Override");
        tb->base_dir_override_check = w;
        gtk_table_attach(GTK_TABLE(parent2), w,
            3, 4,
            0, 1,
            0,
            0,
            2, 2
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(TexBrowserOverrideCheckCB),
            tb
        );
        GUISetWidgetTip(
            w,
            MsgListMatchCaseMessage(   
                msglist, VMA_MSGNAME_TEXBROWSER_BASE_DIR_OVERRIDE
            )
        );
        gtk_widget_show(w);


	return(0);
}

/*
 *      Builds the search ribbon for the texture browser.
 *
 *      Parent assumed to be a handle box.
 */
int TexBrowserBuildSearchRibbon(
        ma_texture_browser_struct *tb,
        GtkWidget *parent
)
{
        GtkWidget *w;
	void *ptr_rtn;

	w = (GtkWidget *)GUIComboCreate(
	    "Find:",		/* Label. */
	    NULL,		/* Initial value. */
	    NULL,		/* Items glist. */
	    10,			/* Max items. */
	    &ptr_rtn,		/* Combo widget return. */
	    tb,			/* Client data. */
	    TexBrowserEnterCB,	/* Activate callback. */
	    NULL		/* List change callback. */
	);
        tb->search_combo = (GtkWidget *)ptr_rtn;
	tb->search_ribbon = w;
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);

	return(0);
}


/*
 *	Initializes and creates a texture browser as needed.
 *
 *	Returns non-zero on error.
 */
int TexBrowserCreate(
	ma_texture_browser_struct *tb,
	void *editor
)
{
	int i;
	int	bw = (100 + (2 * 3)),
		bh = (30 + (2 * 3));
	int	toplevel_width, toplevel_height;
	int	preview_width = 128,
		preview_height = 128;
	int	chwidth[3];
        GtkWidget	*w, *fw, *parent, *parent2, *handle_box,
			*panel, *panel2, *scroll_parent, *menu;
	gchar *heading[4];
        int accel_key;
        void *accel_group;
        unsigned int accel_mods;
        u_int8_t **icon;
        char *label = NULL;
	void *mclient_data;
        void (*func_cb)(GtkWidget *w, void *);


	if(tb == NULL)
	    return(-1);

	/* Already initialized? */
	if(tb->initialized)
	    return(0);


	/* Get last toplevel size of texture browser. */
	toplevel_width = VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_TEXBROWSER_WIDTH
	);
	if(toplevel_width < 1)
	    toplevel_width = VMA_TEXBROWSER_DEF_WIDTH;
        toplevel_height = VMACFGItemListGetValueI(
            option, VMA_CFG_PARM_TEXBROWSER_HEIGHT
        );
        if(toplevel_height < 1) 
            toplevel_height = VMA_TEXBROWSER_DEF_HEIGHT;

	/* Get last preview size. */
        preview_width = VMACFGItemListGetValueI(
            option, VMA_CFG_PARM_TEXBROWSER_PREVIEW_WIDTH
        );
        if(preview_width < 1) 
            preview_width = 128;
        preview_height = VMACFGItemListGetValueI(
            option, VMA_CFG_PARM_TEXBROWSER_PREVIEW_HEIGHT
        );
        if(preview_height < 1)
            preview_height = 128;                     

	/* Clist heading widths. */
	/* Index. */
        chwidth[0] = VMACFGItemListGetValueI(
            option, VMA_CFG_PARM_TEXBROWSER_INDEX_CHWIDTH
        );
        if(chwidth[0] < 1)
            chwidth[0] = 40;
        chwidth[1] = VMACFGItemListGetValueI(
            option, VMA_CFG_PARM_TEXBROWSER_NAME_CHWIDTH
        );
        if(chwidth[1] < 1)   
            chwidth[1] = 100;
        chwidth[2] = VMACFGItemListGetValueI(
            option, VMA_CFG_PARM_TEXBROWSER_PATH_CHWIDTH
        );
        if(chwidth[2] < 1)
            chwidth[2] = 120;


        /* Set up basic values. */
        tb->initialized = TRUE;
        tb->map_state = FALSE;

	tb->list_sort_code = TEXBROWSER_LIST_SORT_ADDED;
	tb->selected_texture = NULL;
	tb->total_selected_textures = 0;

	tb->editor_ptr = editor;


        /* Load cursors. */
        tb->busy_cur = gdk_cursor_new(GDK_WATCH);


        /* Toplevel. */
        w = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        tb->toplevel = w;
        gtk_signal_connect(
            GTK_OBJECT(w), "key_press_event",
            GTK_SIGNAL_FUNC(TexBrowserEventCB),
            tb
        );
        gtk_signal_connect(
            GTK_OBJECT(w), "key_release_event",
            GTK_SIGNAL_FUNC(TexBrowserEventCB),
            tb
        );
        gtk_widget_add_events(
	    w,
            GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
        );
        gtk_widget_realize(w);
        GUISetWMIcon(w->window, (u_int8_t **)texbrowser_xpm);
        gtk_widget_set_usize( 
            w, toplevel_width, toplevel_height
        );
        gtk_window_set_policy(
            GTK_WINDOW(w),
            TRUE, TRUE, FALSE
        );
        if(!GTK_WIDGET_NO_WINDOW(w))
        { 
            GdkGeometry geometry;

            geometry.min_width = 100;
            geometry.min_height = 70;

            geometry.base_width = 0;
            geometry.base_height = 0;

            geometry.width_inc = 1;
            geometry.height_inc = 1;
/*
            geometry.min_aspect = 1.3;
            geometry.max_aspect = 1.3;
 */
            gdk_window_set_geometry_hints(
                w->window,
                &geometry,  
                GDK_HINT_MIN_SIZE |
                GDK_HINT_BASE_SIZE |
                /* GDK_HINT_ASPECT | */
                GDK_HINT_RESIZE_INC
            );
        }
        gtk_signal_connect(
            GTK_OBJECT(w),
            "delete_event",
            GTK_SIGNAL_FUNC(TexBrowserCloseCB),
            tb
        );
        gtk_signal_connect(
            GTK_OBJECT(w),
            "destroy",
            GTK_SIGNAL_FUNC(TexBrowserDestroyCB),
            tb
        );
        gtk_window_set_title(GTK_WINDOW(w), PROG_NAME TITLE);
        gtk_container_set_border_width(GTK_CONTAINER(w), 0);
        parent = w;


        /* Main vbox. */
        w = gtk_vbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(parent), w);
        gtk_widget_show(w);
        parent = w;


        /* Build menu, first creating a handle box to place it in. */
        handle_box = gtk_handle_box_new();
	tb->menu_bar_dock = handle_box;
        gtk_box_pack_start(GTK_BOX(parent), handle_box, FALSE, FALSE, 0);
        gtk_widget_show(handle_box);

        TexBrowserBuildMenuBar(tb, handle_box);


        /* Build general tool ribbon, first creating a handle box to
         * place it in.
         */
        handle_box = gtk_handle_box_new();
	tb->general_tool_ribbon_dock = handle_box;
        gtk_box_pack_start(GTK_BOX(parent), handle_box, FALSE, FALSE, 0);
        gtk_widget_show(handle_box);

        TexBrowserBuildToolRibbon(tb, handle_box);


        /* Build base directory ribbon, first creating a handle box to
         * place it in.
         */
        handle_box = gtk_handle_box_new();
        gtk_box_pack_start(GTK_BOX(parent), handle_box, FALSE, FALSE, 0);
        gtk_widget_show(handle_box);

        TexBrowserBuildBaseDirRibbon(tb, handle_box);


        /* Build search ribbon, first creating a handle box to
         * place it in.
         */
        handle_box = gtk_handle_box_new();
	tb->search_ribbon_dock = handle_box;
        gtk_box_pack_start(GTK_BOX(parent), handle_box, FALSE, FALSE, 0);
        gtk_widget_show(handle_box);

        TexBrowserBuildSearchRibbon(tb, handle_box);



	/* Horizontal separator. */
/*
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 2);
	gtk_widget_show(w);
 */

	/* Panel widget to hold preview widgets and textures
	 * clist widget.
	 */
	w = gtk_vpaned_new();
        gtk_paned_set_handle_size(GTK_PANED(w), VMA_DEF_PANED_HANDLE_SIZE);
        gtk_paned_set_gutter_size(GTK_PANED(w), VMA_DEF_PANED_GUTTER_SIZE);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
        gtk_widget_show(w);
        panel = w;

	w = gtk_hpaned_new();
        gtk_paned_set_handle_size(GTK_PANED(w), VMA_DEF_PANED_HANDLE_SIZE);
        gtk_paned_set_gutter_size(GTK_PANED(w), VMA_DEF_PANED_GUTTER_SIZE);
        gtk_paned_add1(GTK_PANED(panel), w);
        gtk_widget_show(w);
        panel2 = w;

	/* Vbox to hold preview table. */
	w = gtk_vbox_new(FALSE, 0);
	tb->preview_label_vbox = w;
	gtk_paned_add1(GTK_PANED(panel2), w);
	gtk_widget_set_usize(
	    w,
	    MAX(toplevel_width - preview_width, 10),
	    preview_height
	);
	gtk_widget_show(w);

	/* Leave table to hold preview labels NULL for now. */
	tb->preview_label_table = NULL;


	/* Create preview window. */
	w = TexBrowserPreviewCreate(tb);
	gtk_paned_add2(GTK_PANED(panel2), w);
	gtk_widget_show(w);


	/* Textures list. */
        w = gtk_scrolled_window_new(NULL, NULL);
        gtk_scrolled_window_set_policy(
            GTK_SCROLLED_WINDOW(w),
            GTK_POLICY_AUTOMATIC,
            GTK_POLICY_AUTOMATIC
        );
        gtk_paned_add2(GTK_PANED(panel), w);
        gtk_widget_show(w);
        scroll_parent = w;

	heading[0] = strdup("Index");
        heading[1] = strdup("Name");
	heading[2] = strdup("Path");
	heading[3] = strdup("Priority");
        w = gtk_clist_new_with_titles(4, heading);
	free(heading[0]);
        free(heading[1]);
        free(heading[2]);
        free(heading[3]);
        tb->textures_list = w;
        if(!GTK_WIDGET_NO_WINDOW(w))
        {
            /* Tie to button press callback (for mapping menu). */
            gtk_widget_add_events(
                w,
                GDK_BUTTON_PRESS_MASK
            );
            gtk_signal_connect(
                GTK_OBJECT(w),
                "button_press_event",
                GTK_SIGNAL_FUNC(TexBrowserMenuMapCB),
                tb
            );
        }
        gtk_clist_column_titles_active(GTK_CLIST(w));
	gtk_clist_set_sort_column(GTK_CLIST(w), 0);
        gtk_clist_set_selection_mode(GTK_CLIST(w), GTK_SELECTION_EXTENDED);
        gtk_clist_set_row_height(GTK_CLIST(w), VMA_LIST_ROW_SPACING);
        gtk_container_add(GTK_CONTAINER(scroll_parent), w);
	for(i = 0; i < 3; i++)
            gtk_clist_set_column_justification(
                GTK_CLIST(w),
                i, GTK_JUSTIFY_LEFT
            );
	gtk_clist_set_column_width(
	    GTK_CLIST(w),
	    0, chwidth[0]
	);
        gtk_clist_set_column_width(
            GTK_CLIST(w),
            1, chwidth[1]
        );
        gtk_clist_set_column_width(
            GTK_CLIST(w),
            2, chwidth[2]
        );

        gtk_clist_set_shadow_type(GTK_CLIST(w), GTK_SHADOW_IN);
        gtk_signal_connect(
            GTK_OBJECT(w),
            "select_row",
            GTK_SIGNAL_FUNC(TexBrowserListSelectCB),
            tb
        );
        gtk_signal_connect(
            GTK_OBJECT(w),
            "unselect_row",
            GTK_SIGNAL_FUNC(TexBrowserListUnselectCB),  
            tb
        );
        gtk_signal_connect(
            GTK_OBJECT(w),
            "click_column",
            GTK_SIGNAL_FUNC(TexBrowserListColumClickCB),
            tb
        );
        gtk_widget_show(w);


	/* Close on select check button. */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 5);
        gtk_widget_show(w);
        parent2 = w;

	w = gtk_check_button_new_with_label("Close On Select");
	tb->close_on_select_check = w;
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(TexBrowserCloseOnSelectCheckCB),
            tb
        );
	gtk_widget_show(w);


	/* Separator. */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
        gtk_widget_show(w);


	/* Texture select and close button. */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 5);
	gtk_widget_show(w);
	parent2 = w;

        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_select_20x20_xpm, "Select", NULL
        );
	tb->texture_select_btn = w;
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(TexBrowserSelectTextureCB),
            tb
        );
	gtk_widget_show(w);

        w = GUIButtonPixmapLabelH(
            (u_int8_t **)icon_close_20x20_xpm, "Close", NULL
        );
        tb->close_btn = w;
        gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
        gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
        gtk_signal_connect(
            GTK_OBJECT(w), "clicked",
            GTK_SIGNAL_FUNC(TexBrowserCloseMCB),
            tb
        );
        gtk_widget_show(w);



        /* Right click menu for textures list. */
        menu = (GtkWidget *)GUIMenuCreate();
        tb->textures_list_menu = menu;
        accel_group = NULL;
        mclient_data = tb;

#define DO_ADD_MENU_ITEM_LABEL  \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}

#define DO_ADD_MENU_SEP \
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

        icon = (u_int8_t **)icon_select_20x20_xpm;
        label = "Select";
        accel_key = 0;
        accel_mods = 0;
        func_cb = TexBrowserSelectTextureCB;
        DO_ADD_MENU_ITEM_LABEL
	tb->texture_list_select_mi = w;

	DO_ADD_MENU_SEP

        icon = (u_int8_t **)icon_add_20x20_xpm;
        label = "Add";
        accel_key = 0;
        accel_mods = 0;
        func_cb = TexBrowserAddTextureCB;
        DO_ADD_MENU_ITEM_LABEL

        icon = (u_int8_t **)icon_remove_20x20_xpm;
        label = "Remove";
        accel_key = 0;
        accel_mods = 0;
        func_cb = TexBrowserRemoveTextureCB;
        DO_ADD_MENU_ITEM_LABEL
        tb->texture_list_remove_mi = w;

        DO_ADD_MENU_SEP  

        icon = (u_int8_t **)icon_properties_20x20_xpm;
        label = "Properties...";
        accel_key = 0;
        accel_mods = 0;
        func_cb = TexBrowserEditPropertiesCB;
        DO_ADD_MENU_ITEM_LABEL
        tb->texture_list_properties_mi = w;

#undef DO_ADD_MENU_SEP
#undef DO_ADD_MENU_ITEM_LABEL




	/* Reset values. */
	TexBrowserReset(tb, FALSE);

        /* Realize preview. */
/*
        w = tb->preview.preview;
        if((w == NULL) ? 0 : !GTK_WIDGET_REALIZED(w))
            gtk_widget_realize(w);
 */

	return(0);
}

/*
 *      Makes the glarea preview widget on the texture browser
 *	into gl context.
 *
 *      Returns 0 on success or -1 on error.
 */
int TexBrowserPreviewEnableContext(ma_texture_browser_struct *tb)
{
        GtkWidget *w;
        gint status;
	ma_texbrowser_preview_struct *pv;


        if(tb == NULL)
            return(-1);

	pv = &tb->preview;

        if(!pv->realized)
            return(-1);

        w = pv->preview;
        if(w != NULL)
        {
            status = gtk_gl_area_make_current(GTK_GL_AREA(w));
            if(status)
                return(0);
            else
                return(-1);
        }

        return(-1);
}

/*
 *      Updates the menu items on the specified texture browser to
 *      match other values on the texture browser structure.
 */
void TexBrowserUpdateMenus(ma_texture_browser_struct *tb)
{
	static gbool reenterant = FALSE;
	GtkWidget *w, *w2;
	gbool state, sensitivity;
	ma_texbrowser_preview_struct *pv;
        int toolbar_btn_layout = 2;


	if(tb == NULL)
	    return;

        if(!tb->initialized)
            return;

        pv = &tb->preview;   

	if(reenterant)
	    return;
	else
	    reenterant = TRUE;


        /* Get toolbar button style. */
	toolbar_btn_layout = VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_TOOLBAR_STYLE
	);

#define SET_WIDGET_SENSITIVITY	\
{ \
 if(w != NULL) \
  gtk_widget_set_sensitive(w, sensitivity); \
}

#define UPDATE_CHECK_MENU_ITEM_STATE    \
{ \
 if(w != NULL) \
  gtk_check_menu_item_set_active( \
   GTK_CHECK_MENU_ITEM(w), state \
  ); \
}

#define SET_BUTTON_LAYOUT		\
{ \
 if(w != NULL) \
 { \
  switch(toolbar_btn_layout) \
  { \
   case 2: \
    GUIButtonChangeLayout(w, 1, 1); \
    gtk_widget_set_usize(w, 60, 50); \
    break; \
   case 1: \
    GUIButtonChangeLayout(w, 1, 0); \
    gtk_widget_set_usize(w, 30, 30); \
    break; \
   default: \
    GUIButtonChangeLayout(w, 0, 1); \
    gtk_widget_set_usize(w, -1, 30); \
    break; \
  } \
 } \
}

        /* Begin updating tool ribbon buttons. */
        w = tb->general_tool_ribbon;
        if(w != NULL)
        {
            w = tb->add_btn;
            SET_BUTTON_LAYOUT 

            w = tb->remove_btn;
            sensitivity = ((tb->total_selected_textures > 0) ?
                TRUE : FALSE
            );
            SET_WIDGET_SENSITIVITY
            SET_BUTTON_LAYOUT

            w = tb->properties_btn;
            sensitivity = ((tb->total_selected_textures == 1) ?
                TRUE : FALSE
            );
            SET_WIDGET_SENSITIVITY
            SET_BUTTON_LAYOUT

            w = tb->reload_btn;
            SET_BUTTON_LAYOUT 
        }
	w = tb->general_tool_ribbon_dock;
	if(w != NULL)
	    gtk_widget_queue_resize(w);


	/* Begin updating menus. */

	/* Texture menu. */

	/* Select texture. */
        w = tb->select_mi;
	sensitivity = ((tb->total_selected_textures == 1) ?
            TRUE : FALSE
        );
	SET_WIDGET_SENSITIVITY

        /* Remove texture. */
        w = tb->remove_mi;
        sensitivity = ((tb->total_selected_textures > 0) ?
            TRUE : FALSE
        );
        SET_WIDGET_SENSITIVITY


	/* Select all. */
	w = tb->select_all_mi;
	sensitivity = TRUE;
	SET_WIDGET_SENSITIVITY

        /* Unselect all. */
        w = tb->unselect_all_mi;
        sensitivity = TRUE;
        SET_WIDGET_SENSITIVITY

        /* Texture properties. */
        w = tb->properties_mi;
        sensitivity = ((tb->total_selected_textures == 1) ?
            TRUE : FALSE
        );
        SET_WIDGET_SENSITIVITY


	/* View menu. */

	/* Sort list check items. */
	switch(tb->list_sort_code)
	{
	  case TEXBROWSER_LIST_SORT_ADDED:
	    w2 = tb->sort_by_index_micheck;
	    break;

          case TEXBROWSER_LIST_SORT_NAME:
            w2 = tb->sort_by_name_micheck;
            break;

          case TEXBROWSER_LIST_SORT_PATH:
            w2 = tb->sort_by_path_micheck;
            break;

          case TEXBROWSER_LIST_SORT_PRIORITY:
            w2 = tb->sort_by_priority_micheck;
            break;

	  default:
	    w2 = NULL;
	    break;
	}

	w = tb->sort_by_index_micheck;
	state = ((w == w2) ? TRUE : FALSE);
	UPDATE_CHECK_MENU_ITEM_STATE

        w = tb->sort_by_name_micheck;
        state = ((w == w2) ? TRUE : FALSE);
        UPDATE_CHECK_MENU_ITEM_STATE

        w = tb->sort_by_path_micheck;
        state = ((w == w2) ? TRUE : FALSE);
        UPDATE_CHECK_MENU_ITEM_STATE

        w = tb->sort_by_priority_micheck;
        state = ((w == w2) ? TRUE : FALSE);
        UPDATE_CHECK_MENU_ITEM_STATE


	/* Textures list pop up menu. */

	/* Select texture. */
        w = tb->texture_list_select_mi;
        sensitivity = ((tb->total_selected_textures == 1) ?
            TRUE : FALSE
        );
        SET_WIDGET_SENSITIVITY

        /* Remove texture. */
        w = tb->texture_list_remove_mi;
        sensitivity = ((tb->total_selected_textures > 0) ?
            TRUE : FALSE
        );
        SET_WIDGET_SENSITIVITY

        /* Texture properties. */
        w = tb->texture_list_properties_mi;
        sensitivity = ((tb->total_selected_textures == 1) ?
            TRUE : FALSE
        );
        SET_WIDGET_SENSITIVITY


	/* Texture base directory entry and browse button. */
	w = tb->base_dir_entry;
	if(w != NULL)
	{
	    gtk_entry_set_editable(GTK_ENTRY(w), tb->base_dir_override);
	}
	w = tb->base_dir_browse_btn;
	sensitivity = tb->base_dir_override;
	SET_WIDGET_SENSITIVITY


	/* Other buttons. */

        /* Override texture base dir check button. */
        w = tb->base_dir_override_check;
        if(w != NULL)
            gtk_toggle_button_set_active(  
                GTK_TOGGLE_BUTTON(w), tb->base_dir_override
            );

	/* Close on select check button. */
	w = tb->close_on_select_check;
	if(w != NULL)
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w), tb->close_on_select
	    );

	/* Select button. */
	w = tb->texture_select_btn;
        sensitivity = ((tb->total_selected_textures == 1) ?
            TRUE : FALSE
        );
        SET_WIDGET_SENSITIVITY


	/* Widgets on preview window. */
	w = pv->zoom_in_btn;
	sensitivity = ((pv->tex == NULL) ? FALSE : TRUE);
        SET_WIDGET_SENSITIVITY

        w = pv->zoom_out_btn;
        sensitivity = ((pv->tex == NULL) ? FALSE : TRUE); 
        SET_WIDGET_SENSITIVITY

        w = pv->zoom_to_fit_btn;
        sensitivity = ((pv->tex == NULL) ? FALSE : TRUE);
        SET_WIDGET_SENSITIVITY

        w = pv->zoom_one_to_one_btn;
        sensitivity = ((pv->tex == NULL) ? FALSE : TRUE); 
        SET_WIDGET_SENSITIVITY

	/* Preview menu. */
        /* Zoom to fit. */
        w = pv->preview_zoom_to_fit_mi;
	sensitivity = ((pv->tex == NULL) ? FALSE : TRUE);
        SET_WIDGET_SENSITIVITY

        /* Zoom one to one. */
        w = pv->preview_zoom_one_to_one_mi;
        sensitivity = ((pv->tex == NULL) ? FALSE : TRUE);
        SET_WIDGET_SENSITIVITY

        /* Texture properties. */
        w = pv->preview_properties_mi;
        sensitivity = ((tb->total_selected_textures == 1) ?
            TRUE : FALSE
        );
        SET_WIDGET_SENSITIVITY



#undef UPDATE_CHECK_MENU_ITEM_STATE
#undef SET_WIDGET_SENSITIVITY
#undef SET_BUTTON_LAYOUT

	reenterant = FALSE;

	return;
}


/*
 *      Deallocates the texture browser's user loaded data.
 *      If unmap is set to true then the texture browser will
 *	be unmapped as needed.
 */
void TexBrowserReset(ma_texture_browser_struct *tb, gbool unmap)
{
	char *base_dir;
	GtkWidget *w;


        if(tb == NULL)
            return;
        
        if(!tb->initialized)
            return;

	/* Begin resetting values. */
	tb->close_on_select = TRUE;


	/* Override base dir set in model? */
	if(VMACFGItemListGetValueI(
	    option, VMA_CFG_PARM_TEXTURE_BASE_DIR_OVERRIDE
	))
	    tb->base_dir_override = TRUE;
	else
	    tb->base_dir_override = FALSE;

	/* Reset base dir text value. */
	base_dir = VMACFGItemListGetValueS(
            option, VMA_CFG_PARM_TEXTURE_BASE_DIR
        );

	/* Update base dir entry. */
	w = tb->base_dir_entry;
	if(w != NULL)
	{
	    gtk_entry_set_text(
		GTK_ENTRY(w), ((base_dir == NULL) ? "" : base_dir)
	    );
	    gtk_entry_set_position(GTK_ENTRY(w), 0);
	}

	/* Search combo. */
	w = tb->search_combo;
	GUIComboClearAll(w);


	/* Clear textures list. */
	TexBrowserListDeleteAll(tb);

	/* Reset textures list sort code. */
	tb->list_sort_code = TEXBROWSER_LIST_SORT_ADDED;
	TexBrowserListColumClickCB(tb->textures_list, 0, tb);

	/* Unload preview data. */
	TexBrowserPreviewUnload(tb);

        /* Update menus. */
        TexBrowserUpdateMenus(tb);


        /* Unmap the texture browser. */
        if(unmap)
            TexBrowserUnmap(tb);

	return;
}


/*
 *	Destroys a texture browser structure but does not deallocate
 *	the structure itself.
 */
void TexBrowserDestroy(ma_texture_browser_struct *tb)
{
	GtkWidget **w;
        GdkCursor **cur;


	if(tb == NULL)
	    return;

	if(tb->initialized)
	{
#define DO_DESTROY_CURSOR	\
{ \
 if((*cur) != NULL) \
 { \
  GdkCursor *tc = *cur; \
  (*cur) = NULL; \
  gdk_cursor_destroy(tc); \
 } \
}

#define DO_DESTROY_WIDGET	\
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Unload preview data and destroy preview window. */
	    TexBrowserPreviewDestroy(tb);

            /* Clear textures list. */
            TexBrowserListDeleteAll(tb);


	    /* Begin destroying widgets. */

	    /* Textures list menu. */
	    w = &tb->textures_list_menu;
	    tb->sort_by_index_micheck = NULL;
	    tb->sort_by_name_micheck = NULL;
	    tb->sort_by_path_micheck = NULL;
	    tb->sort_by_priority_micheck = NULL;
	    DO_DESTROY_WIDGET

	    /* Textures list. */
	    w = &tb->textures_list;
	    DO_DESTROY_WIDGET

	    /* Search combo. */
	    w = &tb->search_combo;
	    DO_DESTROY_WIDGET

	    w = &tb->search_ribbon;
	    DO_DESTROY_WIDGET


	    /* General tool ribbon. */
	    w = &tb->general_tool_ribbon;
	    DO_DESTROY_WIDGET

            w = &tb->general_tool_ribbon_dock;
            DO_DESTROY_WIDGET


	    /* Menu bar. */
	    w = &tb->menu_bar;
            DO_DESTROY_WIDGET

            w = &tb->menu_bar_dock;
            DO_DESTROY_WIDGET




	    /* Toplevel. */
	    w = &tb->toplevel;
	    DO_DESTROY_WIDGET

            /* Cursors. */   
            cur = &tb->busy_cur;
            DO_DESTROY_CURSOR

#undef DO_DESTROY_WIDGET
#undef DO_DESTROY_CURSOR

	}

	/* Clear texture browser structure. */
	memset(tb, 0x00, sizeof(ma_texture_browser_struct));
}


/*
 *      Maps (shows) the texture browser as needed.
 */
void TexBrowserMap(ma_texture_browser_struct *tb)
{
	GtkWidget *w;


        if(tb == NULL)
            return;

        if(!tb->initialized)
            return;

        if(!tb->map_state)
        {
            w = tb->texture_select_btn;
            if(w != NULL)
            {
                gtk_widget_grab_focus(w);
                gtk_widget_grab_default(w);
            }

	    w = tb->toplevel;
	    if(w != NULL)
		gtk_widget_show(w);

            tb->map_state = TRUE;
        }
}

/*
 *      Unmaps (shows) the texture browser as needed.
 */     
void TexBrowserUnmap(ma_texture_browser_struct *tb)
{
	GtkWidget *w;


        if(tb == NULL)
            return;

        if(!tb->initialized)
            return;

        if(tb->map_state)
        {
	    w = tb->toplevel;
	    if(w != NULL)
		gtk_widget_hide(w);

            tb->map_state = FALSE;
        }
}
