/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <string>
#include <map>
#include <set>
#include <queue>
#include <list>
/* -- */
#include "common/Errors.hpp"
/* -- */
#include "trace/values/Values.hpp"
#include "trace/EntityValue.hpp"
#include "trace/EntityTypes.hpp"
#include "trace/Trace.hpp"
/* -- */
#include "parser/Line.hpp"
#include "parser/Definition.hpp"
#include "parser/ParserEventPaje.hpp"
/* -- */
using namespace std;

ParserEventPaje::ParserEventPaje(){
}

ParserEventPaje::~ParserEventPaje(){
    _containers.clear();
}

void ParserEventPaje::store_event(const Definition &definition, Line &line, Trace &trace){

    String alias;
    bool   alias_is_initialized = false;
    String name;
    bool   name_is_initialized = false;
    String container_type;
    String source_container_type;
    String dest_container_type;
    String entity_type;
    Date   time;
    String type;
    String container;
    String value_string;
    Double value_double;
    String source_container;
    String dest_container;
    String key;

    map<std::string, Value *> extra_fields;

    unsigned int i = 1;
    const vector<Field> fields = definition.get_fields();
    const unsigned int number_of_values = fields.size();

    while(i < number_of_values+1) {

        string current_value;
        const string &current_field_name = fields[i-1]._name;
        const string &current_field_type = fields[i-1]._type;

        if (!line.item(i, current_value)) {
            Error::set(Error::_LINE_TOO_SHORT_EVENT, line.get_line_count(), Error::_WARNING);
            return;
        }

        else if (current_field_name == "Alias") {
            alias = current_value;
            alias_is_initialized = true;
        }

        else if (current_field_name == "Name") {
            name = current_value;
            name_is_initialized = true;
        }

        else if (current_field_name == "ContainerType") {
            container_type = current_value;
        }

        else if (current_field_name == "SourceContainerType") {
            source_container_type = current_value;
        }

        else if (current_field_name == "DestContainerType") {
            dest_container_type = current_value;
        }

        else if (current_field_name == "EntityType") {
            entity_type = current_value;
        }

        else if (current_field_name == "Time") {
            time = current_value;
            if(!time.is_correct()) {
                Error::set(Error::_INCOMPATIBLE_VALUE_IN_EVENT + current_value + " (expecting a \"date\")", line.get_line_count(), Error::_WARNING);
                return;
            }
        }

        else if (current_field_name == "Type") {
            type = current_value;
        }

        else if (current_field_name == "Container") {
            container = current_value;
        }

        else if (current_field_name == "Value") {
            if(current_field_type == "double") {
                value_double = current_value;
                if(!value_double.is_correct()) {
                    Error::set(Error::_INCOMPATIBLE_VALUE_IN_EVENT + current_value + " (expecting a \"double\")", line.get_line_count(), Error::_WARNING);
                    return;
                }
            }
            else {
                value_string = current_value;
            }
        }

        else if (current_field_name == "SourceContainer") {
            source_container = current_value;
        }

        else if (current_field_name == "DestContainer") {
            dest_container = current_value;
        }

        else if (current_field_name == "Key") {
            key = current_value;
        }

        else {
            Value *value = NULL;
            if(current_field_type == "string") {
                value = new String(current_value);
            }
            else if(current_field_type == "double") {
                value = new Double(current_value);
            }
            else if(current_field_type == "hex") {
                value = new Hex(current_value);
            }
            else if(current_field_type == "date") {
                value = new Date(current_value);
            }
            else if(current_field_type == "int") {
                value = new Integer(current_value);
            }
            else if(current_field_type == "color") {
                value = new Color(current_value);
            }
            else {
                Error::set(Error::_FIELD_TYPE_UNKNOWN + current_field_type, line.get_line_count(), Error::_WARNING);
                return;
            }
            
            if(!value->is_correct()) { // Check if the value is correct or not
                Error::set(Error::_INCOMPATIBLE_VALUE_IN_EVENT + current_value + " (expecting a \""+current_field_type+"\")", line.get_line_count(), Error::_WARNING);
                return;
            }
            
            extra_fields[current_field_name] = value;
        }

        i ++;
    }

    if(line.length() > i) {
        Error::set(Error::_EXTRA_TOKEN, line.get_line_count(), Error::_WARNING);
    }

    Name alias_name;

    if(alias_is_initialized){
        alias_name.set_alias(alias.to_string());
    }
    if (name_is_initialized){
        alias_name.set_name(name.to_string());
    }


    const string event_name = definition.get_event_name();

    if(event_name == "PajeDefineContainerType") {
        ContainerType *temp_container_type = trace.search_container_type(container_type);
        if(temp_container_type == 0 && container_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + container_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.define_container_type(alias_name, temp_container_type, extra_fields);
        }
    }

    else if(event_name == "PajeCreateContainer") {
        ContainerType *temp_container_type = trace.search_container_type(type);
        Container *temp_container = trace.search_container(container);
        if(temp_container_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.create_container(time, alias_name, temp_container_type, temp_container, extra_fields);
            // We store the container in the map
            _containers[alias_name.to_string()] = trace.search_container(alias_name.to_string());
        }
    }
    
    else if(event_name == "PajeDestroyContainer") {
        Container *temp_container = trace.search_container(alias_name.to_string());
        ContainerType *temp_container_type = trace.search_container_type(type);
        if(temp_container == 0 && alias_name.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + alias_name.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.destroy_container(time, temp_container, temp_container_type, extra_fields);
        }
    }

    else if(event_name == "PajeDefineEventType") {
        ContainerType *temp_container_type = trace.search_container_type(container_type);
        if(temp_container_type == 0 && container_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + container_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.define_event_type(alias_name, temp_container_type, extra_fields);
        }
    }

    else if(event_name == "PajeDefineStateType") {
        ContainerType *temp_container_type = trace.search_container_type(container_type);
        if(temp_container_type == 0 && container_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + container_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.define_state_type(alias_name, temp_container_type, extra_fields);
        }
    }

    else if(event_name == "PajeDefineVariableType") {
        ContainerType *temp_container_type = trace.search_container_type(container_type);
        if(temp_container_type == 0 && container_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + container_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.define_variable_type(alias_name, temp_container_type, extra_fields);
        }
    }

    else if(event_name == "PajeDefineLinkType") {
        ContainerType *temp_container_type = trace.search_container_type(container_type);
        ContainerType *temp_source_container_type = trace.search_container_type(source_container_type);
        ContainerType *temp_dest_container_type = trace.search_container_type(dest_container_type);
        if(temp_container_type == 0 && container_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + container_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_source_container_type == 0 && source_container_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + source_container_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_dest_container_type == 0 && dest_container_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER_TYPE + dest_container_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.define_link_type(alias_name, temp_container_type, temp_source_container_type, temp_dest_container_type, extra_fields);
        }
    }

    else if(event_name == "PajeDefineEntityValue") {
        EntityType *temp_entity_type = trace.search_entity_type(entity_type);
        if(temp_entity_type == 0 && entity_type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_ENTITY_TYPE + entity_type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.define_entity_value(alias_name, temp_entity_type, extra_fields);
        }
    }

    else if(event_name == "PajeSetState") {
        StateType   *temp_state_type   = trace.search_state_type(type);
        EntityValue *temp_entity_value = trace.search_entity_value(value_string, temp_state_type);

        Container   *temp_container = NULL;
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }

        if(temp_state_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_STATE_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.set_state(time, temp_state_type, temp_container, temp_entity_value, extra_fields);
        }
    }

    else if(event_name == "PajePushState") {
        StateType   *temp_state_type = trace.search_state_type(type);
        EntityValue *temp_entity_value = trace.search_entity_value(value_string, temp_state_type);

        Container   *temp_container = NULL; 
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }

        if(temp_state_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_STATE_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.push_state(time, temp_state_type, temp_container, temp_entity_value, extra_fields);
        }
    }

    else if(event_name == "PajePopState") {
        StateType *temp_state_type = trace.search_state_type(type);
        Container *temp_container = NULL; 
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }

        if(temp_state_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_STATE_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.pop_state(time, temp_state_type, temp_container, extra_fields);
        }
    }

    else if(event_name == "PajeNewEvent") {
        EventType   *temp_event_type = trace.search_event_type(type);
        EntityValue *temp_entity_value = trace.search_entity_value(value_string, temp_event_type);

        Container   *temp_container = NULL; 
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }

        if(temp_event_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_EVENT_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.new_event(time, temp_event_type, temp_container, temp_entity_value, extra_fields);
        }
    }

    else if(event_name == "PajeSetVariable") {
        VariableType *temp_variable_type = trace.search_variable_type(type);

        Container    *temp_container = NULL;
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }
        
        if(temp_variable_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_VARIABLE_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.set_variable(time, temp_variable_type, temp_container, value_double, extra_fields);
        }
    }

    else if(event_name == "PajeAddVariable") {
        VariableType *temp_variable_type = trace.search_variable_type(type);
        Container    *temp_container = NULL;
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }

        if(temp_variable_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_VARIABLE_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.add_variable(time, temp_variable_type, temp_container, value_double, extra_fields);
        }
    }

    else if(event_name == "PajeSubVariable") {
        VariableType *temp_variable_type = trace.search_variable_type(type);
        Container    *temp_container = NULL;
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }

        if(temp_variable_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_VARIABLE_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.sub_variable(time, temp_variable_type, temp_container, value_double, extra_fields);
        }
    }

    else if(event_name == "PajeStartLink") {
        LinkType    *temp_link_type = trace.search_link_type(type);
        EntityValue *temp_entity_value = trace.search_entity_value(value_string, temp_link_type);

        Container   *temp_container = NULL;
        Container   *temp_source_container = NULL;
        // temp_container
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }
        // temp_source_container
        if(_containers.find(source_container) != _containers.end()) {
            temp_source_container = _containers[source_container];
        }
        else {
            temp_source_container = trace.search_container(source_container);
            _containers[source_container] = temp_source_container;
        }


        if(temp_link_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_LINK_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_source_container == 0 && source_container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + source_container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.start_link(time, temp_link_type, temp_container, temp_source_container, temp_entity_value, key, extra_fields);
        }
    }

    else if(event_name == "PajeEndLink") {
        LinkType    *temp_link_type = trace.search_link_type(type);
        EntityValue *temp_entity_value = trace.search_entity_value(value_string, temp_link_type);

        Container   *temp_container = NULL;
        Container   *temp_dest_container = NULL;
        // temp_container
        if(_containers.find(container) != _containers.end()) {
            temp_container = _containers[container];
        }
        else {
            temp_container = trace.search_container(container);
            _containers[container] = temp_container;
        }
        // temp_dest_container
        if(_containers.find(dest_container) != _containers.end()) {
            temp_dest_container = _containers[dest_container];
        }
        else {
            temp_dest_container = trace.search_container(dest_container);
            _containers[dest_container] = temp_dest_container;
        }

        if(temp_link_type == 0 && type.to_string() != "0"){
            Error::set(Error::_UNKNOWN_LINK_TYPE + type.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_container == 0 && container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else if(temp_dest_container == 0 && dest_container.to_string() != "0"){
            Error::set(Error::_UNKNOWN_CONTAINER + dest_container.to_string(), line.get_line_count(), Error::_ERROR);
        }
        else{
            trace.end_link(time, temp_link_type, temp_container, temp_dest_container, temp_entity_value, key, extra_fields);
        }
    }

    else {
        Error::set(Error::_UNKNOWN_EVENT_DEF + event_name, line.get_line_count(), Error::_WARNING);
        return;
    }
}

