/*
    vlad - an LDAP visualisation tool
    Copyright (C) 1999-2001 Robert Norris <rob@nauseum.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <ncurses.h>
#include <stdio.h>
#include <stdarg.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "vlad.h"

char buffer[VLAD_MAX_BUFFER];

static int MAX_Y, MAX_X;

static WINDOW *titlewin, *statuswin, *mainwin;

void set_title(char *format, ...) {
	va_list args;

	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);

	wmove(titlewin, 0, 0);
	wclrtoeol(titlewin);
	waddnstr(titlewin, buffer, MAX_X);

	wrefresh(titlewin);
	}

void clear_title(void) {
	wmove(titlewin, 0, 0);
	wclrtoeol(titlewin);
	wrefresh(titlewin);
	}

void set_status(char *format, ...) {
	va_list args;

	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);

	wmove(statuswin, 0, 0);
	wclrtoeol(statuswin);
	waddnstr(statuswin, buffer, MAX_X);

	wrefresh(statuswin);
	}

void clear_status(void) {
	wmove(statuswin, 0, 0);
	wclrtoeol(statuswin);
	wrefresh(statuswin);
	}

extern void tree_view(WINDOW *);

static void usage(void) {
	fputs("Vlad " VERSION " (" __DATE__ ")\n", stderr);
	fputs("Usage: vlad [options]\n"
	      "Options are:\n"
	      "     -h host       LDAP server hostname (required)\n"
	      "     -p port       LDAP server port (default: 389)\n"
	      "     -b base DN    base search DN (required)\n"
	      "     -D bind DN    DN to bind to server as\n"
	      "     -w password   password to bind to server with\n"
	      ,stderr);

	exit(0);
	};

struct options opt;

static void parse_options(int argc, char **argv) {
	char optchar;

	memset(&opt, 0, sizeof(struct options));

	while((optchar = getopt(argc, argv, "h:p:b:D:w:")) >= 0) {
		switch(optchar) {
			case 'h':
				opt.host = strdup(optarg);
				break;
			case 'p':
				opt.port = atol(optarg);
				break;
			case 'b':
				opt.basedn = strdup(optarg);
				break;
			case 'D':
				opt.binddn = strdup(optarg);
				break;
			case 'w':
				opt.bindpw = strdup(optarg);
				break;
			default:
				exit(1);
			}
		}

	if(!(opt.host && opt.basedn))
		usage();

	if(!opt.port)
		opt.port = 389;

	if(!opt.binddn) {
		opt.binddn = "";
		opt.bindpw = "";
		}

	if(!opt.bindpw) {
		opt.bindpw = "";
		}

	return;
	}

void cleanup_options(void) {
	if(opt.host)
		free(opt.host);

	if(opt.basedn)
		free(opt.basedn);

	if(opt.binddn)
		free(opt.binddn);

	if(opt.bindpw)
		free(opt.bindpw);
	}

int main(int argc, char **argv) {
	parse_options(argc, argv);

	(void) signal(SIGINT, SIG_IGN);

	initscr();

	curs_set(0);

	cbreak();
	noecho();

	if(has_colors()) {
		start_color();

		init_pair(0, COLOR_BLACK, COLOR_WHITE);
		init_pair(1, COLOR_WHITE, COLOR_BLACK);
		init_pair(2, COLOR_YELLOW, COLOR_BLUE);
		init_pair(3, COLOR_WHITE, COLOR_BLUE);
		}

	refresh();
	
	getmaxyx(stdscr, MAX_Y, MAX_X);

	titlewin = newwin(1, MAX_X, 0, 0);
	wbkgd(titlewin, COLOR_PAIR(2));
	wattrset(titlewin, A_BOLD);

	statuswin = newwin(1, MAX_X, MAX_Y - 1, 0);
	wbkgd(statuswin, COLOR_PAIR(2));
	wattrset(statuswin, A_BOLD);

	mainwin = newwin(MAX_Y - 2, MAX_X, 1, 0);
	wbkgd(mainwin, COLOR_PAIR(0));
	
	keypad(mainwin, TRUE);

	wrefresh(titlewin);
	wrefresh(statuswin);
	wrefresh(mainwin);

	tree_view(mainwin);

	endwin();

	cleanup_options();

	exit(0);
	}
