
// vbmakeglm.cpp
// hack to create voxbo glm's
// Copyright (c) 1998-2010 by The VoxBo Development Team

// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg
// written to accomplish the same thing as some original IDL code by
// Geoff Aguirre

using namespace std;

#include <stdio.h>
#include <iostream>
#include <fstream>
#include <sstream>

#include "vbutil.h"
#include "vbjobspec.h"
#include "vbprefs.h"
#include "vbio.h"
#include "glmutil.h"
#include "vbx.h"
#include <sys/wait.h>

VBPrefs vbp;

const int BUFLEN=1024;
bool f_ignorewarnings,f_validateonly,f_setuponly;
int32 f_run;
string glmname;

class GLMParams {
public:
  string name;              // name of the sequence
  string dirname;           // root directory
  string stem;              // stem name for analysis
  vector<string>scanlist;   // list of scans (data files)
  int lows,highs;           // low and high frequencies to remove
  string middles;           // middle frequencies to remove
  int pieces;               // n pieces for matrix jobs
  string kernelname;        // temporal smoothing kernel
  double kerneltr;          // sampling interval of kernel file
  string noisemodel;        // noise model
  string refname;           // file containing reference function
  string glmfile;           // filename of .glm file
  string gmatrix;           // g matrix file
  string email;             // email address
  vector<string> contrasts; // list of contrasts to add to contrasts.txt
  int pri;
  bool auditflag,meannorm,emailflag,meannormset,driftcorrect,driftcorrectset;
  double TR;
  int orderg;
  bool valid;
  bool rfxgflag;            // should we just dummy up a G matrix with all 1's?
  VBSequence seq;
  void init();
  int RunCheck();
  int CreateGLMDir();
  void FixRelativePaths();
  void CreateGLMJobs();
  void CreateGLMJobs2();
  vector<string> CreateGLMScript();
  int WriteGLMFile(string filename="");
  int WriteAndSubmitJobs();
  void Validate();
  int createsamplefiles();
};

class GLMConfig {
private:
  GLMParams global,local,*gp;
public:
  GLMConfig(string fname,double tr,double kr);
  int ParseFile(string filename);
};

void vbmakeglm_help();
void vbmakeglm_sample(FILE *);
void vbmakeglm_make_sample(const string &fname);
void vbmakeglm_show_sample();

int
main(int argc,char **argv)
{
  tokenlist args;
  vbp.init();
  vbp.read_jobtypes();
  f_run=vbp.cores;

  args.Transfer(argc-1,argv+1);
  if (args.size() == 0) {
    vbmakeglm_help();
    exit(0);
  }
  if (args[0]=="-h") {
    vbmakeglm_show_sample();
    exit(0);
  }
  if (args[0]=="-x") {
    if (args.size() > 1)
      vbmakeglm_make_sample(args[1]);
    else
      printf("vbmakeglm -x needs a filename to store the sample glm file\n");
    exit(0);
  }
  
  // okay, let's fly

  f_ignorewarnings=0;
  f_validateonly=0;
  f_setuponly=0;
  vector<string> filelist;
  int i;
  double x_tr=0.0,x_kr=0.0;

  for (i=0; i<args.size(); i++) {
    if (args[i]=="-i") {
      f_ignorewarnings=1;
      continue;
    }
    else if (args[i]=="-v") {
      f_validateonly=1;
      continue;
    }
    else if (args[i]=="-s") {
      f_setuponly=1;
      continue;
    }
    else if (args[i]=="--run") {
      f_run=ncores()-1;
      if (f_run<1) f_run=1;
    }
    else if (args[i].compare(0,6,"--run=")==0) {
      f_run=strtol(args[i].substr(6));
      if (f_run<1) f_run=1;
    }
    else if (args[i]=="-n" && (i+1)<args.size()) {
      glmname=args[i+1];
      i++;
      continue;
    }
    else if (args[i]=="-tr" && (i+1)<args.size()) {
      x_tr=strtod(args[i+1]);
      i++;
      continue;
    }
    else if (args[i]=="-kr" && (i+1)<args.size()) {
      x_kr=strtod(args[i+1]);
      i++;
      continue;
    }
    else {
      filelist.push_back(args[i]);
    }
  }
  for (i=0; i<(int)filelist.size(); i++) {
    GLMConfig *glmc=new GLMConfig(filelist[i],x_tr,x_kr);
    delete glmc;
  }
  exit(0);
}

// this is the sole constructor, it chews on the passed arguments

GLMConfig::GLMConfig(string fname,double tr,double kr)
{
  local.init();
  global.init();
  gp=&global;
  global.glmfile=fname;
  global.TR=tr;
  global.kerneltr=kr;
  ParseFile(fname);
}

void
GLMParams::init()
{
  name="";
  dirname="";
  stem="";
  scanlist.clear();
  lows=highs=0;
  pieces=0;
  kernelname="";
  kerneltr=0.0;
  noisemodel="";
  refname="";
  gmatrix="";
  email="";
  pri=3;
  auditflag=1;
  meannorm=0;
  driftcorrect=0;
  emailflag=1;
  TR=0.0;
  orderg=0;
  valid=0;
  seq.init();
  rfxgflag=0;
}

void
GLMParams::Validate()
{
  int i,errs,warns;
  int timecount;
  Tes *mytes;
  stringstream tmps;

  timecount=0;
  errs=0;
  warns=0;
  valid=0;

  printf("[I] vbmakeglm: validating GLM %s...\n",name.c_str());

  if (!(dirname.size())) {
    printf("[E] vbmakeglm: no dirname specified for glm %s\n",name.c_str());
    errs++;
  }

  for (i=0; i<(int)scanlist.size(); i++) {
    mytes=new Tes;
    mytes->ReadHeader(scanlist[i].c_str());
    if (!(mytes->header_valid)) {
      printf("[E] vbmakeglm: couldn't get info on data file %s.\n",scanlist[i].c_str());
      errs++;
    }
    else if ((string)"tes1" != mytes->GetFileFormat().getSignature()) {
      printf("[E] vbmakeglm: couldn't read 4D data from file %s.\n",scanlist[i].c_str());
      errs++;
    }
    else if (mytes->GetFileFormat().getDimensions() != 4) {
      printf("[E] vbmakeglm: %s isn't a 4D file.\n",scanlist[i].c_str());
      errs++;
    }
    if (mytes->header_valid) {
      timecount += mytes->dimt;
      if (i==0 && TR<1.0) {
        if (mytes->voxsize[3]>0.0)
          TR=mytes->voxsize[3];
      }
    }
    delete mytes;
  }
  // the number of time points ought to be the same as orderg
  if (orderg == 0)
    orderg=timecount;

  if (lows < 0 || lows > (orderg/2)) {
    printf("[W] vbmakeglm: removing %d low frequencies is a little suspect\n",lows);
    warns++;
  }

  if (highs < 0 || highs > 20) {
    printf("[W] vbmakeglm: removing %d high frequencies is a little suspect\n",highs);
    warns++;
  }

  if (noisemodel.size()) {
    Vec p(noisemodel.c_str());
    if (p.size() != 3) {
      printf("[E] vbmakeglm: your 1/f parameter file %s has the wrong number of elements\n",xfilename(noisemodel).c_str());
      errs++;
    }
  }

  if (kernelname.size()) {
    Vec IRF;
    if (IRF.ReadFile(kernelname.c_str())) {
      tmps.str("");
      tmps << "vbmakeglm: your HRF kernel file " << kernelname << " doesn't exist.";
      printErrorMsg(VB_ERROR,tmps.str());
      errs++;
    }
    else if (IRF.size() < 6 || IRF.size() > 20) {
      tmps.str("");
      tmps << "vbmakeglm: your HRF kernel file " << kernelname << " has "
           << IRF.size() << " elements, which seems a little suspect.";
      printErrorMsg(VB_WARNING,tmps.str());
      warns++;
    }
    else if (kerneltr < 1.0) {
      string trline=GetHeader(IRF.header,"TR(msecs)");
      if (trline.size()) {
        tokenlist tmpl;
        tmpl.ParseLine(trline);
        if (tmpl.size()>1)
          kerneltr=strtod(tmpl[1]);
      }
    }
  }

  // load the G matrix, check its order and see if it has an intercept of interest
  if (gmatrix.size()) {
    VBMatrix gmat(gmatrix);
    // gmat.MakeInCore();
    if (gmat.m <=0 || gmat.n <= 0) {
      tmps.str("");
      tmps << "vbmakeglm: couldn't read G (design) matrix " << gmatrix << ".";
      printErrorMsg(VB_ERROR,tmps.str());
      errs++;
    }
    orderg=gmat.m;
    
    if (gmat.n==1 && meannorm) {
      tmps.str("");
      tmps << "vbmakeglm: you have a single covariate and the mean norm flag set - make sure you don't "
           << "mean normalize data for second tier (group rfx) analyses.";
      printErrorMsg(VB_WARNING,tmps.str());
      warns++;
    }
  }
  
  if (rfxgflag && meannorm) {
    tmps.str("");
    tmps << "vbmakeglm: you have the makerrandfxg flag and the mean norm flag set - make sure you don't "
         << "mean normalize data for second tier (group rfx) analyses.";
    printErrorMsg(VB_WARNING,tmps.str());
    warns++;
  }

  if (meannorm && noisemodel.size()==0) {
    tmps.str("");
    tmps << "vbmakeglm: you have the mean norm flag set, and no noise model - make sure you really want"
         << "to mean normalize these data.";
    printErrorMsg(VB_WARNING,tmps.str());
    warns++;
  }

  if (!meannormset) {
    tmps.str("");
    tmps << "vbmakeglm: no meannorm flag set -- defaulting to no mean normalization";
    printErrorMsg(VB_WARNING,tmps.str());
  }

  if (orderg != timecount) {
    printf("[E] vbmakeglm: orderg (%d) doesn't match the number of timepoints (%d)\n",
           orderg,timecount);
    errs++;
  }

  // FIXME - should actually check order of G matrix
  if (gmatrix.size()==0 && !rfxgflag) {
    printf("[E] vbmakeglm: no valid G (design) matrix specified\n");
    errs++;
  }

  if (!errs)
    valid=1;
  if (warns && !f_ignorewarnings)
    valid=0;

  if (valid) {
    tmps.str("");
    tmps << "vbmakeglm: GLM " << name << " is good to go.";
    printErrorMsg(VB_INFO,tmps.str());
  }
  if (TR<1.0) {
    TR=2000;
    tmps.str("");
    tmps << "vbmakeglm: TR not set, using default of 2000ms.";
    printErrorMsg(VB_WARNING,tmps.str());
  }
  if (kerneltr<1.0 && kernelname.size()) {
    kerneltr=2000;
    tmps.str("");
    tmps << "vbmakeglm: HRF TR (sampling rate) not set, using default of 2000ms.";
    printErrorMsg(VB_WARNING,tmps.str());
  }
  // the following added to double-check on the TR bugs
  printf("[I] vbmakeglm: your data TR is %f\n",TR);
  printf("[I] vbmakeglm: your HRF kernel TR (sampling rate) is %f\n",kerneltr);
}

int
GLMParams::CreateGLMDir()
{
  string fname;   // temporary variable for wherever we need to build a filename
  int i;
  struct stat st;
  stringstream tmps;

  tokenlist stemstuff;
  stemstuff.SetSeparator(" \t\n\r/");
  stemstuff.ParseLine(dirname.c_str());
  if (!stemstuff.size()) {
    tmps << "vbmakeglm: couldn't find parent of GLM directory" << dirname << ".";
    printErrorMsg(VB_ERROR,tmps.str());
    return 101;
  }
  stem=dirname+(string)"/"+stemstuff[stemstuff.size()-1];

  // create directory
  mkdir(dirname.c_str(),0777);
  if (stat(dirname.c_str(),&st)) {
    tmps << "vbmakeglm: couldn't get to GLM directory " << dirname << ".";
    printErrorMsg(VB_ERROR,tmps.str());
    return 102;
  }
  // create sub file
  fname=stem+".sub";
  ofstream subfile(fname.c_str());
  if (!subfile) {
    tmps << "vbmakeglm: couldn't create sub file " << fname << ".";
    printErrorMsg(VB_ERROR,tmps.str());
    return 103;
  }
  subfile << ";VB98\n;TXT1\n;\n; Subject list generated by vbmakeglm\n;\n\n";
  for (i=0; i<(int)scanlist.size(); i++)
    subfile << scanlist[i] << endl;
  subfile.close();
  
  // create ref
  if (refname.size()) {
    if (copyfile(refname,stem+".ref")) {
      tmps << "vbmakeglm: couldn't copy reference function " << refname << ".";
      printErrorMsg(VB_ERROR,tmps.str());
      return 104;
    }
  }

  // create glm file for potential future use
  if (glmfile.size()) {
    if (WriteGLMFile(stem+".glm")) {
      tmps << "vbmakeglm: couldn't write glm configuration file " << (stem+".glm") << ".";
      printErrorMsg(VB_WARNING,tmps.str());
      tmps.str("");
    }
  }

  // copy G if necessary
  if (gmatrix.size() > 0) {
    if (copyfile(gmatrix,stem+".G")) {
      printf("[E] vbmakeglm: couldn't copy G matrix file %s.\n",gmatrix.c_str());
      return 105;
    }
  }
  else if (rfxgflag) {
    gmatrix=stem+".G";
    ofstream gstr(gmatrix.c_str(),ios::binary);
    if (gstr) {
      gstr << "VB98\nMAT1\n";
      gstr << "DataType:\tFloat\n";
      gstr << "VoxDims(XY):\t1\t" << orderg << endl << endl;
      gstr << "# This G matrix generated automatically by vbmakeglm\n\n";
      gstr << "Parameter:\t0\tInterest\tEffect\n";
      gstr << "\x0c\n";
      float pts[orderg];
      for (i=0; i<orderg; i++)
        pts[i]=1.0;
      if (my_endian() != ENDIAN_BIG)
        swap(pts,orderg);
      for (i=0; i<(int)(orderg * sizeof(float)); i++) {
        gstr << *((unsigned char *)pts+i);
      }
      gstr.close();
    }
  }
  createsamplefiles();   // after G matrix, so we can count vars of interest
  return 0;   // no error!
}

int
GLMParams::createsamplefiles()
{
  // first find variables of interest
  GLMInfo glmi;
  glmi.stemname=stem;
  glmi.getcovariatenames();
  string fname=dirname+"/contrasts.txt";
  vector<string> interestnames;
  if (access(fname.c_str(),R_OK) || contrasts.size()) {
    ofstream ofile(fname.c_str());
    if (ofile) {
      ofile << "# contrasts.txt\n";
      ofile << "# in this file you can define contrasts among your covariates of interest\n";
      if (glmi.cnames.size()) {
        ofile << "# your covariates of interest are:\n";
        for (size_t i=0; i<glmi.cnames.size(); i++) {
          if (glmi.cnames[i][0]=='I') {
            ofile << "#   "<<strnum(i)<<": "<<glmi.cnames[i].c_str()+1<<endl;
            interestnames.push_back(glmi.cnames[i].substr(1));
          }
        }
      }
      ofile << "# you can specify a complete contrast as follows:\n#\n";
      ofile << "# <name> <scale> vec";  // 0 0 1 0\n";
      ofile << " 1";
      for (size_t i=1; i<interestnames.size(); i++)
        ofile << " 0";
      ofile << endl << "#\n";
      ofile << "# (with one value for each covariate of interest)\n";
      ofile << "#\n";
      ofile << "# lines beginning with a '#' are comments!\n";
      ofile << "#\n";
      ofile << "# the following simple contrasts are provided for your convenience:\n";
      ofile << endl;
      for (size_t i=0; i<interestnames.size(); i++) {
        ofile << interestnames[i] << " t vec";
        for (size_t j=0; j<interestnames.size(); j++) {
          if (j==i) ofile << " 1";
          else ofile << " 0";
        }
        ofile << endl;
      }
      if (contrasts.size()) {
        ofile << "\n# the following contrasts were specified:\n";
        ofile << endl;
        for (size_t i=0; i<contrasts.size(); i++) {
          if (glmi.parsecontrast(contrasts[i]))
            printf("[W] vbgmakeglm: couldn't parse contrast: %s\n",contrasts[i].c_str());
          else
            ofile << contrasts[i] << endl;
        }
      }

      ofile.close();
    }
  }
  fname=dirname+"/averages.txt";
  if (access(fname.c_str(),R_OK)) {
    ofstream ofile(fname.c_str());
    if (ofile) {
      ofile << "# averages.txt\n";
      ofile << "# \n";
      ofile << "# In this file you can specify one or more ways to trial-average your data.\n";
      ofile << "# You can also block-average or whatever else you need, we just call it\n";
      ofile << "# trial averaging generically.\n";
      ofile << "# \n";
      ofile << "# Each trial average needs a separate section that looks like the following:\n";
      ofile << "# \n";
      ofile << "# average <name>\n";
      ofile << "#   units <time/vols>\n";
      ofile << "#   interval <ms/vols>\n";
      ofile << "#   nsamples <n>\n";
      ofile << "#   tr <ms>\n";
      ofile << "#   trial <n>...\n";
      ofile << "#   trialset <first> <interval> <count>\n";
      ofile << "# end\n";
      ofile << "# \n";
      ofile << "# Here's what these parameters mean:\n";
      ofile << "# \n";
      ofile << "# units: whether the other parameters are in volumes or seconds\n";
      ofile << "# interval: interval in time or volumes between samples within the trial\n";
      ofile << "# nsamples: number of time points to include per trial\n";
      ofile << "# tr: sets the TR if you're using time units\n";
      ofile << "#\n";
      ofile << "# The remaining options are two ways to indicate when the trials begin.\n";
      ofile << "# If your trials are evenly spaced, use 'trialset,' otherwise use 'trial'\n";
      ofile << "#\n";
      ofile << "# trialset: specify the start of the first trial, the interval between trial\n";
      ofile << "#     onsets, and the trial count\n";
      ofile << "# trial: each trial line lists one or more start times/vols for a trial\n";
      ofile << "#     (you can include multiple trial lines to help you keep the file neat)\n";
      ofile << "#\n";
      ofile << "# --> for trial and trialsets, the first volume is volume 0 (also time 0)\n";
      ofile << "# --> both time and volume values can be floating point\n";
      ofile << "#\n";
      ofile << "# Total data points for this GLM: "<<orderg<<endl;
      ofile << "# Your TR in ms: "<<TR<<endl;
      ofile << "# \n";

      ofile.close();
    }
  }
  return 0;
}

int
GLMConfig::ParseFile(string filename)
{
  ifstream infile;
  char buf[BUFLEN];
  tokenlist args;
  stringstream tmps;

  infile.open(filename.c_str());
  if (!infile) {
    tmps << "vbmakeglm: couldn't open " << filename;
    printErrorMsg(VB_ERROR,tmps.str());
    return 1;  // error!
  }
  while (infile.getline(buf,BUFLEN,'\n')) {
    args.ParseLine(buf);
    if (args.size()) {
      if (args[0][0]=='#' || args[0][0] == ';')
        continue;
      if (args[0]=="name" || args[0]=="glm") {
        local=global;  // start with a copy of the global
        gp=&local;
        gp->name=args.Tail();
      }
      else if (args[0]=="dirname") {
        if (args.size() == 2)
          gp->dirname=args[1];
        else {
          tmps.str("");
          tmps << "vbmakeglm: dirname takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="scan") {
        if (args.size() != 2) {
          tmps.str("");
          tmps << "vbmakeglm: scan takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
          continue;
        }
        gp->scanlist.push_back(args[1]);
      }
      else if (args[0]=="lows") {
        if (args.size() == 2)
          gp->lows=strtol(args[1]);
        else {
          tmps.str("");
          tmps << "vbmakeglm: lows takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="highs") {
        if (args.size() == 2)
          gp->highs=strtol(args[1]);
        else {
          tmps.str("");
          tmps << "vbmakeglm: highs takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="middles") {
        if (args.size() > 0)
          gp->middles=args[1];
      }
      else if (args[0]=="pieces") {
        if (args.size() == 2)
          gp->pieces=strtol(args[1]);
        else {
          printf("[E] vbmakeglm: pieces takes exactly one argument\n");
        }
      }
      else if (args[0]=="orderg") {
        if (args.size() == 2)
          gp->orderg=strtol(args[1]);
        else {
          tmps.str("");
          tmps << "vbmakeglm: orderg takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="refname") {
        if (args.size() == 2)
          gp->refname=args[1];
        else {
          printf("[E] vbmakeglm: reference takes exactly one argument\n");
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="kernel") {
        if (args.size() == 2)
          gp->kernelname=args[1];
        else if (args.size()==3) {
          gp->kernelname=args[1];
          gp->kerneltr=strtod(args[2]);
        }
        else {
          tmps.str("");
          tmps << "vbmakeglm: kernel takes one or two arguments";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="noisemodel") {
        if (args.size() == 2)
          gp->noisemodel=args[1];
        else {
          tmps.str("");
          tmps << "vbmakeglm: noisemodel takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="gmatrix") {
        if (args.size() == 2)
          gp->gmatrix=args[1];
        else {
          tmps.str("");
          tmps << "vbmakeglm: gmatrix takes exactly one argument"; 
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="makerandfxg") {
        gp->rfxgflag=1;
        if (args.size() > 1) {
          tmps.str("");
          tmps << "vbmakeglm: arguments to makerandfxg ignored";
          printErrorMsg(VB_WARNING,tmps.str());
        }
      }
      else if (args[0]=="tr") {
        if (args.size() == 2)
          gp->TR=strtol(args[1]);
        else {
          tmps.str("");
          tmps << "vbmakeglm: tr takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="pri") {
        if (args.size() == 2)
          gp->pri=strtol(args[1]);
        else {
          tmps.str("");
          tmps << "vbmakeglm: pri takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="audit") {
        if (args.size() != 2) {
          tmps.str("");
          tmps << "vbmakeglm: audit takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
        if (args[1]=="yes" || args[1]=="1")
          gp->auditflag=1;
        else if (args[1]=="no" || args[1]=="0")
          gp->auditflag=0;
        else {
          tmps.str("");
          tmps << "vbmakeglm: unrecognized value for audit flag (should be yes/no)";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="meannorm") {
        if (args.size() != 2) {
          tmps.str("");
          tmps << "vbmakeglm: meannorm takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
        if (args[1]=="yes" || args[1]=="1") {
          gp->meannorm=1;
          gp->meannormset=1;
        }
        else if (args[1]=="no" || args[1]=="0") {
          gp->meannorm=0;
          gp->meannormset=1;
        }
      }
      else if (args[0]=="driftcorrect") {
        if (args.size() != 2) {
          tmps.str("");
          tmps << "vbmakeglm: driftcorrect takes exactly one argument";
          printErrorMsg(VB_ERROR,tmps.str());
        }
        if (args[1]=="yes" || args[1]=="1") {
          gp->driftcorrect=1;
          gp->driftcorrectset=1;
        }
        else if (args[1]=="no" || args[1]=="0") {
          gp->driftcorrect=0;
          gp->driftcorrectset=1;
        }
        else {
          tmps.str("");
          tmps << "vbmakeglm: unrecognized value for driftcorrect flag (should be yes/no)";
          printErrorMsg(VB_ERROR,tmps.str());
        }
      }
      else if (args[0]=="email") {
        gp->emailflag=1;
        if (args.size() == 2)
          gp->email=args[1];
      }
      else if (args[0]=="go" || args[0]=="end") {
        if (!glmname.size() || glmname==gp->name) {
          gp->FixRelativePaths();
          gp->Validate();
          if (gp->valid && !f_validateonly) {
            if (!gp->CreateGLMDir()) {
              if (f_setuponly)
                ;
              else if (f_run) {
                gp->CreateGLMJobs2();
                runseq(vbp,gp->seq,f_run);
              }
              else {
                gp->CreateGLMJobs();
                gp->WriteAndSubmitJobs();
              }
            }
          }
        }
        gp=&global;
      }
      else if (args[0]=="contrast") {
        gp->contrasts.push_back(args.Tail());
      }
      else if (args[0]=="endx" || args[0]=="noend") {
        gp=&global;
      }
      else if (args[0]=="include") {
        if (args.size() == 2)
          ParseFile(args[1]);
      }
      else {
        tmps.str("");
        tmps << "vbmakeglm: unrecognized keyword " << args[0];
        printErrorMsg(VB_ERROR,tmps.str());
      }
    }
  }
  infile.close();
  return 0; // no error!
}

void
GLMParams::CreateGLMJobs()
{
  VBJobSpec js;
  int rowstart,rowfinish,mergeflag;
  char tmp[STRINGLEN];
  int i,jobnum=0;

  seq.name=name;

  // set pieces heuristically if the user didn't
  if (pieces==0) {
    // FIXME is this any good?
    int cellcount=600000;
    pieces=(int)ceil((double)orderg*orderg/cellcount);
    if (pieces > orderg) pieces=orderg;
    if (pieces<1) pieces=1;
  }
  int rowsperjob=orderg/pieces;
  // are we going to need the merge jobs?
  if (rowsperjob < orderg)
    mergeflag=1;
  else
    mergeflag=0;

  // make the exofilt
  js.init();
  js.jobtype="vb_makefilter";
  js.arguments["outfile"]=stem+".ExoFilt";
  // sprintf(tmp,"lowflag -lf %d",lows);
  js.arguments["lowflag"]=(string)"-lf "+strnum(lows);
  if (middles.size())
    js.arguments["middleflag"]=(string)"-mf "+middles;
  else
    js.arguments["middleflag"]="";
  sprintf(tmp,"highflag -hf %d",highs);
  js.arguments["highflag"]=(string)"-hf "+strnum(highs);
  if (kernelname.size())
    js.arguments["kernelflag"]=(string)"-k "+kernelname;
  else
    js.arguments["kernelflag"]="";
  js.arguments["ndata"]=strnum(orderg);
  sprintf(tmp,"tr %f",TR);
  js.arguments["tr"]=strnum(TR);
  js.magnitude=0;    // set it to something!
  js.name="make exofilt";
  js.jnum=jobnum++;
  int n_exofilt=js.jnum;
  seq.addJob(js);

  // make the noisemodel
  js.init();
  js.jobtype="vb_makenoisemodel";
  js.arguments["outfile"]=stem+".IntrinCor";
  if (noisemodel.size())
    js.arguments["noisemodelflag"]="-n "+noisemodel;
  js.arguments["ndata"]=strnum(orderg);
  js.arguments["tr"]=strnum(TR);
  js.magnitude=0;    // set it to something!
  js.name="make noisemodel";
  js.jnum=jobnum++;
  int n_noisemodel=js.jnum;
  seq.addJob(js);

  // make the KG matrix
  js.init();
  js.jobtype="makematkg";
  js.arguments["stem"]=stem;
  js.magnitude=0;    // set it to something!
  js.name="GLM-KG";
  js.waitfor.insert(n_exofilt);
  js.jnum=jobnum++;
  int n_kg=js.jnum;
  seq.addJob(js);

  // make the F1 matrix
  js.init();
  js.jobtype="matpinv";
  js.arguments["in"]=stem+".KG";
  js.arguments["out"]=stem+".F1";
  js.magnitude=0;    // set it to something!
  js.name="GLM-F1";
  js.waitfor.insert(n_kg);
  js.jnum=jobnum++;
  int n_f1=js.jnum;
  seq.addJob(js);

  // make the R matrix = I-(KG)(F1)
  js.init();
  js.jobtype="matimxy";
  js.arguments["in1"]=stem+".KG";
  js.arguments["in2"]=stem+".F1";
  js.arguments["out"]=stem+".R";
  js.magnitude=0;    // set it to something!
  js.name="GLM-R";
  js.waitfor.insert(n_f1);
  js.jnum=jobnum++;
  int n_r=js.jnum;
  seq.addJob(js);

  // create a K matrix
  js.init();
  js.jobtype="makematk";
  js.arguments["stem"]=stem;
  js.magnitude=0;    // set it to something!
  js.name="GLM-K";
  js.waitfor.insert(n_exofilt);
  js.waitfor.insert(n_noisemodel);
  js.jnum=jobnum++;
  int n_k=js.jnum;
  seq.addJob(js);

  // create an empty V matrix
  js.init();
  js.jobtype="matzeros";
  js.arguments["name"]=stem+".V"; 
  js.arguments["cols"]=strnum(orderg);
  js.arguments["rows"]=strnum(orderg);
  js.magnitude=0;    // set it to something!
  js.name="GLM-Vcreate";
  js.jnum=jobnum++;
  int n_vcreate=js.jnum;
  seq.addJob(js);

  // build V matrix (V=KKt)
  set<int32> n_vpieces;
  rowstart=0;
  i=0;
  int n_v;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    js.init();
    js.jobtype="matxxt";
    js.arguments["in"]=stem+".K";
    js.arguments["out"]=stem+".V";
    js.arguments["col1"]=strnum(rowstart);
    js.arguments["col2"]=strnum(rowfinish);
    js.magnitude=0;    // set it to something!
    sprintf(tmp,"GLM-V%d",i++);
    js.name=tmp;
    js.waitfor.insert(n_k);
    js.waitfor.insert(n_vcreate);
    js.jnum=jobnum;
    n_vpieces.insert(js.jnum);
    n_v=js.jnum;  // in case it's just one piece
    seq.addJob(js);
    jobnum++;
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge V
  if (mergeflag) {
    js.init();
    js.jobtype="matmerge";
    js.arguments["out"]=stem+".V";
    js.name="GLM-MergeV";
    js.waitfor=n_vpieces;
    js.jnum=jobnum++;
    n_v=js.jnum;
    seq.addJob(js);
  }
  
  // create F3
  js.init();
  js.jobtype="matf3";
  js.arguments["vmatrix"]=stem+".V";
  js.arguments["kgmatrix"]=stem+".KG";
  js.arguments["outfile"]=stem+".F3";
  js.magnitude=0;    // set it to something!
  js.name="GLM-makeF3";
  js.waitfor.insert(n_kg);
  js.waitfor.insert(n_v);
  js.jnum=jobnum++;
  int n_f3=js.jnum;
  seq.addJob(js);
  
  // create an empty RV matrix
  js.init();
  js.jobtype="matzeros";
  js.arguments["name"]=stem+".RV";
  js.arguments["cols"]=strnum(orderg);
  js.arguments["rows"]=strnum(orderg);
  js.magnitude=0;    // set it to something!
  js.name="GLM-RVcreate";
  js.jnum=jobnum++;
  int n_rvcreate=js.jnum;
  seq.addJob(js);

  // build RV matrix
  set<int32> n_rvpieces;
  rowstart=0;
  int n_rv;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    js.init();
    js.jobtype="matxy";
    js.arguments["in1"]=stem+".R";
    js.arguments["in2"]=stem+".V";
    js.arguments["out"]=stem+".RV";
    js.arguments["col1"]=strnum(rowstart);
    js.arguments["col2"]=strnum(rowfinish);
    js.magnitude=0;    // set it to something!
    sprintf(tmp,"GLM-RV%d",i);
    js.name=tmp;
    js.waitfor.insert(n_r);
    js.waitfor.insert(n_v);
    js.waitfor.insert(n_rvcreate);
    js.jnum=jobnum;
    n_rv=js.jnum;
    seq.addJob(js);
    n_rvpieces.insert(js.jnum);
    n_rv=js.jnum;    // in case it's one piece
    jobnum++;
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge RV
  if (mergeflag) {
    js.init();
    js.jobtype="matmerge";
    js.arguments["out"]=stem+".RV";
    js.name="GLM-MergeRV";
    js.waitfor=n_rvpieces;
    js.jnum=jobnum++;
    n_rv=js.jnum;
    seq.addJob(js);
  }
  
  // create an empty RVRV matrix
  js.init();
  js.jobtype="matzeros";
  js.arguments["name"]=stem+".RVRV";
  js.arguments["cols"]=strnum(orderg);
  js.arguments["rows"]=strnum(orderg);
  js.magnitude=0;    // set it to something!
  js.name="GLM-RVRVcreate";
  js.jnum=jobnum++;
  int n_rvrvcreate=js.jnum;
  seq.addJob(js);

  // build RVRV
  set<int32> n_rvrvpieces;
  int n_rvrv;
  rowstart=0;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    js.init();
    js.jobtype="matxx";
    js.arguments["in"]=stem+".RV";
    js.arguments["out"]=stem+".RVRV";
    js.arguments["col1"]=strnum(rowstart);
    js.arguments["col2"]=strnum(rowfinish);
    js.magnitude=0;    // set it to something!
    sprintf(tmp,"GLM-RVRV%d",i);
    js.name=tmp;
    js.waitfor.insert(n_rvrvcreate);
    js.waitfor.insert(n_rv);
    js.jnum=jobnum;
    n_rvrvpieces.insert(js.jnum);
    n_rvrv=js.jnum;
    seq.addJob(js);
    jobnum++;
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge RVRV
  if (mergeflag) {
    js.init();
    js.jobtype="matmerge";
    js.arguments["out"]=stem+".RVRV";
    js.name="GLM-MergeRVRV";
    js.waitfor=n_rvrvpieces;
    js.jnum=jobnum++;
    n_rvrv=js.jnum;
    seq.addJob(js);
  }

  // traces
  js.init();
  js.jobtype="comptraces";
  js.arguments["stem"]=stem;
  js.magnitude=0;    // set it to something!
  js.name="GLM-traces";
  js.waitfor.insert(n_rvrv);
  js.jnum=jobnum++;
  int n_traces=js.jnum;
  seq.addJob(js);
  
  // tes regression steps
  set<int32> n_regstep;
  for (i=0; i<pieces; i++) {
    js.init();
    js.jobtype="vbregress";
    js.arguments["stem"]=stem;
    js.arguments["steps"]=strnum(pieces);
    js.arguments["index"]=strnum(i+1);
    string flags;
    if (this->meannorm)
      flags+="-m ";
    if (this->driftcorrect)
      flags+="-d";
    js.arguments["flags"]=flags;
    js.magnitude=0;    // set it to something!
    sprintf(tmp,"regress(%d/%d)",i+1,pieces);
    js.name=tmp;
    js.waitfor.insert(n_traces);
    js.waitfor.insert(n_f1);
    js.waitfor.insert(n_r);
    js.waitfor.insert(n_exofilt);
    js.jnum=jobnum;
    n_regstep.insert(js.jnum);
    seq.addJob(js);
    jobnum++;
  } // for i
  
  if (pieces>1) {
    js.init();
    js.jobtype="vbmerge4d";
    js.arguments["stem"]=stem;
    js.arguments["ext"]="prm";
    js.magnitude=0;    // set it to something!
    js.name="mergeparams";
    js.waitfor=n_regstep;
    js.jnum=jobnum;
    seq.addJob(js);
    jobnum++;

    js.init();
    js.jobtype="vbmerge4d";
    js.arguments["stem"]=stem;
    js.arguments["ext"]="res";
    js.magnitude=0;    // set it to something!
    js.name="mergeparams";
    js.waitfor=n_regstep;
    js.jnum=jobnum;
    seq.addJob(js);
    jobnum++;

    // make these two jobs stand in for regstep in the waitfor chain
    n_regstep.clear();
    n_regstep.insert(jobnum-2);
    n_regstep.insert(jobnum-1);
  }
  
  // create smoothness estimate
  js.init();
  js.jobtype="vbse";
  js.arguments["stem"]=stem;
  js.magnitude=0;    // set it to something!
  js.name="vbse";
  js.waitfor=n_regstep;
  js.jnum=jobnum;
  int n_se=js.jnum;
  seq.addJob(js);
  jobnum++;

  // audit?
  int n_audit;
  if (auditflag) {
    js.init();
    js.jobtype="vb_auditglm";
    js.arguments["glmdir"]=xdirname(stem);
    js.magnitude=0;    // set it to something!
    js.name="GLM-audit";
    js.waitfor=n_regstep;
    js.waitfor.insert(n_se);
    js.jnum=jobnum++;
    n_audit=js.jnum;
    seq.addJob(js);
  }
  
  // notify?
  if (emailflag && email.size()) {
    js.init();
    js.jobtype="notify";
    js.arguments["email"]=email;
    js.arguments["msg"]="Your GLM has been solved.";
    js.magnitude=0;    // set it to something!
    js.name="Notify";
    js.waitfor.insert(n_se);
    if (auditflag)
      js.waitfor.insert(n_audit);
    js.waitfor.insert(n_f3);
    js.jnum=jobnum++;
    seq.addJob(js);
  }
}

void
GLMParams::CreateGLMJobs2()
{
  VBJobSpec js;
  int rowstart,rowfinish,mergeflag;
  char tmp[STRINGLEN];
  int i,jobnum=0;

  seq.name=name;

  // set pieces heuristically if the user didn't
  if (pieces==0) {
    // FIXME is this any good?
    int cellcount=600000;
    pieces=(int)ceil((double)orderg*orderg/cellcount);
    if (pieces > orderg) pieces=orderg;
    if (pieces<1) pieces=1;
  }
  int rowsperjob=orderg/pieces;
  // are we going to need the merge jobs?
  if (rowsperjob < orderg)
    mergeflag=1;
  else
    mergeflag=0;

  // make the exofilt
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmakefilter -e %s.ExoFilt -lf %d -hf %d %s %s -t %d %f")
                              %stem%lows%highs%(middles.size()?"-mf "+middles:"")
                              %(kernelname.size()?"-k "+kernelname:"")
                              %orderg%TR);
  js.name="make exofilt";
  js.jnum=jobnum++;
  int n_exofilt=js.jnum;
  seq.addJob(js);

  // make the noisemodel
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmakefilter -i %s.IntrinCor %s -t %d %f")
                              %stem%(noisemodel.size()?"-n "+noisemodel:"")%orderg%TR);
  js.name="make noisemodel";
  js.jnum=jobnum++;
  int n_noisemodel=js.jnum;
  seq.addJob(js);

  // make the KG matrix
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("makematkg -m %s")%stem);
  js.name="GLM-KG";
  js.waitfor.insert(n_exofilt);
  js.jnum=jobnum++;
  int n_kg=js.jnum;
  seq.addJob(js);

  // make the F1 matrix
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmm2 -pinv %1%.KG %1%.F1")%stem);
  js.name="GLM-F1";
  js.waitfor.insert(n_kg);
  js.jnum=jobnum++;
  int n_f1=js.jnum;
  seq.addJob(js);

  // make the R matrix = I-(KG)(F1)
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmm2 -imxy %1%.KG %1%.F1 %1%.R")%stem);
  js.name="GLM-R";
  js.waitfor.insert(n_f1);
  js.jnum=jobnum++;
  int n_r=js.jnum;
  seq.addJob(js);

  // create a K matrix
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("makematk -m %s")%stem);
  js.name="GLM-K";
  js.waitfor.insert(n_exofilt);
  js.waitfor.insert(n_noisemodel);
  js.jnum=jobnum++;
  int n_k=js.jnum;
  seq.addJob(js);

  // create an empty V matrix
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmm2 -zeros %s.V %d %d")%stem%orderg%orderg);
  js.name="GLM-Vcreate";
  js.jnum=jobnum++;
  int n_vcreate=js.jnum;
  seq.addJob(js);

  // build V matrix (V=KKt)
  set<int32> n_vpieces;
  rowstart=0;
  i=0;
  int n_v;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmm2 -xyt %1%.K %1%.K %1%.V %2% %3%")%stem%rowstart%rowfinish);
    sprintf(tmp,"GLM-V%d",i++);
    js.name=tmp;
    js.waitfor.insert(n_k);
    js.waitfor.insert(n_vcreate);
    js.jnum=jobnum;
    n_vpieces.insert(js.jnum);
    n_v=js.jnum;  // in case it's just one piece
    seq.addJob(js);
    jobnum++;
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge V
  if (mergeflag) {
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmm2 -assemblecols %s.V")%stem);
    js.name="GLM-MergeV";
    js.waitfor=n_vpieces;
    js.jnum=jobnum++;
    n_v=js.jnum;
    seq.addJob(js);
  }
  
  // create F3
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmm2 -f3 %1%.V %1%.KG %1%.F3")%stem);
  js.name="GLM-makeF3";
  js.waitfor.insert(n_kg);
  js.waitfor.insert(n_v);
  js.jnum=jobnum++;
  int n_f3=js.jnum;
  seq.addJob(js);
  
  // create an empty RV matrix
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmm2 -zeros %s.RV %d %d")%stem%orderg%orderg);
  js.name="GLM-RVcreate";
  js.jnum=jobnum++;
  int n_rvcreate=js.jnum;
  seq.addJob(js);

  // build RV matrix
  set<int32> n_rvpieces;
  rowstart=0;
  int n_rv;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmm2 -xyt %1%.R %1%.V %1%.RV %2% %3%")%stem%rowstart%rowfinish);
    sprintf(tmp,"GLM-RV%d",i);
    js.name=tmp;
    js.waitfor.insert(n_r);
    js.waitfor.insert(n_v);
    js.waitfor.insert(n_rvcreate);
    js.jnum=jobnum;
    n_rv=js.jnum;
    seq.addJob(js);
    n_rvpieces.insert(js.jnum);
    n_rv=js.jnum;    // in case it's one piece
    jobnum++;
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge RV
  if (mergeflag) {
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmm2 -assemblecols %s.RV")%stem);
    js.name="GLM-MergeRV";
    js.waitfor=n_rvpieces;
    js.jnum=jobnum++;
    n_rv=js.jnum;
    seq.addJob(js);
  }
  
  // create an empty RVRV matrix
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbmm2 -zeros %s.RVRV %d %d")%stem%orderg%orderg);
  js.name="GLM-RVRVcreate";
  js.jnum=jobnum++;
  int n_rvrvcreate=js.jnum;
  seq.addJob(js);

  // build RVRV
  set<int32> n_rvrvpieces;
  int n_rvrv;
  rowstart=0;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmm2 -xy %1%.RV %1%.RV %1%.RVRV %2% %3%")%stem%rowstart%rowfinish);
    sprintf(tmp,"GLM-RVRV%d",i);
    js.name=tmp;
    js.waitfor.insert(n_rvrvcreate);
    js.waitfor.insert(n_rv);
    js.jnum=jobnum;
    n_rvrvpieces.insert(js.jnum);
    n_rvrv=js.jnum;
    seq.addJob(js);
    jobnum++;
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge RVRV
  if (mergeflag) {
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmm2 -assemblecols %s.RVRV")%stem);
    js.name="GLM-MergeRVRV";
    js.waitfor=n_rvrvpieces;
    js.jnum=jobnum++;
    n_rvrv=js.jnum;
    seq.addJob(js);
  }

  // traces
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("comptraces -m %s")%stem);
  js.name="GLM-traces";
  js.waitfor.insert(n_rvrv);
  js.jnum=jobnum++;
  int n_traces=js.jnum;
  seq.addJob(js);
  
  // tes regression steps
  set<int32> n_regstep;
  string flags;
  if (this->meannorm)
    flags+="-m ";
  if (this->driftcorrect)
    flags+="-d";
  for (i=0; i<pieces; i++) {
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbregress %s -p %d %d %s")
                                %stem%(i+1)%pieces%flags);
    sprintf(tmp,"regress(%d/%d)",i+1,pieces);
    js.name=tmp;
    js.waitfor.insert(n_traces);
    js.waitfor.insert(n_f1);
    js.waitfor.insert(n_r);
    js.waitfor.insert(n_exofilt);
    js.jnum=jobnum;
    n_regstep.insert(js.jnum);
    seq.addJob(js);
    jobnum++;
  } // for i
  
  if (pieces>1) {
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmerge4d %1%.prm_part_* -o %1%.prm")%stem);
    js.name="mergeparams";
    js.waitfor=n_regstep;
    js.jnum=jobnum;
    seq.addJob(js);
    jobnum++;

    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("rm %s.prm_part_*")%stem);
    js.name="mergeparams";
    js.waitfor.insert(jobnum-1);
    js.jnum=jobnum;
    seq.addJob(js);
    jobnum++;

    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("vbmerge4d %1%.res_part_* -o %1%.res")%stem);
    js.name="mergeparams";
    js.waitfor=n_regstep;
    js.jnum=jobnum;
    seq.addJob(js);
    jobnum++;

    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("rm %s.res_part_*")%stem);
    js.name="mergeparams";
    js.waitfor.insert(jobnum-1);
    js.jnum=jobnum;
    seq.addJob(js);
    jobnum++;

    // make these four jobs stand in for regstep in the waitfor chain
    n_regstep.clear();
    n_regstep.insert(jobnum-4);
    n_regstep.insert(jobnum-3);
    n_regstep.insert(jobnum-2);
    n_regstep.insert(jobnum-1);
  }
  
  // create smoothness estimate
  js.init();
  js.jobtype="shellcommand";
  js.arguments["command"]=str(format("vbse %1%.res %1%.se")%stem);
  js.name="vbse";
  js.waitfor=n_regstep;
  js.jnum=jobnum;
  int n_se=js.jnum;
  seq.addJob(js);
  jobnum++;

  // audit?
  int n_audit;
  if (auditflag) {
    js.init();
    js.jobtype="shellcommand";
    js.arguments["command"]=str(format("glminfo -r %1% > %1%/audit.txt")%xdirname(stem));
    js.name="GLM-audit";
    js.waitfor=n_regstep;
    js.waitfor.insert(n_se);
    js.jnum=jobnum++;
    n_audit=js.jnum;
    seq.addJob(js);
  }
  
  // notify?
  if (emailflag && email.size()) {
    js.init();
    js.jobtype="notify";
    js.arguments["email"]=email;
    js.arguments["msg"]="Your GLM has been solved.";
    js.magnitude=0;    // set it to something!
    js.name="Notify";
    js.waitfor.insert(n_se);
    if (auditflag)
      js.waitfor.insert(n_audit);
    js.waitfor.insert(n_f3);
    js.jnum=jobnum++;
    seq.addJob(js);
  }
  for (SMI j=seq.specmap.begin(); j!=seq.specmap.end(); j++)
    j->second.dirname=dirname;
}

vector<string>
GLMParams::CreateGLMScript()
{
  VBJobSpec js;
  int rowstart,rowfinish,mergeflag;
  int i;
  string tmps;

  // set pieces heuristically if the user didn't
  if (pieces==0) {
    // FIXME is this any good?
    int cellcount=600000;
    pieces=(int)ceil((double)orderg*orderg/cellcount);
    if (pieces > orderg) pieces=orderg;
    if (pieces<1) pieces=1;
  }
  int rowsperjob=orderg/pieces;
  // are we going to need the merge jobs?
  if (rowsperjob < orderg)
    mergeflag=1;
  else
    mergeflag=0;

  vector<string> commandlist;

  // make the exofilt
  tmps=(format("vbmakefilter -e %s.ExoFilt -lf %d -hf %d %s %s -t %d %f")
        %stem%lows%highs%(middles.size()?"-mf "+middles:"")
        %(kernelname.size()?"-k "+kernelname:"")
        %orderg%TR).str();
  commandlist.push_back(tmps);
  // make the noisemodel
  tmps=(format("vbmakefilter -i %s.IntrinCor %s -t %d %f")
        %stem%(noisemodel.size()?"-n "+noisemodel:"")%orderg%TR).str();
  commandlist.push_back(tmps);
  // make the KG matrix
  tmps=(format("makematkg -m %s")%stem).str();
  commandlist.push_back(tmps);
  // make the F1 matrix
  tmps=(format("vbmm2 -pinv %1%.KG %1%.F1")%stem).str();
  commandlist.push_back(tmps);
  // make the R matrix = I-(KG)(F1)
  tmps=(format("vbmm2 -imxy %1%.KG %1%.F1 %1%.R")%stem).str();
  commandlist.push_back(tmps);
  // create a K matrix
  tmps=(format("makematk -m %s")%stem).str();
  commandlist.push_back(tmps);
  // create an empty V matrix
  tmps=(format("vbmm2 -zeros %s.V %d %d")%stem%orderg%orderg).str();
  commandlist.push_back(tmps);

  // build V matrix (V=KKt)
  set<int32> n_vpieces;
  rowstart=0;
  i=0;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    js.init();
    tmps=(format("vbmm2 -xyt %1%.K %1%.K %1%.V %2% %3%")%stem%rowstart%rowfinish).str();
    commandlist.push_back(tmps);
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge V
  if (mergeflag) {
    tmps=(format("vbmm2 -assemblecols %s.V")%stem).str();
    commandlist.push_back(tmps);
  }
  
  // create F3
  tmps=(format("vbmm2 -f3 %1%.V %1%.KG %1%.F3")%stem).str();
  commandlist.push_back(tmps);
  
  // create an empty RV matrix
  tmps=(format("vbmm2 -zeros %s.RV %d %d")%stem%orderg%orderg).str();
  commandlist.push_back(tmps);

  // build RV matrix
  rowstart=0;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    tmps=(format("vbmm2 -xyt %1%.R %1%.V %1%.RV %2% %3%")%stem%rowstart%rowfinish).str();
    commandlist.push_back(tmps);
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge RV
  if (mergeflag) {
    tmps=(format("vbmm2 -assemblecols %s.RV")%stem).str();
    commandlist.push_back(tmps);
  }
  
  // create an empty RVRV matrix
  tmps=(format("vbmm2 -zeros %s.RVRV %d %d")%stem%orderg%orderg).str();
  commandlist.push_back(tmps);

  // build RVRV
  rowstart=0;
  do {
    rowfinish=rowstart+rowsperjob;
    if (rowfinish > orderg-1)
      rowfinish=orderg-1;
    tmps=(format("vbmm2 -xy %1%.RV %1%.RV %1%.RVRV %2% %3%")%stem%rowstart%rowfinish).str();
    commandlist.push_back(tmps);
    rowstart+=rowsperjob+1;
  } while (rowstart < orderg);

  // merge RVRV
  if (mergeflag) {
    tmps=(format("vbmm2 -assemblecols %s.RVRV")%stem).str();
    commandlist.push_back(tmps);
  }

  // traces
  tmps=(format("comptraces -m %s")%stem).str();
  commandlist.push_back(tmps);
  
  // tes regression steps
  string flags;
  if (this->meannorm)
    flags+="-m ";
  if (this->driftcorrect)
    flags+="-d";
  for (i=0; i<pieces; i++) {
    tmps=(format("vbregress %s -p %d %d %s")
          %stem%(i+1)%pieces%flags).str();
    commandlist.push_back(tmps);
  }
  
  if (pieces>1) {
    tmps=str((format("vbmerge4d %1%.prm_part_* -o %1%.prm")%stem));
    commandlist.push_back(tmps);
    tmps=str((format("rm %s.prm_part_*")%stem));
    commandlist.push_back(tmps);

    tmps=str((format("vbmerge4d %1%.res_part_* -o %1%.res")%stem));
    commandlist.push_back(tmps);
    tmps=str((format("rm %s.res_part_*")%stem));
    commandlist.push_back(tmps);
  }
  
  // create smoothness estimate
  tmps=str((format("vbse %1%.res %1%.se")%stem));
  commandlist.push_back(tmps);

  // audit?
  if (auditflag) {
    tmps=str((format("glminfo -r %1% > %1%/audit.txt")%xdirname(stem)));
    commandlist.push_back(tmps);
  }
  return commandlist;
}

int
GLMParams::WriteAndSubmitJobs()
{
  // copy a few things to the sequence
  seq.priority=pri;
  seq.email=email;
  for (SMI j=seq.specmap.begin(); j!=seq.specmap.end(); j++)
    j->second.dirname=dirname;
  return seq.Submit(vbp);
}

int
GLMParams::WriteGLMFile(string fname)
{
  FILE *fp;
  if (fname.empty())
    fname=stem+".glm";
  fp=fopen(fname.c_str(),"w");
  if (!fp) {
    printf("[E] vbmakeglm: couldn't create glm file %s\n",fname.c_str());
    return 103;
  }
  fprintf(fp,"lows %d\n",lows);
  fprintf(fp,"highs %d\n",highs);
  if (middles.size())
    fprintf(fp,"middles %s\n",middles.c_str());
  fprintf(fp,"orderg %d\n",orderg);
  fprintf(fp,"pieces %d\n",pieces);
  fprintf(fp,"kernel %s\n",kernelname.c_str());
  fprintf(fp,"noisemodel %s\n",noisemodel.c_str());
  if (rfxgflag)
    fprintf(fp,"makerandfxg\n");
  else
    fprintf(fp,"gmatrix %s\n",gmatrix.c_str());
  if (refname.size())
    fprintf(fp,"refname %s\n",refname.c_str());
  fprintf(fp,"pri %d\n",pri);
  fprintf(fp,"audit %s\n",(auditflag ? "yes" : "no"));
  fprintf(fp,"meannorm %s\n",(meannorm ? "yes" : "no"));
  fprintf(fp,"driftcorrect %s\n",(driftcorrect ? "yes" : "no"));
  fprintf(fp,"email %s\n",email.c_str());
  fprintf(fp,"\n");
  fprintf(fp,"glm %s\n",name.c_str());
  fprintf(fp,"dirname %s\n",dirname.c_str());
  for (int i=0; i<(int)scanlist.size(); i++)
    fprintf(fp,"scan %s\n",scanlist[i].c_str());
  fprintf(fp,"end\n");
  fclose(fp);
  return 0;  
}

void
GLMParams::FixRelativePaths()
{
  string path=xgetcwd()+"/";
  dirname=xabsolutepath(dirname);
  kernelname=xabsolutepath(kernelname);
  noisemodel=xabsolutepath(noisemodel);
  refname=xabsolutepath(refname);
  gmatrix=xabsolutepath(gmatrix);
  for (size_t i=0; i<scanlist.size(); i++)
    scanlist[i]=xabsolutepath(scanlist[i]);
}

void
vbmakeglm_help()
{
  printf("\nVoxBo vbmakeglm (v%s)\n",vbversion.c_str());
  printf("usage: vbmakeglm [flags] file [file ...]\n");
  printf("flags:\n");
  printf("    -i            ignore warnings, queue it anyway\n");
  printf("    -v            don't queue, just validate\n");
  printf("    -s            setup only (create filters)\n");
  printf("    -tr <rate>    set data TR (ms)\n");
  printf("    -kr <rate>    set HRF kernel sampling rate (ms)\n");
  printf("    -h            i'm lazy, show me a sample .glm file\n");
  printf("    -n <name>     only actually submit glms matching name\n");
  printf("    -x <filename> i'm really lazy, make me a sample .glm file\n");
  printf("    --run=<n>     don't queue, run now on n cores\n");
  printf("notes:\n");
  printf("  If --run is specified without =n, the default is the total number of\n");
  printf("  available cores minus 1 (or 1, if that would be 0).\n");
  printf("\n");
}

void
vbmakeglm_make_sample(const string &fname)
{
  printf("Creating sample glm file in %s...",fname.c_str());  fflush(stdout);
  FILE *fp=fopen(fname.c_str(),"a");
  if (fp) {
    vbmakeglm_sample(fp);
    fclose(fp);
    printf("done.\n");
  }
  else {
    printf("failed.\n");
  }
}

void
vbmakeglm_show_sample()
{
  printf("Here's what a valid .glm file looks like, more or less:\n\n");

  vbmakeglm_sample(stdout);
}

void
vbmakeglm_sample(FILE *fp)
{
  fprintf(fp,"###############################################################\n");
  fprintf(fp,"# Sample .glm file for VoxBo vbmakeglm.\n");
  fprintf(fp,"# At the top, put stuff that's common to all GLMS in this file.\n");
  fprintf(fp,"###############################################################\n");
  fprintf(fp,"\n");

  fprintf(fp,"# how many low frequencies do you want to filter out?\n");
  fprintf(fp,"lows 1\n\n");
  fprintf(fp,"# how many high frequencies do you want to filter out?\n");
  fprintf(fp,"highs 1\n\n");
  fprintf(fp,"# which random frequencies fo you want to filter out? (separate by spaces)\n");
  fprintf(fp,"# middles 15 16 21\n\n");
  fprintf(fp,"# how many total data points per voxel (typically time points)?\n");
  fprintf(fp,"orderg 160\n\n");
  fprintf(fp,"# how many pieces do you want to break the matrix operations into?\n");
  fprintf(fp,"pieces 10\n\n");
  fprintf(fp,"# specify a kernel for exogenous smoothing by convolution\n");
  fprintf(fp,"kernel /usr/local/VoxBo/elements/filters/Eigen1.ref\n\n");
  fprintf(fp,"# specify a model of intrinsic noise\n");
  fprintf(fp,"noisemodel /usr/local/VoxBo/elements/noisemodels/smooth_params.ref\n\n");
  fprintf(fp,"# specify the location of your G matrix, or just \"makerandfxg\" for makerandfxg\n");
  fprintf(fp,"gmatrix /data/Models/mytask.G\n");
  fprintf(fp,"# makerandfxg\n\n");
  fprintf(fp,"# the location of a reference function to copy into the GLM directory\n");
  fprintf(fp,"refname /data/Models/motor.ref\n\n");
  fprintf(fp,"# priority (1 for overnight, 2 for low, 3 for normal, 4 and 5 for various emergencies)\n");
  fprintf(fp,"pri 3\n\n");
  fprintf(fp,"# do some summary statistics on your GLM?\n");
  fprintf(fp,"audit yes\n\n");
  fprintf(fp,"# mean normalize?  usually say yes only if you have multiple runs of BOLD data\n");
  fprintf(fp,"meannorm yes\n\n");
  fprintf(fp,"# correct linear drift?  \n");
  fprintf(fp,"driftcorrect yes\n\n");
  fprintf(fp,"# your email address here\n");
  fprintf(fp,"email nobody@nowhere.com\n\n");
  fprintf(fp,"\n");

  fprintf(fp,"###############################################################\n");
  fprintf(fp,"# Then create paragraphs of specific stuff for each GLM.\n");
  fprintf(fp,"# (you can also override globals here)\n");
  fprintf(fp,"###############################################################\n");
  fprintf(fp,"\n");

  fprintf(fp,"glm larry-glm1\n");
  fprintf(fp,"dirname /data/study/larry/glm1\n");
  fprintf(fp,"scan /data/study/larry/larry01/larry01.tes\n");
  fprintf(fp,"scan /data/study/larry/larry02/larry02.tes\n");
  fprintf(fp,"end\n");
  fprintf(fp,"\n");

  fprintf(fp,"glm moe-glm1\n");
  fprintf(fp,"dirname /data/study/moe/glm1\n");
  fprintf(fp,"scan /data/study/moe/moe01/moe01.tes\n");
  fprintf(fp,"scan /data/study/moe/moe02/moe02.tes\n");
  fprintf(fp,"end\n");
  fprintf(fp,"\n");

  fprintf(fp,"glm shemp-glm1\n");
  fprintf(fp,"dirname /data/study/shemp/glm1\n");
  fprintf(fp,"scan /data/study/shemp/shemp01/shemp01.tes\n");
  fprintf(fp,"scan /data/study/shemp/shemp02/shemp02.tes\n");
  fprintf(fp,"end\n");
  fprintf(fp,"\n");
}
