// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Jorgen Dahl                 dahlj@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: eclmplPhysicalCommunicationLayer.cpp
// 
//---------------------------------------------------------------------------

#include "eclmplPhysicalCommunicationLayer.h"
#include "warped/SerializedInstance.h" 

#define ONLY_CHECK_PHYSICAL_LAYER_ON_EMPTY_QUEUE
const int maxBuf = 65535;

eclmplPhysicalCommunicationLayer::eclmplPhysicalCommunicationLayer() {
  physicalId = 0;
  physicalSize = 0;
} // End of default constructor.

eclmplPhysicalCommunicationLayer::~eclmplPhysicalCommunicationLayer(){
} // End of desctructor.

void
eclmplPhysicalCommunicationLayer::physicalInit( SimulationConfiguration &configuration ) {
  int argc = configuration.getArguments().size();
  char **argv = new char*[argc];
  int i;
  for( i = 0; i < argc; i++ ){
    argv[i] = strdup( configuration.getArguments()[i].c_str() );
  }

  connInterface->establishConnections(&argc, &argv);
  physicalId = connInterface->getConnectionId();
  ASSERT( (physicalSize = connInterface->getNumberOfConnections()) > 1);
  initializeCommunicationLayerAttributes();

  for( i = 0; i < argc; i++ ){
    free( argv[i] );
  }
  delete [] argv;
} // End of physicalInit(...).

int
eclmplPhysicalCommunicationLayer::physicalGetId() const {
  return physicalId;
} // End of physicalGetId().

void
eclmplPhysicalCommunicationLayer::physicalSend( const SerializedInstance *toSend,
						unsigned int dest){

  const void *buffer = &toSend->getData()[0];
  unsigned int size = toSend->getSize();

  ECLMPL_ASSERT(buffer != NULL);
  ECLMPL_ASSERT(dest <= physicalSize);
  ECLMPL_ASSERT(dest != physicalId);
  ECLMPL_ASSERT(size <= connInterface->getMTU());

  connInterface->send(size, buffer, dest);
} // End of physicalSend(...).

SerializedInstance *
eclmplPhysicalCommunicationLayer::physicalProbeRecv(){
#ifdef ONLY_CHECK_PHYSICAL_LAYER_ON_EMPTY_QUEUE
  if (inOrderMessageQ.empty() == true) {
    probeNetwork();
  }
#else
    probeNetwork();
#endif
    return getNextInSequence();
} // End of physicalProbeRecv().

// const SerializedInstance *
// eclmplPhysicalCommunicationLayer::physicalProbeRecvBuffer(){
//   const SerializedInstance *retval = 0;

//   int msgSize;

//   // Get messages waiting in the network input queues and
//   // insert them in our buffering queues that ensure messages
//   // are delivered in FIFO and sequence order to the kernel.
//   probeNetwork();

//   // Peek next message and get it's size.
//   msgSize = peekNextInSequenceSize();
//   if (msgSize != -1) {
//     retval = getNextInSequence();
//   }

//   return retval;
// } // End of physicalProbeRecvBuf(...).

void
eclmplPhysicalCommunicationLayer::physicalFinalize(){
  connInterface->tearDownConnections();
} // End of physicalFinalize().

int
eclmplPhysicalCommunicationLayer::physicalGetSize() const {
  return physicalSize;
} // End of physicalGetSize().

SerializedInstance *
eclmplPhysicalCommunicationLayer::getNextInSequence() {
  SerializedInstance *retval = 0;

  NetworkMessage *nwMsg;
  char *msg = NULL;

  if (inOrderMessageQ.empty() == false) {
    nwMsg = inOrderMessageQ.front();
    inOrderMessageQ.pop_front();
    msg = nwMsg->getUserData();
    retval = new SerializedInstance( nwMsg->getUserData(), 
				     nwMsg->getUserDataSize() );
    delete nwMsg;
  } // End of if (inOrderMessageQIsEmpty == false).
  return retval;
} // End of getNextInSequence().

// Size -1 will be returned if no in-sequence message is avaiable.
int
eclmplPhysicalCommunicationLayer::peekNextInSequenceSize() {
  int size = -1;
  if (inOrderMessageQ.empty() == false) {
    size = inOrderMessageQ.front()->getUserDataSize();
  }
  return size;
} // End of peekNextInSequenceSize().

#ifdef ONLY_CHECK_PHYSICAL_LAYER_ON_EMPTY_QUEUE
#undef ONLY_CHECK_PHYSICAL_LAYER_ON_EMPTY_QUEUE
#endif
