/**************************************************************************
* Generic, native, relational database
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef TABLEINDEXES_H
#define TABLEINDEXES_H

#include "tabledata.h"

namespace RDB
{

/**
* General index for unique keys.
*
* Use this class for indexing rows of unknown type.
*/
class UniqueIndexBase
{
public:
    /**
    * Constructor.
    */
    UniqueIndexBase( TableData* data ) : m_data( data ) { }

    /**
    * Destructor.
    */
    ~UniqueIndexBase() { }

public:
    /**
    * @internal
    */
    TableData* data() const { return m_data; }

    /**
    * Find the row with the given key value.
    *
    * @param key Value of the key.
    *
    * @return The matching row or \c NULL if it doesn't exist.
    */
    Row* find( int key ) { return m_data->find( key ); }

    /**
    * @overload
    */
    const Row* find( int key ) const { return m_data->find( key ); }

protected:
    TableData* m_data;
};

/**
* Type-safe index for unique keys.
*
* Objects of this class are created by the tables for indexing rows using a unique key.
*/
template<class ROW>
class UniqueIndex : public UniqueIndexBase
{
public:
    /**
    * Constructor.
    */
    UniqueIndex( TableData* data ) : UniqueIndexBase( data ) { }

    /**
    * Destructor.
    */
    ~UniqueIndex() { }

public:
    ROW* find( int key ) { return (ROW*)m_data->find( key ); }
    const ROW* find( int key ) const { return (const ROW*)m_data->find( key ); }
};

/**
* General index for foreign keys.
*
* Use this class for indexing rows of unknown type.
*/
class ForeignIndexBase
{
public:
    /**
    * Constructor.
    */
    ForeignIndexBase( TableData* data ) : m_data( data ) { }

    /**
    * Destructor.
    */
    ~ForeignIndexBase() { }

public:
    /**
    * @internal
    */
    TableData* data() const { return m_data; }

protected:
    TableData* m_data;
};

/**
* Type-safe index for foreign keys.
*
* Objects of this class are created by the tables for indexing rows using a foreign key.
*
* The ForeignIterator and ForeignConstIterator can be used to find all items with a given
* value of the foreign key using the index.
*/
template<class ROW>
class ForeignIndex : public ForeignIndexBase
{
public:
    /**
    * Constructor.
    */
    ForeignIndex( TableData* data ) : ForeignIndexBase( data ) { }

    /**
    * Destructor.
    */
    ~ForeignIndex() { }
};

}

#endif
