/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _COLOR_H
#define _COLOR_H

#include <iosfwd>
#include <SDL/SDL_version.h>
#include <SDL/SDL_video.h>

#include <wftk/resources.h>

namespace wftk {
/// Wrapper class for describing colors
/// using 8-bits per red, green, blue, and alpha channels
class Color
{
 public:
  /**
   * The alpha channel values for transparent and opaque
   **/
  enum {
	WFTK_TRANSPARENT = SDL_ALPHA_TRANSPARENT, ///< \hideinitializer The alpha value for transparent
	WFTK_OPAQUE = SDL_ALPHA_OPAQUE ///< \hideinitializer The alpha value for opaque
  };

  /// Default constructor.  (opaque black)
  Color() : r(0), g(0), b(0), a(WFTK_OPAQUE) {}
  /// Construct from a triple/quadruple. (R, G, B, A)
  Color(Uint8 red, 
	Uint8 green,
	Uint8 blue,
	Uint8 alpha = WFTK_OPAQUE) : r(red), g(green), b(blue),a(alpha) {}
  /// Construct from SDL_Color
  Color(const SDL_Color& c) : r(c.r), g(c.g), b(c.b), a(WFTK_OPAQUE) {}
  /// Construct a named color (as find())
  Color(const std::string& name) {*this = Color::find(name);}
  /// same thing, so compiler will autocreate temporaries from "foo"
  Color(const char* name) {*this = Color::find(name);}
  /// Use the find method to retrieve a color by name (replaces use of static colors in wftk namespace)
  static const Color& find(const std::string&);
  
  /// Compares colors (without alpha channel !)
  friend bool operator==(const Color &c1, const Color &c2)
  {
    return c1.r==c2.r && c1.g==c2.g && c1.b==c2.b;
  }
  /// Inequality comparison
  friend bool operator!=(const Color &c1, const Color &c2)
  {
    return c1.r!=c2.r || c1.g!=c2.g || c1.b!=c2.b;
  }
  /// for use with std::map
  friend bool operator<(const Color& c1, const Color& c2)
  {
    return c1.r < c2.r || (c1.r == c2.r &&
          (c1.g < c2.g || (c1.g == c2.g && c1.b < c2.b)));
  }

  /// Operator for output streams
  friend std::ostream& operator<<(std::ostream& s, const Color& c);

  struct ResLoad {
    std::pair<Color,bool> operator()(const std::string&);
  };
  struct ResInval {
    typedef const Color& OutType;
    OutType operator()(const std::string& resname) const {return Color::find(resname);}
  };
  /** Load bitmapped surface from file
   *
   * Use the \b load() function to register a color resource, using a string specification
   * for the color, giving values for RGB or RGBA, as integers from 0 to 255, e.g.:
   * \code
   *   Color::registry.load("blue", "0 0 255");
   *   Color::registry.load("grey25", "128 128 128 64");
   * \endcode
   * Use the \b find() macro to retrieve the color Resource.  If the
   * named color has not been registered, find() will return a transparent color.
   * \code
   *   myWidget->setColor(&Color::registry.find("blue"));
   * \endcode
   */
  static ResourceRegistry<Color,ResLoad,ResInval> registry;
  /// refcounted resource
  typedef Resource<Color> Resource;

  /// Red
  Uint8 r;
  /// Green
  Uint8 g;
  /// Blue
  Uint8 b;
  /// Alpha
  Uint8 a;

};

}

#endif // !_COLOR_H
