/*
  libwftk - a C++ widget library based on SDL (Simple DirectMedia Layer)
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, May 1999  

#include "surface.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include "region.h"
#include "application.h"
#include "sdlhandler.h" // for SDLFatal
#include "rotozoom.h"

#include <math.h>
#ifndef M_PI
 #define M_PI 3.14159265358979323846 /* stupid, stupid w32 crosscompiler */
#endif

#include "sge_blib.h"
#include "sge_draw.h"

#include <assert.h>

#ifdef USE_SDL_OPENGL
#include "rootwindow.h"
  // shut up a redefinition warning
  #ifdef __WIN32__
  #undef NOMINMAX
  #endif
#include <SDL/SDL_opengl.h>
#endif

#if SDL_BYTEORDER == SDL_BIG_ENDIAN
#define TRANS_MASK 0xffffff00
#else
#define TRANS_MASK 0x00ffffff
#endif

namespace wftk {

ResourceRegistry<Surface*,Surface::ResLoad> Surface::registry;

static void
do_lock(SDL_Surface* sdlSurface_)
{
  /* hmmm.. I am unsure, it seems SDL_MUSTLOCK returns false
     on RLE-encoded surface which is not correct !
     so we lock all RLE encoded surfaces

     this has been fixed in SDL-1.1.4 !
  */

  //lock surface
  if(SDL_MUSTLOCK(sdlSurface_))
     {
      if(SDL_LockSurface(sdlSurface_) < 0)
  {
    SDL_Delay(10);
    if(SDL_LockSurface(sdlSurface_) < 0)
      throw Fatal("Surface::lock on surface failed twice");
  }
     }
}

static void
do_unlock(SDL_Surface* sdlSurface_)
{
  //unlock
  if(SDL_MUSTLOCK(sdlSurface_))
    {
      SDL_UnlockSurface(sdlSurface_);
    }
}

static SDL_Surface*
clone_surface(SDL_Surface* surf)
{
  if(!surf)
    return 0;

  SDL_Surface* new_surf
    = SDL_CreateRGBSurface(surf->flags,
         surf->w,
         surf->h,
         surf->format->BitsPerPixel,
         surf->format->Rmask,
         surf->format->Gmask,
         surf->format->Bmask,
         surf->format->Amask);

  if(!new_surf)
    throw SDLFatal("SDL_CreateRGBSurface");

  assert(new_surf->h == surf->h);
  assert(new_surf->pitch == surf->pitch);

  do_lock(new_surf);
  do_lock(surf);

  // don't copy the full pitch for the last row, we don't
  // need it and it may cause seg faults if the originating
  // surface is a child surface
  if(surf->h > 0)
    memcpy(new_surf->pixels, surf->pixels,
      (surf->h - 1) * surf->pitch + surf->w * surf->format->BytesPerPixel);

  do_unlock(surf);
  do_unlock(new_surf);

  SDL_SetColorKey(new_surf, surf->flags & (SDL_SRCCOLORKEY | SDL_RLEACCEL),
    surf->format->colorkey);
  SDL_SetAlpha(new_surf, surf->flags & SDL_SRCALPHA, surf->format->alpha);

  if(surf->format->palette && surf->format->BitsPerPixel == 8)
    SDL_SetColors(new_surf, surf->format->palette->colors, 0,
      surf->format->palette->ncolors);

  return new_surf;
}

Surface::Surface():
  sdlSurface_(0),
	glSurface_(0),
	glTexture_(0),
 	glTexMaxX_(0.0f),
	glTexMaxY_(0.0f),
 parent_(0)
{

}

Surface::Surface(unsigned w, unsigned h, const Pixelformat& pixelformat) :
  sdlSurface_(0),
	glSurface_(0),
	glTexture_(0),
	glTexMaxX_(0.0f),
	glTexMaxY_(0.0f),
  parent_(0)
{
  setSurface(w, h, pixelformat);
}

void
Surface::setSurface(unsigned w, unsigned h, const Pixelformat& pixelformat)
{
//  Debug out(Debug::DRAWING);

  if(!pixelformat.valid())
    return;

//  out << "Inside setSurface()" << Debug::endl;

  if(sdlSurface_)
    SDL_FreeSurface(sdlSurface_);

//  out << "About to allocate an SDL surface" << Debug::endl;

  sdlSurface_ = SDL_AllocSurface(SDL_SWSURFACE |SDL_SRCALPHA ,
         w,
         h,
         pixelformat.bitspp(),
         pixelformat.rMask(),
         pixelformat.gMask(),
         pixelformat.bMask(),
         pixelformat.aMask());

//  out << "Filling new surface with transparent color" << Debug::endl;

  if(sdlSurface_) {
    if(sdlSurface_->flags & SDL_SRCCOLORKEY)
      SDL_FillRect(sdlSurface_, 0, sdlSurface_->format->colorkey);
    else if(sdlSurface_->flags & SDL_SRCALPHA)
      fill(Color(0, 0, 0, Color::WFTK_TRANSPARENT));
  }
}

Surface::Surface(Surface& parent, const Rect& rect) :
  sdlSurface_(0),
	glSurface_(0),
	glTexture_(0),
	glTexMaxX_(0.0f),
	glTexMaxY_(0.0f),
  parent_(0)
{
  setAsChild(parent, rect);
}

Surface::Surface(const Surface& surf):
  sdlSurface_(clone_surface(surf.sdlSurface_)),
	glSurface_(clone_surface(surf.glSurface_)),
	glTexture_(surf.glTexture_),
 	glTexMaxX_(surf.glTexMaxX_),
	glTexMaxY_(surf.glTexMaxY_),
 parent_(0)
{

}
  
Surface::~Surface()
{
#ifdef USE_SDL_OPENGL
  if(glTexture_) 
		glDeleteTextures(1, &glTexture_);
#endif
	if(glSurface_)
		SDL_FreeSurface(glSurface_);
  if(sdlSurface_)
    SDL_FreeSurface(sdlSurface_);
  if(parent_)
    SDL_FreeSurface(parent_);
};

void 
Surface::setPalette(const SDL_Palette* pal)
{  
  if(!pal || !sdlSurface_)
    return;
    
  unsigned int count = pal->ncolors;

  if(count == 0)
    return;

  //accept only max 256 entries, there are only 8 bits for color indices
  const unsigned int max_count = 256;

  if(count > max_count)
    count = max_count;

  Uint32 have_alpha = sdlSurface_->flags & SDL_SRCALPHA;
  Uint8 alpha = have_alpha ? sdlSurface_->format->alpha : SDL_ALPHA_OPAQUE;

  Uint32 colorkey_mask = sdlSurface_->flags & (SDL_SRCCOLORKEY | SDL_RLEACCEL);
  Uint32 colorkey = sdlSurface_->format->colorkey;

  if(!SDL_SetColors(sdlSurface_,pal->colors,0,count)) {
    Debug::channel(Debug::DRAWING)
      << "Surface: unable to set palette properly. "<< Debug::endl;
  }

  SDL_SetAlpha(sdlSurface_, have_alpha, alpha);
  
  if(colorkey_mask & SDL_SRCCOLORKEY)
    SDL_SetColorKey(sdlSurface_, colorkey_mask, colorkey);
}

static bool
IsOpaque(Uint32 pixel, SDL_PixelFormat* format, Uint8 cutoff)
{
  Uint8 red, green, blue, alpha;
  SDL_GetRGBA(pixel, format, &red, &green, &blue, &alpha);
  return (SDL_ALPHA_OPAQUE) ? (alpha > cutoff) : (alpha < cutoff);
}

Region
Surface::opaqueRegion(Uint8 alpha_cutoff) const
{
  if(!sdlSurface_)
    return Region();

  if(!hasAlphaChannel() && !usesColorKey())
    return rect();

  Region r;

  lock();

  for(unsigned y = 0; y < height(); ++y) {
    unsigned begin_x = 0; // initial value unused, keep compiler happy
    bool current_opaque = false;
    Uint32 data = y * sdlSurface_->pitch;

    for(unsigned x = 0; x < width(); ++x) {
      Uint32 pixel = readPixel(data);
      bool is_opaque = hasAlphaChannel() ?
	IsOpaque(pixel, sdlSurface_->format, alpha_cutoff) :
        (pixel != sdlSurface_->format->colorkey);

      if(is_opaque && !current_opaque) {
        current_opaque = true;
        begin_x = x;
      }
      else if(!is_opaque && current_opaque) {
        current_opaque = false;
        r |= Rect(begin_x, y, x - begin_x, 1);
      }

      data += sdlSurface_->format->BytesPerPixel;
    }
    // end of row
    if(current_opaque)
      r |= Rect(begin_x, y, width() - begin_x, 1);
  }

  unlock();

  return r;
}



Surface& 
Surface::operator = (const Surface& surf)
{
  if(this != &surf)
    {
      if(sdlSurface_)
        SDL_FreeSurface(sdlSurface_);

      if(parent_)
        SDL_FreeSurface(parent_);
      parent_ = 0;

      //create a new surface
      sdlSurface_ = clone_surface(surf.sdlSurface_);
    }

  return *this;
}  

void
Surface::setAsChild(Surface& parent, const Rect& r)
{
  if(sdlSurface_)
    SDL_FreeSurface(sdlSurface_);
  sdlSurface_ = 0;

  if(parent_)
    SDL_FreeSurface(parent_);
  parent_ = 0;

  if(!parent.sdlSurface_)
    return;

  Rect rect = parent.rect().intersect(r);

  const SDL_PixelFormat &format = *parent.sdlSurface_->format;

  Uint8* addr = 
    (Uint8*)(parent.sdlSurface_->pixels) + 
    rect.origin().x * format.BytesPerPixel +
    rect.origin().y * parent.sdlSurface_->pitch;

  sdlSurface_ = SDL_CreateRGBSurfaceFrom(addr,
           rect.width(), rect.height(),
           format.BitsPerPixel,
           parent.sdlSurface_->pitch,
           format.Rmask,
           format.Gmask,
           format.Bmask,
           format.Amask);

  if(!sdlSurface_)
    throw SDLFatal("SDL_CreateRGBSurfaceFrom");

  SDL_SetColorKey(sdlSurface_, parent.sdlSurface_->flags &
    (SDL_SRCCOLORKEY | SDL_RLEACCEL), parent.sdlSurface_->format->colorkey);
  SDL_SetAlpha(sdlSurface_, parent.sdlSurface_->flags & SDL_SRCALPHA,
    parent.sdlSurface_->format->alpha);
      
  // add a reference to the underlying surface to the child,
  // SDL_FreeSurface() is really a refcount decrement
  parent_ = parent.parent_ ? parent.parent_ : parent.sdlSurface_;
  ++parent_->refcount;

  // Child surfaces are not an issue for passing sdlSurface_ to
  // Pixelformat(). True, if the wtfk::Surface is destroyed
  // before the Pixelformat is, the Pixelformat may end up
  // holding an SDL_Surface* with invalid pixel data. However,
  // Pixelformat only touches the 'format' and 'refcount'
  // members of the SDL_Surface, so keeping the pixel data
  // alive is irrelevant for Pixelformat.
}

void
Surface::doSetColorKey(const Color* color)
{
  if(sdlSurface_ == 0)
    return;

  Uint32 colorkey = color ? SDL_MapRGB(sdlSurface_->format, color->r,
  color->g, color->b) : sdlSurface_->format->colorkey;

  Uint32 flags = SDL_RLEACCEL & sdlSurface_->flags;
  if(color)
    flags |= SDL_SRCCOLORKEY;

  SDL_SetColorKey(sdlSurface_, flags, colorkey);
}

void
Surface::useRLEAcceleration(bool flag)
{
  if(!sdlSurface_)
    return;

  Uint32 flags = SDL_SRCCOLORKEY & sdlSurface_->flags;
  if(flag)
    flags |= SDL_RLEACCEL;

  SDL_SetColorKey(sdlSurface_, flags, sdlSurface_->format->colorkey);
}
  

Color
Surface::transparentColor() const
{
  Color out;

  if(sdlSurface_ && (sdlSurface_->flags & SDL_SRCCOLORKEY))
    SDL_GetRGBA(sdlSurface_->format->colorkey, sdlSurface_->format,
      &out.r, &out.g, &out.b, &out.a);

  return out;
}

void 
Surface::setAlpha(unsigned char alpha)
{
  if(!sdlSurface_)
    return;

  Uint32 flag = (alpha != SDL_ALPHA_OPAQUE || (sdlSurface_->flags & SDL_SRCALPHA))
    ? SDL_SRCALPHA : 0;

  SDL_SetAlpha(sdlSurface_, flag, alpha);
}

 
void 
Surface::lock() const
{
  do_lock(sdlSurface_);
}

void 
Surface::unlock() const
{
  do_unlock(sdlSurface_);
}

void
Surface::writePixel(Uint32 pixeladdr, Uint32 pixel)
{
  Uint8* screen_loc = (Uint8*)pixels()+pixeladdr;

  switch(sdlSurface_->format->BytesPerPixel)
    {
    case 1:
      *((Uint8 *)screen_loc) = pixel;
      break;
    case 2:
      *((Uint16 *)screen_loc) = pixel;
      break;
    case 3:      
      int shift;
      shift = sdlSurface_->format->Rshift;
      *(screen_loc+shift/8) = pixel>>shift;
      shift = sdlSurface_->format->Gshift;
      *(screen_loc+shift/8) = pixel>>shift;
      shift = sdlSurface_->format->Bshift;
      *(screen_loc+shift/8) = pixel>>shift;
      break;
    case 4:
      *((Uint32 *)screen_loc) = pixel;    
      break;
    default:
      break;
    }
}

Uint32 
Surface::readPixel(Uint32 pixeladdr) const
{
  Uint8* screen_loc = (Uint8*)pixels()+pixeladdr;
  
  Uint32 pixel=0;

  switch(sdlSurface_->format->BytesPerPixel)
    {
    case 1:
      pixel = *((Uint8 *)screen_loc);
      break;
    case 2:
      pixel = *((Uint16 *)screen_loc);
      break;
    case 3:
      int shift;
      // Im not sure on this, Karsten 01/1999
      shift = sdlSurface_->format->Rshift;
      pixel = *(screen_loc+shift/8) << shift;
      shift = sdlSurface_->format->Gshift;
      pixel |= *(screen_loc+shift/8) << shift;
      shift = sdlSurface_->format->Bshift;
      pixel |= *(screen_loc+shift/8) << shift;
      break;
    case 4:
      pixel = *((Uint32 *)screen_loc);    
      break;
    default:
      break;
    }

  return pixel;
}

// Lighten (g > 0) or darken (g < 0) the shade of
// all pixels in a surface. The value of g is added
// to each of the (red,green,blue) color values,
// where the colors are treated as if they were
// eight bit values (i.e. adding 127 will take black
// to grey and grey to white)
void
Surface::gammaShift(char g)
{
  if(g == 0 || !sdlSurface_)
    return;

  const SDL_PixelFormat *format = sdlSurface_->format;

  assert(format);

  int bytespp = format->BytesPerPixel;
  assert(bytespp == 2 || bytespp == 3 || bytespp == 4);

  int pitch = sdlSurface_->pitch;

  // deal with 3 bytes per pixel as a special case to avoid
  // alignment problems
  if(bytespp == 3) {
    // Assume one byte per color
    assert(format->Rmask >> format->Rshift == 0xff);
    assert(format->Gmask >> format->Gshift == 0xff);
    assert(format->Bmask >> format->Bshift == 0xff);

    lock();

    unsigned char* pixels = (unsigned char*) sdlSurface_->pixels;

    for(int x = 0; x < sdlSurface_->w * 3; ++x) {
      for(int y = 0; y < sdlSurface_->h; ++y) {
        unsigned char *pixelpos = pixels + x + y * pitch;

        if(g > 0) {
          Uint16 tmp = ((Uint16) g ) + *pixelpos;
          *pixelpos = (tmp > 0xff) ? 0xff : tmp;
        }
        else {
          *pixelpos = (*pixelpos > -g) ? *pixelpos + g : 0;
        }
      }
    }

    unlock();
    return;
  }

  Uint32 Rmask = format->Rmask;
  Uint32 Gmask = format->Gmask;
  Uint32 Bmask = format->Bmask;
  Uint32 Amask = format->Amask;

  // Instead of shifting the red bits in the pixel down to get
  // an eight bit value, adding, and shifting up again, we
  // shift g up so its most significant bit lines up with the
  // most significant bit in the red mask. Same for green, blue.
  // Since we need g > 0 to pull this trick, for the case g < 0
  // we shift -g to get a similar effect.
  
  Uint32 Rgam, Ggam, Bgam;

  if(g > 0) {
    Rgam = (((Uint32) g) >> format->Rloss) << format->Rshift;
    Ggam = (((Uint32) g) >> format->Gloss) << format->Gshift;
    Bgam = (((Uint32) g) >> format->Bloss) << format->Bshift;
  }
  else {
    Rgam = (((Uint32) (-g)) >> format->Rloss) << format->Rshift;
    Ggam = (((Uint32) (-g)) >> format->Gloss) << format->Gshift;
    Bgam = (((Uint32) (-g)) >> format->Bloss) << format->Bshift;
  }

  lock();

  unsigned char* pixels = (unsigned char*) sdlSurface_->pixels;

  // Walk through the surface, add g for each color
  for(Uint16 x = 0; x < sdlSurface_->w; ++x) {
    for(Uint16 y = 0; y < sdlSurface_->h; ++y) {
      unsigned char* pixelpos = pixels + x * bytespp + y * pitch;

      Uint32 pixel;

      if(bytespp == 2)
        pixel = *((const Uint16*) pixelpos);
      else // bytespp == 4
        pixel = *((const Uint32*) pixelpos);

      Uint32 red, green, blue, tmp;

      if(g > 0) {
        tmp = (pixel & Rmask);
        red = (Rmask - tmp > Rgam) ? Rgam + tmp : Rmask;

        tmp = (pixel & Gmask);
        green = (Gmask - tmp > Ggam) ? Ggam + tmp : Gmask;

        tmp = (pixel & Bmask);
        blue = (Bmask - tmp > Bgam) ? Bgam + tmp : Bmask;
      }
      else {
        tmp = pixel & Rmask;
        red = (tmp > Rgam) ? tmp - Rgam : 0;

        tmp = pixel & Gmask;
        green = (tmp > Ggam) ? tmp - Ggam : 0;

        tmp = pixel & Bmask;
        blue = (tmp > Bgam) ? tmp - Bgam : 0;
      }

      Uint32 outpixel = red | green | blue | (pixel & Amask);

      if(bytespp == 2)
        *((Uint16*) pixelpos) = outpixel;
      else // bytespp == 4
        *((Uint32*) pixelpos) = outpixel;
    }
  }

  unlock();
}

GammaFunction::GammaFunction() : offset_x(0), offset_y(0), shift_x(0), shift_y(0)
{
  memset(values, 0, 256*256);
}

void GammaFunction::read(char** xpm)
{
  offset_x = 0;
  offset_y = 0;

  Surface tmp;
  tmp.readFromXPM(xpm);

  // readFromXPM() is a libwftk function, so we know what format
  // we're reading into
  assert(tmp.pixelformat() == Pixelformat::ABGR8888);

  tmp.lock();

  unsigned char* pixels = (unsigned char*) tmp.pixels();
  int rowstride = tmp.pitch();
  int width = tmp.width();
  int height = tmp.height();

  for(int y = 0; y < 256; ++y)
    for(int x = 0; x < 256; ++x)
      values[x + y*256] = (x < width && y < height) ?
        (int)pixels[1 + x*4 + y*rowstride] - 128 : 0;

  tmp.unlock();
}

GammaFunction::GammaFunction(const GammaFunction& g)
  : offset_x(g.offset_x), offset_y(g.offset_y),
    shift_x(g.shift_x), shift_y(g.shift_y)
{
  memcpy(values, g.values, 256*256);
}

GammaFunction& GammaFunction::operator=(const GammaFunction& g)
{
  offset_x = g.offset_x;
  offset_y = g.offset_y;
  shift_x = g.shift_x;
  shift_y = g.shift_y;
  memcpy(values, g.values, 256*256);
  return *this;
}

// This works the same as the other gammaShift() function,
// except we use a texture instead of a constant
// value for g.
void
Surface::gammaShift(const GammaFunction& gamma)
{
  if(!sdlSurface_)
    return;

  const SDL_PixelFormat *format = sdlSurface_->format;

  assert(format);

  int bytespp = format->BytesPerPixel;
  assert(bytespp == 2 || bytespp == 3 || bytespp == 4);

  int pitch = sdlSurface_->pitch;

  // deal with 3 bytes per pixel as a special case to avoid
  // alignment problems
  if(bytespp == 3) {
    // Assume one byte per color
    assert(format->Rmask >> format->Rshift == 0xff);
    assert(format->Gmask >> format->Gshift == 0xff);
    assert(format->Bmask >> format->Bshift == 0xff);

    lock();

    unsigned char* pixels = (unsigned char*) sdlSurface_->pixels;

    for(int x = 0; x < sdlSurface_->w; ++x) {
      for(int y = 0; y < sdlSurface_->h; ++y) {
        Uint16 g = gamma.getPoint(x, y);
        if(g == 0)
          continue;

        for(int i = 0; i < 3; ++i) {
          unsigned char *pixelpos = pixels + i + x * 3 + y * pitch;

          if(g > 0) {
            Uint16 tmp = g + *pixelpos;
            *pixelpos = (tmp > 0xff) ? 0xff : tmp;
          }
          else {
            *pixelpos = (*pixelpos > -g) ? *pixelpos + g : 0;
          }
        }
      }
    }

    unlock();
    return;
  }

  Uint32 Rmask = format->Rmask;
  Uint32 Gmask = format->Gmask;
  Uint32 Bmask = format->Bmask;
  Uint32 Amask = format->Amask;

  // Start Rgam, Ggam, Bgam with values that match the
  // starting value of old_g
  Uint32 Rgam = 0, Ggam = 0, Bgam = 0;
  char old_g = 0;

  lock();

  unsigned char* pixels = (unsigned char*) sdlSurface_->pixels;

  for(Uint16 x = 0; x < sdlSurface_->w; ++x) {
    for(Uint16 y = 0; y < sdlSurface_->h; ++y) {

      char g = gamma.getPoint(x, y);
      if(g == 0)
        continue;

      if(g != old_g) { // only recompute if we need to

        // Instead of shifting the red bits in the pixel down to get
        // an eight bit value, adding, and shifting up again, we
        // shift g up so its most significant bit lines up with the
        // most significant bit in the red mask. Same for green, blue.
        // Since we need g > 0 to pull this trick, for the case g < 0
        // we shift -g to get a similar effect.

        if(g > 0) {
          Rgam = (((Uint32) g) >> format->Rloss) << format->Rshift;
          Ggam = (((Uint32) g) >> format->Gloss) << format->Gshift;
          Bgam = (((Uint32) g) >> format->Bloss) << format->Bshift;
        }
        else {
          Rgam = (((Uint32) (-g)) >> format->Rloss) << format->Rshift;
          Ggam = (((Uint32) (-g)) >> format->Gloss) << format->Gshift;
          Bgam = (((Uint32) (-g)) >> format->Bloss) << format->Bshift;
        }

        old_g = g;
      }

      unsigned char* pixelpos = pixels + x * bytespp + y * pitch;

      Uint32 pixel;

      if(bytespp == 2)
        pixel = *((const Uint16*) pixelpos);
      else // bytespp == 4
        pixel = *((const Uint32*) pixelpos);

      Uint32 red, green, blue, tmp;

      if(g > 0) {
        tmp = (pixel & Rmask);
        red = (Rmask - tmp > Rgam) ? Rgam + tmp : Rmask;

        tmp = (pixel & Gmask);
        green = (Gmask - tmp > Ggam) ? Ggam + tmp : Gmask;

        tmp = (pixel & Bmask);
        blue = (Bmask - tmp > Bgam) ? Bgam + tmp : Bmask;
      }
      else {
        Uint32 tmp;

        tmp = pixel & Rmask;
        red = (tmp > Rgam) ? tmp - Rgam : 0;

        tmp = pixel & Gmask;
        green = (tmp > Ggam) ? tmp - Ggam : 0;

        tmp = pixel & Bmask;
        blue = (tmp > Bgam) ? tmp - Bgam : 0;
      }

      Uint32 outpixel = red | green | blue | (pixel & Amask);

      if(bytespp == 2)
        *((Uint16*) pixelpos) = outpixel;
      else // bytespp == 4
        *((Uint32*) pixelpos) = outpixel;
    }
  }

  unlock();
}

Rect
Surface::blit(Surface& destBM) const
{
  if(!sdlSurface_ || !destBM.sdlSurface_)
    return Rect::invalid;

  SDL_Rect dst = {0, 0, 0, 0};
#ifdef USE_SDL_OPENGL
  if (destBM.sdlSurface_->flags & SDL_OPENGL) {
    Debug::channel(Debug::OPENGL) << "FIXME: Undefined openGL blit!" << Debug::endl;
  }
  else {
#endif
  // need this to return blitted area,
  // we're blitting to (0, 0), and the width and height
  // of this are ignored as input
    
    SDL_BlitSurface(sdlSurface_, 0, destBM.sdlSurface_, &dst);
#ifdef USE_SDL_OPENGL
  }
#endif
 
  return Rect(dst);
}

Rect
Surface::blit(Surface& destBM, const Rect& dest) const
{
  if(!sdlSurface_ || !destBM.sdlSurface_)
    return Rect::invalid;

  SDL_Rect dst(dest);
#ifdef USE_SDL_OPENGL 
  if (destBM.sdlSurface_->flags & SDL_OPENGL) {
    // Blit the entire surface to the x and y offset in the destination
    Debug::channel(Debug::OPENGL) << "FIXME: Undefined openGL blit #2!\n" << Debug::endl;
  }
  else {
#endif

    SDL_BlitSurface(sdlSurface_, 0, destBM.sdlSurface_, &dst);
#ifdef USE_SDL_OPENGL
  }
#endif

  return Rect(dst);

}

Rect
Surface::blit(Surface& destBM, const Rect& dest, const Rect& src_) const
{
  if(!sdlSurface_  || !destBM.sdlSurface_)
    return Rect::invalid;  
  
  Debug::channel(Debug::DRAWING)<<src_<<">>"<< dest<<Debug::endl;

  SDL_Rect src(src_), dst(dest);

#ifdef USE_SDL_OPENGL
  if (destBM.sdlSurface_->flags & SDL_OPENGL) {
    drawGL(src, dst);
  }
  else {
#endif
    SDL_BlitSurface(sdlSurface_, &src, destBM.sdlSurface_, &dst);
#ifdef USE_SDL_OPENGL
	}
#endif
  
  return Rect(dst);
}

void
Surface::blit(Surface& destBM, const Point& dest, const Region& destMask) const
{
   if(!sdlSurface_ || !destBM.sdlSurface_)
       return;

  Debug out(Debug::DRAWING);

  // just a quick check with the clipbox, to optimize away cases
  // where it's easy to tell there's nothing to draw

  Rect clip = destMask.getClipbox();
  Rect overlap = clip.intersect(Rect(dest.x, dest.y, width(), height()));
  if(overlap.isEmpty() || !overlap.isValid()) {
    out << "Region doesn't overlap source, bailing" << Debug::endl;
    return;
  }
  overlap = clip.intersect(Rect(0, 0, destBM.width(), destBM.height()));
  if(overlap.isEmpty() || !overlap.isValid()) {
    out << "Region doesn't overlap dest, bailing" << Debug::endl;
    return;
  }
   
  for(unsigned long i = 0; i < destMask.nRects(); ++i) {
    // need to copy rect into dst, since SDL modifies dst to match the
    // region actually blitted
    Rect dst = destMask.getRect(i), src(dst);
    src.x -= dest.x;
    src.y -= dest.y;
#ifdef USE_SDL_OPENGL
    // Have to check if GL is enabled
    if (destBM.sdlSurface_->flags & SDL_OPENGL) {
      // Blit with GL
      drawGL(src, dst);
    }
    else {
#endif
 			out << "dest Rect(" << dest.x << "+" <<  dst.x << ", " << dest.y <<"+"<< dst.y << ");\n";
      SDL_BlitSurface(sdlSurface_, &src, destBM.sdlSurface_, &dst);
#ifdef USE_SDL_OPENGL
		}
#endif
  }
}

#ifdef USE_SDL_OPENGL
void 
Surface::drawGL(SDL_Rect src, SDL_Rect dst) const
{
    // create the GL texture if it does not exist
    if (!glTexture_) {
     // Debug::channel(Debug::OPENGL) << "GL blit without texture, dst " << Rect(dst) << Debug::endl;
			// Draw this surface directly to screen
	    // create a temp SDL surface and blit to it
  	  Uint32 rmask, gmask, bmask, amask;

    	/* SDL interprets each pixel as a 32-bit number, so our
	     * masks must depend on the endianness (byte order) of the  machine */
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
           rmask = 0xff000000;
           gmask = 0x00ff0000;
           bmask = 0x0000ff00;
           amask = 0x000000ff;
#else
           rmask = 0x000000ff;
           gmask = 0x0000ff00;
           bmask = 0x00ff0000;
           amask = 0xff000000;
#endif
    
      SDL_Surface * temp = SDL_CreateRGBSurface(SDL_SWSURFACE,
        dst.w, dst.h, 32, rmask, gmask, bmask, amask); 
  
      Uint32 saved_flags = sdlSurface_->flags & (SDL_SRCALPHA|SDL_RLEACCELOK);
      Uint8 saved_alpha = sdlSurface_->format->alpha;
      if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
	      SDL_SetAlpha(sdlSurface_, 0, 0);
      }

      // put the SDL surface into this surface
      SDL_Rect area;
      area.x = 0;
      area.y = 0;
      area.w = sdlSurface_->w;
      area.h = sdlSurface_->h;
	

			SDL_BlitSurface(sdlSurface_, &area, temp, &area);

      if ((saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA) {
        SDL_SetAlpha(sdlSurface_, saved_flags, saved_alpha);
      }
	    // blit to the GL surface
	    int rowlength = temp->pitch / temp->format->BytesPerPixel;
	
	    //glPixelTransferi(GL_MAP_COLOR, 0);
	    glPixelStorei(GL_UNPACK_ROW_LENGTH, rowlength);

	    glEnable(GL_ALPHA_TEST);
	    glAlphaFunc(GL_GREATER, 0.0f);

			glRasterPos2i(dst.x - src.x, dst.y - src.y);
			glDisable(GL_DEPTH_TEST);
			glPixelZoom(1.0, -1.0);
			SDL_LockSurface(temp);
			glDrawPixels(temp->w, temp->h, GL_RGBA, GL_UNSIGNED_BYTE, (const GLvoid *)temp->pixels);
			SDL_UnlockSurface(temp);

			SDL_FreeSurface(temp);

			// Add this surface as a candidate for converting into a 
			// GL texture
			RootWindow::instance()->addSurface((Surface *)this);

		}
		else {
			// surface is available as a GL texture
      //Debug::channel(Debug::OPENGL) << "Blitting using texture " << glTexture_ << Debug::endl;

			if (sdlSurface_->w > 0 && sdlSurface_->h > 0) {
				// make sure we're not going to divide by zero
				int x_offset = dst.x - src.x;
				int y_offset = dst.y - src.y;
				
	

				glBindTexture( GL_TEXTURE_2D, glTexture_ );
				glBegin (GL_QUADS);
  			glTexCoord2f(src.x / sdlSurface_->w * glTexMaxX_, src.y / sdlSurface_->h * glTexMaxY_);
				glTexCoord2f(0.0f, 0.0f);
				glVertex3i(x_offset, y_offset, 0);
				glTexCoord2f((src.x + src.w) / sdlSurface_->w * glTexMaxX_, src.y / sdlSurface_->h * glTexMaxY_);
				glTexCoord2f(glTexMaxX_, 0.0f);
				glVertex3i(x_offset + sdlSurface_->w, y_offset, 0);
				glTexCoord2f((src.x + src.w) / sdlSurface_->w * glTexMaxX_, (src.y + src.h) /
					sdlSurface_->h * glTexMaxY_);
				glTexCoord2f(glTexMaxX_, glTexMaxY_);
				glVertex3i(x_offset + sdlSurface_->w, y_offset + sdlSurface_->h, 0);
				glTexCoord2f(src.x / sdlSurface_->w * glTexMaxX_, (src.y + src.h) / sdlSurface_->h * glTexMaxY_);
				glTexCoord2f(0.0f, glTexMaxY_);
				glVertex3i(x_offset, y_offset + sdlSurface_->h, 0);
				glEnd();
			}
		}

}
#endif

void
Surface::clear()
{
  if(sdlSurface_ == 0)
    return;

  SDL_FillRect(sdlSurface_, NULL, sdlSurface_->format->colorkey);
}

void
Surface::clear(const Rect& r)
{
  if(sdlSurface_ == 0 || !r.isValid())
    return;

  SDL_Rect rin(r);

  SDL_FillRect(sdlSurface_, &rin, sdlSurface_->format->colorkey);
}

void
Surface::fill(const Color &color)
{
  if(sdlSurface_ == 0)
    return;

  Uint32 bits = SDL_MapRGBA(sdlSurface_->format, color.r, color.g, color.b, color.a);

  SDL_FillRect(sdlSurface_, NULL, bits);
}


void
Surface::fill(const Rect &dest_, const Color &color)
{
   if(sdlSurface_ == 0)
     {
       Debug::channel(Debug::DRAWING) << "Surface.fill() on empty Surface has no effect." << Debug::endl;
       return;
     }

   if(!dest_.isValid())
     {
       Debug::channel(Debug::DRAWING) << "Surface.fill() was given an invalid rect." << Debug::endl;
       return;
     }
   
  //do clipping if neccessary
  Rect dest = dest_.intersect(Rect(0,0,width(),height()));

#ifdef USE_SDL_OPENGL
  Debug::channel(Debug::OPENGL) << "FIXME: Not implemented yet!" << Debug::endl;
#else
  Uint32 bits = SDL_MapRGBA(sdlSurface_->format, color.r, color.g, color.b, color.a);
  SDL_FillRect(sdlSurface_, &dest, bits);
#endif
  
}

void
Surface::fill(const Region &r, const Color &color)
{
  if(sdlSurface_ == 0)
    return;
  Uint32 bits = 0;
  
  Rect overlap = r.getClipbox().intersect(Rect(0, 0, width(), height()));
  if(overlap.isEmpty() || !overlap.isValid())
    return;

#ifdef USE_SDL_OPENGL
  if (sdlSurface_->flags & SDL_OPENGL) { 
    glPushAttrib(GL_CURRENT_BIT | GL_TEXTURE_BIT);
    glColor4ub(color.r, color.g, color.b, color.a);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    for(unsigned long i = 0; i < r.nRects(); ++i) {
      Rect rect = r.getRect(i);
      Debug::channel(Debug::OPENGL) << "Drawing OpenGL rectangle at " << rect <<
              ", color " << color << Debug::endl;
      glBegin(GL_POLYGON);
      glVertex2f(rect.x, rect.y);
      glVertex2f(rect.x + rect.w, rect.y);
      glVertex2f(rect.x + rect.w, rect.y + rect.h);
      glVertex2f(rect.x, rect.y + rect.h);
      glVertex2f(rect.x, rect.y);
      glEnd();

    }
    glPopAttrib();
  }
  else {
#endif
    bits = pixelformat().mapToPixel(color);
    for(unsigned i = 0; i < r.nRects(); ++i) {
      SDL_Rect rect(r.getRect(i));
      SDL_FillRect(sdlSurface_, &rect, bits);
    }
#ifdef USE_SDL_OPENGL
  }
#endif
}

void
Surface::blend(const Rect &r, const Color &color)
{
  if(sdlSurface_ == 0)
    return;

  switch(color.a) {
    case Color::WFTK_TRANSPARENT:
      return;
    case Color::WFTK_OPAQUE:
      fill(r, color);
      return;
    default:
      break;
  }

  Rect overlap = r.intersect(Rect(0, 0, width(), height()));
  if(overlap.isEmpty() || !overlap.isValid())
    return;
	
#ifdef USE_SDL_OPENGL
  if (sdlSurface_->flags & SDL_OPENGL) {
    // easy - just create an overlapping surface with the alpha
    // channel set.
    
		// FIXME: the 'a' channel in openGL probably goes from 0-100, not 0-255
    glPushAttrib(GL_CURRENT_BIT | GL_TEXTURE_BIT);
    glColor4ub((GLubyte)color.r, (GLubyte)color.g, (GLubyte)color.b, (GLubyte)color.a);
		Debug::channel(Debug::OPENGL) << "Blending GL alpha to "<< (int)color.r << "," << (int)color.g << "," << (int)color.b << ","  << (int)color.a << Debug::endl;
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

    glBegin(GL_POLYGON);
    glVertex2f(overlap.x, overlap.y);
    glVertex2f(overlap.x + overlap.w, overlap.y);
    glVertex2f(overlap.x + overlap.w, overlap.y + overlap.h);
    glVertex2f(overlap.x, overlap.y + overlap.h);
    glVertex2f(overlap.x, overlap.y);
    glEnd();
    glPopAttrib(); 

  }
  else {
#endif
    lock();

#if SDL_ALPHA_TRANSPARENT
        // transparent == 255
        Uint32 old_part = color.a;
        Uint32 new_part = 255 - color.a;
#else
        // transparent == 0
        Uint32 old_part = 255 - color.a;
        Uint32 new_part = color.a;
#endif

    Color c_pixel;
    int bpp = sdlSurface_->format->BytesPerPixel;
    for(int y = overlap.y; y < overlap.y + overlap.h; ++y) {
      Uint32 address = y * sdlSurface_->pitch + overlap.x * bpp;
      for(int x = overlap.x; x < overlap.x + overlap.w; ++x) {
        SDL_GetRGBA(readPixel(address), sdlSurface_->format,
          &c_pixel.r, &c_pixel.g, &c_pixel.b, &c_pixel.a);
        c_pixel.r = (new_part * color.r + old_part * c_pixel.r) / 255;
        c_pixel.g = (new_part * color.g + old_part * c_pixel.g) / 255;
        c_pixel.b = (new_part * color.b + old_part * c_pixel.b) / 255;
        writePixel(address, SDL_MapRGBA(sdlSurface_->format,
          c_pixel.r, c_pixel.g, c_pixel.b, c_pixel.a));
        address += bpp;
    }
  }

  unlock();
#ifdef USE_SDL_OPENGL
  }
#endif
}

void
Surface::blend(const Region &r, const Color &color)
{
  if(sdlSurface_ == 0)
    return;

  switch(color.a) {
    case Color::WFTK_TRANSPARENT:
      return;
    case Color::WFTK_OPAQUE:
      fill(r, color);
      return;
    default:
      break;
  }

  Rect overlap = r.getClipbox().intersect(Rect(0, 0, width(), height()));
  if(overlap.isEmpty() || !overlap.isValid())
    return;
   
  for(unsigned long i = 0; i < r.nRects(); ++i)
    blend(r.getRect(i), color);
}

#define CLIP(v) ((v) <= 0 ? 0 : ((v) >= 255 ? 255 : (v)))   

bool
Surface::convert(const Pixelformat& pixelformat, bool dither)
{
  if(!pixelformat.valid())
    {
      Debug::channel(Debug::DRAWING) << "Surface: convert to unknown Pixelformat requested." << Debug::endl;
      return false;
    }

  if(!sdlSurface_)
    return false;

  Debug::channel(Debug::DRAWING) << "Surface converting " << Pixelformat(sdlSurface_).asString() 
  <<" to " << pixelformat.asString() << " ... "<< Debug::endl;

  // we only dither if target is palettized
  if(pixelformat.bpp() != 1)
    dither = false;

  SDL_Surface *tmp = SDL_CreateRGBSurface(SDL_SWSURFACE | SDL_SRCALPHA ,
              width(),
              height(),
              pixelformat.bitspp(),
              pixelformat.rMask(),
              pixelformat.gMask(),
              pixelformat.bMask(),
              pixelformat.aMask());
    
  if(!tmp)
    throw SDLFatal("SDL_CreateRGBSurface");

  const SDL_Palette* palette = static_cast<const SDL_PixelFormat&>(pixelformat).palette;

  // set palette for target if needed
  if(palette && palette->ncolors > 0)
    SDL_SetColors(tmp, palette->colors, 0, palette->ncolors);
     
  /* converter blit.
     Handle special case, so that any semitransparencies
     get full transparent for the target, if target does not
     has an alpha channel.
  */
        
  SDL_LockSurface(tmp);
  SDL_LockSurface(sdlSurface_);

  Pixelformat format(sdlSurface_);

  Uint32 pixel;
  Uint32 src = 0; 
  Uint8 *dst = (Uint8*)tmp->pixels;
  Uint32 src_skip = sdlSurface_->pitch - format.bpp() * width();
  Uint32 dst_skip = tmp->pitch - pixelformat.bpp()*width();
  Color col, transCol_(format.mapToColor(sdlSurface_->format->colorkey));
  int ind;
  int* errors = (int*)calloc(width()+1, sizeof(int)* 3);

  for(unsigned y=0; y < height(); y++)
    {               
      int r, r0, r1, r2;
      int g, g0, g1, g2;
      int b, b0, b1, b2;
      int* e = errors;

      r = r0 = r1 = r2 = 0;
      g = g0 = g1 = g2 = 0;
      b = b0 = b1 = b2 = 0;  

      for(unsigned x = 0; x < width() ; x++)
        {      
          int d2;

          pixel = readPixel(src);
          col = format.mapToColor(pixel);

          // set any semitransparent areas to full transparency
          if(col.a != SDL_ALPHA_OPAQUE)
      {
        col = transCol_;
      }

          if(!dither)
      {
        pixel = pixelformat.mapToPixel(col);

        switch(pixelformat.bpp())
          {
          case 1:
            *dst = pixel;
            break;
          case 2:
            *((Uint16*)dst) = pixel;
            break;
          case 3:
            int shift;
            shift = tmp->format->Rshift;
            *(dst+shift/8) = pixel>>shift;
            shift = tmp->format->Gshift;
            *(dst+shift/8) = pixel>>shift;
            shift = tmp->format->Bshift;
            *(dst+shift/8) = pixel>>shift;        
            break;
          default:
            *((Uint32*)dst) = pixel;
            break;
          }
      }
          else
      {
        r = CLIP(col.r + (r + e[3+0])/16);
        g = CLIP(col.g + (g + e[3+1])/16);
        b = CLIP(col.b + (b + e[3+2])/16);

        // get best matching color
        ind = pixelformat.mapToPixel(Color(r,g,b));

        /* write data to target */
        *dst = ind;

        Color col = pixelformat.getColor(ind);

        r -= (int) col.r;
        g -= (int) col.g;
        b -= (int) col.b;

        /* propagate errors (don't ask ;-)*/
        r2 = r; d2 = r + r; r += d2; e[0] = r + r0;
        r += d2; r0 = r + r1; r1 = r2; r += d2;
        g2 = g; d2 = g + g; g += d2; e[1] = g + g0;
        g += d2; g0 = g + g1; g1 = g2; g += d2;
        b2 = b; d2 = b + b; b += d2; e[2] = b + b0;
        b += d2; b0 = b + b1; b1 = b2; b += d2;

        e += 3; 
      } // if(!dither ..)

          src+= format.bpp();
          dst+= pixelformat.bpp();
        }  // for (int x= ...

      e[0] = b0;
      e[1] = b1;
      e[2] = b2;  

      src += src_skip;
      dst += dst_skip;
    } // for (int y = ...

  free(errors);
  
  SDL_UnlockSurface(sdlSurface_);
  SDL_UnlockSurface(tmp);


  //remove old surface
  SDL_FreeSurface(sdlSurface_);
  //store new one
  sdlSurface_ = tmp;

  Debug::channel(Debug::DRAWING) << "Surface.convert() done." << Debug::endl;

  return true;
}


Rect 
Surface::textureBlit(Surface& dst, 
         const Point& p1, const Point& p2,
         const Point& p3, const Point& p4) const
{
  if(sdlSurface_ == 0 || dst.sdlSurface_ == 0)
    return Rect::invalid;
  return textureBlit(dst,p1,p2,p3,p4,Rect(0,0,width(),height()));
}

Rect
Surface::textureBlit(Surface& dst, 
         const Point& p1, const Point& p2,
         const Point& p3, const Point& p4, const Rect& src) const
{
  if(sdlSurface_ == 0 || dst.sdlSurface_ == 0)
    return Rect::invalid;

  lock(); 
  sge_TexturedRect(dst.sdlSurface_,
       p1.x,p1.y,
       p2.x,p2.y,
       p3.x,p3.y,
       p4.x,p4.y,
       sdlSurface_,
       src.x,src.y,
       src.x+src.w,src.y,
       src.x,src.y+src.h,
       src.x+src.w,src.y+src.h);
  unlock();

  // Get a bounding box for the changed area

  int low_x, low_y, high_x, high_y;

  low_x = high_x = p1.x;
  low_y = high_y = p1.y;

  if(p2.x < low_x)
    low_x = p2.x;
  else if(p2.x > high_x)
    high_x = p2.x;

  if(p2.y < low_y)
    low_y = p2.y;
  else if(p2.y > high_y)
    high_y = p2.y;

  if(p3.x < low_x)
    low_x = p3.x;
  else if(p3.x > high_x)
    high_x = p3.x;

  if(p3.y < low_y)
    low_y = p3.y;
  else if(p3.y > high_y)
    high_y = p3.y;

  if(p4.x < low_x)
    low_x = p4.x;
  else if(p4.x > high_x)
    high_x = p4.x;

  if(p4.y < low_y)
    low_y = p4.y;
  else if(p4.y > high_y)
    high_y = p4.y;

  Rect r(low_x, low_y, 1 + high_x - low_x, 1 + high_y - low_y);
  return r.intersect(Rect(0,0,dst.width(), dst.height()));
}

Rect 
Surface::scaledBlit(Surface& dst, bool smooth) const
{
  if(sdlSurface_ == 0 || dst.sdlSurface_ == 0)
    return Rect::invalid;
  
  return scaledBlit(dst,
        Rect(0,0,dst.width(),dst.height()),
        Rect(0,0,width(),height()), smooth);
}

Rect 
Surface::scaledBlit(Surface& dst, const Rect& dest, bool smooth) const
{
  if(sdlSurface_ == 0 || dst.sdlSurface_ == 0)
    return Rect::invalid;

  int sm = smooth ? SMOOTHING_ON : SMOOTHING_OFF;
  // get scale factors
  float x_scale = (float)dest.w / (float)width();
  float y_scale = (float)dest.h / (float)height();
  
  // use the rotozoom scaled blit to use the entire surface and overlay on 
  // the destination
  SDL_Surface* after = zoomSurface(sdlSurface_, x_scale, y_scale, sm);
  SDL_Rect dstRect((SDL_Rect)dest);
  SDL_BlitSurface(after, NULL, dst.sdlSurface_, &dstRect);
  SDL_FreeSurface(after);

  return dest;
}

Rect
Surface::scaledBlit(Surface& dst, const Rect& dest, const Rect& src, bool
    smooth) const
{
  if(sdlSurface_ == 0 || dst.sdlSurface_ == 0)
    return Rect::invalid;

  if(dest.width() == src.width() && dest.height() == src.height())
    return blit(dst, dest, src);
  
  int sm = smooth ? SMOOTHING_ON : SMOOTHING_OFF;
  // get scale factors
  float x_scale = (float)dest.w / (float)src.w;
  float y_scale = (float)dest.h / (float)src.h;
  
  // create two temp surfaces: one to blit the portion of
  // the source, and one to hold the zoomed result
  SDL_Surface* before = SDL_CreateRGBSurface(sdlSurface_->flags, src.w, src.h,
      sdlSurface_->format->BitsPerPixel,
      sdlSurface_->format->Rmask,
      sdlSurface_->format->Bmask,
      sdlSurface_->format->Gmask,
      sdlSurface_->format->Amask);
  SDL_Rect srcRect(src);
  SDL_Rect dstRect(dest);
  SDL_BlitSurface(sdlSurface_, &srcRect, before, NULL);
  SDL_Surface* after = zoomSurface(before, x_scale, y_scale, sm);
  SDL_BlitSurface(after, NULL, dst.sdlSurface_, &dstRect);
  SDL_FreeSurface(before);
  SDL_FreeSurface(after);

  return dest;
}

bool 
Surface::scale(float n)
{
  return scale((int)(width()*n + 0.5), (int)(height()*n + 0.5));
}


bool 
Surface::scale(unsigned new_width, unsigned new_height)
{
  if(!sdlSurface_) {
    return false;
  }
  
  if(new_width == width() && new_height == height())
    return true;

  SDL_PixelFormat& format = *sdlSurface_->format;

  SDL_Surface *tmp = SDL_AllocSurface(SDL_SWSURFACE |SDL_SRCALPHA ,
              new_width,
              new_height,
              format.BitsPerPixel,
              format.Rmask,
              format.Gmask,
              format.Bmask,
              format.Amask);
  assert(tmp);
  if(sdlSurface_->flags & SDL_SRCCOLORKEY)
    SDL_SetColorKey(tmp, SDL_SRCCOLORKEY, sdlSurface_->format->colorkey);

  //the source surface may be RLE-encoded, thus we need to lock it ! 
  lock();
  sge_TexturedRect(tmp,0,0,new_width,0,0,new_height,new_width,new_height,
       sdlSurface_,0,0,width(),0,0,height(),width(),height());
  unlock();

  SDL_FreeSurface(sdlSurface_);
  sdlSurface_ = tmp;
  
  return true;
}

bool
Surface::mirror()
{

  if(!sdlSurface_ || width() == 0)
    {
      return false;
    }

  unsigned bytes = sdlSurface_->format->BytesPerPixel;
  // indexed or truecolor is irrelevant, we're just swapping bytecodes

  lock();

  for(unsigned y = 0; y < height(); y++) {
    Uint8* pixeladdr = (Uint8*)pixels() + pitch() * y;
    Uint8* pixeladdr_end = pixeladdr + (width() - 1) * bytes;
    while(pixeladdr < pixeladdr_end) {
      for(unsigned byte = 0; byte < bytes; ++byte) {
        // this is just a raw data swap, so endian-ness is irrelevant
        Uint8 tmp = *(pixeladdr + byte);
        *(pixeladdr + byte) = *(pixeladdr_end + byte);
        *(pixeladdr_end + byte) = tmp;
      }
      pixeladdr += bytes;
      pixeladdr_end -= bytes;
    }
  }

  unlock();
      
  return true;
} 

void
Surface::makeGLTexture() {
#ifdef USE_SDL_OPENGL
  // create a GL texture from this surface
	if (!glTexture_) {
		glGenTextures(1, (GLuint *)&glTexture_);
	}
	
	if (!glSurface_) {
		unsigned gl_x = 1;
		unsigned gl_y = 1;
		while ((gl_x < width()) && (gl_x < 1024))
			gl_x = gl_x << 1;
		while ((gl_y < height()) && (gl_y < 1024))
			gl_y = gl_y << 1;

    // create a temp SDL surface and blit to it
    Uint32 rmask, gmask, bmask, amask;

    /* SDL interprets each pixel as a 32-bit number, so our
     * masks must depend on the endianness (byte order) of the  machine */
#if SDL_BYTEORDER == SDL_BIG_ENDIAN
           rmask = 0xff000000;
           gmask = 0x00ff0000;
           bmask = 0x0000ff00;
           amask = 0x000000ff;
#else
           rmask = 0x000000ff;
           gmask = 0x0000ff00;
           bmask = 0x00ff0000;
           amask = 0xff000000;
#endif
    glSurface_ = SDL_CreateRGBSurface(SDL_SWSURFACE,
        gl_x, gl_y, 32, rmask, gmask, bmask, amask); 
	}
  /* Save the alpha blending attributes */
  Uint32 saved_flags = sdlSurface_->flags & (SDL_SRCALPHA|SDL_RLEACCELOK);
  Uint8 saved_alpha = sdlSurface_->format->alpha;
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
	  SDL_SetAlpha(sdlSurface_, 0, 0);
  }
	
	// put the SDL surface into this surface
	SDL_Rect area;
	area.x = 0;
	area.y = 0;
	area.w = sdlSurface_->w;
	area.h = sdlSurface_->h;
	
	SDL_BlitSurface(sdlSurface_, &area, glSurface_, &area);

	/* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(sdlSurface_, saved_flags, saved_alpha);
  }

	// set up openGL to read the texture
//	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	int rowlength = glSurface_->pitch / glSurface_->format->BytesPerPixel;
	glPixelStorei(GL_UNPACK_ROW_LENGTH, rowlength);
  
	if ((saved_flags & SDL_SRCCOLORKEY) == SDL_SRCCOLORKEY) {
	      Debug::channel(Debug::OPENGL) << "Warning, got colorkey in alpha";
	        }
  
  // apply an alpha mask to our new texture, if necessary
/*	SDL_Color ck;
	Uint32 ckey;
	if (sdlSurface_->format->BitsPerPixel == 8) {
		ck = sdlSurface_->format->palette->colors[sdlSurface_->format->colorkey];
		ckey = SDL_MapRGB(glSurface_->format, ck.r, ck.g, ck.b);
	}
	else {
		ckey = sdlSurface_->format->colorkey;
	}
	SDL_LockSurface(glSurface_);
	for (int j = 0; j < (glSurface_->w * glSurface_->h); ++j) {
		if (ckey & TRANS_MASK == ((Uint32 *)glSurface_->pixels)[j] & TRANS_MASK) {
			((Uint32 *)glSurface_->pixels)[j] = ((Uint32 *)glSurface_->pixels)[j] & TRANS_MASK;
		}
	}
	SDL_UnlockSurface(glSurface_);
	*/

	// bind the texture
	glBindTexture(GL_TEXTURE_2D, glTexture_);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, glSurface_->w, glSurface_->h, 0,
			GL_RGBA, GL_UNSIGNED_BYTE, (GLubyte*)glSurface_->pixels);
	
	// set the max bound
	glTexMaxX_ = (float)sdlSurface_->w / glSurface_->w;
	glTexMaxY_ = (float)sdlSurface_->h / glSurface_->h;
#else
	// stub
	;
#endif
}

std::pair<Surface*,bool>
Surface::ResLoad::operator()(const std::string& filename_)
{
  std::pair<Surface*, bool> out;

  out.first = new Surface();
  out.first->readFromFile(filename_);
  out.second = !out.first->empty();

  if(!out.second) {
    delete out.first;
    out.first = 0;
  } else if(!out.first->hasAlphaChannel())
    out.first->setColorKey("black");

  return out;
}

}
