/*
 * MFFloat.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include <stdio.h> 
#include "stdafx.h"

#include "MFFloat.h"
#include "DuneApp.h"

MFFloat::MFFloat(float *values, int len, int /* stride */)
{
    _value.setData(values, len);
}

MFFloat::MFFloat(double *values, int len, int /* stride */)
{
    float* floats = new float[len];
    for (int i = 0; i < len; i++)
         floats[i] = values[i];
    _value.setData(floats, len);    
}

MFFloat::MFFloat(int stride)
{
    _value.resize(0);
}

MFFloat::MFFloat(const MFFloat &value)
{
    _value.setData(value.getValues(), value.getSize());
}

MFFloat::MFFloat(const float value)
{
    _value.resize(0);
    _value[0] = value;
}

MFFloat::~MFFloat()
{
    _value.resize(0);
}

FieldValue *
MFFloat::copy()
{
    float *value = new float[_value.size()];
    for (int i = 0;i < _value.size(); i++)
        value[i] = _value[i];
    return new MFFloat(value, _value.size());
}

bool
MFFloat::equals(const FieldValue *value) const
{
    return value->getType() == MFFLOAT && equals((const MFFloat *) value);
}

bool
MFFloat::equals(const MFFloat *value) const
{
    if (_value.size() == value->getSize()) {
	for (int i = 0; i < _value.size(); i++)
	    if (_value[i] != value->getValue(i))
		return false;
	return true;
    }
    return false;
}

int MFFloat::write(int f, int indent) const
{
    int		stride = getStride();
    if (!TheApp->GetkrFormating()) {
        RET_ONERROR( mywritestr(f, "\n") )
        TheApp->incSelectionLinenumber();
        RET_ONERROR( indentf(f, indent + TheApp->GetIndent()) )
    }
    RET_ONERROR( mywritestr(f, "[\n") )
    TheApp->incSelectionLinenumber();
    for (int i = 0; i < _value.size(); i += stride) {
	RET_ONERROR( indentf(f, indent + TheApp->GetIndent()) )
	for (int j = 0; j < stride; j++) {
	    RET_ONERROR( mywritef(f, "%g ", _value[i+j]) )
	}
	RET_ONERROR( mywritestr(f, "\n") )
        TheApp->incSelectionLinenumber();
    }
    if (!TheApp->GetkrFormating())
        RET_ONERROR( indentf(f, indent + TheApp->GetIndent()) )
    else
        RET_ONERROR( indentf(f, indent) )
    RET_ONERROR( mywritestr(f, "]\n") )
    TheApp->incSelectionLinenumber();
    return(0);
}

void MFFloat::clamp(const FieldValue *min, const FieldValue *max)
{
    if (min) {
	float fmin = ((SFFloat *) min)->getValue();
	for (int i = 0; i < _value.size(); i ++) {
	    if (_value[i] < fmin) _value[i] = fmin;
	}
    }
    if (max) {
	float fmax = ((SFFloat *) max)->getValue();
	for (int i = 0; i < _value.size(); i ++) {
	    if (_value[i] > fmax) _value[i] = fmax;
	}
    }
}

void 
MFFloat::setSFValue(int index, FieldValue *value)
{
    assert(index<getSFSize());
    _value[index] = ((SFFloat *) value)->getValue();
}

void 
MFFloat::setSFValue(int index, float value)
{
    assert(index<getSFSize());
    _value[index] = value;
}

FieldValue *
MFFloat::getSFValue(int index) const
{
    assert(index<getSFSize());
    return new SFFloat(_value[index]); 
}

MyString
MFFloat::getEcmaScriptComment(MyString name, int flags) const
{
    const char *indent = ((FieldValue *)this)->getEcmaScriptIndent(flags);
    MyString ret;
    ret = "";
    if (TheApp->GetEcmaScriptAddAllowedValues()) {
        ret += indent;
        ret += "// allowed values:\n";

        ret += indent;
        ret += "   // array ([0] [1] [2] ...) of single-precision floating point numbers\n";
    }
    if (TheApp->GetEcmaScriptAddAvailableFunctions()) {
        ret += indent;
        ret += "// available functions:\n";
        if (flags != EL_EVENT_IN) {
            ret += indent;
            ret += "   // ";
            ret += name;
            ret += " = new MFFloat(float_f1, float_f2, ...);\n";
        }
        if (flags != EL_EVENT_OUT) {
            ret += indent;
            ret += "   // int_i = ";
            ret += name;
            ret += ".length();\n";

            ret += indent;
            ret += "   // string_str = ";
            ret += name;
            ret += ".toString();\n";
       }
    }
    if (TheApp->GetEcmaScriptAddExampleUsage()) {
        ret += indent;
        ret += "// example usage:\n";
        if (flags != EL_EVENT_IN) {
             ret += indent;
             ret += "   // ";
             ret += name;
             ret += " = new MFFloat(0.2, 3.7, 8, -123.34);\n";

             ret += indent;
             ret += "   // ";
             ret += name;
             ret += "[0] = 1 + Math.cos(Math.PI * 2) - Math.LN2;\n";
        } 
        if (flags != EL_EVENT_OUT) {
             ret += indent;
             ret += "   // float_number = ";
             ret += name;
             ret += "[0];\n";

             ret += indent;
             ret += "   // float_number = Math.abs(";
             ret += name;
             ret += "[3]);\n";
        }
        if (flags == EL_FIELD_DEF) {
             ret += indent;
             ret += "   // ";
             ret += name;
             ret += "[0] = Math.floor(";
             ret += name;
             ret += "[1]);\n";
        } 
    }
    return ret;
}

void 
MFFloat::insertSFValue(int index, FieldValue *value)
{
    _value.insert(((SFFloat *)value)->getValue(), index);
}

void 
MFFloat::insertSFValue(int index, float value)
{
    _value.insert(value, index);
}

// search for biggest value
float
MFFloat::getMaxValue(void)
{
    float retMax = 0;
    const float *values = getValues();
    if (getSize() > 0) {
       retMax = values[0];
       for (int i = 1; i < getSize(); i++)
           if (values[i] > retMax)
               retMax = values[i];
    }
    return retMax;
}

// search for smallest value
float
MFFloat::getMinValue(void)
{
    float retMin = 0;
    const float *values = getValues();
    if (getSize() > 0) {
       retMin = values[0];
       for (int i = 1; i < getSize(); i++)
           if (values[i] < retMin)
               retMin = values[i];
    }
    return retMin;
}

