// WideMargin. Simple fast bible software.
// Copyright (C) 2011  Daniel Hughes
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

using System;
using System.Data;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using Mono.Data.Sqlite;
using WideMargin.MVCInterfaces;

namespace WideMargin.Database
{

	public class BibleDatabase : IDisposable
	{
		private IDbConnection _connection;
		private IDbCommand _searchCommand;
		private IDbCommand _chapterCommand;
		private IDbCommand _searchCount;
		private IDbCommand _getDaylyReadings;
		
		/// <summary>
		/// Default constructor to enable unit testing.
		/// </summary>
		public BibleDatabase()
		{
		}
		
		public BibleDatabase(string file)
		{
			if(!File.Exists(file))
			{
				OSISImporter.CreateDatabase(file);
				OSISImporter.Import(@"kjv.rawtxt", file, @"ReadingPlanner.txt");
			}
			_connection = new SqliteConnection (string.Format("DbLinqProvider=Sqlite;Data Source={0}",file));
			_connection.Open();
		}
		
		/// <summary>
		/// Create a bible database using an existing open connection
		/// </summary>
		/// <param name="connection">open connection to use</param>
		public BibleDatabase(IDbConnection connection)
		{
			_connection = connection;
		}
		
		public SearchPageResult SearchFirstPage(string searchText)
		{
			int count = SearchCount(searchText);
			return Search(searchText, count, 10, 0);
		}
		
		/// <summary>
		/// Searches for searchText in the bible.
		/// </summary>
		/// <param name="searchText">Text to search for</param>
		/// <param name="pageSize">Number of results to show</param>
		/// <param name="pageNumber">Page number to get (Zero based)</param>
		/// <returns>search results</returns>
		public SearchPageResult Search(string searchText, int resultCount, int pageSize, int pageNumber)
		{
			if(_searchCommand == null)
			{
				_searchCommand = _connection.CreateCommand();
				
				_searchCommand.CommandText = "SELECT * FROM Verse WHERE Contents MATCH @searchTerm LIMIT @limit OFFSET @offset";
				
				//searchTerm parameter
				IDbDataParameter searchTerm =_searchCommand.CreateParameter();
				searchTerm.ParameterName = "@searchTerm";
				searchTerm.DbType = DbType.String;
  				_searchCommand.Parameters.Add(searchTerm);
				
				//limit paramter
				IDbDataParameter limit = _searchCommand.CreateParameter();
				limit.ParameterName = "@limit";
				limit.DbType = DbType.Int32;
				_searchCommand.Parameters.Add(limit);
				
				//offset paramter
				IDbDataParameter offset = _searchCommand.CreateParameter();
				offset.ParameterName = "@offset";
				offset.DbType = DbType.Int32;
				_searchCommand.Parameters.Add(offset);
			}
			
			((IDataParameter)_searchCommand.Parameters["@searchTerm"]).Value = searchText;			
			((IDataParameter)_searchCommand.Parameters["@limit"]).Value = pageSize;
			((IDataParameter)_searchCommand.Parameters["@offset"]).Value = pageNumber * pageSize;
			
			List<IVerse> verses = new List<IVerse>();
			using(IDataReader reader = _searchCommand.ExecuteReader())
			{
				while (reader.Read())
		        {
		            verses.Add(new Verse(reader));
		        }	
			}
			return new SearchPageResult(verses, resultCount, pageSize, pageNumber, searchText);
		}
		
		private int SearchCount(string searchText)
		{
			if(_searchCount == null)
			{
				_searchCount = _connection.CreateCommand();
				
				_searchCount.CommandText = "SELECT count(*) FROM Verse WHERE Contents MATCH @searchTerm";
				var searchTerm =_searchCount.CreateParameter();
				searchTerm.ParameterName = "@searchTerm";
				searchTerm.DbType = DbType.String;
  				_searchCount.Parameters.Add(searchTerm);
			}
			((IDataParameter)_searchCount.Parameters["@searchTerm"]).Value = searchText;
			object result = _searchCount.ExecuteScalar();
			return (int)(long)result;
		}
		
		public IEnumerable<IVerse> GetChapter(string book, int chapter)
		{
			if(_chapterCommand == null)
			{
				_chapterCommand = _connection.CreateCommand();
				
				_chapterCommand.CommandText = "SELECT * FROM Verse WHERE Book=@book AND Chapter=@chapter";
				
				IDataParameter bookTerm = _chapterCommand.CreateParameter();
				bookTerm.ParameterName = "@book";
				bookTerm.DbType = DbType.String;
  				_chapterCommand.Parameters.Add(bookTerm);
				
				IDataParameter chapterTerm = _chapterCommand.CreateParameter();
				chapterTerm.ParameterName = "@chapter";
				chapterTerm.DbType = DbType.Int32;
  				_chapterCommand.Parameters.Add(chapterTerm);
			}

			((IDataParameter)_chapterCommand.Parameters["@book"]).Value = book;
			((IDataParameter)_chapterCommand.Parameters["@chapter"]).Value = chapter;
			
			List<IVerse> verses = new List<IVerse>();
			using(IDataReader reader = _chapterCommand.ExecuteReader())
			{
				while (reader.Read())
		        {
		            verses.Add(new Verse(reader));
		        }	
			}
			return verses;
		}
		
		public void AddVerses(IEnumerable<Verse> verses)
		{
			using(var command = _connection.CreateCommand())
			{
				command.CommandText = "INSERT INTO Verse VALUES (@book, @chapter, @verse, @contents)";
				
				var bookParam = command.CreateParameter();
				bookParam.ParameterName = "@book";
				bookParam.DbType = DbType.String;
				command.Parameters.Add(bookParam);
				
				var chapterParam = command.CreateParameter();
				chapterParam.ParameterName = "@chapter";
				chapterParam.DbType = DbType.Int32;
				command.Parameters.Add(chapterParam);
				
				var verseParam = command.CreateParameter();
				verseParam.ParameterName = "@verse";
				verseParam.DbType = DbType.Int32;
				command.Parameters.Add(verseParam);
				
				var contentsParam = command.CreateParameter();
				contentsParam.ParameterName = "@contents";
				contentsParam.DbType = DbType.String;
				command.Parameters.Add(contentsParam);
				
				
				
				using(IDbTransaction transaction = _connection.BeginTransaction())
				{
					foreach(var verse in verses)
					{
						bookParam.Value = verse.Book;
						chapterParam.Value = verse.Chapter;	
						verseParam.Value = verse.VerseNumber;
						contentsParam.Value = verse.Contents;
						command.Transaction = transaction;
						command.ExecuteNonQuery();
					}
					transaction.Commit();
				}
			}
		}
		
		/// <summary>
		/// Gets the dayly readings for the specified date.
		/// </summary>
		/// <param name="dateTime">date to get readings for</param>
		/// <returns>dayly readings or empty if none found for given date</returns>
		public IEnumerable<DaylyReading> GetDaylyReading(DateTime dateTime)
		{
			if(_getDaylyReadings == null)
			{
				_getDaylyReadings = _connection.CreateCommand();
				_getDaylyReadings.CommandText = "SELECT FirstPortion, SecondPortion, ThirdPortion FROM ReadingPlanner WHERE Day=@day AND Month=@month";
				
				var dayParam = _getDaylyReadings.CreateParameter();
				dayParam.ParameterName = "@day";
				dayParam.DbType = DbType.Int32;
				_getDaylyReadings.Parameters.Add(dayParam);
				
				var monthParam = _getDaylyReadings.CreateParameter();
				monthParam.ParameterName = "@month";
				monthParam.DbType = DbType.Int32;
				_getDaylyReadings.Parameters.Add(monthParam);
			}
			
			((IDataParameter)_getDaylyReadings.Parameters["@day"]).Value = dateTime.Day;
			((IDataParameter)_getDaylyReadings.Parameters["@month"]).Value = dateTime.Month;
			
			using(IDataReader reader = _getDaylyReadings.ExecuteReader())
			{
				
				if(!reader.Read())
				{
					return new DaylyReading[]{};//return empty
				}
				
				List<DaylyReading> readings = new List<DaylyReading>();
				
				
				int firstIndex = reader.GetOrdinal("FirstPortion");
		    	string firstPortion = reader.GetString(firstIndex);
				readings.Add(DecodeDaylyReading(firstPortion));
				
				int secondIndex = reader.GetOrdinal("SecondPortion");
		    	string secondPortion = reader.GetString(secondIndex);
				readings.Add(DecodeDaylyReading(secondPortion));
				
				int thirdIndex = reader.GetOrdinal("ThirdPortion");
		    	string thirdPortion = reader.GetString(thirdIndex);
				readings.Add(DecodeDaylyReading(thirdPortion));
				
				return readings;
			}
		}
		
		/// <summary>
		/// Converts the dayly reading string into a DaylyReading
		/// </summary>
		/// <param name="reading">string in format Gen 2-4</param>
		/// <returns>Object representing the reading</returns>
		public DaylyReading DecodeDaylyReading(string reading)
		{
			if(reading == "2-3John")
			{
				//special 2 book case (only one)
				return new DaylyReading(new VerseIdentifier("2 John", 1, 1), "2-3 John");
			}
			int verse = 1;
			var parts = reading.Split(' ');
			string abbreviatedBookName = parts[0];
			string chapters = parts[1];
			int chapter;
			if(chapters.Contains(':'))
			{
				//verse range (ie psa 119:1-56)
				var chapterVerses = chapters.Split(':');
				chapter = int.Parse(chapterVerses[0]);
				verse = int.Parse(chapterVerses[1].Split('-')[0]);
			}
			else if(chapters.Contains("-"))
			{
				chapter = int.Parse(chapters.Split('-')[0]);	
			}
			else
			{
				chapter = int.Parse(chapters);	
			}

			string book = NameAbbreviationLookup.Get().Lookup(abbreviatedBookName);
			var verseId = new VerseIdentifier(book, chapter, verse);
			string fullReading = string.Format("{0} {1}", book, chapters);
			return new DaylyReading(verseId, fullReading);		                                  
		}
		
		internal void AddReadingPlannerData(string readingPlannerFile)
		{
			string[] days = File.ReadAllLines(readingPlannerFile);
			var dayRows = 
				from day in days
				let parts = day.Split(',')
				let dateParts = parts[0].Split('/')
				let dayOfMonth = int.Parse(dateParts[0])
				let month = int.Parse(dateParts[1])
				let firstPortion = parts[1]
				let secondPortion = parts[2]
				let thirdPortion = parts[3]
				select new
				{
					Day = dayOfMonth,
					Month = month,
					FirstPortion = firstPortion,
					SecondPortion = secondPortion,
					ThirdPortion = thirdPortion
				};
			
			using(var command = _connection.CreateCommand())
			{
				command.CommandText = "INSERT INTO ReadingPlanner VALUES (@day, @month, @firstPortion, @secondPortion, @thirdPortion)";
				
				var dayParam = command.CreateParameter();
				dayParam.ParameterName = "@day";
				dayParam.DbType = DbType.Int32;
				command.Parameters.Add(dayParam);
				
				var monthParam = command.CreateParameter();
				monthParam.ParameterName = "@month";
				monthParam.DbType = DbType.Int32;
				command.Parameters.Add(monthParam);
				
				var firstPortionParam = command.CreateParameter();
				firstPortionParam.ParameterName = "@firstPortion";
				firstPortionParam.DbType = DbType.String;
				command.Parameters.Add(firstPortionParam);
				
				var secondPortionParam = command.CreateParameter();
				secondPortionParam.ParameterName = "@secondPortion";
				secondPortionParam.DbType = DbType.String;
				command.Parameters.Add(secondPortionParam);
						
				var thirdPortionParam = command.CreateParameter();
				thirdPortionParam.ParameterName = "@thirdPortion";
				thirdPortionParam.DbType = DbType.String;
				command.Parameters.Add(thirdPortionParam);
				
				using(IDbTransaction transaction = _connection.BeginTransaction())
				{
					foreach(var day in dayRows)
					{
						dayParam.Value = day.Day;
						monthParam.Value = day.Month;
						firstPortionParam.Value = day.FirstPortion;
						secondPortionParam.Value = day.SecondPortion;
						thirdPortionParam.Value = day.ThirdPortion;
						command.Transaction = transaction;
						command.ExecuteNonQuery();
					}
					transaction.Commit();
				}
			}
				
		}
		
		protected virtual void Dispose(bool disposing)
		{
			if(disposing)
			{
				if(_chapterCommand != null)
				{
					_chapterCommand.Dispose();	
					_chapterCommand = null;
				}
				if(_connection != null)
				{
					_connection.Dispose();
					_connection = null;
				}
				if(_searchCommand != null)
				{
					_searchCommand.Dispose();	
					_searchCommand = null;
				}
				if(_searchCount != null)
				{
					_searchCount.Dispose();
					_searchCount = null;
				}
				if(_getDaylyReadings != null)
				{
					_getDaylyReadings.Dispose();
					_getDaylyReadings = null;
				}
			}			
		}
		
		public void Dispose()
		{
			GC.SuppressFinalize(this);
			Dispose(true);
		}
	}
}
