/* 
 * WMX10 - A WindowMaker dockapp to control 
 * 	   the X10 Firecracker Unit
 *
 * Copyright (C) 1999  Joshua Hawkins (jhawkins@osiris.978.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
 * 02111-1307, USA.
 *
 */

/* Standard include files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>

/* X include files */
#include <X11/xpm.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

/* Program specific includes */
#include "../wmgeneral/wmgeneral.h"
#include "wmx10.h"
#include "wmx10-master.xpm"
#include "br_cmd.h"



/* Print out usage information */
void usage() 
{
   fprintf(stderr,"\nwmx10 - WindowMaker X10 Firecracker Dock App\n");
   fprintf(stderr,"\nusage:  wmx10 <switches> [houseletter]\n");
   fprintf(stderr,"-h				this help file\n");
   fprintf(stderr,"-v				print version number\n");
   fprintf(stderr,"A thru P			house letter\n\n");
   exit(1);
}

/* Parse the command line, set up the initial display,
 * and assign the house letter.
 */
int init(int argc, char *argv[], char *house)
{
   char wmx10_mask_bits[64*64];
   int mask_size = 64;
   int i;
   char *arg;

   *house = 'A';

   /* Do some parsing... "Hey baby, wanna parse?" */
   for(i=1; i<argc; i++)
   {
      arg = argv[i];
      if (*arg == '-')
      {
	 arg++;
	 /* They want the version number! */
         if (*arg == 'v') 
	 {
	    fprintf(stderr,"wmx10 version %s by Joshua Hawkins\n",WMX10_VERSION);
	    exit(1);
	 }
	 else 
	    usage();
      }
      else 
      {
        *house = toupper(*argv[1]);
	/* The Firecracker kit allows for house letters between
	 * A and P, so lets make sure we stay in those bounds 
	 */ 
        if ((*house < 'A') || (*house > 'P')) 
        {
           fprintf(stderr, "House must be between \"A\" and \"P\".");
	   usage();
        }
      }
   }

   /* Take our xpm and conver it so we can mess with it */
   createXBMfromXPM(wmx10_mask_bits,wmx10_master_xpm,mask_size,mask_size);

   /* Create the initial gui */
   openXwindow(argc,argv,wmx10_master_xpm,wmx10_mask_bits,mask_size,mask_size);

   /* Add clickable regions for our buttons */
   for(i=0;i<6;i++)
   {
      AddMouseRegion(i,buttonArray[i].tlx,
		       buttonArray[i].tly,
		       buttonArray[i].brx,
		       buttonArray[i].bry);
   }

   return 0;
}


/* Copies the "pressed" image to the location of the
 * pressed button.
 */
int pressButton(int region)
{
   copyXPMArea(buttonDownArray[region].tlx,
	       buttonDownArray[region].tly,
	       buttonDownArray[region].brx-buttonDownArray[region].tlx,
	       buttonDownArray[region].bry-buttonDownArray[region].tly,
	       buttonArray[region].tlx,
	       buttonArray[region].tly);
   return 0;
}


/* Copies the "released" image to the location of the
 * pressed button.
 */
int releaseButton(int region)
{
   copyXPMArea(buttonUpArray[region].tlx,
	       buttonUpArray[region].tly,
	       buttonUpArray[region].brx-buttonUpArray[region].tlx,
	       buttonUpArray[region].bry-buttonUpArray[region].tly,
	       buttonArray[region].tlx,
	       buttonArray[region].tly);
   return 0;
}


/* Copies the lamp number image to its
 * location on the app.
 */ 
int updateNumDisplay(int lampNum)
{

   int ten = (lampNum+1) / 10;
   int one = (lampNum+1) % 10; 

#ifdef DEBUG
   printf("lampnum=%d, ten=%d, one=%d\n",lampNum,ten,one);
#endif

     copyXPMArea(numberArray[ten].tlx,
  	         numberArray[ten].tly,
  	         numberArray[ten].brx-numberArray[ten].tlx,
	         numberArray[ten].bry-numberArray[ten].tly,
	         26,
	         15);

   if ((one == 0) && (ten == 1))
     copyXPMArea(numberArray[0].tlx,
	         numberArray[0].tly,
	         numberArray[0].brx-numberArray[0].tlx,
	         numberArray[0].bry-numberArray[0].tly,
	         32,
	         15);
   else
     copyXPMArea(numberArray[one].tlx,
	         numberArray[one].tly,
	         numberArray[one].brx-numberArray[one].tlx,
	         numberArray[one].bry-numberArray[one].tly,
	         32,
	         15);
#ifdef DEBUG
   fprintf(stderr,"%d, %d, %d, %d\n",
	       numberArray[lampNum].tlx,
	       numberArray[lampNum].tly,
	       numberArray[lampNum].brx-numberArray[lampNum].tlx,
	       numberArray[lampNum].bry-numberArray[lampNum].tly);
#endif
   return 0;
}


/* Turns on and off the Christmas lights...
 * Bright Green means the appliance or lamp is on.
 * ************************************************************
 * The indicator will dim appropriately when a device is dimmed,
 * and brighten when the device is brightened.
 * Because all lamps vary in brightness and the x10 kit can't
 * monitor the brightness of each of the lamps, there's no way
 * to accurately display a dimmed/brightened indicator.
 * ************************************************************
 * Blank means the lamp or appliance is off.
 */ 
int updateStatusDisplay(int lampNum,
			int cmd)
{
    switch(cmd) {
    case six:
	copyXPMArea(greenLight6.tlx,
		    greenLight6.tly,
		    greenLight6.brx-greenLight6.tlx,
		    greenLight6.bry-greenLight6.tly,
		    lightArray[lampNum].tlx,
		    lightArray[lampNum].tly);
	break;
    case five:
	copyXPMArea(greenLight5.tlx,
		    greenLight5.tly,
		    greenLight5.brx-greenLight5.tlx,
		    greenLight5.bry-greenLight5.tly,
		    lightArray[lampNum].tlx,
		    lightArray[lampNum].tly);
	break;

    case four:
	copyXPMArea(greenLight4.tlx,
		    greenLight4.tly,
		    greenLight4.brx-greenLight4.tlx,
		    greenLight4.bry-greenLight4.tly,
		    lightArray[lampNum].tlx,
		    lightArray[lampNum].tly);
	break;
	
    case three:
	copyXPMArea(greenLight3.tlx,
		    greenLight3.tly,
		    greenLight3.brx-greenLight3.tlx,
		    greenLight3.bry-greenLight3.tly,
		    lightArray[lampNum].tlx,
		    lightArray[lampNum].tly);
	break;
    case two:
	copyXPMArea(greenLight2.tlx,
		    greenLight2.tly,
		    greenLight2.brx-greenLight2.tlx,
		    greenLight2.bry-greenLight2.tly,
		    lightArray[lampNum].tlx,
		    lightArray[lampNum].tly);
	break;
	
    case one:
	copyXPMArea(greenLight1.tlx,
		    greenLight1.tly,
		    greenLight1.brx-greenLight1.tlx,
		    greenLight1.bry-greenLight1.tly,
		    lightArray[lampNum].tlx,
		    lightArray[lampNum].tly);
	break;
	
    case zero:
	copyXPMArea(greenLight0.tlx,
		    greenLight0.tly,
		    greenLight0.brx-greenLight0.tlx,
		    greenLight0.bry-greenLight0.tly,
		    lightArray[lampNum].tlx,
		    lightArray[lampNum].tly);
	break;
      
    }
    return 0;
}


/* Copy the letter image of the current house
 * to its respective location on the app.
 */ 
int houseDisplay(char house)
{
   int let;

   let = house-'A';
   copyXPMArea(letterArray[let].tlx,
 	       letterArray[let].tly,
	       letterArray[let].brx-letterArray[let].tlx,
	       letterArray[let].bry-letterArray[let].tly,
	       50,
	       4);
   return 0; 
}

/* Send a command to the firecracker interface.
 * This routine calls the bottlerocket command:
 * x10_br_out()
 * to accomplish its task.
 */ 

int sendCommand(char house, 
		int lampNum, 
		int cmd)
{
   unsigned short addr;
   int fd;
   unsigned char let;
   char serial_port[256] = SERIAL_PORT;

   /* The house letter we're using */
   let = ((unsigned char)house)-'A';

   /* Address of the lamp we want - see the bottlerocket source */
   addr = (unsigned char)(((let)<<4)|(lampNum));

   /* Open up our serial port */
   fd = open(serial_port,O_RDONLY|O_NONBLOCK);

   if (fd < 0)
      fprintf(stderr,"Failed to open serial port!");

   /* And use the bottlerocket code to send the command */
   x10_br_out(fd, addr, (unsigned char)cmd);

   close(fd);

   return 0;
}
   
   

int main(int argc, char *argv[])
{
    XEvent Event;     /* X event status */
    int lampNum = lamp1;  /* The lamp number we're currently working with */
    int region=-1;    /* Clickable region as defined by AddMouseRegion() */
    int oldregion=-1;
    int i;
    char house='A';   /* The house letter - 'A' by default */
    int light_intense[16][2];
    
    /* init array setting each intensity to off. */
    for (i=0;i<16;i++)
    {
       light_intense[i][1] = OFF;
       light_intense[i][0] = 0;
    }
    
    /* Initialization routines */
    init(argc,argv,&house);
    
    /* Start by default at lamp 1 */
    updateNumDisplay(lamp1);
    
    /* Write out the house letter */
    houseDisplay(house);
    
    /* Our main loop */
    while(1) 
    {
	XNextEvent(display, &Event);
	switch (Event.type)
      {
      case Expose:
	  RedrawWindow();
	  break;
      case DestroyNotify:
	  XCloseDisplay(display);
	  exit(0);
	  break;
      case ButtonPress:		/* We've pressed a button! */
	  if ((region = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y)) != -1) 
	  {
	      pressButton(region);
	      oldregion=region;
	      
   	      switch(region)
	      {
	      case 0:		/* The "lamp off" button */
		  if( light_intense[lampNum][1] == ON ) {
		     RedrawWindow();
		     releaseButton(region);
		     RedrawWindow();
		     sendCommand( house, lampNum, OFF );
		     updateStatusDisplay( lampNum, zero );
		     light_intense[lampNum][0] = zero;
		     light_intense[lampNum][1] = OFF;
		  }
		  break;
	      case 1:		/* The "lamp on" button */
		  if( light_intense[lampNum][1] == OFF ) {
		     /* if lamp has been dimmed, turning it on does not restore
		     full brightness.  so don't change the status until light
		     is turned on and off.*/
		     RedrawWindow();
		     releaseButton(region);
		     RedrawWindow();
		     sendCommand(house,lampNum,ON);
		     updateStatusDisplay(lampNum,six);
		     light_intense[lampNum][0] = six; 
		     light_intense[lampNum][1] = ON;
		  }
		  break;
	      case 2:		/* The "dim" button */
		  if (light_intense[lampNum][1] == ON) {
		     RedrawWindow();
		     releaseButton(region);
		     RedrawWindow();
	       	     sendCommand(house,lampNum,DIM);
		     if (light_intense[lampNum][0]-1 > zero)
		        light_intense[lampNum][0] -= 1;
		     light_intense[lampNum][1] = ON;
		     updateStatusDisplay(lampNum,light_intense[lampNum][0]);
		  }
		  break;
	      case 3:		/* The "brighten" button */
		  if (light_intense[lampNum][1] == ON) {
		     RedrawWindow();
		     releaseButton(region);
		     RedrawWindow();
		     sendCommand(house,lampNum,BRIGHT);
		     if (light_intense[lampNum][0] < six)
		        light_intense[lampNum][0] += 1;
		     light_intense[lampNum][1] = ON;
		     updateStatusDisplay(lampNum,light_intense[lampNum][0]);
		  }
		  break;
	      case 4:		/* The "previous device" button */
		  if (--lampNum < lamp1) lampNum = lamp16;
		  updateNumDisplay(lampNum);
		  break;
	      case 5:		/* The "next device" button */
		  if (++lampNum > lamp16) lampNum = lamp1;
		  updateNumDisplay(lampNum);
		  break;
	      default:
	      }
	  }
      case ButtonRelease:
	  if ((region = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y)) != -1) 
	  {
	      RedrawWindow();
	      releaseButton(oldregion);
	  }
	  break;
      }
    }
    return 0;
}

