/*
 * $Id: draw_symbols.c,v 1.20 2001/05/03 20:04:10 kg4ijb Exp $
 *
 * XASTIR, Amateur Station Tracking and Information Reporting
 * Copyright (C) 1999,2000  Frank Giannandrea
 * Copyright (C) 2000,2001  The Xastir Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Look at the README for more information on the program.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <Xm/XmAll.h>

#include "draw_symbols.h"
#include "xastir.h"
#include "main.h"
#include "util.h"


int symbols_loaded;

/*** symbol data ***/

void clear_symbol_data(void) {
    int my_size;
    int i;
    char *data_ptr;

    data_ptr = (char *)symbol_data;
    my_size = (int)sizeof(SymbolData);
    for (i=0;i<my_size;i++)
        *data_ptr++ = '\0';
    symbols_loaded = 0;
}



/* draw letters */
/* this routine need to be made quicker and less efficent but for now this quick and
     gives me the type of look I want */
void draw_nice_string(Widget w, Pixmap where, int style, long x, long y, char *text, int bgcolor, int fgcolor, int length) {
    if(style==0) {
        /* make outline style */
        (void)XSetForeground(XtDisplay(w),gc,colors[bgcolor]);

        /* draw sides */
        (void)XDrawString(XtDisplay(w),where,gc,x+1,y-1,text,length);
        (void)XDrawString(XtDisplay(w),where,gc,x+1,y,text,length);
        (void)XDrawString(XtDisplay(w),where,gc,x+1,y+1,text,length);
        (void)XDrawString(XtDisplay(w),where,gc,x-1,y,text,length);
        (void)XDrawString(XtDisplay(w),where,gc,x-1,y-1,text,length);
        (void)XDrawString(XtDisplay(w),where,gc,x-1,y+1,text,length);
        (void)XDrawString(XtDisplay(w),where,gc,x,y+1,text,length);
        (void)XDrawString(XtDisplay(w),where,gc,x,y-1,text,length);
    } else {
        /* Draw old way */
        (void)XSetForeground(XtDisplay(w),gc,colors[0xff]);
        (void)XFillRectangle(XtDisplay(w),where,gc,x-1,(y-10),(length*6)+2,11);
        (void)XSetForeground(XtDisplay(w),gc,colors[bgcolor]);
        (void)XDrawString(XtDisplay(w),where,gc,x+1,y+1,text,length);
    }

    /* draw text */
    (void)XSetForeground(XtDisplay(w),gc,colors[fgcolor]);
    (void)XDrawString(XtDisplay(w),where,gc,x,y,text,length);
}



/* symbol drawing routines */

void draw_phg(long x_long, long y_lat, char *phg, time_t sec_heard, Pixmap where) {
    double range, rad;
    int offx,offy;
    long xx,yy,xx1,yy1,fxx,fyy;
    long max_x, max_y, x, y, x_origin, y_origin;
    double deg,tilt;
    double a,b;

    xx=0l;
    yy=0l;
    xx1=0l;
    yy1=0l;
    fxx=0l;
    fyy=0l;
    tilt=0.0;

    /* max off screen values */
    max_x = screen_width+800l;
    max_y = screen_height+800l;
    if ((sec_old+sec_heard)>sec_now()) {
        if ((x_long>=0) && (x_long<=129600000l)) {
            if ((y_lat>=0) && (y_lat<=64800000l)) {
                if ((x_long>x_long_offset) && (x_long<(x_long_offset+(long)(screen_width*size)))) {
                    if ((y_lat>y_lat_offset) && (y_lat<(y_lat_offset+(long)(screen_height*size)))) {
                        range = phg_range(phg[3],phg[4],phg[5]);
                        rad = range/((double)x_screen_distance/(double)50);
                        a=rad;
                        b=rad/2;
                        switch (phg[6]-'0') {
                            case(0):
                                offx=0;
                                offy=0;
                                break;

                            case(1):    //  45
                                offx=-1*(rad/3);
                                offy=rad/3;
                                tilt=5.49778;
                                break;

                            case(2):    //  90
                                offx=-1*(rad/3);
                                offy=0;
                                tilt=0;
                                break;

                            case(3):    // 135
                                offx=-1*(rad/3);
                                offy=-1*(rad/3);
                                tilt=.78539;
                                break;

                            case(4):    // 180
                                offx=0;
                                offy=-1*(rad/3);
                                tilt=1.5707;
                                break;

                            case(5):    // 225
                                offx=rad/3;
                                offy=-1*(rad/3);
                                tilt=2.3561;
                                break;

                            case(6):    // 270
                                offx=rad/3;
                                offy=0;
                                tilt=3.14159;
                                break;

                            case(7):    // 315
                                offx=rad/3;
                                offy=rad/3;
                                tilt=3.92699;
                                break;

                            case(8):    // 360
                                offx=0;
                                offy=rad/3;
                                tilt=4.71238;
                                break;

                            default:
                                offx=0;
                                offy=0;
                                break;
                        }
                        //printf("PHG=%02f %0.2f %0.2f %0.2f pix %0.2f\n",range,power,height,gain,rad);
                        if (rad>4.0) {
                            (void)XSetLineAttributes(XtDisplay(da), gc, 1, LineSolid, CapButt,JoinMiter);
                            if ((sec_old+sec_heard)>sec_now())
                                (void)XSetForeground(XtDisplay(da),gc,colors[0x0a]);
                            else
                                (void)XSetForeground(XtDisplay(da),gc,colors[0x52]);

                            if (phg[6]=='0') {
                                (void)XDrawArc(XtDisplay(da),where,gc,((x_long-x_long_offset)/size)-(rad/2),
                                         ((y_lat-y_lat_offset)/size)-(rad/2),rad,rad,0,64*360);
                            } else {
                                for (deg=0; deg<(M_PI*2);deg+=.1) {
                                    xx=(long)( a*cos(deg)*cos(tilt)-(b*sin(deg)*sin(tilt)) );
                                    yy=(long)( a*cos(deg)*sin(tilt)+(b*sin(deg)*cos(tilt)) );
                                    if (deg!=0) {
                                        x_origin = xx+((x_long-x_long_offset)/size)-offx;
                                        y_origin = yy+((y_lat-y_lat_offset)/size)-offy;
                                        x = xx1+((x_long-x_long_offset)/size)-offx;
                                        y = yy1+((y_lat-y_lat_offset)/size)-offy;
                                        if (x_origin > (-800l) && (x_origin< max_x)) {
                                            if (y_origin > (-800l) && (y_origin< max_y)) {
                                                if (x > (-800l) && (x< max_x)) {
                                                    if (y > (-800l) && (y< max_y))
                                                        (void)XDrawLine(XtDisplay(da),where,gc,x_origin, y_origin, x, y);
                                                }
                                            }
                                        }
                                    } else {
                                        fxx=xx;
                                        fyy=yy;
                                    }
                                    xx1=xx;
                                    yy1=yy;
                                }
                                (void)XDrawLine(XtDisplay(da),where,gc,xx+((x_long-x_long_offset)/size)-offx,yy+
                                        ((y_lat-y_lat_offset)/size)-offy,fxx+((x_long-x_long_offset)/size)-offx,fyy+
                                        ((y_lat-y_lat_offset)/size)-offy);
                            }
                        }
                    }
                }
            }
        }
    }
}



/* DK7IN: Statistics for colors in all symbols (as of 16.03.2001)
60167 .
 6399 q
 3686 m
 3045 c
 2034 j
 1903 h
 1726 l
 1570 k
 1063 g
 1051 #
  840 p
  600 ~
  477 i
  443 n
  430 a
  403 o
  337 f
  250 b
  207 e
  169 d
*/

// read pixels from file, speeding it up by smart ordering of switches
void read_symbol_from_file(FILE *f, char *pixels) {
    int x,y;                
    int color;
    char line[21];

    for (y=0;y<20;y++) {
        (void)get_line(f,line,20);
        for (x=0;x<20;x++) {
            switch (line[x]) {
                case('.'):       // transparent
                    color=0xff;
                    break;
                case('q'):       // #000000  black   0%
                    color=0x51;
                    break;
                case('m'):       // #FFFFFF  white 100%
                    color=0x4d;
                    break;
                case('c'):       // #CCCCCC  gray80 80%
                    color=0x43;
                    break;
                case('j'):       // #EE0000  red2
                    color=0x4a;
                    break;
                case('h'):       // #00BFFF  Deep sky blue
                    color=0x48;
                    break;
                case('l'):       // #0000CD  mediumblue
                    color=0x4c;
                    break;
                case('k'):       // #00CD00  green3
                    color=0x4b;
                    break;
                case('g'):       // #00008B  blue4
                    color=0x47;
                    break;
                case('#'):       // #FFFF00  yellow
                    color=0x40;
                    break;
                case('p'):       // #454545  gray27 27%
                    color=0x50;
                    break;
                case('~'):       // used in the last two symbols in the file
                    color=0xff;  // what should it be? was transparent before...
                    break;
                case('i'):       // #006400  Dark Green
                    color=0x49;
                    break;
                case('n'):       // #878787  gray53 52%
                    color=0x4e;
                    break;
                case('a'):       // #CD6500  darkorange2
                    color=0x41;
                    break;
                case('o'):       // #5A5A5A  gray59 35%
                    color=0x4f;
                    break;
                case('f'):       // #CD3333  brown3
                    color=0x46;
                    break;
                case('b'):       // #A020F0  purple
                    color=0x42;
                    break;
                case('e'):       // #FF4040  brown1
                    color=0x45;
                    break;
                case('d'):       // #CD0000  red3
                    color=0x44;
                    break;
                case('r'):       //          LimeGreen  DK7IN: saw this in the color definitions...
                    color=0x52;                         // so we could use it
                    break;
                default:
                    color=0xff;
                    break;
            }
            pixels[y*20+x] = (char)(color);
        }
    }
}



/* read in symbol table */
void load_pixmap_symbol_file(char *filename) {
    FILE *f;
    char filen[500];
    char line[21];
    char table_char;
    char symbol_char;
    int done;
    char pixels[400];
    char orient;
    
    busy_cursor(appshell);
    symbols_loaded = 0;
    table_char = '\0';
    symbol_char = '\0';
    done=0;
    sprintf(filen,"%s/%s",SYMBOLS_DIR,filename);
    f=fopen(filen,"r");
    if (f!=NULL) {
        while (!feof(f) && !done) {
            (void)get_line(f,line,20);
            if (strncasecmp("TABLE ",line,6)==0) {
                table_char=line[6];
                /*printf("TABLE %c\n",table_char);*/
            } else {
                if (strncasecmp("DONE",line,4)==0) {
                    done=1;
                    /*printf("DONE\n");*/
                } else {
                    if (strncasecmp("APRS ",line,5)==0) {
                        symbol_char=line[5];
                        if (strlen(line)>=20 && line[19] == 'l')     // symbol with orientation ?
                            orient = 'l';   // should be 'l' for left
                        else
                            orient = ' ';
                        read_symbol_from_file(f, pixels);                         // read pixels for one symbol
                        insert_symbol(table_char,symbol_char,pixels,270,orient);  // always have normal orientation
                        if (orient == 'l') {
                            insert_symbol(table_char,symbol_char,pixels,  0,'u'); // create other orientations
                            insert_symbol(table_char,symbol_char,pixels, 90,'r');
                            insert_symbol(table_char,symbol_char,pixels,180,'d');
                        }
                    }
                }
            }
        }
    } else
        printf("Error opening symbol file %s\n",filen);
    if (f != NULL)
        (void)fclose(f);
}



/* add a symbol to the end of the symbol table */
void insert_symbol(char table, char symbol, char *pixel, int deg, char orient) {
    int x,y,idx,old_next,color;

    if (symbols_loaded < MAX_SYMBOLS) {
        symbol_data[symbols_loaded].pix=XCreatePixmap(XtDisplay(appshell),
                    RootWindowOfScreen(XtScreen(appshell)),20,20,DefaultDepthOfScreen(XtScreen(appshell)));
        symbol_data[symbols_loaded].pix_mask=XCreatePixmap(XtDisplay(appshell),
                    RootWindowOfScreen(XtScreen(appshell)),20,20,1);
        symbol_data[symbols_loaded].pix_mask_old=XCreatePixmap(XtDisplay(appshell),
                    RootWindowOfScreen(XtScreen(appshell)),20,20,1);
        old_next=0;
        for (y=0;y<20;y++) {
            for (x=0;x<20;x++) {
                switch (deg) {
                    case(0):
                        idx = 20* (19-x) +   y;
                        break;
                    case(90):
                        idx = 20*   y    + (19-x);
                        break;
                    case(180):
                        idx = 20* (19-x) + (19-y);
                        break;
                    default:
                        idx = 20*   y    +   x;
                        break;
                }
                color = (int)(pixel[idx]);
                if (color<0)
                    color = 0xff;

                // DK7IN: is (da) correct or should this be (appshell) ?
                (void)XSetForeground(XtDisplay(da),gc,colors[color]);
                (void)XDrawPoint(XtDisplay(da),symbol_data[symbols_loaded].pix,gc,x,y);
                // DK7IN

                if (color != 0xff)     // create symbol mask
                    (void)XSetForeground(XtDisplay(appshell),gc2,1);  // active bit
                else
                    (void)XSetForeground(XtDisplay(appshell),gc2,0);  // transparent

                (void)XDrawPoint(XtDisplay(appshell),symbol_data[symbols_loaded].pix_mask,gc2,x,y);

                old_next++;        // create ghost symbol mask
                if (old_next>1) {  // by setting every 2nd bit to transparent
                    old_next=0;
                    (void)XSetForeground(XtDisplay(appshell),gc2,0);
                }
                (void)XDrawPoint(XtDisplay(appshell),symbol_data[symbols_loaded].pix_mask_old,gc2,x,y);
            }
            old_next++;    // shift one bit every scan line for ghost image
            if (old_next>1)
                old_next=0;
        }
        symbol_data[symbols_loaded].active = SYMBOL_ACTIVE;
        symbol_data[symbols_loaded].table  = table;
        symbol_data[symbols_loaded].symbol = symbol;
        symbol_data[symbols_loaded].orient = orient;
        symbols_loaded++;
    }
}



/* calculate symbol orientation from course */
#define ANGLE_UPDOWN 30
char symbol_orient(char *course) {
    char orient;
    float mydir;

    orient = ' ';
    if (strlen(course)) {
        mydir = (float)atof(course);
        if (mydir > 0) {
            if (mydir < (float)( 180+ANGLE_UPDOWN ) )
                orient = 'd';
            if (mydir < (float)( 180-ANGLE_UPDOWN ) )
                orient = 'r';
            if (mydir < (float)ANGLE_UPDOWN || mydir > (float)( 360-ANGLE_UPDOWN) )
                orient = 'u';
        }
    }
    return(orient);
}



void symbol(Widget w, int ghost, char symbol_table, char symbol_id, char symbol_overlay, Pixmap where,
            int mask, long x_offset, long y_offset, char orient) {
    int i;
    int found;
    int nosym;
    int alphanum;

    /* DK7IN: orient  is ' ','l','r','u','d'  for left/right/up/down symbol orientation */
    // if symbol could be rotated, normal symbol orientation in symbols.dat is to the left
    found = nosym = alphanum = -1;

    if (symbol_overlay == '\0' || symbol_overlay == ' ')
        alphanum = 0;          // we don't want an overlay

    for (i=0;(i<symbols_loaded);i++) {
        if (symbol_data[i].active==SYMBOL_ACTIVE) {
            if (symbol_data[i].table == symbol_table && symbol_data[i].symbol == symbol_id)
                if (found==-1) {found=i;}       // index of symbol

            if (symbol_data[i].table=='!' && symbol_data[i].symbol=='#')
                nosym=i;       // index of special symbol (if none available)

            if (symbol_data[i].table=='#' && symbol_data[i].symbol==symbol_overlay)
                alphanum=i;    // index of symbol for character overlay

            if (alphanum != -1 && nosym != -1 && found != -1)
                break;

        }
    }

    if (found == -1) {
        found=nosym;
        if (symbol_table && symbol_id && debug_level & 128)
            printf("No Symbol Yet! %2x:%2x\n", (unsigned int)symbol_table, (unsigned int)symbol_id);
    } else {                    // maybe we want a rotated symbol
        if (!(orient == ' ' || orient == 'l' || symbol_data[found].orient == ' ' || ghost)) {
            for (i=found;(i<symbols_loaded);i++) {
                if (symbol_data[i].active==SYMBOL_ACTIVE) {
                    if (symbol_data[i].table == symbol_table && symbol_data[i].symbol == symbol_id
                             && symbol_data[i].orient == orient) {
                        found=i;  // index of rotated symbol
                        break;
                    }
                }
            }
        }
    }

    if (mask) {
        if (ghost)
            (void)XSetClipMask(XtDisplay(w),gc,symbol_data[found].pix_mask_old);
        else
            (void)XSetClipMask(XtDisplay(w),gc,symbol_data[found].pix_mask);
    }
    (void)XSetClipOrigin(XtDisplay(w),gc,x_offset,y_offset);
    (void)XCopyArea(XtDisplay(w),symbol_data[found].pix,where,gc,0,0,20,20,x_offset,y_offset);

    if(alphanum>0) {
        if (ghost)
            (void)XSetClipMask(XtDisplay(w),gc,symbol_data[alphanum].pix_mask_old);
        else
            (void)XSetClipMask(XtDisplay(w),gc,symbol_data[alphanum].pix_mask);

        (void)XSetClipOrigin(XtDisplay(w),gc,x_offset,y_offset);
        (void)XCopyArea(XtDisplay(w),symbol_data[alphanum].pix,where,gc,0,0,20,20,x_offset,y_offset); // rot
    }

    (void)XSetClipMask(XtDisplay(w),gc,None);

}



void draw_symbol(Widget w, char symbol_table, char symbol_id, char symbol_overlay, long x_long,long y_lat,
                 char *callsign_text, char *alt_text, char *course_text, char *speed_text, char *my_distance,
                 char *my_course, char *wx_temp, char* wx_wind, time_t sec_heard, Pixmap where, char orient) {
    long x_offset,y_offset;
    int length;
    int ghost;
    int posyl;
    int posyr;

    if ((sec_clear+sec_heard)>sec_now()) {
        if((x_long+10>=0) && (x_long-10<=129600000l)) {      // 360 deg
            if((y_lat+10>=0) && (y_lat-10<=64800000l)) {     // 180 deg
                if((x_long>x_long_offset) && (x_long<(x_long_offset+(long)(screen_width*size)))) {
                    if((y_lat>y_lat_offset) && (y_lat<(y_lat_offset+(long)(screen_height*size)))) {
                        x_offset=((x_long-x_long_offset)/size)-(10);
                        y_offset=((y_lat-y_lat_offset)/size)-(10);
                        ghost = (int)(((sec_old+sec_heard)) < sec_now());
                        symbol(w,ghost,symbol_table,symbol_id,symbol_overlay,where,1,x_offset,y_offset,orient);

                        posyr = 10;      // align symbols vertically centered to the right
                        if (!ghost && strlen(alt_text)>0)
                            posyr -= 6;
                        if (strlen(callsign_text)>0)
                            posyr -= 6;
                        if (!ghost && strlen(speed_text)>0)
                            posyr -= 6;
                        if (!ghost && strlen(course_text)>0)
                            posyr -= 6;

                        length=(int)strlen(alt_text);
                        if (!ghost && length>0) {
                            x_offset=((x_long-x_long_offset)/size)+12;
                            y_offset=((y_lat-y_lat_offset)/size)+posyr;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,alt_text,0x08,0x48,length);
                            posyr += 12;
                        }

                        length=(int)strlen(callsign_text);
                        if (length>0) {
                            x_offset=((x_long-x_long_offset)/size)+12;
                            y_offset=((y_lat-y_lat_offset)/size)+posyr;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,callsign_text,0x08,0x0f,length);
                            posyr += 12;
                        }

                        length=(int)strlen(speed_text);
                        if (!ghost && length>0) {
                            x_offset=((x_long-x_long_offset)/size)+12;
                            y_offset=((y_lat-y_lat_offset)/size)+posyr;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,speed_text,0x08,0x4a,length);
                            posyr += 12;
                        }

                        length=(int)strlen(course_text);
                        if (!ghost && length>0) {
                            x_offset=((x_long-x_long_offset)/size)+12;
                            y_offset=((y_lat-y_lat_offset)/size)+posyr;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,course_text,0x08,0x52,length);
                            posyr += 12;
                        }

                        posyl = 10;       // distance and direction goes to the left
                        if (!ghost && strlen(my_distance)>0)
                            posyl -= 6;
                        if (!ghost && strlen(my_course)>0)
                            posyl -= 6;

                        length=(int)strlen(my_distance);
                        if (!ghost && length>0) {
                            x_offset=(((x_long-x_long_offset)/size)-(length*6))-12;
                            y_offset=((y_lat-y_lat_offset)/size)+posyl;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,my_distance,0x08,0x0f,length);
                            posyl += 12;
                        }
                        length=(int)strlen(my_course);
                        if (!ghost && length>0) {
                            x_offset=(((x_long-x_long_offset)/size)-(length*6))-12;
                            y_offset=((y_lat-y_lat_offset)/size)+posyl;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,my_course,0x08,0x0f,length);
                            posyl += 12;
                        }
                                                                                                                    
                        if (posyr < posyl)  // weather goes to the bottom, centered horizontally
                            posyr = posyl;
                        if (posyr < 18)
                            posyr = 18;
                                        
                        length=(int)strlen(wx_temp);
                        if (!ghost && length>0) {
                            x_offset=((x_long-x_long_offset)/size)-(length*3);
                            y_offset=((y_lat-y_lat_offset)/size)+posyr;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,wx_temp,0x08,0x40,length);
                            posyr += 12;
                        }
                        length=(int)strlen(wx_wind);
                        if (!ghost && length>0) {
                            x_offset=((x_long-x_long_offset)/size)-(length*3);
                            y_offset=((y_lat-y_lat_offset)/size)+posyr;
                            draw_nice_string(w,where,letter_style,x_offset,y_offset,wx_wind,0x08,0x40,length);
                        }
                    }
                }
            }
        }
    }
}

