/*
 * $Id: maps.c,v 1.40 2001/05/01 14:33:45 kg4ijb Exp $
 *
 * XASTIR, Amateur Station Tracking and Information Reporting
 * Copyright (C) 1999,2000  Frank Giannandrea
 * Copyright (C) 2000,2001  The Xastir Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Look at the README for more information on the program.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#ifdef USE_IMAGEMAGICK
#include <time.h>
#include <sys/types.h>
#include <magick/api.h>
#endif // USE_IMAGEMAGICK

#include <dirent.h>
#include <netinet/in.h>
#include <Xm/XmAll.h>
#include <X11/xpm.h>

#ifdef USE_GEOTIFF
#include <math.h>
#include "xtiffio.h"
#include "geotiffio.h"
#include "geo_normalize.h"
#include "projects.h"
#endif // USE_GEOTIFF

#include "xastir.h"
#include "maps.h"
#include "alert.h"
#include "util.h"
#include "main.h"

#define DOS_HDR_LINES 8
#define GRID_MORE 513


#define CHECKMALLOC(m)  if (!m) { fprintf(stderr, "***** Malloc Failed *****\n"); exit(0); }


typedef struct {
    int img_x;
    int img_y;
    unsigned long x_long;
    unsigned long y_lat;
} tiepoint;


int mag;
int npoints;    /* number of points in a line */


/* MAP pointers */

map_vectors *map_vectors_ptr;
text_label *map_text_label_ptr;
symbol_label *map_symbol_label_ptr;

/* MAP counters */

long vectors_num;
long text_label_num;
long object_label_num;

float geotiff_map_intensity = 1.0;  // Map color intensity, set from Maps->Map Intensity





/** MAP DRAWING ROUTINES **/


/**********************************************************
 * draw_grid()
 *
 * Draws a lat/lon grid on top of the view.
 **********************************************************/
void draw_grid (Widget w) {
    int place;
    char place_str[10];
    int grid_place;
    long xx, yy, xx1, yy1;

    /* Set the line width in the GC */
    (void)XSetLineAttributes (XtDisplay (w), gc, 1, LineOnOffDash, CapButt,JoinMiter);
    (void)XSetForeground (XtDisplay (w), gc, colors[0x08]);

    if (size < GRID_MORE)
        grid_place = 1;
    else
        grid_place = 10;

    if (long_lat_grid) {
        for (place = 180; place >= 0; place -= grid_place) {
            sprintf (place_str, "%03d00.00W", place);
            /*printf("Place %s\n",place_str); */
            xx1 = xx = ((convert_lon_s2l (place_str) - x_long_offset) / size);
            if (xx > 0 && xx < screen_width) {
                yy  = (convert_lat_s2l ("9000.00N") - y_lat_offset) / size;
                yy1 = (convert_lat_s2l ("9000.00S") - y_lat_offset) / size;
                if (yy < 0)
                    yy = 0;

                if (yy1 > screen_height)
                    yy1 = screen_height;

                (void)XDrawLine (XtDisplay (w), pixmap_final, gc, xx, yy, xx1, yy1);
            }
        }
        for (place = grid_place; place < 181; place += grid_place) {
            sprintf (place_str, "%03d00.00E", place);
            /*printf("Place %s\n",place_str); */
            xx1 = xx = ((convert_lon_s2l (place_str) - x_long_offset) / size);
            if (xx > 0 && xx < screen_width) {
                yy  = (convert_lat_s2l ("9000.00N") - y_lat_offset) / size;
                yy1 = (convert_lat_s2l ("9000.00S") - y_lat_offset) / size;
                if (yy < 0)
                    yy = 0;

                if (yy1 > screen_height)
                    yy1 = screen_height;

                (void)XDrawLine (XtDisplay (w), pixmap_final, gc, xx, yy, xx1, yy1);
            }
        }
        for (place = 90; place >= 0; place -= grid_place) {
            sprintf (place_str, "%02d00.00N", place);
            /*printf("Place %s\n",place_str); */
            yy1 = yy = ((convert_lat_s2l (place_str) - y_lat_offset) / size);

            if (yy > 0 && yy < screen_height) {
                xx  = (convert_lon_s2l ("18000.00W") - x_long_offset) / size;
                xx1 = (convert_lon_s2l ("18000.00E") - x_long_offset) / size;
                if (xx < 0)
                    xx = 0;

                if (xx1 > screen_width)
                    xx1 = screen_width;

                (void)XDrawLine (XtDisplay (w), pixmap_final, gc, xx, yy, xx1, yy1);
            }
        }
        for (place = grid_place; place < 91; place += grid_place) {
            sprintf (place_str, "%02d00.00S", place);
            /*printf("Place %s\n",place_str); */
            yy1 = yy = ((convert_lat_s2l (place_str) - y_lat_offset) / size);

            if (yy > 0 && yy < screen_height) {
                xx  = (convert_lon_s2l ("18000.00W") - x_long_offset) / size;
                xx1 = (convert_lon_s2l ("18000.00E") - x_long_offset) / size;
                if (xx < 0)
                    xx = 0;

                if (xx1 > screen_width)
                    xx1 = screen_width;

                (void)XDrawLine (XtDisplay (w), pixmap_final, gc, xx, yy, xx1, yy1);
            }
        }
    }
}





/**********************************************************
 * map_plot()
 *
 * Plots vectors on the map.  If "color" is non-zero,
 * then it draws filled polygons in the color of
 * "object_behavior"?  Weird.
 **********************************************************/
void map_plot (Widget w, long max_x, long max_y, long x_long_cord,long y_lat_cord,
    unsigned char color, long object_behavior, int destination_pixmap) {
    static int redraw_check;
    static XPoint points[MAX_MAP_POINTS];
    static unsigned char last_color = (unsigned char)0;
    static unsigned char last_behavior = (unsigned char)0, first_behavior = (unsigned char)0;
    long x, y;
    int draw_ok;
    unsigned char line_behavior, fill_color;
    char warning[200];

    /* don't ever go over MAX_MAP_POINTS have a bad map not a crashed program */
    if (npoints > MAX_MAP_POINTS) {
        sprintf (warning, "Warning line point count overflow: map_plot\b\n");
        XtAppWarning (app_context, warning);
        npoints = MAX_MAP_POINTS;
    }

    /* if map_color_levels are on see if we should draw the line? */
    draw_ok = 0;
    if (map_color_levels)
        switch (color) {
            case (0x01):
            case (0x14):
            case (0x18):
                if (mag < 100)
                    draw_ok = 1;
                break;
            case (0x15):
            case (0x19):
                if (mag < 600)
                    draw_ok = 1;
                break;
            case (0x16):
                if (mag < 800)
                    draw_ok = 1;
                break;
            default:
                draw_ok = 1;
                break;
        }
    else
        draw_ok = 1;

    if (draw_ok) {
        x = ((x_long_cord - x_long_offset) / size);
        y = ((y_lat_cord - y_lat_offset) / size);
        if (x < -MAX_OUTBOUND)
            x = -MAX_OUTBOUND;

        if (y < -MAX_OUTBOUND)
            y = -MAX_OUTBOUND;

        if (x > max_x)
            x = max_x;

        if (y > max_y)
            y = max_y;

        if (debug_level & 16)
            printf(" MAP Plot - max_x: %ld, max_y: %ld, x: %ld, y: %ld, color: %d, behavior: %lx, points: %d\n",
                    max_x, max_y, x, y, (int)color, (unsigned long)object_behavior, npoints);

        if ( (last_color != color) || (color == (unsigned char)0xff) ) {
            if (npoints && (last_color != (unsigned char)0xff) ) {
                line_behavior = last_behavior;
                if (last_behavior & 0x80) {
                    if (color) {
                        fill_color = (last_behavior & ~0x80) + (unsigned char)0x60;
                        if (fill_color > (unsigned char)0x69)
                            fill_color = (unsigned char)0x60;
                        } else
                            fill_color = (unsigned char)object_behavior;

                        (void)XSetForeground (XtDisplay (w), gc, colors[(int)fill_color]);

                        // This GC is used only for pixmap_alerts
                        (void)XSetForeground (XtDisplay (w), gc_tint, colors[(int)fill_color]);

                        // This is how we tint it instead of obscuring the whole map
                        (void)XSetFunction (XtDisplay (w), gc_tint, GXor);
                        /*
                        Options are:
                            GXclear         0                       (Don't use)
                            GXand           src AND dst             (Darker colors, black can result from overlap)
                            GXandReverse    src AND (NOT dst)       (Darker colors)
                            GXcopy          src                     (Don't use)
                            GXandInverted   (NOT src) AND dst       (Pretty colors)
                            GXnoop          dst                     (Don't use)
                            GXxor           src XOR dst             (Don't use, overlapping areas cancel each other out)
                            GXor            src OR dst              (More pastel colors, too bright?)
                            GXnor           (NOT src) AND (NOT dst) (Darker colors, very readable)
                            GXequiv         (NOT src) XOR dst       (Bright, very readable)
                            GXinvert        (NOT dst)               (Don't use)
                            GXorReverse     src OR (NOT dst)        (Bright, not as readable as others)
                            GXcopyInverted  (NOT src)               (Don't use)
                            GXorInverted    (NOT src) OR dst        (Bright, not very readable)
                            GXnand          (NOT src) OR (NOT dst)  (Bright, not very readable)
                            GXset           1                       (Don't use)
                        */


// Here's where we draw filled areas.

                        switch (destination_pixmap) {

                            // We must be drawing maps 'cuz this is the pixmap we use for it.
                            case DRAW_TO_PIXMAP:
//WE7U
                                if (map_color_fill)
                                    (void)XFillPolygon (XtDisplay (w), pixmap, gc, points, npoints, Complex,CoordModeOrigin);
                                break;

                            // We must be drawing weather alert maps 'cuz this is the pixmap we use for it.
                            case DRAW_TO_PIXMAP_ALERTS:
                                // Here we wish to tint the existing map instead of obscuring it.  We'll use
                                // gc_tint here instead of gc.
                                (void)XFillPolygon (XtDisplay (w), pixmap_alerts, gc_tint, points, npoints, Complex,CoordModeOrigin);
                                break;
 

                            // We must be drawing symbols/tracks 'cuz this is the pixmap we use for it.
                            case DRAW_TO_PIXMAP_FINAL:
                                (void)XFillPolygon (XtDisplay (w), pixmap_final, gc, points, npoints, Complex,CoordModeOrigin);
                                break;
                        }

                        line_behavior = first_behavior;
                    }
                    if (line_behavior & 0x01)
                        (void)XSetLineAttributes (XtDisplay (w), gc, 2, LineSolid, CapButt,JoinMiter);
                    else
                        (void)XSetLineAttributes (XtDisplay (w), gc, 1, LineSolid, CapButt,JoinMiter);

                    if (color == (unsigned char)0x56)
                        (void)XSetLineAttributes (XtDisplay (w), gc, 10, LineSolid, CapButt,JoinMiter);

                    (void)XSetForeground (XtDisplay (w), gc, colors[(int)last_color]);

                    // This GC is used only for pixmap_alerts
                    (void)XSetForeground (XtDisplay (w), gc_tint, colors[(int)last_color]);

                    switch (destination_pixmap) {

                        case DRAW_TO_PIXMAP_FINAL:
                            (void)XDrawLines (XtDisplay (w), pixmap_final, gc, points, npoints,CoordModeOrigin);
                            break;

                        case DRAW_TO_PIXMAP:
                            (void)XDrawLines (XtDisplay (w), pixmap, gc, points, npoints,CoordModeOrigin);
                            break;

                        case DRAW_TO_PIXMAP_ALERTS:
                            // Here we wish to tint the existing map instead of obscuring it.  We'll use
                            // gc_tint here instead of gc.
                            (void)XDrawLines (XtDisplay (w), pixmap_alerts, gc_tint, points, npoints,CoordModeOrigin);
                            break;
                    }

                    npoints = 0;

                    /* check to see if we have been away from the screen too long */
                    if (redraw_check > 1000) {
                        redraw_check = 0;
                        XmUpdateDisplay (XtParent (da));
                    }
                    redraw_check++;
                }
                last_color = color;
                if (color == (unsigned char)0xff) {
                    npoints = 0;
                    first_behavior = (unsigned char)object_behavior;
                }
                points[npoints].x = (short)x;
                points[npoints].y = (short)y;
                if ( (points[npoints].x > (-MAX_OUTBOUND))
                        && (points[npoints].x < (short)max_x)
                        && (points[npoints].y > (-MAX_OUTBOUND))
                        && (points[npoints].y < (short)max_y)
                        && (color != (unsigned char)0) )

                    npoints++;

                last_behavior = (unsigned char)object_behavior;
                return;
            }
            points[npoints].x = (short)x;
            points[npoints].y = (short)y;
            last_behavior = (unsigned char)object_behavior;

            if (points[npoints].x != points[npoints - 1].x || points[npoints].y != points[npoints - 1].y) {
                if (last_behavior & 0x80)
                    npoints++;

            else if (points[npoints].x > (-MAX_OUTBOUND)
                    && points[npoints].x < (short)max_x
                    && points[npoints].y > (-MAX_OUTBOUND)
                    && points[npoints].y < (short)max_y)
                npoints++;

        }
    } else
        npoints = 0;
}   /* map_plot */





/**********************************************************
 * get_map_ext()
 *
 * Returns the extension for the filename.  We use this to
 * determine which sort of map file it is.
 **********************************************************/
char *get_map_ext (char *filename) {
    int len;
    int i;
    char *ext;

    ext = NULL;
    len = (int)strlen (filename);
    for (i = len; i >= 0; i--) {
        if (filename[i] == '.') {
            ext = filename + (i + 1);
            break;
        }
    }
    return (ext);
}





/**********************************************************
 * get_map_dir()
 *
 * Used to snag just the pathname from a complete filename.
 **********************************************************/
char *get_map_dir (char *fullpath) {
    int len;
    int i;

    len = (int)strlen (fullpath);
    for (i = len; i >= 0; i--) {
        if (fullpath[i] == '/') {
            fullpath[i + 1] = (char)0;
            break;
        }
    }
    return (fullpath);
}





/***********************************************************
 * map_visible()
 *
 * Tests whether a particular path/filename is within our
 * current view.  We use this to decide whether to plot or
 * skip a particular image file (major speed-up!).
 * Input coordinates are in the Xastir coordinate system.
 *
 * Had to fix a bug here where the viewport glanced over the
 * edge of the earth, causing strange results like this.
 * Notice the View Edges Top value is out of range:
 *
 *
 *                Bottom         Top          Left       Right
 * View Edges:  31,017,956  4,290,923,492  35,971,339  90,104,075
 *  Map Edges:  12,818,482     12,655,818  64,079,859  64,357,110
 *
 * Left map boundary inside view
 * Right map boundary inside view
 * map_inside_view: 1  view_inside_map: 0  parallel_edges: 0
 * Map not within current view.
 * Skipping map: /usr/local/xastir/maps/tif/uk/425_0525_bng.tif
 *
 *
 * I had to check for out-of-bounds numbers for the viewport and
 * set them to min or max values so that this function always
 * works properly.  Here are the bounds of the earth (Xastir
 * Coordinate System):
 *
 *              0 (90 deg. or 90N)
 *
 * 0 (-180 deg. or 180W)      129,600,000 (180 deg. or 180E)
 *
 *          64,800,000 (-90 deg. or 90S)
 *
 ***********************************************************/
int map_visible (unsigned long bottom_map_boundary,
                    unsigned long top_map_boundary,
                    unsigned long left_map_boundary,
                    unsigned long right_map_boundary)
{

    unsigned long view_min_x, view_max_x;
    unsigned long view_min_y, view_max_y;
    int map_inside_view = 0;
    int view_inside_map = 0;
    int parallel_edges = 0;

    view_min_x = (unsigned long)x_long_offset;                         /*   left edge of view */
    if (view_min_x > 129600000ul)
        view_min_x = 0;

    view_max_x = (unsigned long)(x_long_offset + (screen_width * size)); /*  right edge of view */
    if (view_max_x > 129600000ul)
        view_max_x = 129600000ul;

    view_min_y = (unsigned long)y_lat_offset;                          /*    top edge of view */
    if (view_min_y > 64800000ul)
        view_min_y = 0;

    view_max_y = (unsigned long)(y_lat_offset + (screen_height * size)); /* bottom edge of view */
    if (view_max_y > 64800000ul)
        view_max_y = 64800000ul;

    if (debug_level & 2) {
        printf ("              Bottom     Top       Left     Right\n");

        printf ("View Edges:  %lu  %lu  %lu  %lu\n",
            view_max_y,
            view_min_y,
            view_min_x,
            view_max_x);

        printf (" Map Edges:  %lu  %lu  %lu  %lu\n",
            bottom_map_boundary,
            top_map_boundary,
            left_map_boundary,
            right_map_boundary);

        if ((left_map_boundary <= view_max_x) && (left_map_boundary >= view_min_x))
            printf ("Left map boundary inside view\n");

        if ((right_map_boundary <= view_max_x) && (right_map_boundary >= view_min_x))
            printf ("Right map boundary inside view\n");

        if ((top_map_boundary <= view_max_y) && (top_map_boundary >= view_min_y))
            printf ("Top map boundary inside view\n");

        if ((bottom_map_boundary <= view_max_y) && (bottom_map_boundary >= view_min_y))
            printf ("Bottom map boundary inside view\n");

        if ((view_max_x <= right_map_boundary) && (view_max_x >= left_map_boundary))
            printf ("Right view boundary inside map\n");

        if ((view_min_x <= right_map_boundary) && (view_min_x >= left_map_boundary))
            printf ("Left view boundary inside map\n");

        if ((view_max_y <= bottom_map_boundary) && (view_max_y >= top_map_boundary))
            printf ("Bottom view boundary inside map\n");

        if ((view_min_y <= bottom_map_boundary) && (view_min_y >= top_map_boundary))
            printf ("Top view boundary inside map\n");
    }


    /* In order to determine whether the two rectangles intersect,
    * we need to figure out if any TWO edges of one rectangle are
    * contained inside the edges of the other.
    */

    /* Look for left or right map boundaries inside view */
    if (   (( left_map_boundary <= view_max_x) && ( left_map_boundary >= view_min_x)) ||
            ((right_map_boundary <= view_max_x) && (right_map_boundary >= view_min_x)))
    {
        map_inside_view++;
    }


    /* Look for top or bottom map boundaries inside view */
    if (   ((   top_map_boundary <= view_max_y) && (   top_map_boundary >= view_min_y)) ||
            ((bottom_map_boundary <= view_max_y) && (bottom_map_boundary >= view_min_y)))
    {

        map_inside_view++;
    }


    /* Look for right or left view boundaries inside map */
    if (   ((view_max_x <= right_map_boundary) && (view_max_x >= left_map_boundary)) ||
            ((view_min_x <= right_map_boundary) && (view_min_x >= left_map_boundary)))
    {
        view_inside_map++;
    }


    /* Look for top or bottom view boundaries inside map */
    if (   ((view_max_y <= bottom_map_boundary) && (view_max_y >= top_map_boundary)) ||
        ((view_min_y <= bottom_map_boundary) && (view_min_y >= top_map_boundary)))
    {
        view_inside_map++;
    }


    /*
    * Look for left/right map boundaries both inside view, but top/bottom
    * of map surround the viewport.  We have a column of the map going
    * through from top to bottom.
    */
    if (   (( left_map_boundary <= view_max_x) && ( left_map_boundary >= view_min_x)) &&
            ((right_map_boundary <= view_max_x) && (right_map_boundary >= view_min_x)) &&
            ((view_max_y <= bottom_map_boundary) && (view_max_y >= top_map_boundary)) &&
            ((view_min_y <= bottom_map_boundary) && (view_min_y >= top_map_boundary)))
    {
        parallel_edges++;
    }


    /*
    * Look for top/bottom map boundaries both inside view, but left/right
    * of map surround the viewport.  We have a row of the map going through
    * from left to right.
    */
    if (   ((   top_map_boundary <= view_max_y) && (   top_map_boundary >= view_min_y)) &&
        ((bottom_map_boundary <= view_max_y) && (bottom_map_boundary >= view_min_y)) &&
        ((view_max_x <= right_map_boundary) && (view_max_x >= left_map_boundary)) &&
        ((view_min_x <= right_map_boundary) && (view_min_x >= left_map_boundary)))
    {
        parallel_edges++;
    }


    if (debug_level & 2)
        printf("map_inside_view: %d  view_inside_map: %d  parallel_edges: %d\n",
                map_inside_view,
                view_inside_map,
                parallel_edges);

    if ((map_inside_view >= 2) || (view_inside_map >= 2) || (parallel_edges) )
        return (1); /* Draw this pixmap onto the screen */
    else
        return (0); /* Skip this pixmap */
}





/**********************************************************
 * draw_geo_image_map()
 *
 * If we have found a ".geo" file, we read it here and plot
 * the .xpm, .xpm.gz or .gif map into the current viewport.
 * We check first to see whether the map should be plotted
 * and skip it if it's not in our viewport.  These images
 * are expected to be aligned in the lat/lon directions
 * (not rotated).
 **********************************************************/
void draw_geo_image_map (Widget w, char *dir, char *filenm) {
    char file[2000];        /* Complete path/name of image file */
    FILE *f;            /* Filehandle of image file */
    char line[400];        /* One line from GEO file */
    char fileimg[400];        /* Ascii name of image file, read from GEO file */
    XpmAttributes atb;        /* Map attributes after map's read into an XImage */
    tiepoint tp[2];        /* Calibration points for map, read in from .geo file */
    int n_tp;            /* Temp counter for number of tiepoints read */
    float temp_long, temp_lat;
    register long x, y;        /* DNN: this was a float, chg to long */
    register long xx, yy, xw, yh;    /* DNN: these should be signed        */
    double xxx, yyy;        /* LFM: needs more accuracy here */
    long sxx, syy;        /* X Y screen plot positions          */
    long oxx, oyy;        /* previous X Y screen plot positions */
    long x_min, x_max, y_min, y_max; /* boundaries for copy loop counters   */
    unsigned long bxx, byy;    /* begin xxx and yyy points           */
    double stepw, steph;
    int stepwc, stephc;
    XImage *xi;            /* Temp XImage used for reading in current image */

#ifdef USE_IMAGEMAGICK
    ExceptionInfo exception;
    Image *image;
    ImageInfo *image_info;
    PixelPacket pixel_pack;
    //int a = 0;
    int b = 0;
    int l;
    XColor my_colors[256];
    Colormap cmap;
#endif // USE_IMAGEMAGICK

    char map_it[300];
    int geo_image_width;        /* Image width from GEO file */
    int geo_image_height;        /* Image height from GEO file */

    sprintf (file, "%s/%s", dir, filenm);
    xi = NULL;

    /* Read the .geo file to find out map filename and tiepoint info */
    n_tp = 0;
    f = fopen (file, "r");
    if (f != NULL) {
        while (!feof (f)) {
            (void)get_line (f, line, 399);
            if (strncasecmp (line, "FILENAME", 8) == 0)
                (void)sscanf (line + 9, "%s", fileimg);

            if (n_tp < 2) {
                if (strncasecmp (line, "TIEPOINT", 8) == 0) {
                    (void)sscanf (line + 9, "%d %d %f %f",&tp[n_tp].img_x,&tp[n_tp].img_y,&temp_long,&temp_lat);

                    /* Convert tiepoints from lat/lon to Xastir coordinates */
                    tp[n_tp].x_long = 64800000l + (360000.0 * temp_long);
                    tp[n_tp].y_lat = 32400000l + (360000.0 * (-temp_lat));
                    n_tp++;
                }
            }

            if (strncasecmp (line, "IMAGESIZE", 9) == 0)
                (void)sscanf (line + 10, "%d %d",&geo_image_width,&geo_image_height);

        }
        (void)fclose (f);
    }
    else
        printf("Couldn't open file: %s\n", file);

    (void)get_map_dir (file);
    strcat (file, fileimg);

    /*
    * Here are the corners of our viewport, using the Xastir
    * coordinate system.  Notice that Y is upside down:
    *
    *   left edge of view = x_long_offset
    *  right edge of view = x_long_offset + (screen_width  * size)
    *    top edge of view =  y_lat_offset
    * bottom edge of view =  y_lat_offset + (screen_height * size)
    *
    * The corners of our map will soon be (after converting the
    * tiepoints to the corner of the map):
    *
    *   left edge of map = tp[0].x_long
    *  right edge of map = tp[1].x_long
    *    top edge of map = tp[0].y_lat
    * bottom edge of map = tp[1].y_lat
    *
    */
    xx = tp[0].x_long - x_long_offset;
    yy = tp[0].y_lat - y_lat_offset;

    xw = (long)(tp[1].x_long - tp[0].x_long);    /*  Width between tiepoints */
    yh = (long)(tp[1].y_lat - tp[0].y_lat);    /* Height between tiepoints */

    /* Calculate step size */
    stepw = ((double) xw / (tp[1].img_x - tp[0].img_x));
    steph = ((double) yh / (tp[1].img_y - tp[0].img_y));

    /* Scaled screen step size for use with XFillRectangle below */
    stepwc = (int) (stepw / size) + 1;
    stephc = (int) (steph / size) + 1;

    if (debug_level & 2) {
        printf ("\nImage: %s\n", file);
        printf ("Image size %d %d\n", geo_image_width, geo_image_height);
        printf ("XX: %ld YY:%ld Sx %f %d Sy %f %d\n", xx, yy, stepw,(int) (stepw / size), steph, (int) (steph / size));
    }

    /* reset tie points to corners of the image */
    /* If first tiepoint not at top left corner (0,0)*/
    if (tp[0].img_x != 0) {
        tp[0].x_long -= (tp[0].img_x * stepw);
        xx = tp[0].x_long - x_long_offset;
        tp[0].img_x = 0;
    }

    /* If first tiepoint not at top left corner (0,0)*/
    if (tp[0].img_y != 0) {
        tp[0].y_lat -= (tp[0].img_y * steph);
        yy = tp[0].y_lat - y_lat_offset;
        tp[0].img_y = 0;
    }

    /* If second tiepoint not at bottom right corner
    * (geo_image_width,geo_image_height)
    */
    if (tp[1].img_x != geo_image_width) {
        tp[1].x_long += (tp[1].img_x * stepw);
        tp[1].img_x = geo_image_width;
    }

    /* If second tiepoint not at bottom right corner
    * (geo_image_width,geo_image_height)
    */
    if (tp[1].img_y != geo_image_height) {
        tp[1].y_lat += (tp[1].img_y * steph);
        tp[1].img_y = geo_image_height;
    }

    /* Check whether map is inside our current view */
    /*                bottom        top    left        right */
    if (!map_visible (tp[1].y_lat, tp[0].y_lat, tp[0].x_long, tp[1].x_long)) {
        if (debug_level & 2)
            printf ("Map not in current view, skipping: %s\n", file);

        /* Map isn't inside our current view.  We're done.
         * Free the memory used for the image and return
         */
        if (xi)
            XDestroyImage (xi);

        return;            /* Skip this map */
    } else if (debug_level & 1)
        printf ("Loading imagemap: %s\n", file);

    sprintf (map_it, langcode ("BBARSTA028"), filenm);
    statusline(map_it,0);       // Loading ...

    atb.valuemask = 0;

    /*  XpmReadFileToImage is the call we wish to avoid if at all
     *  possible.  On large images this can take quite a while.  We
     *  check above to see whether the image is inside our viewport,
     *  and if not we skip loading the image.
     */
    if (XpmReadFileToImage (XtDisplay (w), file, &xi, NULL, &atb) == XpmSuccess) {
        if (debug_level & 2) {
            printf ("Image size %d %d\n", (int)atb.width, (int)atb.height);
            printf ("XX: %ld YY:%ld Sx %f %d Sy %f %d\n", xx, yy, stepw,(int) (stepw / size), steph, (int) (steph / size));
        }

        /* draw the image from the file out to the map screen */
        /*
        * Get the border values for the X and Y for loops used
        * for the XFillRectangle call later.
        */
        /* WE7U:  It looks like we're trying to determine how much
        * of the image is inside the viewport, so that we only
        * copy that much of it into the pixmap, saving time?
        */
        bxx = 0ul;
        x_min = x_max = 0l;
        for (x = 0, xxx = 0; x < (long)atb.width; x++, xxx += stepw) {
            sxx = (xx + xxx) / size;        /* sxx = screenX */
            if (sxx > 0) {
                if (sxx < screen_width)
                    x_max = x;
                else
                    break;
             } else {
                x_min = x;
                bxx = (unsigned long)xxx;
            }
        }

        byy = 0ul;
        y_min = y_max = 0l;
        for (y = 0, yyy = 0; y < (long)atb.height; y++, yyy += steph) {
            syy = (yy + yyy) / size;        /* syy = screenY */
            if (syy > 0) {
                if (syy < screen_height)
                    y_max = y;
                else
                    break;
            } else {
                y_min = y;
                byy = (unsigned long)yyy;
            }
        }

        oxx = -1;
        oyy = -1;
        for (x = x_min, xxx = (double)bxx; x <= x_max; x++, xxx += stepw) {
            sxx = (xx + xxx) / size;
            /*
            * If this row just handled before, don't overwrite the row
            * Just proceed with the next x.
            *
            * Reason:
            * it is useless to stack points on top of each other
            * so don't waste time on overwriting the row.
            */
            if (sxx != oxx) {
                for (y = y_min, yyy = (double)byy; y <= y_max; y++, yyy += steph) {
                    syy = (yy + yyy) / size;
                    /*
                    * If this pixel handled before, don't overwrite the pixel
                    * Just proceed with the next y.
                    *
                    * Reason:
                    * it is useless to stack points on top of each other
                    * so don't waste time on this.
                    */
                    if (syy != oyy) {
                        /* Here's where the actual drawing occurs from the temporary
                        * image onto the real map image
                        */
                        (void)XSetForeground (XtDisplay (w), gc, XGetPixel (xi, x, y));
//printf ("Pixel at %li, %li is %li\n", x, y, XGetPixel(xi, x, y));
                        (void)XFillRectangle (XtDisplay (w), pixmap, gc, sxx, syy, stepwc,stephc);
                        oyy = syy;
                    }
                }
                oxx = sxx;
            }
        }
    }

#ifdef USE_IMAGEMAGICK
    else if ((im = gdImageCreateFromGif(fopen(file, "rb")))) {
        /*
         This section attempts to read in an image file and place it in the display, 
         just like the code above does for XPMs.  It uses the same .geo file and 
         tiepoint criteria.
        */    
        atb.width = gdImageSX(im);
        atb.height = gdImageSY(im);

// Change this to allow different types of images (not just GIF)
        //  Need to get the Colormap so that we can map the 256 colors from GIF to the display
        cmap = DefaultColormap( XtDisplay (w), DefaultScreen( XtDisplay (w) ) );

        for (l = 0; l < gdImageColorsTotal(im); l++)
        {

// Need to check how to do this for ANY image, as ImageMagick can read in all sorts
// of image files
            // Multiply by 255 since GIF only has 255 as the highest while X11 has 65535
            my_colors[l].red   =   (unsigned short)(gdImageRed(im,l)*255*geotiff_map_intensity); 
            my_colors[l].green = (unsigned short)(gdImageGreen(im,l)*255*geotiff_map_intensity);
            my_colors[l].blue  =  (unsigned short)(gdImageBlue(im,l)*255*geotiff_map_intensity);
      
            //  Get the color allocated.  Allocated pixel color is written to my_colors.pixel
            b = XAllocColor( XtDisplay (w), cmap, &my_colors[l] );
            if (debug_level & 2) 
                printf("Color allocated is %li  %i  %i  %i \n", my_colors[l].pixel, my_colors[l].red, my_colors[l].blue, my_colors[l].green);
        }

        if (debug_level & 2) {
            printf ("Image size %d %d\n", atb.width, atb.height);
            printf("Total colors = %i\n", gdImageColorsTotal(im));
            printf ("XX: %ld YY:%ld Sx %f %d Sy %f %d\n", xx, yy, stepw,(int) (stepw / size), steph, (int) (steph / size));
            printf("Total colors = %i\n", gdImageColorsTotal(im));
            for (a=0;a<gdImageColorsTotal(im);a++)
            {
                printf ("RGB for index %i= %i, %i, %i\n", a, gdImageRed(im, a), gdImageGreen(im, a), gdImageBlue(im,a));
            }
        }

        /* draw the image from the file out to the map screen */
        /*
        * Get the border values for the X and Y for loops used
        * for the XFillRectangle call later.
        */
        /* WE7U:  It looks like we're trying to determine how much
        * of the image is inside the viewport, so that we only
        * copy that much of it into the pixmap, saving time?
        */
        x_min = x_max = bxx = 0;
        for (x = 0, xxx = 0; x < atb.width; x++, xxx += stepw) {
            sxx = (xx + xxx) / size;        /* sxx = screenX */
            if (sxx > 0) {
                if (sxx < screen_width)
                    x_max = x;
                else
                    break;
             } else {
                x_min = x;
                bxx = xxx;
            }
        }

        y_min = y_max = byy = 0;
        for (y = 0, yyy = 0; y < atb.height; y++, yyy += steph) {
            syy = (yy + yyy) / size;        /* syy = screenY */
            if (syy > 0) {
                if (syy < screen_height)
                    y_max = y;
                else
                    break;
            } else {
                y_min = y;
                byy = yyy;
            }
        }

        oxx = -1;
        oyy = -1;
        for (x = x_min, xxx = bxx; x <= x_max; x++, xxx += stepw) {
            sxx = (xx + xxx) / size;
            /*
            * If this row just handled before, don't overwrite the row
            * Just proceed with the next x.
            *
            * Reason:
            * it is useless to stack points on top of each other
            * so don't waste time on overwriting the row.
            */
            if (sxx != oxx) {
                for (y = y_min, yyy = byy; y <= y_max; y++, yyy += steph) {
                    syy = (yy + yyy) / size;
                    /*
                    * If this pixel handled before, don't overwrite the pixel
                    * Just proceed with the next y.
                    *
                    * Reason:
                    * it is useless to stack points on top of each other
                    * so don't waste time on this.
                    */
                    if (syy != oyy) {
                        /* Here's where the actual drawing occurs from the temporary
                        * image onto the real map image
                        */
                        b = gdImageGetPixel(im, x, y);
                        XSetForeground (XtDisplay (w), gc, my_colors[b].pixel);
                        XFillRectangle (XtDisplay (w), pixmap, gc, sxx, syy, stepwc,stephc);
                        oyy = syy;
                    }
                }
                oxx = sxx;
            }
        }
    }
#endif // USE_IMAGEMAGICK

    else
        printf ("ERROR loading %s\n", file);

    /* if we are done.. then free the memory */
    if (xi)
        XDestroyImage (xi);

#ifdef USE_IMAGEMAGICK
    if (im)
        gdImageDestroy(im);
#endif // USE_IMAGEMAGICK
}





/***********************************************************
 * convert_to_xastir_coordinates()
 *
 * Converts from lat/lon to Xastir coordinate system.
 * First two parameters are the output Xastir X/Y values, 
 * 2nd two are the input floating point lat/lon values.
 *
 *              0 (90 deg. or 90N)
 *
 * 0 (-180 deg. or 180W)      129,600,000 (180 deg. or 180E)
 *
 *          64,800,000 (-90 deg. or 90S)
 ***********************************************************/
void convert_to_xastir_coordinates ( unsigned long* x,
                                    unsigned long* y,
                                    float f_longitude,
                                    float f_latitude )
{
    *y = (unsigned long)(32400000l + (360000.0 * (-f_latitude)));
    *x = (unsigned long)(64800000l + (360000.0 * f_longitude));
}




#ifdef USE_GEOTIFF
/***********************************************************
 * read_fgd_file()
 *
 * Read in the "*.fgd" file associated with the geoTIFF
 * file.  Get the corner points from it and return.  If
 * no fgd file exists for this map, return a 0.
 ***********************************************************/
int read_fgd_file ( char* tif_filename,
                    float* f_west_bounding,
                    float* f_east_bounding,
                    float* f_north_bounding,
                    float* f_south_bounding)
{
    char fgd_file[2000];        /* Complete path/name of .fgd file */
    FILE *fgd;                  /* Filehandle of .fgd file */
    char line[400];             /* One line from .fgd file */
    int length;
    char *ptr;                  /* Substring pointer */
    int num_coordinates = 0;


    /* Read the .fgd file to find corners of the map neat-line */
    strcpy( fgd_file, tif_filename);
    length = strlen(fgd_file);

    /* Change the extension to ".fgd" */
    fgd_file[length-3] = 'f';
    fgd_file[length-2] = 'g';
    fgd_file[length-1] = 'd';

    if (debug_level & 2)
        printf("%s\n",fgd_file);

    /*
     * Search for the WEST/EAST/NORTH/SOUTH BOUNDING COORDINATES
     * in the .fgd file.
     */
    fgd = fopen (fgd_file, "r");
    if (fgd != NULL)
    {
        while ( ( !feof (fgd) ) && ( num_coordinates < 4 ) )
        {
            get_line (fgd, line, 399);

            if (*f_west_bounding == 0.0)
            {
                if ( (ptr = strstr(line, "WEST BOUNDING COORDINATE:") ) != NULL)
                {
                    sscanf (ptr + 25, " %f", f_west_bounding);
                    if (debug_level & 2)
                        printf("West Bounding:  %f\n",*f_west_bounding);
                    num_coordinates++;
                }
            }

            else if (*f_east_bounding == 0.0)
            {
                if ( (ptr = strstr(line, "EAST BOUNDING COORDINATE:") ) != NULL)
                {
                    sscanf (ptr + 25, " %f", f_east_bounding);
                    if (debug_level & 2)
                        printf("East Bounding:  %f\n",*f_east_bounding);
                    num_coordinates++;
                }
            }

            else if (*f_north_bounding == 0.0)
            {
                if ( (ptr = strstr(line, "NORTH BOUNDING COORDINATE:") ) != NULL)
                {
                    sscanf (ptr + 26, " %f", f_north_bounding);
                    if (debug_level & 2)
                        printf("North Bounding: %f\n",*f_north_bounding);
                    num_coordinates++;
                }
            }

            else if (*f_south_bounding == 0.0)
            {
                if ( (ptr = strstr(line, "SOUTH BOUNDING COORDINATE:") ) != NULL)
                {
                    sscanf (ptr + 26, " %f", f_south_bounding);
                    if (debug_level & 2)
                        printf("South Bounding: %f\n",*f_south_bounding);
                    num_coordinates++;
                }
            }

        }
        fclose (fgd);
    }
    else
    {
        if (debug_level & 2)
            printf("Couldn't open '.fgd' file, assuming no map collar to chop %s\n",
                    tif_filename);
        return(0);
    }


    /*
     * We should now have exactly four bounding coordinates.
     * These specify the map neat-line corners.  We can use
     * them to chop off the white collar from around the map.
     */
    if (num_coordinates != 4)
    {
        printf("Couldn't find 4 bounding coordinates in '.fgd' file, map %s\n",
                tif_filename);
        return(0);
    }


    if (debug_level & 2)
    {
        printf("%f %f %f %f\n",
        *f_south_bounding,
        *f_north_bounding,
        *f_west_bounding,
        *f_east_bounding);
    }

    return(1);    /* Successful */
}





/***********************************************************
 * datum_shift_to_wgs84()
 *
 * Attempt to convert from whatever datum the image is in
 * to WGS84 datum (the normal APRS map datum).
 *
 * TODO:  Generalize this code to take a pointer to the
 * data area and a count of points to convert.
 ***********************************************************/
int datum_shift_to_wgs84 (  float* f_west_bounding,
                            float* f_east_bounding,
                            float* f_north_bounding,
                            float* f_south_bounding,
                            char* original_datum,
                            geocode_t datum )
{
    /* For the moment we'll assume that NAD27 is the initial datum
     * Note that some USGS DRG maps are NAD83.  Will need a large
     * number of possible datum translations for world-wide coverage.
     * I'm not currently looking at the "original_datum" or "datum"
     * input parameters.  They're for future expansion.
     * WE7U
     */

    /* Here is the datum definition that we are translating from */
    static char* src_parms[] =
    {
        "proj=latlong",     /* Might change based on geotiff file */
        "datum=NAD27",      /* Needs to be "original_datum" */
    };

    /* Here is the datum definition that we want to translate to */
    static char* dest_parms[] =
    {
        "proj=latlong",
        "datum=WGS84",
    };

    PJ *src, *dest;
    double* x_ptr;
    double* y_ptr;
    double* z_ptr;
    long point_count = 2l;  /* That's an 'L', not a '1' */
    int point_offset = 1;
    double x[2];
    double y[2];
    double z[2];
    int status;
  

    x_ptr = x;
    y_ptr = y;
    z_ptr = z;


    z[0] = (double)0.0;
    z[1] = (double)0.0;


    if ( ! (src  = pj_init(sizeof(src_parms) /sizeof(char *), src_parms )) )
    {
        printf("datum_shift_to_wgs84: Initialization failed.\n");
        if (src)
            printf("Source: %s\n", pj_strerrno((int)src) );
        return(0);
    }


    if ( ! (dest = pj_init(sizeof(dest_parms)/sizeof(char *), dest_parms)) )
    {
        printf("datum_shift_to_wgs84: Initialization failed.\n");
        if (dest)
            printf("Destination: %s\n", pj_strerrno((int)dest) );
        return(0);
    }


    y[0] = (double)( DEG_TO_RAD * *f_north_bounding );
    x[0] = (double)( DEG_TO_RAD * *f_west_bounding );

    y[1] = (double)( DEG_TO_RAD * *f_south_bounding );
    x[1] = (double)( DEG_TO_RAD * *f_east_bounding );


    /*
     * This call seems to fail the first time it is used, quite
     * often other times as well.  Try it again if it fails the
     * first time.  The datums appear to be defined in pj_datums.c
     * in the proj.4 source code.  Currently defined datums are:
     * WGS84, GGRS87, NAD83, NAD27.
     */
    status = pj_transform( src, dest, point_count, point_offset, x_ptr, y_ptr, z_ptr);
    if (status)
    {
        status = pj_transform( src, dest, point_count, point_offset, x_ptr, y_ptr, z_ptr);
        if (status)
        {
            printf( "datum_shift_to_wgs84: Non-zero status from pj_transform: %d\n",status );
            printf( "datum_shift_to_wgs84: %s\n", pj_strerrno(status) );

            // May or may not be a good idea to skip this, but we'll try to recover.
            // Datum translation failed for some reason, but let's try to load the
            // maps anyway.
            //return(0);
        }
    }


    y[0] = RAD_TO_DEG * y[0];
    x[0] = RAD_TO_DEG * x[0];
    y[1] = RAD_TO_DEG * y[1];
    x[1] = RAD_TO_DEG * x[1];



    if (0 && debug_level & 2)
        printf( "Datum shifted values:  %f\t%f\t%f\t%f\n",
                x[0],
                y[0],
                x[1],
                y[1] );


    /* Free up memory that we used */
    pj_free(src);
    pj_free(dest);


    /* Plug our new values back in */
    *f_north_bounding = (float)y[0];
    *f_west_bounding = (float)x[0];

    *f_south_bounding = (float)y[1];
    *f_east_bounding = (float)x[1];


    return(1);
}





/***********************************************************
 * draw_geotiff_image_map()
 *
 * Here's where we handle geoTIFF files, such as USGS DRG
 * topo maps.  The .fgd file gives us the lat/lon of the map
 * neat-line corners for USGS maps.  We use this info to
 * chop off the white map border.  If no .fgd file is present,
 * we assume there is no map collar to be cropped and display
 * every pixel.
 * We also translate from the map datum to WGS84.  We use
 * libgeotiff/libtiff/libproj for these operations.

 * TODO:
 * Provide support for datums other than NAD27/NAD83/WGS84.
 * Libproj doesn't currently support many datums.
 *
 * Provide support for handling different map projections.
 * Perhaps by reprojecting the map data and storing it on
 * disk in another format.
 *
 * Select 'o', 'f', 'k', or 'c' maps based on zoom level.
 * Might also put some hysteresis in this so that it keeps
 * the current type of map through one extra zoom each way.
 * 'c': Good from x256 to x064.
 * 'f': Good from x128 to x032.  Not very readable at x128.
 * 'k': Good from x??? to x???.
 * 'o': Good from x064 to x004.  Not very readable at x64.
 ***********************************************************/
void draw_geotiff_image_map (Widget w, char *dir, char *filenm)
{
    char file[1000];            /* Complete path/name of image file */
    TIFF *tif = (TIFF *) 0;     /* Filehandle for tiff image file */
    GTIF *gtif = (GTIF *) 0;    /* GeoKey-level descriptor */
    /* enum { VERSION = 0, MAJOR, MINOR }; */
    int versions[3];
    uint32 width;               /* Width of the image */
    uint32 height;              /* Height of the image */
    uint16 bitsPerSample;       /* Should be 8 for USGS DRG's */
    uint16 samplesPerPixel = 1; /* Should be 1 for USGS DRG's.  Some maps
                                    don't have this tag so we default to 1 */
    uint32 rowsPerStrip;        /* Should be 1 for USGS DRG's */
    uint16 planarConfig;        /* Should be 1 for USGS DRG's */
    int    bytesPerRow;            /* Bytes per scanline row of tiff file */
    GTIFDefn defn;              /* Stores geotiff details */
    u_char *imageMemory;        /* Fixed pointer to same memory area */
    uint32 row;                 /* My row counter for the loop */
    int num_colors;             /* Number of colors in the geotiff colormap */
    uint16 *red_orig, *green_orig, *blue_orig; /* Used for storing geotiff colors */
    XColor my_colors[256];      /* Used for translating colormaps */
    Colormap cmap;
    unsigned long west_bounding = 0;
    unsigned long east_bounding = 0;
    unsigned long north_bounding = 0;
    unsigned long south_bounding = 0;
    float f_west_bounding = 0.0;
    float f_east_bounding = 0.0;
    float f_north_bounding = 0.0;
    float f_south_bounding = 0.0;

    unsigned long west_bounding_wgs84 = 0;
    unsigned long east_bounding_wgs84 = 0;
    unsigned long north_bounding_wgs84 = 0;
    unsigned long south_bounding_wgs84 = 0;

    float f_NW_x_bounding;
    float f_NW_y_bounding;
    float f_NE_x_bounding;
    float f_NE_y_bounding;
    float f_SW_x_bounding;
    float f_SW_y_bounding;
    float f_SE_x_bounding;
    float f_SE_y_bounding;

    unsigned long NW_x_bounding_wgs84 = 0;
    unsigned long NW_y_bounding_wgs84 = 0;
    float f_NW_x_bounding_wgs84 = 0.0;
    float f_NW_y_bounding_wgs84 = 0.0;

    unsigned long NE_x_bounding_wgs84 = 0;
    unsigned long NE_y_bounding_wgs84 = 0;
    float f_NE_x_bounding_wgs84 = 0.0;
    float f_NE_y_bounding_wgs84 = 0.0;

    unsigned long SW_x_bounding_wgs84 = 0;
    unsigned long SW_y_bounding_wgs84 = 0;
    float f_SW_x_bounding_wgs84 = 0.0;
    float f_SW_y_bounding_wgs84 = 0.0;

    unsigned long SE_x_bounding_wgs84 = 0;
    unsigned long SE_y_bounding_wgs84 = 0;
    float f_SE_x_bounding_wgs84 = 0.0;
    float f_SE_y_bounding_wgs84 = 0.0;

    int NW_x = 0;               /* Store pixel values for map neat-line */
    int NW_y = 0;               /* ditto */
    int NE_x = 0;               /* ditto */
    int NE_y = 0;               /* ditto */
    int SW_x = 0;               /* ditto */
    int SW_y = 0;               /* ditto */
    int SE_x = 0;               /* ditto */
    int SE_y = 0;               /* ditto */
    int left_crop;              /* Pixel cropping value */
    int right_crop;             /* Pixel cropping value */
    int top_crop;               /* Pixel cropping value */
    int bottom_crop;            /* Pixel cropping value */
    double xxx, yyy;            /* LFM: needs more accuracy here */
    register long sxx, syy;              /* X Y screen plot positions          */
    float steph;
    register float stepw;
    int stepwc, stephc;
    char map_it[300];           /* Used to hold filename for status line */
    int have_fgd;               /* Tells where we have an associated *.fgd file */
    //short datum;
    char *datum_name;           /* Points to text name of datum */
    //double *GeoTie;
    int crop_it = 0;            /* Flag which tells whether the image should be cropped */

    register uint32 column;

    float xastir_left_x_increment;
    float left_x_increment;
    float xastir_left_y_increment;
    float left_y_increment;
    float xastir_right_x_increment;
    float right_x_increment;
    float xastir_right_y_increment;
    float right_y_increment;
    float xastir_top_y_increment;
    float top_y_increment;
    float xastir_bottom_y_increment;
    float bottom_y_increment;
    float xastir_avg_y_increment;
    float avg_y_increment;
    int row_offset;
    register unsigned long current_xastir_left;
    unsigned long current_xastir_right;
    register uint32 current_left;
    uint32 current_right;
    uint32 current_line_width;
    register unsigned long xastir_current_y;
    register uint32 column_offset;
    register unsigned long xastir_current_x;
    double *PixelScale;
    uint16 qty;
    int SkipRows;
    unsigned long view_min_x, view_max_x;
    unsigned long view_min_y, view_max_y;

    register unsigned long xastir_total_y;
    int NW_line_offset;
    int NE_line_offset;
    int NW_xastir_x_offset;
    int NE_xastir_x_offset;
    int NW_xastir_y_offset;
    int NW_x_offset;
    int NE_x_offset;
    float xastir_avg_left_right_y_increment;
    register float total_avg_y_increment;
    unsigned long view_left_minus_pixel_width;
    unsigned long view_top_minus_pixel_height;



    if (debug_level & 2)
        printf ("%s/%s\n", dir, filenm);


    sprintf (file, "%s/%s", dir, filenm);


    /* Check whether we have an associated *.fgd file.  This
     * file contains the neat-line corner points for USGS DRG
     * maps, which allows us to chop off the map collar.
     */
    have_fgd = read_fgd_file( file,
                              &f_west_bounding,
                              &f_east_bounding,
                              &f_north_bounding,
                              &f_south_bounding );


    /*
     * If we are able to read the fgd file then we have the lat/lon
     * corner points in floating point variables.  If there isn't
     * an fgd file then we must get the info from the geotiff
     * tags themselves and we assume that there's no map collar to
     * chop off.
     */


    /*
     * What we NEED to do (implemented a bit later in this function
     * in order to support geotiff files created with other map
     * datums, is to open up the geotiff file and get the map datum
     * used for the data.  Then convert the corner points to WGS84
     * and check to see whether the image is inside our viewport.
     * Some USGS geotiff maps have map data in NAD83 datum and the
     * .fgd file incorrectly specifying NAD27 datum.  There are also
     * some USGS geotiff maps created with WGS84 datum.
     */


    /* convert_to_xastir_coordinates( x,y,longitude,latitude ); */
    if (have_fgd)   /* Must be a USGS file */
    {
        crop_it = 1;        /* The map collar needs to be cropped */

        convert_to_xastir_coordinates(  &west_bounding,
                                        &north_bounding,
                                        f_west_bounding,
                                        f_north_bounding );


        convert_to_xastir_coordinates(  &east_bounding,
                                        &south_bounding,
                                        f_east_bounding,
                                        f_south_bounding );


        /*
         * Check whether map is inside our current view.  It'd be
         * good to do a datum conversion first, but we don't know
         * what the datum is by this point in the code.  I'm just
         * doing this check here for speed, so that I can eliminate
         * maps that aren't even close to our viewport area, without
         * having to open those map files.  All other maps that pass
         * this test (at the next go-around later in the code) must
         * have their corner points datum-shifted so that we can
         * REALLY tell whether a map fits within the viewport.
         *
         * Perhaps add a bit to the corners (the max datum shift?)
         * to do our quick check?  I decided to add about 10 seconds
         * to the map edges, which equates to 1000 in the Xastir
         * coordinate system.  That should be greater than any datum
         * shift in North America for USGS topos.  I'm artificially
         * inflating the size of the map just for this quick
         * elimination check.
         *
         *   bottom          top             left           right
         */
        if (!map_visible( south_bounding + 1000,
                             north_bounding - 1000,
                             west_bounding - 1000,
                             east_bounding + 1000 ) )
        {
            if (debug_level & 2)
            {
                printf ("Map not within current view.\n");
                printf ("Skipping map: %s\n", file);
            }

            /* Map isn't inside our current view.  We're done.
             * Free any memory used and return.
             */
            return;                     /* Skip this map */
        }
    }


    /*
     * At this point the map MAY BE in our current view.
     * We don't know for sure until we do a datum translation
     * on the bounding coordinates and check again.  Note that
     * if there's not an accompanying .fgd file, we don't have
     * the bounding coordinates yet by this point.
     */


    /* Open TIFF descriptor to read GeoTIFF tags */
    tif = XTIFFOpen (file, "r");
    if (!tif)
        return;


    /* Open GTIF Key parser.  Keys will be read at this time */
    gtif = GTIFNew (tif);
    if (!gtif)
    {
        /* Close the TIFF file descriptor */
        XTIFFClose (tif);
        return;
    }


    /*
     * Get the GeoTIFF directory info.  Need this for
     * some of the operations further down in the code.
     */
    GTIFDirectoryInfo (gtif, versions, 0);

    /*
    if (versions[MAJOR] > 1)
    {
        printf ("This file is too new for me\n");
        GTIFFree (gtif);
        XTIFFClose (tif);
        return;
    }
    */


    /* I might want to attempt to avoid the GTIFGetDefn
     * call, as it takes a bit of time per file.  It
     * normalizes the info.  Try getting just the tags
     * or keys that I need individually instead.  I
     * need "defn" for the GTIFProj4ToLatLong calls though.
     */
    if (GTIFGetDefn (gtif, &defn))
    {
        if (debug_level & 2)
            GTIFPrintDefn (&defn, stdout);
    }


    /* Fetch a few TIFF fields for this image */
    TIFFGetField (tif, TIFFTAG_IMAGEWIDTH, &width);
    TIFFGetField (tif, TIFFTAG_IMAGELENGTH, &height);


    /*
     * If we don't have an associated .fgd file for this map,
     * check for corner points in the ImageDescription
     * tag (proposed new USGS DRG standard).  Boundary
     * coordinates will be the outside corners of the image
     * unless I can find some other proof.
     * Currently I assume that the map has no
     * map collar to chop off and set the neat-line corners
     * to be the outside corners of the image.
     *
     * NOTE:  For the USGS files (with a map collar), the
     * image must be cropped and rotated and is slightly
     * narrower at one end (top for northern hemisphere, bottom
     * for southern hemisphere).  For other files with no map
     * collar, the image is rectangular but the lat/lon
     * coordinates may be rotated.
     */
    if (!have_fgd)      // Not a USGS map or perhaps a newer spec
    {
        crop_it = 0;        /* Do NOT crop this map image */

        /*
         * Snag and parse ImageDescription tag here.
         */

        /* Code goes here for getting ImageDescription tag... */


        /* Figure out the bounding coordinates for this map */
        if (debug_level & 2)
            printf("\nCorner Coordinates:\n");

        /* Find lat/lon for NW corner of image */
        xxx = 0.0;
        yyy = 0.0;
        if ( GTIFImageToPCS( gtif, &xxx, &yyy ) )   // Do all 4 of these in one call?
        {
            if (debug_level & 2)
            {
                printf( "%-13s ", "Upper Left" );
                printf( "(%11.3f,%11.3f)\n", xxx, yyy );
            }
        }
        if ( GTIFProj4ToLatLong( &defn, 1, &xxx, &yyy ) )   // Do all 4 of these in one call?
        {
            if (debug_level & 2)
            {
                printf( "  (%s,", GTIFDecToDMS( xxx, "Long", 2 ) );
                printf( "%s)\n", GTIFDecToDMS( yyy, "Lat", 2 ) );
                printf("%f  %f\n", xxx, yyy);
            }
        }
        f_NW_x_bounding = (float)xxx;
        f_NW_y_bounding = (float)yyy;


        /* Find lat/lon for NE corner of image */
        xxx = width - 1;
        yyy = 0.0;
        if ( GTIFImageToPCS( gtif, &xxx, &yyy ) )
        {
            if (debug_level & 2)
            {
                printf( "%-13s ", "Lower Right" );
                printf( "(%11.3f,%11.3f)\n", xxx, yyy );
            }
        }
        if ( GTIFProj4ToLatLong( &defn, 1, &xxx, &yyy ) )
        {
            if (debug_level & 2)
            {
                printf( "  (%s,", GTIFDecToDMS( xxx, "Long", 2 ) );
                printf( "%s)\n", GTIFDecToDMS( yyy, "Lat", 2 ) );
                printf("%f  %f\n", xxx, yyy);
            }
        }
        f_NE_x_bounding = (float)xxx;
        f_NE_y_bounding = (float)yyy;

        /* Find lat/lon for SW corner of image */
        xxx = 0.0;
        yyy = height - 1;
        if ( GTIFImageToPCS( gtif, &xxx, &yyy ) )
        {
            if (debug_level & 2)
            {
                printf( "%-13s ", "Lower Right" );
                printf( "(%11.3f,%11.3f)\n", xxx, yyy );
            }
        }
        if ( GTIFProj4ToLatLong( &defn, 1, &xxx, &yyy ) )
        {
            if (debug_level & 2)
            {
                printf( "  (%s,", GTIFDecToDMS( xxx, "Long", 2 ) );
                printf( "%s)\n", GTIFDecToDMS( yyy, "Lat", 2 ) );
                printf("%f  %f\n", xxx, yyy);
            }
        }
        f_SW_x_bounding = (float)xxx;
        f_SW_y_bounding = (float)yyy;

        /* Find lat/lon for SE corner of image */
        xxx = width - 1;
        yyy = height - 1;
        if ( GTIFImageToPCS( gtif, &xxx, &yyy ) )
        {
            if (debug_level & 2)
            {
                printf( "%-13s ", "Lower Right" );
                printf( "(%11.3f,%11.3f)\n", xxx, yyy );
            }
        }
        if ( GTIFProj4ToLatLong( &defn, 1, &xxx, &yyy ) )
        {
            if (debug_level & 2)
            {
                printf( "  (%s,", GTIFDecToDMS( xxx, "Long", 2 ) );
                printf( "%s)\n", GTIFDecToDMS( yyy, "Lat", 2 ) );
                printf("%f  %f\n", xxx, yyy);
            }
        }
        f_SE_x_bounding = (float)xxx;
        f_SE_y_bounding = (float)yyy;
    }


    // Handle special USGS geoTIFF case here.  We only have
    // four boundaries because the edges are aligned with
    // lat/long.
    if (have_fgd)
    {
        f_NW_x_bounding = f_west_bounding;
        f_NW_y_bounding = f_north_bounding;

        f_SW_x_bounding = f_west_bounding;
        f_SW_y_bounding = f_south_bounding;

        f_NE_x_bounding = f_east_bounding;
        f_NE_y_bounding = f_north_bounding;

        f_SE_x_bounding = f_east_bounding;
        f_SE_y_bounding = f_south_bounding;
    }


    // Fill in the wgs84 variables so we can do a datum
    // conversion but keep our original values also.
    f_NW_x_bounding_wgs84 = f_NW_x_bounding;
    f_NW_y_bounding_wgs84 = f_NW_y_bounding;

    f_SW_x_bounding_wgs84 = f_SW_x_bounding;
    f_SW_y_bounding_wgs84 = f_SW_y_bounding;

    f_NE_x_bounding_wgs84 = f_NE_x_bounding;
    f_NE_y_bounding_wgs84 = f_NE_y_bounding;

    f_SE_x_bounding_wgs84 = f_SE_x_bounding;
    f_SE_y_bounding_wgs84 = f_SE_y_bounding;


    /* Get the datum */
    // GTIFKeyGet( gtif, GeogGeodeticDatumGeoKey, &datum, 0, 1 );
    // if (debug_level & 2)
    //     printf( "GeogGeodeticDatumGeoKey: %d\n", datum );


    /* Get the tiepoints (in UTM coordinates always?)
     * In our case they look like:
     *
     * 0.000000         Y
     * 0.000000         X
     * 0.000000         Z
     * 572983.025771    Y in UTM (longitude for some maps?)
     * 5331394.085064   X in UTM (latitude for some maps?)
     * 0.000000         Z
     *
     */
    // if (debug_level & 2)
    //     printf("Tiepoints:\n");
    // TIFFGetField( tif, TIFFTAG_GEOTIEPOINTS, &qty, &GeoTie );
    // for ( i = 0; i < qty; i++ )
    //   if (debug_level & 2)
    //     printf( "%f\n", *(GeoTie + i) );


    /* Get the geotiff horizontal datum name */
    GTIFGetDatumInfo( defn.Datum, &datum_name, NULL );
    if (debug_level & 2)
        printf("Datum: %d/%s\n", defn.Datum, datum_name );


    /*
     * Perform a datum shift on the bounding coordinates before we
     * check whether the map is inside our viewport.  At the moment
     * this is still hard-coded to NAD27 datum.  If the map is already
     * in WGS84 or NAD83 datum, skip the datum conversion code.
     */
    if (   (defn.Datum != 6030)     /* DatumE_WGS84 */
        && (defn.Datum != 6326)     /*  Datum_WGS84 */
        && (defn.Datum != 6269) )   /* Datum_North_American_Datum_1983 */
    {
        if (debug_level & 2)
            printf("***** Attempting Datum Conversions\n");

        // Change datum_shift to use arrays and counts and make
        // only one call for all 4 corners
        if (   (! datum_shift_to_wgs84 ( &f_NW_x_bounding_wgs84,
                                        &f_NE_x_bounding_wgs84,
                                        &f_NW_y_bounding_wgs84,
                                        &f_NE_y_bounding_wgs84,
                                        datum_name,
                                        defn.Datum) )
            || (! datum_shift_to_wgs84 ( &f_SW_x_bounding_wgs84,
                                        &f_SE_x_bounding_wgs84,
                                        &f_SW_y_bounding_wgs84,
                                        &f_SE_y_bounding_wgs84,
                                        datum_name,
                                        defn.Datum) ) )
        {
            /* Problem doing the datum shift */
            printf("Problem with datum shift.  Perhaps that conversion is not implemented?\n");
            /*
            GTIFFree (gtif);
            XTIFFClose (tif);
            return;
            */
        }
    }
    else
        if (debug_level & 2)
            printf("***** Skipping Datum Conversion\n");


    /*
     * Convert new datum-translated bounding coordinates to the
     * Xastir coordinate system.
     * convert_to_xastir_coordinates( x,y,longitude,latitude )
     */
    // NW corner
    convert_to_xastir_coordinates(  &NW_x_bounding_wgs84,
                                    &NW_y_bounding_wgs84,
                                    f_NW_x_bounding_wgs84,
                                    f_NW_y_bounding_wgs84 );

    // NE corner
    convert_to_xastir_coordinates(  &NE_x_bounding_wgs84,
                                    &NE_y_bounding_wgs84,
                                    f_NE_x_bounding_wgs84,
                                    f_NE_y_bounding_wgs84 );

    // SW corner
    convert_to_xastir_coordinates(  &SW_x_bounding_wgs84,
                                    &SW_y_bounding_wgs84,
                                    f_SW_x_bounding_wgs84,
                                    f_SW_y_bounding_wgs84 );

    // SE corner
    convert_to_xastir_coordinates(  &SE_x_bounding_wgs84,
                                    &SE_y_bounding_wgs84,
                                    f_SE_x_bounding_wgs84,
                                    f_SE_y_bounding_wgs84 );


    /*
     * Check whether map is inside our current view.  These
     * are the real datum-shifted bounding coordinates now,
     * so this is the final decision as to whether the map
     * should be loaded.
     */

    // Find the largest dimensions
    if (NW_y_bounding_wgs84 <= NE_y_bounding_wgs84)
        north_bounding_wgs84 = NW_y_bounding_wgs84;
    else
        north_bounding_wgs84 = NE_y_bounding_wgs84;

    if (NW_x_bounding_wgs84 <= SW_x_bounding_wgs84)
        west_bounding_wgs84 = NW_x_bounding_wgs84;
    else
        west_bounding_wgs84 = SW_x_bounding_wgs84;

    if (SW_y_bounding_wgs84 >= SE_y_bounding_wgs84)
        south_bounding_wgs84 = SW_y_bounding_wgs84;
    else
        south_bounding_wgs84 = SE_y_bounding_wgs84;

    if (NE_x_bounding_wgs84 >= SE_x_bounding_wgs84)
        east_bounding_wgs84 = NE_x_bounding_wgs84;
    else
        east_bounding_wgs84 = SE_x_bounding_wgs84;

    // bottom top left right
    if (!map_visible( south_bounding_wgs84,
                         north_bounding_wgs84,
                         west_bounding_wgs84,
                         east_bounding_wgs84 ) )
    {
        if (debug_level & 2)
        {
            printf ("Map not within current view.\n");
            printf ("Skipping map: %s\n", file);
        }

        /*
         * Map isn't inside our current view.  We're done.
         * Free any memory used and return
         */

        /* We're finished with the geoTIFF key parser, so get rid of it */
        GTIFFree (gtif);

        /* Close the TIFF file descriptor */
        XTIFFClose (tif);

        return;         /* Skip this map */
    }


    /* Fetch a few TIFF fields for this image */
    TIFFGetField (tif, TIFFTAG_ROWSPERSTRIP, &rowsPerStrip);
    TIFFGetField (tif, TIFFTAG_BITSPERSAMPLE, &bitsPerSample);
    TIFFGetField (tif, TIFFTAG_SAMPLESPERPIXEL, &samplesPerPixel);
    TIFFGetField (tif, TIFFTAG_PLANARCONFIG, &planarConfig);


    if (debug_level & 2)
    {
        printf ("            Width: %ld\n", width);
        printf ("           Height: %ld\n", height);
        printf ("   Rows Per Strip: %ld\n", rowsPerStrip);
        printf ("  Bits Per Sample: %d\n", bitsPerSample);
        printf ("Samples Per Pixel: %d\n", samplesPerPixel);
        printf ("    Planar Config: %d\n", planarConfig);
    }


    /*
     * Check for properly formatted geoTIFF file.  If it isn't
     * in the standard format we're looking for, spit out an
     * error message and return.
     *
     * Should we also check compression method here?
     */
    /* if ( (   rowsPerStrip != 1) */
    if ( (samplesPerPixel != 1)
        || (  bitsPerSample != 8)
        || (   planarConfig != 1) )
    {
        printf("*** geoTIFF file %s is not in the proper format.\n", file);
        printf("*** Please reformat it and try again.\n");
        XTIFFClose(tif);
        return;
    }


    if (debug_level & 2)
        printf ("Loading geoTIFF map: %s\n", file);


    /* Put "Loading..." message on status line */
    sprintf (map_it, langcode ("BBARSTA028"), filenm);
    statusline(map_it,0);       // Loading ...


    /*
     * Snag the original map colors out of the colormap embedded
     * inside the tiff file.
     */
    if (!TIFFGetField(tif, TIFFTAG_COLORMAP, &red_orig, &green_orig, &blue_orig))
    {
        TIFFError(TIFFFileName(tif), "Missing required \"Colormap\" tag");
        GTIFFree (gtif);
        XTIFFClose (tif);
        return;
    }


    /* Here are the number of possible colors.  It turns out to
     * be 256 for a USGS geotiff file, of which only the first
     * 13 are used.  Other types of geotiff's may use more
     * colors (and do).  A proposed revision to the USGS DRG spec
     * allows using more colors.
     */
    num_colors = (1L << bitsPerSample);


    /* Print out the colormap info */
    //    if (debug_level & 2)
    //    {
    //        int l;
    //
    //        for (l = 0; l < num_colors; l++)
    //            printf("   %5u: %5u %5u %5u\n",
    //                        l,
    //                        red_orig[l],
    //                        green_orig[l],
    //                        blue_orig[l]);
    //    }


    if (crop_it)    // USGS geoTIFF map
    {
         /*
         * Next:
         * Convert the map neat-line corners to image x/y coordinates.
         * This will give the map neat-line coordinates in pixels.
         * Use this data to chop the image at these boundaries
         * and to stretch the shorter lines to fit a rectangle.
         *
         * Note that at this stage we're using the bounding coordinates
         * that are in the map original datum so that the translations
         * to pixel coordinates will be correct.
         *
         * Note that we already have the datum-shifted values for all
         * the corners in the *_wgs84 variables.  In short:  We use the
         * non datum-shifted values to work with the tiff file, and the
         * datum-shifted values to plot the points in Xastir.
         */

        if (debug_level & 2)
            printf("\nNW neat-line corner = %f\t%f\n",
                    f_NW_x_bounding,
                    f_NW_y_bounding);

        xxx = (double)f_NW_x_bounding;
        yyy = (double)f_NW_y_bounding;

        /* Convert lat/long to projected coordinates */
        if ( GTIFProj4FromLatLong( &defn, 1, &xxx, &yyy ) )     // Do all 4 in one call?
        {
            if (debug_level & 2)
                printf("%11.3f,%11.3f\n", xxx, yyy);

            /* Convert from PCS coordinates to image pixel coordinates */
            if ( GTIFPCSToImage( gtif, &xxx, &yyy ) )           // Do all 4 in one call?
            {
                if (debug_level & 2)
                    printf("X/Y Pixels: %f, %f\n", xxx, yyy);

                NW_x = (int)(xxx + 0.5);    /* Tricky way of rounding */
                NW_y = (int)(yyy + 0.5);    /* Tricky way of rounding */

                if (debug_level & 2)
                    printf("X/Y Pixels: %d, %d\n", NW_x, NW_y);
            }
        }
        else
            printf("Problem in translating\n");


        if (debug_level & 2)
            printf("NE neat-line corner = %f\t%f\n",
                    f_NE_x_bounding,
                    f_NE_y_bounding);

        xxx = (double)f_NE_x_bounding;
        yyy = (double)f_NE_y_bounding;

        /* Convert lat/long to projected coordinates */
        if ( GTIFProj4FromLatLong( &defn, 1, &xxx, &yyy ) )
        {
            if (debug_level & 2)
                printf("%11.3f,%11.3f\n", xxx, yyy);

            /* Convert from PCS coordinates to image pixel coordinates */
            if ( GTIFPCSToImage( gtif, &xxx, &yyy ) )
            {
                if (debug_level & 2)
                    printf("X/Y Pixels: %f, %f\n", xxx, yyy);

                NE_x = (int)(xxx + 0.5);    /* Tricky way of rounding */
                NE_y = (int)(yyy + 0.5);    /* Tricky way of rounding */

                if (debug_level & 2)
                    printf("X/Y Pixels: %d, %d\n", NE_x, NE_y);
            }
        }
        else
            printf("Problem in translating\n");


        if (debug_level & 2)
            printf("SW neat-line corner = %f\t%f\n",
                    f_SW_x_bounding,
                    f_SW_y_bounding);

        xxx = (double)f_SW_x_bounding;
        yyy = (double)f_SW_y_bounding;

        /* Convert lat/long to projected coordinates */
        if ( GTIFProj4FromLatLong( &defn, 1, &xxx, &yyy ) )
        {
            if (debug_level & 2)
                printf("%11.3f,%11.3f\n", xxx, yyy);

            /* Convert from PCS coordinates to image pixel coordinates */
            if ( GTIFPCSToImage( gtif, &xxx, &yyy ) )
            {
                if (debug_level & 2)
                    printf("X/Y Pixels: %f, %f\n", xxx, yyy);

                SW_x = (int)(xxx + 0.5);    /* Tricky way of rounding */
                SW_y = (int)(yyy + 0.5);    /* Tricky way of rounding */

                if (debug_level & 2)
                    printf("X/Y Pixels: %d, %d\n", SW_x, SW_y);
            }
        }
        else
            printf("Problem in translating\n");


        if (debug_level & 2)
            printf("SE neat-line corner = %f\t%f\n",
                    f_SE_x_bounding,
                    f_SE_y_bounding);

        xxx = (double)f_SE_x_bounding;
        yyy = (double)f_SE_y_bounding;

        /* Convert lat/long to projected coordinates */
        if ( GTIFProj4FromLatLong( &defn, 1, &xxx, &yyy ) )
        {
            if (debug_level & 2)
                printf("%11.3f,%11.3f\n", xxx, yyy);

        /* Convert from PCS coordinates to image pixel coordinates */
        if ( GTIFPCSToImage( gtif, &xxx, &yyy ) )
        {
            if (debug_level & 2)
                printf("X/Y Pixels: %f, %f\n", xxx, yyy);

            SE_x = (int)(xxx + 0.5);    /* Tricky way of rounding */
            SE_y = (int)(yyy + 0.5);    /* Tricky way of rounding */

            if (debug_level & 2)
                printf("X/Y Pixels: %d, %d\n", SE_x, SE_y);
            }
        }
        else
            printf("Problem in translating\n");
    }
    else    /*
             * No map collar to crop off, so we already know
             * where the corner points are.  This is for non-USGS
             * maps.
             */
    {
        NW_x = 0;
        NW_y = 0;

        NE_x = width - 1;
        NE_y = 0;

        SW_x = 0;
        SW_y = height - 1;

        SE_x = width - 1;
        SE_y = height - 1;
    }


    // Here's where we crop off part of the black border for USGS maps.
    if (crop_it)    // USGS maps only
    {
        int i = 3;

        NW_x += i;
        NW_y += i;
        NE_x -= i;
        NE_y += i;
        SW_x += i;      
        SW_y -= i;
        SE_x -= i;
        SE_y -= i;
    }


    // Now figure out the rough pixel crop values from what we know.
    // Image rotation means a simple rectangular crop isn't sufficient.
    if (NW_y < NE_y)
        top_crop = NW_y;
    else
        top_crop = NE_y;


    if (SW_y > SE_y)
        bottom_crop = SW_y;
    else
        bottom_crop = SE_y;

 
    if (NE_x > SE_x)
        right_crop = NE_x;
    else
        right_crop = SE_x;

 
    if (NW_x < SW_x)
        left_crop = NW_x;
    else
        left_crop = SW_x;


    if (!crop_it)       /* If we shouldn't crop the map collar... */
    {
      top_crop = 0;
      bottom_crop = height - 1;
      left_crop = 0;
      right_crop = width - 1;
    }

    // The four crop variables are the maximum rectangle that we
    // wish to keep, rotation notwithstanding (we may want to crop
    // part of some lines due to rotation).  Crop all lines/pixels
    // outside these ranges.


    if (debug_level & 2)
    {
        printf("Crop points (pixels):\n");
        printf("Top: %d\tBottom: %d\tLeft: %d\tRight: %d\n",
        top_crop,
        bottom_crop,
        left_crop,
        right_crop);
    }


    /*
     * The color map is embedded in the geoTIFF file as TIFF tags.
     * We get those tags out of the file and translate to our own
     * colormap.
     * Allocate colors for the map image.  We allow up to 256 colors
     * and allow only 8-bits per pixel in the original map file.  We
     * get our 24-bit RGB colors right out of the map file itself, so
     * the colors should look right.
     * We're picking existing colormap colors that are closest to
     * the original map colors, so we shouldn't run out of colors
     * for other applications.
     *
     * Brightness adjust for the colors?  Implemented in the
     * "geotiff_map_intensity" variable below.
     */

    cmap = DefaultColormap( XtDisplay (w), DefaultScreen( XtDisplay (w) ) );
    {
        int l;
        // float geotiff_map_intensity = 1.00;    // Change this to reduce the
                                    // intensity of the map colors

        for (l = 0; l < num_colors; l++)
        {
            my_colors[l].red   =   (uint16)(red_orig[l] * geotiff_map_intensity);
            my_colors[l].green = (uint16)(green_orig[l] * geotiff_map_intensity);
            my_colors[l].blue  =  (uint16)(blue_orig[l] * geotiff_map_intensity);
      
            XAllocColor( XtDisplay (w), cmap, &my_colors[l] );
        }
    }


    // Each data value should be an 8-bit value, which is a
    // pointer into a color
    // table.  Later we perform a translation from the geoTIFF
    // color table to our current color table (matching values
    // as close as possible), at the point where we're writing
    // the image to the pixmap.


    /* We should be ready now to actually read in some
     * pixels and deposit them on the screen.  We will
     * allocate memory for the data area based on the
     * sizes of fields and data in the geoTIFF file.
     */

    bytesPerRow = TIFFScanlineSize(tif);

    if (debug_level & 2)
    {
        printf("\nInitial Bytes Per Row: %d\n", bytesPerRow);
    }


    // Here's a tiny malloc that'll hold only one scanline worth of pixels
    imageMemory = (u_char *) malloc(bytesPerRow + 2);

    CHECKMALLOC(imageMemory);


    // TODO:  Figure out the middle boundary on each edge for
    // lat/long and adjust the crop values to match the largest
    // of either the middle or the corners for each edge.  This
    // will help to handle edges that are curved.


    /*
     * There are some optimizations that can still be done:
     *
     * 1) Read in all scanlines but throw away unneeded pixels,
     *    paying attention not to lose smaller details.  Compare
     *    neighboring pixels?
     *
     * 3) Keep a map cache or a screenmap cache to reduce need
     *    for reading map files so often.
     */



    // Here we wish to start at the top line that may have
    // some pixels of interest and proceed to the bottom line
    // of interest.  Process scanlines from top_crop to bottom_crop.
    // Start at the left/right_crop pixels, compute the lat/long
    // of each, using x/y increments so we can quickly scan across
    // the line.
    // Iterate across the line checking whether each pixel is
    // within the viewport.  If so, plot it on the pixmap at
    // the correct scale.

    // Later I may wish to get the lat/lon of each pixel and plot
    // it at the correct point, to handle the curvature of each
    // line (this might be VERY slow).  Right now I treat them as
    // straight lines.

    // At this point we have these variables defined.  The
    // first column contains map corners in Xastir coordinates,
    // the second column contains map corners in pixels:
    //
    // NW corner:
    // NW_x_bounding_wgs84  <-> NW_x
    // NW_y_bounding_wgs84  <-> NW_y
    //
    // NE corner:
    // NE_x_bounding_wgs84  <-> NE_x
    // NE_y_bounding_wgs84  <-> NE_y
    //
    // SW corner:
    // SW_x_bounding_wgs84  <-> SW_x
    // SW_y_bounding_wgs84  <-> SW_y
    //
    // SE corner:
    // SE_x_bounding_wgs84  <-> SE_x
    // SE_y_bounding_wgs84  <-> SE_y

    // I should be able to use these variables to figure out
    // the xastir coordinates of each scanline pixel using
    // linear interpolation along each edge.
    //
    // I don't want to use the crop values in general.  I'd
    // rather crop properly along the neat line instead of a
    // rectangular crop.
    //
    // Define lines along the left/right edges so that I can
    // compute the Xastir coordinates of each pixel along these
    // two lines.  These will be the start/finish of each of my
    // scanlines, and I can use these values to compute the
    // x/y_increment values for each line.  This way I can
    // stretch short lines as I go along, and auto-crop the
    // white border as well.

    // Left_line goes from (top to bottom):
    // NW_x,NW_y -> SW_x,SW_y
    // and from:
    // west_bounding_wgs84,north_bounding_wgs84 -> west_bounding_wgs84,south_bounding_wgs84
    //
    // Right_line goes from(top to bottom):
    // NE_x,NE_y -> SE_x,SE-Y
    // and from:
    // east_bounding_wgs84,north_bounding_wgs84 -> east_bounding_wgs84,south_bounding_wgs84
    //
    // Simpler:  Along each line, Xastir coordinates change how much
    // and in what direction as we move down one scanline?


    // These increments are how much we change in Xastir coordinates and
    // in pixel coordinates as we move down either the left or right
    // neatline one pixel.
    // Be prepared for 0 angle of rotation as well (x-increments = 0).


    // Xastir Coordinate System:
    //
    //              0 (90 deg. or 90N)
    //
    // 0 (-180 deg. or 180W)      129,600,000 (180 deg. or 180E)
    //
    //          64,800,000 (-90 deg. or 90S)


    // Watch out for division by zero here.


    //
    // Left Edge X Increment Per Scanline (Going from top to bottom).
    // This increment will help me to keep track of the left edge of
    // the image, both in Xastir coordinates and in pixel coordinates.
    //
    if (SW_y != NW_y)
    {
        // Xastir coordinates
        xastir_left_x_increment = (float)
            (1.0 * abs(SW_x_bounding_wgs84 - NW_x_bounding_wgs84)   // Need to add one pixel worth here yet
            / abs(SW_y - NW_y));

        // Pixel coordinates
        left_x_increment = (float)(1.0 * abs(SW_x - NW_x)
                            / abs(SW_y - NW_y));

        if (SW_x_bounding_wgs84 < NW_x_bounding_wgs84)
            xastir_left_x_increment = -xastir_left_x_increment;

        if (SW_x < NW_x)
            left_x_increment = -left_x_increment;

        if (debug_level & 2)
             printf("xastir_left_x_increment: %f  %ld  %ld     %f  %d  %d  %d  %d\n",
             xastir_left_x_increment,
             SW_x_bounding_wgs84,
             NW_x_bounding_wgs84,
             left_x_increment,
             SW_x,
             NW_x,
             bottom_crop,
             top_crop);
    }
    else
    {
        // Xastir coordinates
        xastir_left_x_increment = 0;

        // Pixel coordinates
        left_x_increment = 0;
    }


    //
    // Left Edge Y Increment Per Scanline (Going from top to bottom)
    // This increment will help me to keep track of the left edge of
    // the image, both in Xastir coordinates and in pixel coordinates.
    //
    if (SW_y != NW_y)
    {
        // Xastir coordinates
        xastir_left_y_increment = (float)
            (1.0 * abs(SW_y_bounding_wgs84 - NW_y_bounding_wgs84)   // Need to add one pixel worth here yet
            / abs(SW_y - NW_y));

        // Pixel coordinates
        left_y_increment = (float)1.0; // Aren't we going down one pixel each time?

        if (SW_y_bounding_wgs84 < NW_y_bounding_wgs84)  // Ain't gonn'a happen
            xastir_left_y_increment = -xastir_left_y_increment;

        if (debug_level & 2)
             printf("xastir_left_y_increment: %f  %ld  %ld     %f  %d  %d  %d  %d\n",
             xastir_left_y_increment,
             SW_y_bounding_wgs84,
             NW_y_bounding_wgs84,
             left_y_increment,
             SW_y,
             NW_y,
             bottom_crop,
             top_crop);
    }
    else
    {
        // Xastir coordinates
        xastir_left_y_increment = 0;

        // Pixel coordinates
        left_y_increment = 0;
    }


    //
    // Right Edge X Increment Per Scanline (Going from top to bottom)
    // This increment will help me to keep track of the right edge of
    // the image, both in Xastir coordinates and image coordinates.
    //
    if (SE_y != NE_y)
    {
        // Xastir coordinates
        xastir_right_x_increment = (float)
            (1.0 * abs(SE_x_bounding_wgs84 - NE_x_bounding_wgs84)   // Need to add one pixel worth here yet
            / abs(SE_y - NE_y));

        // Pixel coordinates
        right_x_increment = (float)(1.0 * abs(SE_x - NE_x)
                            / abs(SE_y - NE_y));

        if (SE_x_bounding_wgs84 < NE_x_bounding_wgs84)
            xastir_right_x_increment = -xastir_right_x_increment;

        if (SE_x < NE_x)
            right_x_increment = -right_x_increment;

        if (debug_level & 2)
            printf("xastir_right_x_increment: %f  %ld  %ld     %f  %d  %d  %d  %d\n",
            xastir_right_x_increment,
            SE_x_bounding_wgs84,
            NE_x_bounding_wgs84,
            right_x_increment,
            SE_x,
            NE_x,
            bottom_crop,
            top_crop);
    }
    else
    {
        // Xastir coordinates
        xastir_right_x_increment = 0;

        // Pixel coordinates
        right_x_increment = 0;
    }


    //
    // Right Edge Y Increment Per Scanline (Going from top to bottom)
    // This increment will help me to keep track of the right edge of
    // the image, both in Xastir coordinates and in image coordinates.
    //
    if (SE_y != NE_y)
    {
        // Xastir coordinates
        xastir_right_y_increment = (float)
            (1.0 * abs(SE_y_bounding_wgs84 - NE_y_bounding_wgs84)   // Need to add one pixel worth here yet
            / abs(SE_y - NE_y));

        // Pixel coordinates
        right_y_increment = (float)1.0;    // Aren't we going down one pixel each time?

        if (SE_y_bounding_wgs84 < NE_y_bounding_wgs84)  // Ain't gonn'a happen
            xastir_right_y_increment = -xastir_right_y_increment;

        if (debug_level & 2)
            printf("xastir_right_y_increment: %f  %ld  %ld     %f  %d  %d  %d  %d\n",
            xastir_right_y_increment,
            SE_y_bounding_wgs84,
            NE_y_bounding_wgs84,
            right_y_increment,
            SE_y,
            NE_y,
            bottom_crop,
            top_crop);
    }
    else
    {
        // Xastir coordinates
        xastir_right_y_increment = 0;

        // Pixel coordinates
        right_y_increment = 0;
    }


    if (debug_level & 2)
    {
        printf(" Left x increments: %f %f\n", xastir_left_x_increment, left_x_increment);
        printf(" Left y increments: %f %f\n", xastir_left_y_increment, left_y_increment);
        printf("Right x increments: %f %f\n", xastir_right_x_increment, right_x_increment);
        printf("Right y increments: %f %f\n", xastir_right_y_increment, right_y_increment);
    }


    // Compute how much "y" changes per pixel as we traverse from left to right
    // along a scanline along the top of the image.
    //
    // Top Edge Y Increment Per X-Pixel Width (Going from left to right).
    // This increment will help me to get rid of image rotation.
    //
    if (NE_x != NW_x)
    {
        // Xastir coordinates
        xastir_top_y_increment = (float)
            (1.0 * abs(NE_y_bounding_wgs84 - NW_y_bounding_wgs84)   // Need to add one pixel worth here yet
            / abs(NE_x - NW_x));    // And a "+ 1.0" here?

        // Pixel coordinates
        top_y_increment = (float)(1.0 * abs(NE_y - NW_y)
                    / abs(NE_x - NW_x));

        if (NE_y_bounding_wgs84 < NW_y_bounding_wgs84)
            xastir_top_y_increment = -xastir_top_y_increment;

        if (NE_y < NW_y)
            top_y_increment = -top_y_increment;

        if (debug_level & 2)
            printf("xastir_top_y_increment: %f  %ld  %ld     %f  %d  %d  %d  %d\n",
            xastir_top_y_increment,
            NE_y_bounding_wgs84,
            NW_y_bounding_wgs84,
            top_y_increment,
            NE_y,
            NW_y,
            right_crop,
            left_crop);
    }
    else
    {
        // Xastir coordinates
        xastir_top_y_increment = 0;

        // Pixel coordinates
        top_y_increment = 0;
    }


    // Compute how much "y" changes per pixel as you traverse from left to right
    // along a scanline along the bottom of the image.
    //
    // Bottom Edge Y Increment Per X-Pixel Width (Going from left to right).
    // This increment will help me to get rid of image rotation.
    //
    if (SE_x != SW_x)
    {
        // Xastir coordinates
        xastir_bottom_y_increment = (float)
            (1.0 * abs(SE_y_bounding_wgs84 - SW_y_bounding_wgs84)   // Need to add one pixel worth here yet
            / abs(SE_x - SW_x));    // And a "+ 1.0" here?

        // Pixel coordinates
        bottom_y_increment = (float)(1.0 * abs(SE_y - SW_y)
                        / abs(SE_x - SW_x));

        if (SE_y_bounding_wgs84 < SW_y_bounding_wgs84)  
            xastir_bottom_y_increment = -xastir_bottom_y_increment;

        if (SE_y < SW_y)
            bottom_y_increment = -bottom_y_increment;

        if (debug_level & 2)
            printf("xastir_bottom_y_increment: %f  %ld  %ld     %f  %d  %d  %d  %d\n",
            xastir_bottom_y_increment,
            SE_y_bounding_wgs84,
            SW_y_bounding_wgs84,
            bottom_y_increment,
            SE_y,
            SW_y,
            right_crop,
            left_crop);
    }
    else
    {
        // Xastir coordinates
        xastir_bottom_y_increment = 0;

        // Pixel coordinates
        bottom_y_increment = 0;
    }


    // Find the average change in Y as we traverse from left to right one pixel
    xastir_avg_y_increment = (float)(xastir_top_y_increment + xastir_bottom_y_increment) / 2.0;
    avg_y_increment = (float)(top_y_increment + bottom_y_increment) / 2.0;


    // Find edges of current viewport in Xastir coordinates
    //
    view_min_x = x_long_offset;                         /*   left edge of view */
    if (view_min_x > 129600000l)
        view_min_x = 0;

    view_max_x = x_long_offset + (screen_width * size); /*  right edge of view */
    if (view_max_x > 129600000l)
        view_max_x = 129600000l;

    view_min_y = y_lat_offset;                          /*    top edge of view */
    if (view_min_y > 64800000l)
        view_min_y = 0;

    view_max_y = y_lat_offset + (screen_height * size); /* bottom edge of view */
    if (view_max_y > 64800000l)
        view_max_y = 64800000l;


    /* Get the pixel scale */
    TIFFGetField( tif, TIFFTAG_GEOPIXELSCALE, &qty, &PixelScale );
    if (debug_level & 2)
        printf("PixelScale: %f %f %f\n",
            *PixelScale,
            *(PixelScale + 1),
            *(PixelScale + 2) );


    // Use PixelScale to determine lines to skip at each
    // zoom level?
    // O-size map:
    // ModelPixelScaleTag (1,3):
    //   2.4384           2.4384           0
    //
    // F-size map:
    // ModelPixelScaleTag (1,3):
    //   10.16            10.16            0                
    //
    // C-size map:
    // ModelPixelScaleTag (1,3):
    //   25.400001        25.400001        0                
    //

    if (debug_level & 2)
        printf("Size: %ld\n", size);


    // I tried to be very aggressive with the scaling factor
    // below (3.15) in order to skip the most possible rows
    // to speed things up.  If you see diagonal lines across
    // the maps, increase this number (up to a max of 4.0
    // probably).  A higher number means less rows skipped,
    // which improves the look but slows the map drawing down.
    //
    SkipRows = (int)( size / ( *PixelScale * 3.15 ) );
    if (SkipRows < 1)
        SkipRows = 1;
    if (SkipRows > (height / 10) )
        SkipRows = height / 10;
    if (debug_level & 2)
        printf("SkipRows: %d\n", SkipRows);

    // Use SkipRows to set increments for the loops below.


    if ( top_crop <= 0 )
        top_crop = 0;               // First row of image

    if ( ( bottom_crop + 1) >= height )
        bottom_crop = height - 1;   // Last row of image


    // Here I pre-compute some of the values I'll need in the
    // loops below in order to save some time.
    NW_line_offset = (int)(NW_y - top_crop);
    NE_line_offset = (int)(NE_y - top_crop);

    NW_xastir_x_offset =  (int)(xastir_left_x_increment * NW_line_offset);
    NE_xastir_x_offset = (int)(xastir_right_x_increment * NE_line_offset);
    NW_xastir_y_offset =  (int)(xastir_left_y_increment * NW_line_offset);

    NW_x_offset =  (int)(1.0 * left_x_increment * NW_line_offset);
    NE_x_offset = (int)(1.0 * right_x_increment * NE_line_offset);
    xastir_avg_left_right_y_increment = (float)((xastir_right_y_increment + xastir_left_y_increment) / 2.0);
    total_avg_y_increment = (float)(xastir_avg_left_right_y_increment * avg_y_increment);



    // (Xastir bottom - Xastir top) / height
    //steph = (double)( (left_y_increment + right_y_increment) / 2); 
    // NOTE:  This one does not take into account current height
    steph = (float)( (SW_y_bounding_wgs84 - NW_y_bounding_wgs84)
                      / (1.0 * (SW_y - NW_y) ) );


    // Compute scaled pixel size for XFillRectangle
    stephc = (int)( (1.0 * steph / size) + 1.0);


    view_top_minus_pixel_height = (unsigned long)(view_min_y - steph);


    // Iterate over the rows of interest only.  Using the rectangular
    // top/bottom crop values for these is ok at this point.
    //
    // Put row multipliers above loops.  Try to get as many
    // multiplications as possible outside the loops.  Adds and
    // subtracts are ok.  Try to do as little floating point stuff
    // as possible inside the loops.  I also declared a lot of
    // the inner loop stuff as register variables.  Saved me about
    // a second per map (not much, but I'll take what I can get!)
    //
    for ( row = top_crop; row < bottom_crop + 1; row+= SkipRows )
    {
        int skip = 0;


        // Our offset from the top row of the map neatline
        // (kind of... ignoring rotation anyway).
        row_offset = row - top_crop;
        //printf("row_offset: %d\n", row_offset);


        // Compute the line end-points in Xastir coordinates
        // Initially was a problem here:  Offsetting from NW_x_bounding but
        // starting at top_crop line.  Fixed by last term added to two
        // equations below.

        current_xastir_left = (unsigned long)
              ( NW_x_bounding_wgs84
            + ( 1.0 * xastir_left_x_increment * row_offset )
            -   NW_xastir_x_offset );

        current_xastir_right = (unsigned long)
              ( NE_x_bounding_wgs84
            + ( 1.0 * xastir_right_x_increment * row_offset )
            -   NE_xastir_x_offset );


        //if (debug_level & 2)
        //  printf("Left: %ld  Right:  %ld\n",
        //      current_xastir_left,
        //      current_xastir_right);


        // In pixel coordinates:
        current_left = (int)
                          ( NW_x
                        + ( 1.0 * left_x_increment * row_offset ) + 0.5
                        -   NW_x_offset );

        current_right = (int)
                          ( NE_x
                        + ( 1.0 * right_x_increment * row_offset ) + 0.5
                        -   NE_x_offset );

        current_line_width = current_right - current_left + 1;  // Pixels


        // if (debug_level & 2)
        //     printf("Left: %ld  Right: %ld  Width: %ld\n",
        //         current_left,
        //         current_right, current_line_width);


        // Compute original pixel size in Xastir coordinates.  Note
        // that this can change for each scanline in a USGS geoTIFF.

        // (Xastir right - Xastir left) / width-of-line
        // Need the "1.0 *" or the math will be incorrect (won't be a float)
        stepw = (float)( (current_xastir_right - current_xastir_left)
                      / (1.0 * (current_right - current_left) ) );



        // if (debug_level & 2)
        //     printf("\t\t\t\t\t\tPixel Width: %f\n",stepw);

        // Compute scaled pixel size for XFillRectangle
        stepwc = (int)( (1.0 * stepw / size) + 1.0);


        // In Xastir coordinates
        xastir_current_y = (unsigned long)(NW_y_bounding_wgs84
                  + (xastir_left_y_increment * row_offset) );

        xastir_current_y = (unsigned long)(xastir_current_y - NW_xastir_y_offset);


        view_left_minus_pixel_width = view_min_x - stepw;


        // Check whether any part of the scanline will be within the
        // view.  If so, read the scanline from the file and iterate
        // across the pixels.  If not, skip this line altogether.

        // Compute right edge of image
        xastir_total_y = (unsigned long)
                           ( xastir_current_y
                         - ( total_avg_y_increment * (current_right - current_left) ) );

        // Check left edge y-value then right edge y-value.
        // If either are within view, process the line, else skip it.
        if ( ( ( xastir_current_y <= view_max_y) && (xastir_total_y >= view_top_minus_pixel_height) )
            || ( ( xastir_total_y <= view_max_y ) && ( xastir_total_y >= view_top_minus_pixel_height ) ) )
        {
            // Read one geoTIFF scanline
            if (TIFFReadScanline(tif, imageMemory, row, 0) < 0)
                break;  // No more lines to read or we couldn't read the file at all



            // Iterate over the columns of interest, skipping the left/right
            // cropped pixels, looking for pixels that fit within our viewport.
            //
            for ( column = current_left; column < (current_right + 1); column++ )
            {
                skip = 0;

                column_offset = column - current_left;  // Pixels

                //printf("Column Offset: %ld\n", column_offset);  // Pixels
                //printf("Current Left: %ld\n", current_left);    // Pixels

                xastir_current_x = (unsigned long)
                                    current_xastir_left
                                    + (stepw * column_offset);    // In Xastir coordinates

                // Left line y value minus
                // avg y-increment per scanline * avg y-increment per x-pixel * column_offset
                xastir_total_y = (unsigned long)
                                  ( xastir_current_y
                                - ( total_avg_y_increment * column_offset ) );

                //printf("Xastir current: %ld %ld\n", xastir_current_x, xastir_current_y);


                // Check whether pixel fits within boundary lines (USGS maps)
                // This is how we get rid of the last bit of white border at
                // the top and bottom of the image.
                if (have_fgd)   // USGS map
                {
                    if (   (xastir_total_y > SW_y_bounding_wgs84)
                        || (xastir_total_y < NW_y_bounding_wgs84) )
                    skip++;


                    // Here's a trick to make it look like the map pages join better.
                    // If we're within a certain distance of a border, change any black
                    // pixels to white (changes map border to less obtrusive color).
                    if ( *(imageMemory + column) == 0x00 )  // If pixel is Black
                    {
                        if ( (xastir_total_y > (SW_y_bounding_wgs84 - 25) )
                            || (xastir_total_y < (NW_y_bounding_wgs84 + 25) )
                            || (xastir_current_x < (SW_x_bounding_wgs84 + 25) )
                            || (xastir_current_x > (SE_x_bounding_wgs84 - 25) ) )
                        {
                            *(imageMemory + column) = 0x01;     // Change to White
                        }
                    }
                }


                /* Look for left or right map boundaries inside view */    
                if ( !skip
                    && ( xastir_current_x <= view_max_x )
                    && ( xastir_current_x >= view_left_minus_pixel_width )    
                    && ( xastir_total_y <= view_max_y )
                    && ( xastir_total_y >= view_top_minus_pixel_height ) )
                {
                    // Here are the corners of our viewport, using the Xastir
                    // coordinate system.  Notice that Y is upside down:
                    // 
                    // left edge of view = x_long_offset
                    // right edge of view = x_long_offset + (screen_width  * size)
                    // top edge of view =  y_lat_offset
                    // bottom edge of view =  y_lat_offset + (screen_height * size)


                    // Compute the screen position of the pixel and scale it
                    sxx = (long)( ( ( xastir_current_x - x_long_offset ) / size) + 0.5);
                    syy = (long)( ( ( xastir_total_y - y_lat_offset ) / size) + 0.5);

                    // Set the color for the pixel
                    XSetForeground (XtDisplay (w), gc, my_colors[*(imageMemory + column)].pixel);
                    // And draw the pixel
                    XFillRectangle (XtDisplay (w), pixmap, gc, sxx, syy, stepwc, stephc);
                }
            }
        }
    }



    /* Free up any malloc's that we did */
    if (imageMemory)
        free(imageMemory);


    if (debug_level & 2)
        printf ("%d rows read in\n", (int) row);

    /* We're finished with the geoTIFF key parser, so get rid of it */
    GTIFFree (gtif);

    /* Close the TIFF file descriptor */
    XTIFFClose (tif);
}
#endif /* USE_GEOTIFF */





/**********************************************************
 * draw_label_text()
 *
 * Does what it says.  Used to draw strings onto the
 * display.
 **********************************************************/
void draw_label_text (Widget w, int x, int y, int label_length, int color, char *label_text) {
    (void)XSetForeground (XtDisplay (w), gc, colors[0x0ff]);
    (void)XFillRectangle (XtDisplay (w), pixmap, gc, x - 1, (y - 10),(label_length * 6) + 2, 11);

    (void)XSetForeground (XtDisplay (w), gc, color);
    (void)XDrawString (XtDisplay (w), pixmap, gc, x, y, label_text, label_length);
}





/**********************************************************
 * draw_map()
 *
 * Function which tries to figure out what type of map or
 * image file we're dealing with, and takes care of getting
 * it onto the screen.  Calls other functions to deal with
 * .geo or .tif maps.  This function deals with DOS/Windows
 * vector maps itself.
 **********************************************************/
void draw_map (Widget w, char *dir, char *filenm, int tn, alert_entry * alert,
                unsigned char alert_color, int destination_pixmap) {
    FILE *f;
    char file[2000];
    char map_it[300];

    /* map header info */
    char map_type[5];
    char map_version[5];
    char file_name[33];
    char *ext;
    char map_title[33];
    char map_creator[8];
    unsigned long creation_date;
    unsigned long left_boundary;
    unsigned long right_boundary;
    unsigned long top_boundary;
    unsigned long bottom_boundary;
    char map_reserved1[9];
    long total_vector_points;
    long total_labels;
    char map_reserved2[141];
    char Buffer[2049];
    char *ptr;
    int dos_labels;
    int dos_flag;
    long temp;
    int points_per_degree;
    int map_range;

    /* vector info */
    unsigned char vector_start;
    unsigned char object_behavior;
    unsigned long x_long_cord;
    unsigned long y_lat_cord;

    /* label data */
    char label_type[3];
    unsigned long label_x_cord;
    unsigned long label_y_cord;
    int temp_mag;
    int label_mag;
    char label_symbol_del;
    char label_symbol_char;
    char label_text_color;
    char label_text[50];

    unsigned long year;
    unsigned long days;
    long count;
    int label_length;
    int i;
    int map_maxed_vectors;
    int map_maxed_text_labels;
    int map_maxed_symbol_labels;
    map_vectors *vectors_ptr;
    text_label *text_ptr;
    symbol_label *symbol_ptr;
    int line_width;
    int x, y;
    unsigned long max_x_long_offset;
    unsigned long max_y_lat_offset;
    long map_border_min_x;
    long map_border_max_x;
    long map_border_min_y;
    long map_border_max_y;
    int color;
    long max_x, max_y;
    long x_test, y_test;
    int in_window = 0;

    x = 0;
    y = 0;
    color = -1;
    line_width = 1;
    mag = (1 * size) / 2;

    /* MAP counters */
    vectors_ptr = map_vectors_ptr;
    text_ptr = map_text_label_ptr;
    symbol_ptr = map_symbol_label_ptr;

    map_maxed_vectors = 0;
    map_maxed_text_labels = 0;
    map_maxed_symbol_labels = 0;
    npoints = 0;

    sprintf (file, "%s/%s", dir, filenm);

    ext = get_map_ext (filenm);
    if (ext != NULL && strcasecmp (ext, "MAP") == 0) {
        f = fopen (file, "r");
        if (f != NULL) {
            (void)fread (map_type, 4, 1, f);
            map_type[4] = '\0';
            dos_labels = FALSE;
            points_per_degree = 300;
            if (strtod (map_type, &ptr) != 0.0 && (*ptr == '\0' || *ptr == ' ' || *ptr == ',')) {
                /* DOS type map */
                int j;
                top_boundary = left_boundary = bottom_boundary = right_boundary = 0;
                rewind (f);
                map_title[0] = map_creator[0] = Buffer[0] = '\0';
                strncpy (map_type, "DOS ", 4);          // substr ??
                strncpy (file_name, filenm, 32);
                total_vector_points = 30000;
                total_labels = 2000;

// Lclint can't handle this structure for some reason
#ifndef __LCLINT__
                for (j = 0; j < DOS_HDR_LINES; strlen (Buffer) ? : j++) {
#else
// So we do it this way for Lclint:
                for (j = 0; j < DOS_HDR_LINES;) {
                    if (strlen(Buffer))
                        j++;
#endif // __LCLINT__

                    (void)fgets (&Buffer[strlen (Buffer)],(int)sizeof (Buffer) - (strlen (Buffer)), f);
                    while ((ptr = strpbrk (Buffer, "\r\n")) != NULL && j < DOS_HDR_LINES) {
                        *ptr = '\0';
                        for (ptr++; *ptr == '\r' || *ptr == '\n'; ptr++) ;
                        switch (j) {
                            case 0:
                                top_boundary = (unsigned long) (-atof (Buffer) * 360000 + 32400000);
                                break;

                            case 1:
                                left_boundary = (unsigned long) (-atof (Buffer) * 360000 + 64800000);
                                break;

                            case 2:
                                points_per_degree = (int) atof (Buffer);
                                break;

                            case 3:
                                bottom_boundary = (unsigned long) (-atof (Buffer) * 360000 + 32400000);
                                bottom_boundary = bottom_boundary + bottom_boundary - top_boundary;
                                break;

                            case 4:
                                right_boundary = (unsigned long) (-atof (Buffer) * 360000 + 64800000);
                                right_boundary = right_boundary + right_boundary - left_boundary;
                                break;

                            case 5:
                                map_range = (int) atof (Buffer);
                                break;

                            case 7:
                                strncpy (map_version, Buffer, 4);
                                break;
                        }
                        strcpy (Buffer, ptr);
                        if (strlen (Buffer))
                            j++;
                    }
                }   // End of DOS-type map
            } else {    // Windows map
                (void)fread (map_version, 4, 1, f);
                map_version[4] = '\0';

                (void)fread (file_name, 32, 1, f);
                file_name[32] = '\0';

                (void)fread (map_title, 32, 1, f);
                map_title[32] = '\0';
                if (debug_level & 2)
                    printf ("Map Title %s\n", map_title);

                (void)fread (map_creator, 8, 1, f);
                map_creator[8] = '\0';
                if (debug_level & 2)
                    printf ("Map Creator %s\n", map_creator);

                (void)fread (&temp, 4, 1, f);
                creation_date = ntohl (temp);
                if (debug_level & 2)
                    printf ("Creation Date %lX\n", creation_date);

                year = creation_date / 31536000l;
                days = (creation_date - (year * 31536000l)) / 86400l;
                if (debug_level & 2)
                    printf ("year is %ld + days %ld\n", 1904l + year, (long)days);

                (void)fread (&temp, 4, 1, f);
                left_boundary = ntohl (temp);

                (void)fread (&temp, 4, 1, f);
                right_boundary = ntohl (temp);

                (void)fread (&temp, 4, 1, f);
                top_boundary = ntohl (temp);

                (void)fread (&temp, 4, 1, f);
                bottom_boundary = ntohl (temp);

                if (strcmp (map_version, "2.00") != 0) {
                    left_boundary *= 10;
                    right_boundary *= 10;
                    top_boundary *= 10;
                    bottom_boundary *= 10;
                }
                (void)fread (map_reserved1, 8, 1, f);
                (void)fread (&temp, 4, 1, f);
                total_vector_points = (long)ntohl (temp);
                (void)fread (&temp, 4, 1, f);
                total_labels = (long)ntohl (temp);
                (void)fread (map_reserved2, 140, 1, f);
            }   // End of windows map

            if (debug_level & 2) {
                printf ("Map Type %s, Version: %s, Filename %s\n", map_type,map_version, file_name);
                printf ("Left Boundary %ld, Right Boundary %ld\n", (long)left_boundary,(long)right_boundary);
                printf ("Top Boundary %ld, Bottom Boundary %ld\n", (long)top_boundary,(long)bottom_boundary);
                printf ("Total vector points %ld, total labels %ld\n",total_vector_points, total_labels);
            }

            if (alert) {
                strncpy (alert->filename, filenm, sizeof (alert->filename));
                strcpy (alert->title, map_title);
                alert->top_boundary = top_boundary;
                alert->bottom_boundary = bottom_boundary;
                alert->left_boundary = left_boundary;
                alert->right_boundary = right_boundary;
            }

            max_x_long_offset = (unsigned long)(x_long_offset + (screen_width * size));
            max_y_lat_offset = (unsigned long)(y_lat_offset + (screen_height * size));

            if (debug_level & 4)
                printf("x_long_offset: %ld, y_lat_offset: %ld, max_x_long_offset: %ld, max_y_lat_offset: %ld, tn: %d\n",
                        x_long_offset, y_lat_offset, (long)max_x_long_offset, (long)max_y_lat_offset,tn);

            /*
             * Got rid of the autocheck stuff here.  Better to check whether
             * maps fit our viewport and load the ones that do, irregardless
             * of the automaps setting.  Much faster.

            if (!tn) // check if auto map
                in_window = 1;
            else {  // check edges and corners - quick check
            */

            if (1) {    // Always check whether map fits our viewport
                if ((((long)left_boundary <= x_long_offset) && (x_long_offset <= (long)right_boundary) &&
                        ((long)top_boundary <= y_lat_offset) && (y_lat_offset <= (long)bottom_boundary)) ||
                        (((long)left_boundary <= x_long_offset) && (x_long_offset <= (long)right_boundary) &&
                        (top_boundary <= max_y_lat_offset) && (max_y_lat_offset <= bottom_boundary)) ||
                        ((left_boundary <= max_x_long_offset) && (max_x_long_offset <= right_boundary) &&
                        ((long)top_boundary <= y_lat_offset) && (y_lat_offset <= (long)bottom_boundary)) ||
                        ((left_boundary <= max_x_long_offset) && (max_x_long_offset <= right_boundary) &&
                        (top_boundary <= max_y_lat_offset) && (max_y_lat_offset <= bottom_boundary)) ||
                        ((x_long_offset <= (long)left_boundary) && (left_boundary <= max_x_long_offset) &&
                        (y_lat_offset <= (long)top_boundary) && (top_boundary <= max_y_lat_offset)) ||
                        ((x_long_offset <= (long)left_boundary) && (left_boundary <= max_x_long_offset) &&
                        (y_lat_offset <= (long)bottom_boundary) && (bottom_boundary <= max_y_lat_offset)) ||
                        ((x_long_offset <= (long)right_boundary) && (right_boundary <= max_x_long_offset) &&
                        (y_lat_offset <= (long)top_boundary) && (top_boundary <= max_y_lat_offset)) ||
                        ((x_long_offset <= (long)right_boundary) && (right_boundary <= max_x_long_offset) &&
                        (y_lat_offset <= (long)bottom_boundary) && (bottom_boundary <= max_y_lat_offset)))

                    in_window = 1;
                else {
                    // find min and max borders to look at
                    //this routine are for those odd sized maps
                    if ((long)left_boundary > x_long_offset)
                        map_border_min_x = (long)left_boundary;
                    else
                        map_border_min_x = x_long_offset;

                    if (right_boundary < max_x_long_offset)
                        map_border_max_x = (long)right_boundary;
                    else
                        map_border_max_x = (long)max_x_long_offset;

                    if ((long)top_boundary > y_lat_offset)
                        map_border_min_y = (long)top_boundary;
                    else
                        map_border_min_y = y_lat_offset;

                    if (bottom_boundary < max_y_lat_offset)
                        map_border_max_y = (long)bottom_boundary;
                    else
                        map_border_max_y = (long)max_y_lat_offset;

                    // do difficult check in side map
                    for (x_test = map_border_min_x;(x_test <= map_border_max_x && !in_window); x_test += ((size * screen_width) / 10)) {
                        for (y_test = map_border_min_y;(y_test <= map_border_max_y && !in_window);y_test += ((size * screen_height) / 10)) {
                            if ((x_long_offset <= x_test) && (x_test <= (long)max_x_long_offset) && (y_lat_offset <= y_test) &&
                                    (y_test <= (long)max_y_lat_offset))

                                in_window = 1;
                        }
                    }
                }
            }
            if (alert)
                alert->flags[0] = in_window ? 'Y' : 'N';

            if (in_window && !alert) {
                unsigned char last_behavior, special_fill = (unsigned char)FALSE;
                object_behavior = '\0';
                sprintf (map_it, langcode ("BBARSTA028"), filenm);
                statusline(map_it,0);       // Loading ...
                if (debug_level & 1)
                    printf ("in Boundary %s\n", map_it);

                (void)XSetLineAttributes (XtDisplay (w), gc, line_width, LineSolid, CapButt,JoinMiter);

                /* read vectors */
                max_x = screen_width + MAX_OUTBOUND;
                max_y = screen_height + MAX_OUTBOUND;

                x_long_cord = 0;
                y_lat_cord = 0;
                color = 0;
                dos_flag = 0;
                for (count = 0l;count < total_vector_points && !feof (f) && !dos_labels; count++) {
                    if (strncmp ("DOS ", map_type, 4) == 0) {
                        (void)fgets (&Buffer[strlen (Buffer)],(int)sizeof (Buffer) - (strlen (Buffer)), f);
                        while ((ptr = strpbrk (Buffer, "\r\n")) != NULL && !dos_labels) {
                            long LatHld = 0, LongHld;
                            char *trailer;
                            *ptr = '\0';
                            for (ptr++; *ptr == '\r' || *ptr == '\n'; ptr++) ;

                            process:

                            if (strncasecmp ("Line", map_version, 4) == 0) {
                                int k;
                                color = (int)strtol (Buffer, &trailer, 0);
                                if (trailer && (*trailer == ',' || *trailer == ' ')) {
                                    trailer++;
                                    if (color == -1) {
                                        dos_labels = (int)TRUE;
                                        strcpy (Buffer, ptr);
                                        break;
                                    }
                                    for (k = strlen (trailer) - 1; k >= 0; k--)
                                        trailer[k] = (char)( (int)trailer[k] - 27 );

                                    while (*trailer) {
                                        LongHld = (long)( (int)(*(char *)trailer) * 16);
                                        trailer++;
                                        LatHld = (long)( (int)(*(char *)trailer) * 8);
                                        trailer++;
                                        LongHld += (long)((*(unsigned char*)trailer >> 3) & 0xf);
                                        LatHld += (long)( (*trailer) & 0x7);
                                        trailer++;
                                        LatHld = LatHld * (360000 / points_per_degree);
                                        LongHld = LongHld * (360000 / points_per_degree);
                                        x_long_cord = LongHld + left_boundary;
                                        y_lat_cord = LatHld + top_boundary;
                                        map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, (unsigned char)color, 0, destination_pixmap);
                                    }
                                    map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, '\0', 0, destination_pixmap);
                                }
                            } else if (strncasecmp ("ASCII", map_version, 4) == 0) {
                                if (color == 0) {
                                    color = (int)strtol (Buffer, &trailer, 0);
                                    if (trailer && strpbrk (trailer, ", ")) {
                                        for (; *trailer == ',' || *trailer == ' '; trailer++) ;
                                        dos_flag = (int)strtol (trailer, &trailer, 0);
                                        if (dos_flag == -1)
                                            dos_labels = (int)TRUE;
                                    }
                                } else {
                                    LongHld = strtol (Buffer, &trailer, 0);
                                    if (trailer && strpbrk (trailer, ", ")) {
                                        for (; *trailer == ',' || *trailer == ' '; trailer++) ;
                                        LatHld = strtol (trailer, &trailer, 0);
                                    } else if (LongHld == 0 && *trailer != '\0') {
                                        strncpy (map_version, "Comp", 4);
                                        goto process;
                                    }
                                    if (LongHld == 0 && LatHld == 0) {
                                        color = 0;
                                        map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, (unsigned char)color,0, destination_pixmap);
                                    } else if (LongHld == 0 && LatHld == -1) {
                                        dos_labels = (int)TRUE;
                                        map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, '\0', 0, destination_pixmap);
                                    } else {
                                        LatHld = LatHld * (360000 / points_per_degree);
                                        LongHld = LongHld * (360000 / points_per_degree);
                                        x_long_cord = LongHld + left_boundary;
                                        y_lat_cord = LatHld + top_boundary;
                                        map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, (unsigned char)color,0, destination_pixmap);
                                    }
                                }
                            } else if (strncasecmp ("Comp", map_version, 4) == 0) {
                                char Tag[81];
                                int k;
                                Tag[80] = '\0';
                                if (color == 0) {
                                    color = (int)strtol (Buffer, &trailer, 0);
                                    if (trailer && strpbrk (trailer, ", ")) {
                                        for (; *trailer == ',' || *trailer == ' '; trailer++) ;
                                        dos_flag = (int)strtol (trailer, &trailer, 0);
                                        strncpy (Tag, trailer, 80);
                                        if (dos_flag == -1)
                                            dos_labels = (int)TRUE;
                                    }
                                } else {
                                    LongHld = strtol (Buffer, &trailer, 0);
                                    for (; *trailer == ',' || *trailer == ' '; trailer++) ;
                                    LatHld = strtol (trailer, &trailer, 0);
                                    if (LatHld == 0 && *trailer != '\0')
                                        LatHld = 1;

                                    if (LongHld == 0 && LatHld == 0) {
                                        color = 0;
                                        map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, (unsigned char)color,0, destination_pixmap);
                                    } else if (LongHld == 0 && LatHld == -1) {
                                        dos_labels = (int)TRUE;
                                        map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, (unsigned char)color,0, destination_pixmap);
                                    }

                                    if (color && !dos_labels) {
                                        trailer = Buffer;
                                        for (k = strlen (trailer) - 1; k >= 0; k--)
                                            trailer[k] = (char)((int)trailer[k] - 27);

                                        while (*trailer) {
                                            LongHld = (long)( (int)(*(char *)trailer) * 16);
                                            trailer++;
                                            LatHld = (long)( (int)(*(char *)trailer) * 8);
                                            trailer++;
                                            LongHld += (long)((*(unsigned char *)trailer >> 3) & 0xf);
                                            LatHld += (*trailer) & 7l;
                                            trailer++;
                                            LatHld = LatHld * (360000 / points_per_degree);
                                            LongHld = LongHld * (360000 / points_per_degree);
                                            x_long_cord = LongHld + left_boundary;
                                            y_lat_cord = LatHld + top_boundary;
                                            map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, (unsigned char)color, 0, destination_pixmap);
                                        }
                                    }
                                }
                            } else {
                                LongHld = strtol (Buffer, &trailer, 0);
                                if (trailer) {
                                    if (*trailer == ',' || *trailer == ' ') {
                                        if (LongHld == 0)
                                            strncpy (map_version, "ASCII", 4);

                                        trailer++;
                                        dos_flag = (int)strtol (trailer, &trailer, 0);
                                        if (dos_flag == -1)
                                            dos_labels = (int)TRUE;

                                        if (dos_flag == 0 && *trailer != '\0') {
                                            strncpy (map_version, "Line", 4);
                                            goto process;
                                        }
                                        color = (int)LongHld;
                                    }
                                } else
                                    strncpy (map_version, "Comp", 4);
                            }
                            strcpy (Buffer, ptr);
                        }
                    } else {
                        last_behavior = object_behavior;
                        (void)fread (&vector_start, 1, 1, f);
                        (void)fread (&object_behavior, 1, 1, f);      // Fill Color?
                        if (strcmp (map_type, "COMP") == 0) {
                            short temp_short;
                            long LatOffset, LongOffset;
                            LatOffset = (long)(top_boundary - top_boundary % 6000);
                            LongOffset = (long)(left_boundary - left_boundary % 6000);
                            (void)fread (&temp_short, 2, 1, f);
                            x_long_cord = (ntohs (temp_short) * 10 + LongOffset);
                            (void)fread (&temp_short, 2, 1, f);
                            y_lat_cord = (ntohs (temp_short) * 10 + LatOffset);
                        } else {
                            (void)fread (&temp, 4, 1, f);
                            x_long_cord = ntohl (temp);
                            if (strcmp (map_version, "2.00") != 0)
                                x_long_cord *= 10;

                            (void)fread (&temp, 4, 1, f);
                            y_lat_cord = ntohl (temp);
                            if (strcmp (map_version, "2.00") != 0)
                                y_lat_cord *= 10;
                        }
                        if (alert_color && last_behavior & 0x80 && (int)vector_start == 0xff) {
                            map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, '\0',(long)alert_color, destination_pixmap);
                            //special_fill = TRUE;
                        }
                        map_plot (w, max_x, max_y, (long)x_long_cord, (long)y_lat_cord, vector_start,(long)object_behavior, destination_pixmap);
                    }
                }
                if (alert_color)
                    map_plot (w, max_x, max_y, 0, 0, '\0',special_fill ? (long)0xfd : (long)alert_color, destination_pixmap);
                else
                    map_plot (w, max_x, max_y, 0, 0, (unsigned char)0xff, 0, destination_pixmap);

                (void)XSetForeground (XtDisplay (w), gc, colors[20]);
                line_width = 2;
                (void)XSetLineAttributes (XtDisplay (w), gc, line_width, LineSolid, CapButt,JoinMiter);

                /* read labels */
                for (count = 0l; count < total_labels && !feof (f); count++) {
                    if (strcmp (map_type, "DOS ") == 0) {
                        char *trailer;
                        (void)fgets (&Buffer[strlen (Buffer)],(int)sizeof (Buffer) - (strlen (Buffer)), f);
                        for (; (ptr = strpbrk (Buffer, "\r\n")) != NULL;strcpy (Buffer, ptr)) {
                            *ptr = '\0';
                            label_type[0] = (char)0x08;
                            for (ptr++; *ptr == '\r' || *ptr == '\n'; ptr++) ;
                            trailer = strchr (Buffer, ',');
                            if (trailer && strncmp (Buffer, "0", 1) != 0) {
                                *trailer = '\0';
                                trailer++;
                                strcpy (label_text, Buffer);
                                label_length = (int)strlen (label_text);
                                label_y_cord = (unsigned long) (-strtod (trailer, &trailer) * 360000) + 32400000;
                                trailer++;
                                label_x_cord = (unsigned long) (-strtod (trailer, &trailer) * 360000) + 64800000;
                                trailer++;
                                label_mag = (int)strtol (trailer, &trailer, 0) * 20;
                                if ((label_type[0] & 0x80) == '\0') /* left of coords */
                                    x = ((label_x_cord - x_long_offset) / size) - (label_length * 6);
                                else  /* right of coords */
                                    x = ((label_x_cord - x_long_offset) / size);

                                y = ((label_y_cord - y_lat_offset) / size);
                                if (x > (0) && (x < (int)screen_width)) {
                                    if (y > (0) && (y < (int)screen_height)) {
                                        /*printf("Label mag %d mag %d\n",label_mag,(size*2)-1); */
                                        if (label_mag > (int)((size * 2) - 1) || label_mag == 0)
                                            draw_label_text (w, x, y, label_length,colors[(int)(label_type[0] & 0x7f)],label_text);
                                    }
                                }
                            }
                        }
                    } else {
                        (void)fread (label_type, 1, 2, f);
                        if (label_type[1] == '\0') {               /* label is text */
                            /* label is text */
                            (void)fread (&temp, 4, 1, f);             /* x */
                            label_x_cord = ntohl (temp);
                            if (strcmp (map_version, "2.00") != 0)
                                label_x_cord *= 10;

                            (void)fread (&temp, 4, 1, f);             /* y */
                            label_y_cord = ntohl (temp);
                            if (strcmp (map_version, "2.00") != 0)
                                label_y_cord *= 10;

                            (void)fread (&temp_mag, 2, 1, f);         /* mag */
                            label_mag = (int)ntohs (temp_mag);
                            if (strcmp (map_version, "2.00") != 0)
                                label_mag *= 10;

                            if (strcmp (map_type, "COMP") == 0)
                                for (i = 0; i < 32; i++) {
                                    (void)fread (&label_text[i], 1, 1, f);
                                    if (label_text[i] == '\0')
                                        break;
                                }
                            else
                                (void)fread (label_text, 32, 1, f);   /* text */

                            label_length = (int)strlen (label_text);

                            for (i = (label_length - 1); i > 0; i--) {
                                if (label_text[i] == ' ')
                                    label_text[i] = '\0';
                                else
                                    break;
                            }

                            label_length = (int)strlen (label_text);
                            /*printf("labelin:%s\n",label_text); */

                            if ((label_type[0] & 0x80) == '\0') {
                                /* left of coords */
                                x = ((label_x_cord - x_long_offset) / size) - (label_length * 6);
                                x = 0;
                            } else {
                                /* right of coords */
                                x = ((label_x_cord - x_long_offset) / size);
                            }

                            y = ((label_y_cord - y_lat_offset) / size);

                            if (x > (0) && (x < (int)screen_width)) {
                                if (y > (0) && (y < (int)screen_height)) {
                                    /*printf("Label mag %d mag %d\n",label_mag,(size*2)-1); */
                                    if (label_mag > (int)((size * 2) - 1) || label_mag == 0)
                                        draw_label_text (w, x, y, label_length,colors[(int)(label_type[0] & 0x7f)],label_text);
                                }
                            }
                        } else {
                            /* label is symbol */
                            (void)fread (&temp, 4, 1, f);
                            label_x_cord = ntohl (temp);
                            if (strcmp (map_version, "2.00") != 0)
                                label_x_cord *= 10;

                            (void)fread (&temp, 4, 1, f);
                            label_y_cord = ntohl (temp);
                            if (strcmp (map_version, "2.00") != 0)
                                label_y_cord *= 10;

                            (void)fread (&temp_mag, 2, 1, f);
                            label_mag = (int)ntohs (temp_mag);
                            if (strcmp (map_version, "2.00") != 0)
                                label_mag *= 10;

                            (void)fread (&label_symbol_del, 1, 1, f);
                            (void)fread (&label_symbol_char, 1, 1, f);
                            (void)fread (&label_text_color, 1, 1, f);
                            if (strcmp (map_type, "COMP") == 0)
                                for (i = 0; i < 32; i++) {
                                    (void)fread (&label_text[i], 1, 1, f);
                                    if (label_text[i] == '\0')
                                        break;
                                  }
                            else
                                (void)fread (label_text, 29, 1, f);

                            /* Do object */
                        }
                    }
                }
            }
            (void)fclose (f);
        }
        else
            printf("Couldn't open file: %s\n", file);
    } else {
        /* this is another type, XPM imagemap files */
        if (ext != NULL && strcasecmp (ext, "geo") == 0) {
            /* this is a geo image map */
            draw_geo_image_map (w, dir, filenm);
        }

#ifdef USE_GEOTIFF
        /* and yet another type, USGS DRG geoTIFF files */
        if (ext != NULL && strcasecmp (ext, "tif") == 0)
        {
            /* This should be a geoTIFF map */
            draw_geotiff_image_map (w, dir, filenm);
        }
#endif /* USE_GEOTIFF */

    }
    XmUpdateDisplay (XtParent (da));
}





/**********************************************************
 * map_search()
 *
 * Function which recurses through map directories, finding
 * map files.
 **********************************************************/
void map_search (Widget w, char *dir, alert_entry * alert, int *alert_count,int warn, int destination_pixmap) {
    struct dirent *dl;
    DIR *dm;
    char fullpath[8000];
    struct stat nfile;
    const time_t *ftime;
    char this_time[40];
    char *ptr;
    int line_width;
    char *map_dir;
    int map_dir_length;

    line_width = 1;

    /* Set the line width in the GC */
    (void)XSetLineAttributes (XtDisplay (w), gc, line_width, LineSolid, CapButt,JoinMiter);

    map_dir = alert ? ALERT_MAP_DIR : WIN_MAP_DIR;
    map_dir_length = (int)strlen (map_dir);

    dm = opendir (dir);
    if (!dm) {
        sprintf (fullpath, "aprsmap %s", dir);
        if (warn)
            perror (fullpath);
    } else {
        int count = 0;
        while ((dl = readdir (dm))) {
            sprintf (fullpath, "%s/%s", dir, dl->d_name);
            /*printf("FULL PATH %s\n",fullpath); */
            if (stat (fullpath, &nfile) == 0) {
                ftime = (time_t *)&nfile.st_ctime;
                switch (nfile.st_mode & S_IFMT) {
                    case (S_IFDIR):
                        /*printf("file %c letter %c\n",dl->d_name[0],letter); */
                        if ((strcmp (dl->d_name, ".") != 0) && (strcmp (dl->d_name, "..") != 0)) {
                            strcpy (this_time, ctime (ftime));
                            map_search(w, fullpath, alert, alert_count, warn, destination_pixmap);
                        }
                        break;
                    case (S_IFREG):
                        /*printf("FILE %s\n",dl->d_name); */
                        if (strncmp (fullpath, map_dir, (size_t)map_dir_length) != 0) {
                            draw_map (w, dir, dl->d_name, 1,alert ? &alert[*alert_count] : NULL, '\0', destination_pixmap);
                            if (alert_count && *alert_count)
                                (*alert_count)--;
                        } else {
                            for (ptr = &fullpath[map_dir_length]; *ptr == '/'; ptr++) ;
                            draw_map (w, map_dir, ptr, 1,alert ? &alert[*alert_count] : NULL, '\0', destination_pixmap);
                            if (alert_count && *alert_count)
                                (*alert_count)--;
                        }
                        count++;
                        break;

                    default:
                        break;
                }
            }
        }
        if (debug_level & 8)
            printf ("Number of maps queried: %d\n", count);

        (void)closedir (dm);
    }
}





/* moved these here and made them static so it will function on FREEBSD */
#define MAX_ALERT 7000
static alert_entry alert[MAX_ALERT];
static int alert_count;





/**********************************************************
 * load_alert_maps()
 *
 * Used to load weather alert maps, based on NWS weather
 * alerts that are received.
 **********************************************************/
void load_alert_maps (Widget w, char *dir) {
    int i, level;
    char alert_scan[400], *dir_ptr;

    /* gray86, red2, yellow2, cyan2, RoyalBlue, ForestGreen, orange3 */
    unsigned char fill_color[] = { (unsigned char)0x69, (unsigned char)0x4a, (unsigned char)0x63, (unsigned char)0x66, (unsigned char)0x61, (unsigned char)0x64, (unsigned char)0x62 };

    alert_count = MAX_ALERT - 1;

    if (alert_message_scan ()) {
        memset (alert_scan, 0, sizeof (alert_scan));
        strncpy (alert_scan, dir, 390);
        strcat (alert_scan, "/");
        dir_ptr = &alert_scan[strlen (alert_scan)];
        for (i = 0; i < (int)strlen (alert_tag); i += 3) {
            *dir_ptr = alert_tag[i];
            *(dir_ptr + 1) = alert_tag[i + 1];

            // The last parameter denotes loading into pixmap_alerts instead of pixmap or pixmap_final
            map_search (w, alert_scan, alert, &alert_count,(int)(alert_tag[i + 2] == DATA_VIA_TNC || alert_tag[i + 2] == DATA_VIA_LOCAL), DRAW_TO_PIXMAP_ALERTS);
        }
    }
    if (!alert_count)
        XtAppWarning (app_context, "Alert Map count overflow: load_alert_maps\b\n");

    for (i = MAX_ALERT - 1; i > alert_count; i--)
        if (alert[i].flags[0] == 'Y' || alert[i].flags[0] == 'N')
            alert_update_list (&alert[i], ALERT_TITLE);

    for (i = 0; i < alert_list_count; i++)
        if (alert_list[i].filename[0]) {
            alert[0] = alert_list[i];

            // The last parameter denotes drawing into pixmap_final instead of pixmap or pixmap_final
            draw_map (w, dir, alert_list[i].filename, 1, &alert[0], '\0', DRAW_TO_PIXMAP_ALERTS);

            alert_update_list (&alert[0], ALERT_ALL);
        }

    alert_sort_active ();
    for (i = alert_list_count - 1; i >= 0; i--)
        if (alert_list[i].flags[0] == 'Y' && (level = alert_active (&alert_list[i], ALERT_ALL))) {
            if (level >= (int)sizeof (fill_color))
                level = 0;

            // The last parameter denotes drawing into pixmap_final instead of pixmap or pixmap_final
            draw_map (w, dir, alert_list[i].filename, 0, NULL, fill_color[level], DRAW_TO_PIXMAP_ALERTS);
        }

    (void)alert_display_request ();
}





/**********************************************************
 * load_auto_maps()
 *
 * Recurses through the map directories looking for maps
 * to load.
 **********************************************************/
void load_auto_maps (Widget w, char *dir) {
    map_search (w, dir, NULL, NULL, (int)TRUE, DRAW_TO_PIXMAP);
}





/**********************************************************
 * load_maps()
 *
 * Loads maps, draws grid, updates the display.
 **********************************************************/
void load_maps (Widget w) {
    FILE *f;
    char mapname[300];

    if (debug_level & 1)
        printf ("Load maps start\n");

    (void)filecreate(WIN_MAP_DATA);   // Create empty file if it doesn't exist
    f = fopen (WIN_MAP_DATA, "r");
    if (f != NULL) {
        if (debug_level & 1)
            printf ("Load maps Open map file\n");

        while (!feof (f)) {
            if (fscanf (f, "%299s", mapname) == 1) {
                if (mapname[0] != '#') {
                    draw_map (w, WIN_MAP_DIR, mapname, 0, NULL, '\0', DRAW_TO_PIXMAP);
                    if (debug_level & 1)
                        printf ("Load maps -%s\n", mapname);

                    XmUpdateDisplay (da);
                }
            }
        }
        (void)fclose (f);
        statusline(" ",1);      // delete status line
    }
    else
        printf("Couldn't open file: %s\n", WIN_MAP_DATA);

    if (debug_level & 1)
        printf ("Load maps stop\n");

}

